within PropulsionSystem.Elements.BasicElements;

model FlightToEngine
  import Modelica.Constants;
  /********************************************************
               Declaration
            ********************************************************/
  //********** Package **********
  replaceable package Medium = Modelica.Media.Interfaces.PartialMedium annotation(
    choicesAllMatching = true);
  //********** Parameters **********
  parameter Modelica.SIunits.Length alt = 0.0 "altitude" annotation(
    Evaluate = true,
    Dialog(enable = not use_alt_in));
  parameter Real MN = 0.0 "flight mach number" annotation(
    Evaluate = true,
    Dialog(enable = not use_MN_in));
  parameter Modelica.SIunits.TemperatureDifference dTamb = 0.0 "deviation from std atmospheric temperature" annotation(
    Evaluate = true,
    Dialog(enable = not use_dTamb_in));
  parameter Real relHum = 0.0 "relative humidity" annotation(
    Evaluate = true,
    Dialog(enable = not use_dTamb_in));
  parameter Modelica.SIunits.Angle AoA = 0.0 "" annotation(
    Evaluate = true,
    Dialog(enable = not use_AOA_in));
  parameter Modelica.SIunits.Angle sideSlip = 0.0 "" annotation(
    Evaluate = true,
    Dialog(enable = not use_sideSlip_in));
  parameter Integer n_fluidAmb = 1 "" annotation(
    Dialog(group = "misc setting"));
  //----- switches -----
  parameter Boolean allowFlowReversal = true "= true to allow flow reversal, false restricts to design direction (port_a -> port_b)" annotation(
    Dialog(tab = "Assumptions"),
    Evaluate = true);
  parameter PropulsionSystem.Types.switches.switch_parameter_input switchInput_alt = PropulsionSystem.Types.switches.switch_parameter_input.use_desValue "" annotation(
    Dialog(group = "switch"),
    choicesAllMatching = true,
    Evaluate = true,
    HideResult = true);
  parameter PropulsionSystem.Types.switches.switch_parameter_input switchInput_MN = PropulsionSystem.Types.switches.switch_parameter_input.use_desValue "" annotation(
    Dialog(group = "switch"),
    choicesAllMatching = true,
    Evaluate = true,
    HideResult = true);
  parameter PropulsionSystem.Types.switches.switch_parameter_input switchInput_dTamb = PropulsionSystem.Types.switches.switch_parameter_input.use_desValue "" annotation(
    Dialog(group = "switch"),
    choicesAllMatching = true,
    Evaluate = true,
    HideResult = true);
  parameter PropulsionSystem.Types.switches.switch_parameter_input switchInput_AoA = PropulsionSystem.Types.switches.switch_parameter_input.use_desValue "" annotation(
    Dialog(group = "switch"),
    choicesAllMatching = true,
    Evaluate = true,
    HideResult = true);
  parameter PropulsionSystem.Types.switches.switch_parameter_input switchInput_sideSlip = PropulsionSystem.Types.switches.switch_parameter_input.use_desValue "" annotation(
    Dialog(group = "switch"),
    choicesAllMatching = true,
    Evaluate = true,
    HideResult = true);
  parameter PropulsionSystem.Types.switches.switch_parameter_input switchInput_relHum = PropulsionSystem.Types.switches.switch_parameter_input.use_desValue "" annotation(
    Dialog(group = "switch"),
    choicesAllMatching = true,
    Evaluate = true,
    HideResult = true);
  //********** Internal variables **********
  //***** constants *****
  constant Modelica.SIunits.Acceleration gAccel = 9.81;
  // gound
  constant Modelica.SIunits.Length alt_ground = 0.0;
  constant Modelica.SIunits.Temperature T_ground = 288.16;
  constant Modelica.SIunits.AbsolutePressure p_ground = 101323.7;
  // upper bound of troposhere
  constant Modelica.SIunits.Length alt_UpBdTropos = 11019.1;
  constant Modelica.SIunits.Temperature T_UpBdTropos = 216.67;
  constant Modelica.SIunits.AbsolutePressure p_UpBdTropos = 22632.9;
  constant Real LapseR1 = -0.0064878;
  // upper bound of stratosphere
  constant Modelica.SIunits.Length alt_UpBdStratos = 25099.1;
  constant Modelica.SIunits.Temperature T_UpBdStratos = 216.67;
  constant Modelica.SIunits.AbsolutePressure p_UpBdStratos = 2486.4;
  constant Real LapseR2 = 0.0;
  //*****
  //
  Modelica.SIunits.AbsolutePressure pAmb;
  Modelica.SIunits.Temperature Tamb;
  Real Xamb[Medium.nX] = Medium.X_default;
  Modelica.SIunits.AbsolutePressure pAmbStd;
  Modelica.SIunits.Temperature TambStd;
  Real XambStd[Medium.nX] = Medium.X_default;
  // fluid stations
  Medium.BaseProperties fluid_tot "flow station, total";
  Medium.BaseProperties fluid_ambStd;
  Medium.BaseProperties fluid_amb;
  Modelica.SIunits.Velocity V_tot(start = 100) "";
  Modelica.SIunits.Force Fram "ram drag";
  Modelica.Fluid.Sources.Boundary_pT fluid2Eng(redeclare package Medium = Medium, nPorts = 1, use_C_in = false, use_T_in = true, use_X_in = true, use_p_in = true) annotation(
    Placement(visible = true, transformation(origin = {80, 50}, extent = {{-10, -10}, {10, 10}}, rotation = 90)));
  Modelica.Fluid.Sources.Boundary_pT fluidAmb(redeclare package Medium = Medium, nPorts = 1, use_C_in = false, use_T_in = true, use_X_in = true, use_p_in = true) annotation(
    Placement(visible = true, transformation(origin = {0, 50}, extent = {{-10, -10}, {10, 10}}, rotation = 90)));
  //********** Interfaces **********
  Modelica.Blocks.Interfaces.RealInput alt_in if switchInput_alt == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal annotation(
    Placement(visible = true, transformation(origin = {-120, 80}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {-110, 80}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Modelica.Blocks.Interfaces.RealInput MN_in if switchInput_MN == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal annotation(
    Placement(visible = true, transformation(origin = {-120, 50}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {-110, 50}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Modelica.Blocks.Interfaces.RealInput dTamb_in if switchInput_dTamb == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal annotation(
    Placement(visible = true, transformation(origin = {-120, 20}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {-110, 20}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Modelica.Blocks.Interfaces.RealInput AoA_in if switchInput_AoA == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal annotation(
    Placement(visible = true, transformation(origin = {-120, -20}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {-110, -20}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Modelica.Blocks.Interfaces.RealInput sideSlip_in if switchInput_sideSlip == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal annotation(
    Placement(visible = true, transformation(origin = {-120, -50}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {-110, -50}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Modelica.Blocks.Interfaces.RealInput relHum_in if switchInput_relHum == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal annotation(
    Placement(visible = true, transformation(origin = {-120, -80}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {-110, -80}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Modelica.Fluid.Interfaces.FluidPort_b port_fluid2Eng(redeclare package Medium = Medium, m_flow(min = if allowFlowReversal then -Constants.inf else 0.0, start = m_flow_start)) annotation(
    Placement(visible = true, transformation(origin = {100, 80}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {100, 80}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Modelica.Fluid.Interfaces.FluidPort_b port_fluidAmb(redeclare package Medium = Medium, m_flow(min = if allowFlowReversal then -Constants.inf else 0.0, start = m_flow_start)) annotation(
    Placement(visible = true, transformation(origin = {0, 100}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {0, 100}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Modelica.Blocks.Interfaces.RealOutput Fram_out annotation(
    Placement(visible = true, transformation(origin = {30, -90}, extent = {{-10, -10}, {10, 10}}, rotation = -90), iconTransformation(origin = {30, -110}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
  Modelica.Blocks.Interfaces.RealOutput V_tot_out annotation(
    Placement(visible = true, transformation(origin = {100, -40}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {110, -40}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  PropulsionSystem.Types.ElementBus elementBus1 annotation(
    Placement(visible = true, transformation(origin = {70, -90}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {70, -90}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  //********** Initialization **********
  parameter Medium.AbsolutePressure p_a_start = 101.3 * 1000 "Start value of pressure at port a" annotation(
    Dialog(tab = "Initialization"));
  parameter Medium.AbsolutePressure p_b_start = 101.3 * 1000 * 0.99 "Start value of pressure at port b" annotation(
    Dialog(tab = "Initialization"));
  parameter Medium.MassFlowRate m_flow_start = 10.0 "Start value for mass flow rate" annotation(
    Evaluate = true,
    Dialog(tab = "Initialization"));
protected
  //********** Protected objects **********
  //##### none #####
algorithm
//##### none #####
equation
  connect(fluid2Eng.ports[1], port_fluid2Eng) annotation(
    Line(points = {{80, 60}, {80, 71}, {102, 71}, {102, 80}}, color = {0, 127, 255}));
  connect(fluid_tot.X, fluid2Eng.X_in) annotation(
    Line);
  connect(fluid_tot.T, fluid2Eng.T_in) annotation(
    Line);
  connect(fluid_tot.p, fluid2Eng.p_in) annotation(
    Line);
//-- fluid_Amb --
  connect(fluidAmb.ports[1], port_fluidAmb) annotation(
    Line(points = {{0, 60}, {0, 60}, {0, 100}, {0, 100}}, color = {0, 127, 255}, thickness = 0.5));
  connect(fluid_amb.p, fluidAmb.p_in) annotation(
    Line);
  connect(fluid_amb.T, fluidAmb.T_in) annotation(
    Line);
  connect(fluid_amb.X, fluidAmb.X_in) annotation(
    Line);
//-- fluid_tot --
//********** Geometries **********
//##### none #####
//********** Connections, interface <-> internal variables **********
//-- alt --
  if switchInput_alt == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal then
    alt = alt_in;
  end if;
//-- MN --
  if switchInput_MN == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal then
    MN = MN_in;
  end if;
//-- dTamb --
  if switchInput_dTamb == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal then
    dTamb = dTamb_in;
  end if;
//-- relHum --
  if switchInput_relHum == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal then
    relHum = relHum_in;
  end if;
//-- AoA --
  if switchInput_AoA == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal then
    AoA = AoA_in;
  end if;
//-- sieslip --
  if switchInput_sideSlip == PropulsionSystem.Types.switches.switch_parameter_input.use_inputSignal then
    sideSlip = sideSlip_in;
  end if;
/*
  for i in 1:n_fluidAmb loop
    assert
    (cardinality(ports_fluidAmb[i]) <= 1,"
    each ports[i] of boundary shall at most be connected to one component.
    If two or more connections are present, ideal mixing takes
    place with these connections, which is usually not the intention
    of the modeller. Increase nPorts to add an additional port.
    ");
    
    ports_fluidAmb[i].p          = fluid_amb.p;
    ports_fluidAmb[i].h_outflow  = fluid_amb.h;
    ports_fluidAmb[i].Xi_outflow = fluid_amb.Xi;
  end for;
  */
/*
  ports_fluidAmb.p          = fluid_amb.p;
  ports_fluidAmb.h_outflow  = fluid_amb.h;
  ports_fluidAmb.Xi_outflow = fluid_amb.Xi;
  */
//***** output signal *****
  Fram_out = Fram;
  V_tot_out = V_tot;
//********** Eqns describing physics **********
//-- regression curve of atmospheric table --
// curves of T, P
  if alt_ground <= alt and alt < alt_UpBdTropos then
// troposphere, temperature gradient layer model
    TambStd = T_ground - 0.0064878 * (alt - alt_ground);
    Tamb = TambStd + dTamb;
    pAmbStd = p_ground * (TambStd / T_ground) ^ (-1.0 * gAccel / (LapseR1 * fluid_ambStd.R));
    pAmb = p_ground * (Tamb / T_ground) ^ (-1.0 * gAccel / (LapseR1 * fluid_amb.R));
  elseif alt_UpBdTropos <= alt and alt < alt_UpBdStratos then
// stratosphere, temperature is constant
    TambStd = T_UpBdTropos;
    Tamb = TambStd + dTamb;
    pAmbStd = p_UpBdTropos * exp(-1.0 * gAccel / (fluid_ambStd.R * TambStd) * (alt - T_UpBdTropos));
    pAmb = p_UpBdTropos * exp(-1.0 * gAccel / (fluid_amb.R * Tamb) * (alt - T_UpBdTropos));
  end if;
// set fluid station states
  fluid_ambStd.state = Medium.setState_pTX(pAmbStd, TambStd, XambStd);
  fluid_amb.state = Medium.setState_pTX(pAmb, Tamb, Xamb);
//-- figure out velocity --
  V_tot = MN * Medium.velocityOfSound(fluid_amb.state);
//-- velocity to total pressure --
  fluid_tot.h = fluid_amb.h + V_tot ^ 2.0 / 2.0;
  fluid_tot.h = Medium.isentropicEnthalpy(fluid_tot.p, fluid_amb.state);
//-- ram drag --
  Fram = (-1.0) * port_fluid2Eng.m_flow * V_tot;
/********************************************************
  Graphics
********************************************************/
  annotation(
    Icon(coordinateSystem(initialScale = 0.1), graphics = {Rectangle(origin = {-79, 61}, fillColor = {229, 229, 229}, pattern = LinePattern.None, fillPattern = FillPattern.Solid, extent = {{-21, 39}, {179, -161}}), Bitmap(origin = {-17, 21}, extent = {{-83, 59}, {117, -101}}, imageSource = "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"),
    
    
    Rectangle(lineColor = {255, 0, 0}, fillColor = {255, 0, 0}, lineThickness = 4, extent = {{-100, 100}, {100, -100}}), Line(points = {{-100, 100}, {100, -100}}, color = {255, 0, 0}, thickness = 4)
    
    }));
end FlightToEngine;
