﻿within ClaRa.SubSystems.Furnace;
model ConvectiveSlice_L4 "Convective furnaces slice"

//__________________________________________________________________________//
// Component of the ClaRa library, version: 1.9.0                           //
//                                                                          //
// Licensed by the ClaRa development team under the 3-clause BSD License.   //
// Copyright  2013-2024, ClaRa development team.                            //
//                                                                          //
// The ClaRa development team consists of the following partners:           //
// TLK-Thermo GmbH (Braunschweig, Germany),                                 //
// XRG Simulation GmbH (Hamburg, Germany).                                  //
//__________________________________________________________________________//
// Contents published in ClaRa have been contributed by different authors   //
// and institutions. Please see model documentation for detailed information//
// on original authorship and copyrights.                                   //
//__________________________________________________________________________//

  extends ClaRa.Basics.Icons.FurnaceSlice;
  import rad2deg = Modelica.Units.Conversions.from_deg;
  outer ClaRa.SimCenter simCenter;

  //Furnace
  parameter ClaRa.Basics.Media.FuelTypes.BaseFuel fuelModel=simCenter.fuelModel1 "Fuel definition" annotation (choicesAllMatching, Dialog(group="Media Definitions"));
  parameter TILMedia.Gas.Types.BaseGas flueGas=simCenter.flueGasModel "Flue gas model" annotation (choicesAllMatching,
      Dialog(group="Media Definitions", groupImage=
          "modelica://ClaRa/Resources/Images/ParameterDialog/FurnaceSketch.png"));
  parameter ClaRa.Basics.Media.Slag.PartialSlag slagType=simCenter.slagModel "Slag properties" annotation (choices(choice=simCenter.slagModel "Slag model 1 as defined in simCenter"),Dialog(group="Media Definitions"));

  replaceable model GasHeatTransfer_TubeBundle =
      ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.Gas_HT.Convection.Convection_tubeBank_L2                                            constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.HeatTransferBaseGas "HT from Gas to TB" annotation (choicesAllMatching, Dialog(group="Fundamental Definitions"));
  replaceable model GasHeatTransfer_Wall = ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.Gas_HT.Convection.Convection_flatWall_L2 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.HeatTransferBaseGas "HT from Gas to FTW" annotation (choicesAllMatching, Dialog(group="Fundamental Definitions"));
  replaceable model GasHeatTransfer_CarrierTubes =
      ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.Gas_HT.Convection.Convection_carrierTubes_turbulent_L2                                              constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.HeatTransferBaseGas "HT from Gas to top" annotation (choicesAllMatching, Dialog(group="Fundamental Definitions"));

  replaceable model GasHeatTransfer_Top = ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.Generic_HT.Adiabat_L2 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.HeatTransferBaseGas "HT from Gas to CT" annotation (choicesAllMatching, Dialog(group="Fundamental Definitions"));

  replaceable model GasPressureLoss = ClaRa.Basics.ControlVolumes.Fundamentals.PressureLoss.Generic_PL.LinearPressureLoss_L2 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.PressureLoss.PressureLossBaseGas_L2 "Gas pressure loss" annotation (choicesAllMatching, Dialog(group="Fundamental Definitions"));

  replaceable model Burning_time = ClaRa.Components.Furnace.GeneralTransportPhenomena.BurningTime.ConstantBurningTime
    constrainedby ClaRa.Components.Furnace.GeneralTransportPhenomena.BurningTime.PartialBurningTime "Model for the buring time"
                                annotation (Dialog(tab="Combustion Settings",group="Combustion"),choicesAllMatching=true);
  replaceable model ParticleMigration = ClaRa.Components.Furnace.GeneralTransportPhenomena.ParticleMigration.FixedMigrationSpeed_simple (w_fixed=10)
    constrainedby ClaRa.Components.Furnace.GeneralTransportPhenomena.ParticleMigration.PartialMigrationSpeed "Model for the particle migration speed"
                                             annotation (Dialog(tab="Combustion Settings",group="Combustion"), choicesAllMatching=true);
  replaceable model ReactionZone = ClaRa.Components.Furnace.ChemicalReactions.CoalReactionZone
     constrainedby ClaRa.Components.Furnace.ChemicalReactions.PartialReactionZone "Model to regard chemical reactions"
                                          annotation (Dialog(tab="Combustion Settings",group="Combustion"), choicesAllMatching=true);

  parameter ClaRa.Basics.Units.Length length_furnace=10 "Length of the component" annotation (Dialog(group="Geometry", groupImage="modelica://ClaRa/Resources/Images/ParameterDialog/HollowBlock.png"));
  parameter ClaRa.Basics.Units.Length width_furnace=length_furnace "Width of the component" annotation (Dialog(group="Geometry"));
  parameter ClaRa.Basics.Units.Length z_in_furnace=0 "Height of inlet ports" annotation (Dialog(tab="General", group="Geometry"));
  parameter ClaRa.Basics.Units.Length z_out_furnace=1 "Height of outlet ports" annotation (Dialog(tab="General", group="Geometry"));
  parameter ClaRa.Basics.Choices.GeometryOrientation flowOrientation_furnace=ClaRa.Basics.Choices.GeometryOrientation.vertical "Orientation of shell side flow" annotation (Dialog(group="Geometry"));

  parameter ClaRa.Basics.Units.Pressure p_start_flueGas_out=1e5 "Start pressure at outlet" annotation (Dialog(tab="Initialisation"));
  parameter ClaRa.Basics.Units.Temperature T_start_flueGas_out=700 "Start temperature at outlet" annotation (Dialog(tab="Initialisation"));
  parameter Modelica.Units.SI.Temperature T_top_initial=flameRoom.T_start_flueGas_out "Initial temperature of top volume" annotation (Dialog(tab="Initialisation"));
  parameter ClaRa.Basics.Units.MassFraction xi_start_flueGas_out[flueGas.nc - 1]=flueGas.xi_default "Start composition of flue gas" annotation (Dialog(tab="Initialisation"));

  parameter Modelica.Units.SI.MassFlowRate m_flow_nom_gas=1000 "Nominal mass flow rates at inlet" annotation (Dialog(group="Nominal Values"));

  parameter Integer slagTemperature_calculationType=1 "Calculation type of outflowing slag temperature" annotation (Dialog(tab="Combustion Settings", group="Slag Definitions"), choices(
      choice=1 "Fixed slag temperature",
      choice=2 "Outlet flue gas temperature",
      choice=3 "Mean flue gas temperature",
      choice=4 "Inlet flue gas temperature"));
  parameter ClaRa.Basics.Units.Temperature T_slag=900 "Constant slag outlet temperature" annotation (Dialog(tab="Combustion Settings", group="Slag Definitions"));

// Finned Tube Walls (FTW)///
  parameter Boolean FTWisCooled = true "True if finned walls are cooled (else omitted)" annotation (choices(checkBox=true), Dialog(tab="Finned Tube Walls (FTW)", group="Fundamental Definitions"));
  parameter TILMedia.VLEFluid.Types.BaseVLEFluid medium_FTW=simCenter.fluid1 "FTW medium model" annotation (
      choicesAllMatching, Dialog(
      enable=FTWisCooled,
      tab="Finned Tube Walls (FTW)",
      group="Fundamental Definitions",
      groupImage="modelica://ClaRa/Resources/Images/ParameterDialog/FinnedWallSketch.png"));
  replaceable model Material_FTW = Basics.Media.Solids.Steel16Mo3 constrainedby TILMedia.Solid.Types.BaseSolid "FTW material" annotation (choicesAllMatching, Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Fundamental Definitions"));
  parameter Boolean frictionAtInlet_FTW = false "True if pressure loss at inlet" annotation (choices(checkBox=true), Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Fundamental Definitions"));
  parameter Boolean frictionAtOutlet_FTW = false "True if pressure loss at outlet" annotation (choices(checkBox=true), Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Fundamental Definitions"));
  replaceable model PressureLoss_FTW = ClaRa.Basics.ControlVolumes.Fundamentals.PressureLoss.Generic_PL.LinearPressureLoss_L4 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.PressureLoss.PressureLossBaseVLE_L4 "Pressure loss model" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Fundamental Definitions"), choicesAllMatching=true);
  replaceable model HeatTransfer_FTW = ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.Generic_HT.CharLine_L4 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.HeatTransferBaseVLE_L4 "Heat Transfer Model" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Fundamental Definitions"), choicesAllMatching=true);
  replaceable model MechanicalEquilibrium_FTW = ClaRa.Basics.ControlVolumes.Fundamentals.SpacialDistribution.Homogeneous_L4 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.SpacialDistribution.MechanicalEquilibrium_L4 "Mechanic flow model" annotation (Dialog(tab="Finned Tube Walls (FTW)", group="Fundamental Definitions"), choicesAllMatching=true);
  parameter Modelica.Units.NonSI.Angle_deg psi_FTW=23 "Inclination of pipes" annotation (Dialog(
      enable=FTWisCooled,
      tab="Finned Tube Walls (FTW)",
      group="Geometry"));
  parameter ClaRa.Basics.Units.Length length_FTW=abs(z_out_furnace - z_in_furnace)/sin(rad2deg(psi_FTW)) "Length of the pipe" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length diameter_o_FTW=57e-3 "Outer diameter" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length diameter_i_FTW=49e-3 "Inner diameter of the pipe" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length z_in_FTW = z_in_furnace "Height of inlet above ground" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group = "Geometry"));
  parameter ClaRa.Basics.Units.Length z_out_FTW = z_out_furnace "Height of outlet above ground" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group = "Geometry"));
  parameter Integer N_tubes_FTW=1050 "Number Of parallel pipes" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Geometry"));
  final parameter Integer N_passes_FTW=1 "Number of passes of the tubes" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Geometry"));
  parameter Integer N_cv_FTW=3 "Number of finite volumes" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)",group="Discretisation"));
  parameter ClaRa.Basics.Units.Length Delta_x_FTW[N_cv_FTW]=ClaRa.Basics.Functions.GenerateGrid(
      {0},
      length_FTW,
      N_cv_FTW) "Discretisation scheme" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)",group="Discretisation"));
  parameter Integer initOption_FTW=0 "Type of initialisation" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Initialisation"), choices(choice = 0 "Use guess values", choice = 208 "Steady pressure and enthalpy", choice=201 "Steady pressure", choice = 202 "Steady enthalpy"));
  parameter ClaRa.Basics.Units.EnthalpyMassSpecific h_start_FTW[N_cv_FTW]=ones(N_cv_FTW)*800e3 "Initial specific enthalpy for single tube" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Initialisation"));
  parameter ClaRa.Basics.Units.Pressure p_start_FTW[N_cv_FTW]=ones(N_cv_FTW)*1e5 "Initial pressure" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Initialisation"));
  parameter ClaRa.Basics.Units.MassFraction xi_start_FTW[medium_FTW.nc - 1]=zeros(medium_FTW.nc - 1) "Initial composition" annotation (Dialog(enable = FTWisCooled, tab="Finned Tube Walls (FTW)", group="Initialisation"));
  parameter ClaRa.Basics.Units.EnthalpyMassSpecific h_nom_FTW[N_cv_FTW]=ones(N_cv_FTW)*1e5 "Nominal specific enthalpy for single tube" annotation (Dialog(tab="Finned Tube Walls (FTW)", group="Nominal Values"));
  parameter ClaRa.Basics.Units.Pressure p_nom_FTW[N_cv_FTW]=ones(N_cv_FTW)*1e5 "Nominal pressure" annotation (Dialog(tab="Finned Tube Walls (FTW)", group="Nominal Values"));
  parameter ClaRa.Basics.Units.MassFlowRate m_flow_nom_FTW=100 "Nominal mass flow w.r.t. all parallel tubes" annotation (Dialog(tab="Finned Tube Walls (FTW)", group="Nominal Values"));
  parameter ClaRa.Basics.Units.PressureDifference Delta_p_nom_FTW=1e4 "Nominal pressure loss w.r.t. all parallel tubes" annotation (Dialog(tab="Finned Tube Walls (FTW)", group="Nominal Values"));

/// TUbeBundle///
  parameter TILMedia.VLEFluid.Types.BaseVLEFluid medium_TB=simCenter.fluid1 "TB medium model" annotation (
      choicesAllMatching, Dialog(
      tab="Tube Bundle (TB)",
      group="Fundamental Definitions",
      groupImage="modelica://ClaRa/Resources/Images/ParameterDialog/TubeBundleSketch.png"));
  replaceable model Material_TB = Basics.Media.Solids.Steel16Mo3 constrainedby TILMedia.Solid.Types.BaseSolid "TB material" annotation (choicesAllMatching, Dialog(tab="Tube Bundle (TB)", group="Fundamental Definitions"));
  parameter Boolean frictionAtInlet_TB=false "True if pressure loss at intlet" annotation (choices(checkBox=true), Dialog(tab="Tube Bundle (TB)", group="Fundamental Definitions"));
  parameter Boolean frictionAtOutlet_TB=false "True if pressure loss at outlet" annotation (choices(checkBox=true), Dialog(tab="Tube Bundle (TB)", group="Fundamental Definitions"));
  replaceable model PressureLoss_TB = ClaRa.Basics.ControlVolumes.Fundamentals.PressureLoss.Generic_PL.LinearPressureLoss_L4 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.PressureLoss.PressureLossBaseVLE_L4 "Pressure loss model" annotation (Dialog(tab="Tube Bundle (TB)", group="Fundamental Definitions"), choicesAllMatching=true);
  replaceable model HeatTransfer_TB = ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.Generic_HT.CharLine_L4 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.HeatTransferBaseVLE_L4 "Heat transfer model" annotation (Dialog(tab="Tube Bundle (TB)", group="Fundamental Definitions"), choicesAllMatching=true);
  replaceable model MechanicalEquilibrium_TB = ClaRa.Basics.ControlVolumes.Fundamentals.SpacialDistribution.Homogeneous_L4 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.SpacialDistribution.MechanicalEquilibrium_L4 "Mechanic flow model" annotation (Dialog(tab="Tube Bundle (TB)", group="Fundamental Definitions"), choicesAllMatching=true);

  parameter ClaRa.Basics.Units.Length length_TB=length_furnace "Length of the pipe (one pass)" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length diameter_o_TB=0.06 "Outer diameter of the pipe" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length diameter_i_TB=0.05 "Inner diameter of the pipe" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length z_in_TB=z_out_furnace "Height of inlet above ground" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length z_out_TB=z_in_furnace "Height of outlet above ground" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter Integer N_tubes_TB=800 "Number Of parallel pipes" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter Integer N_passes_TB=1 "Number of passes of the tubes" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length Delta_z_par=1.5*diameter_o_TB "Distance between tubes parallel to flow direction" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length Delta_z_ort=1.5*diameter_o_TB "Distance between tubes orthogonal to flow direction" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter Boolean staggeredAlignment=true "True, if the tubes are aligned staggeredly, false otherwise" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
  parameter Integer N_rows_TB=integer((N_tubes_TB/integer(width_furnace/Delta_z_ort))*N_passes_TB) "Number of pipe rows in flow direction (minimum = N_passes)" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"));
      parameter Integer tubeOrientation_TB=0 "Tube orientation" annotation (Dialog(tab="Tube Bundle (TB)", group="Geometry"), choices(
      choice=0 "Lengthwise",
      choice=1 "Widthwise",
      choice=2 "Heightwise"));
  parameter Integer N_cv_TB=3 "Number of finite volumes" annotation (Dialog(tab="Tube Bundle (TB)", group="Discretisation"));
  parameter ClaRa.Basics.Units.Length Delta_x_TB[N_cv_TB]=ClaRa.Basics.Functions.GenerateGrid(
      {0},
      length_TB*N_passes_TB,
      N_cv_TB) "Discretisation scheme" annotation (Dialog(tab="Tube Bundle (TB)", group="Discretisation"));
  parameter Integer initOption_TB=0 "Type of initialisation" annotation (Dialog(tab="Tube Bundle (TB)", group="Initialisation"), choices(choice = 0 "Use guess values", choice = 208 "Steady pressure and enthalpy", choice=201 "Steady pressure", choice = 202 "Steady enthalpy"));
  parameter ClaRa.Basics.Units.EnthalpyMassSpecific h_start_TB[N_cv_TB]=fill(800e3, N_cv_TB) "Initial specific enthalpy for single tube" annotation (Dialog(tab="Tube Bundle (TB)", group="Initialisation"));
  parameter ClaRa.Basics.Units.Pressure p_start_TB[N_cv_TB]=fill(1e5, N_cv_TB) "Initial pressure" annotation (Dialog(tab="Tube Bundle (TB)", group="Initialisation"));
  parameter ClaRa.Basics.Units.MassFraction xi_start_TB[medium_TB.nc - 1]=medium_CT.xi_default "Initial composition" annotation (Dialog(tab="Tube Bundle (TB)", group="Initialisation"));
  parameter ClaRa.Basics.Units.EnthalpyMassSpecific h_nom_TB[N_cv_TB]=ones(N_cv_TB)*1e5 "Nominal specific enthalpy for single tube" annotation (Dialog(tab="Tube Bundle (TB)", group="Nominal Values"));
  parameter ClaRa.Basics.Units.Pressure p_nom_TB[N_cv_TB]=ones(N_cv_TB)*1e5 "Nominal pressure" annotation (Dialog(tab="Tube Bundle (TB)", group="Nominal Values"));
  parameter ClaRa.Basics.Units.MassFlowRate m_flow_nom_TB=100 "Nominal mass flow w.r.t. all parallel tubes" annotation (Dialog(tab="Tube Bundle (TB)", group="Nominal Values"));
  parameter ClaRa.Basics.Units.PressureDifference Delta_p_nom_TB=1e4 "Nominal pressure loss w.r.t. all parallel tubes" annotation (Dialog(tab="Tube Bundle (TB)", group="Nominal Values"));

/// CarrierTubes///

  parameter Boolean CTisCooled = true "True if carrier tubes are cooled (else omitted)" annotation (choices(checkBox=true), Dialog(tab="Carrier Tubes (CT)", group="Fundamental Definitions"));
  parameter TILMedia.VLEFluid.Types.BaseVLEFluid medium_CT=simCenter.fluid1 "CT medium model" annotation (
      choicesAllMatching, Dialog(
      enable=CTisCooled,
      tab="Carrier Tubes (CT)",
      group="Fundamental Definitions",
      groupImage="modelica://ClaRa/Resources/Images/ParameterDialog/CarrierTubesSketch.png"));
  replaceable model Material_CT = Basics.Media.Solids.Steel16Mo3 constrainedby TILMedia.Solid.Types.BaseSolid "CT material" annotation (choicesAllMatching, Dialog(tab="Carrier Tubes (CT)", group="Fundamental Definitions", enable=CTisCooled));

  parameter Boolean frictionAtInlet_CT=false "True if pressure loss at inlet" annotation (choices(checkBox=true), Dialog(enable=CTisCooled,tab="Carrier Tubes (CT)", group="Fundamental Definitions"));
  parameter Boolean frictionAtOutlet_CT=false "True if pressure loss at outlet" annotation (choices(checkBox=true), Dialog(enable=CTisCooled,tab="Carrier Tubes (CT)", group="Fundamental Definitions"));
  replaceable model PressureLoss_CT = ClaRa.Basics.ControlVolumes.Fundamentals.PressureLoss.Generic_PL.LinearPressureLoss_L4 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.PressureLoss.PressureLossBaseVLE_L4 "Pressure loss model"  annotation (choicesAllMatching, Dialog(enable=CTisCooled,tab="Carrier Tubes (CT)", group="Fundamental Definitions"));
  replaceable model HeatTransfer_CT = ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.Generic_HT.CharLine_L4 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.HeatTransport.HeatTransferBaseVLE_L4 "Heat transfer model"  annotation (choicesAllMatching, Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Fundamental Definitions"));
  replaceable model MechanicalEquilibrium_CT = ClaRa.Basics.ControlVolumes.Fundamentals.SpacialDistribution.Homogeneous_L4 constrainedby ClaRa.Basics.ControlVolumes.Fundamentals.SpacialDistribution.MechanicalEquilibrium_L4 "Mechanic flow model"  annotation (choicesAllMatching, Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Fundamental Definitions"));
  parameter ClaRa.Basics.Units.Length length_CT=abs(z_in_furnace - z_out_furnace) "Length of the pipe (one pass)" annotation (Dialog(enable=CTisCooled,tab="Carrier Tubes (CT)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length diameter_o_CT=0.06 "Outner diameter of the pipe" annotation (Dialog(enable=CTisCooled,tab="Carrier Tubes (CT)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length diameter_i_CT=0.05 "Inner diameter of the pipe" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length z_in_CT=z_out_furnace "Height of inlet above ground" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Geometry"));
  parameter ClaRa.Basics.Units.Length z_out_CT=z_in_furnace "Height of outlet above ground" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Geometry"));
  parameter Integer N_tubes_CT=800 "Number Of parallel pipes" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Geometry"));
  parameter Integer N_passes_CT=1 "Number of passes of the tubes" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Geometry"));
  parameter Integer initOption_CT=0 "Type of initialisation" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Initialisation"), choices(choice = 0 "Use guess values", choice = 208 "Steady pressure and enthalpy", choice=201 "Steady pressure", choice = 202 "Steady enthalpy"));
  parameter ClaRa.Basics.Units.EnthalpyMassSpecific h_start_CT=2500e3 "Initial specific enthalpy for single tube" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Initialisation"));
  parameter ClaRa.Basics.Units.Pressure p_start_CT=100e5 "Initial pressure" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Initialisation"));
  parameter ClaRa.Basics.Units.MassFraction xi_start_CT[medium_CT.nc - 1] = medium_CT.xi_default "Initial composition" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Initialisation"));
  parameter ClaRa.Basics.Units.MassFlowRate m_flow_nom_CT=100 "Nominal mass flow w.r.t. all parallel tubes" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Nominal Values"));
  parameter ClaRa.Basics.Units.EnthalpyMassSpecific h_nom=1e5 "Nominal specific enthalpy for single tube" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Nominal Values"));
  parameter ClaRa.Basics.Units.Pressure p_nom_CT=1e5 "Nominal pressure" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Nominal Values"));
  parameter ClaRa.Basics.Units.PressureDifference Delta_p_nom_CT=1e4 "Nominal pressure loss w.r.t. all parallel tubes" annotation (Dialog(enable=CTisCooled, tab="Carrier Tubes (CT)", group="Nominal Values"));

  ///Summary and Visualisation ///
  parameter Boolean showExpertSummary=simCenter.showExpertSummary "True, if an extended summary shall be shown, else false" annotation (Dialog(tab="Summary and Visualisation"));
  parameter Boolean showData= true "True if data connectors shall be shown" annotation (Dialog(tab="Summary and Visualisation"));

  ClaRa.Components.VolumesValvesFittings.Pipes.PipeFlowVLE_L4_Simple pipeFlow_FTW(
    medium=medium_FTW,
    frictionAtInlet=frictionAtInlet_FTW,
    frictionAtOutlet=frictionAtOutlet_FTW,
    redeclare model PressureLoss = PressureLoss_FTW,
    redeclare model HeatTransfer = HeatTransfer_FTW,
    redeclare model MechanicalEquilibrium = MechanicalEquilibrium_FTW,
    p_nom=p_nom_FTW,
    h_nom=h_nom_FTW,
    m_flow_nom=m_flow_nom_FTW,
    Delta_p_nom=Delta_p_nom_FTW,
    initOption=initOption_FTW,
    h_start=h_start_FTW,
    p_start=p_start_FTW,
    xi_start=xi_start_FTW,
    showExpertSummary=showExpertSummary,
    showData=true,
    length=length_FTW,
    diameter_i=diameter_i_FTW,
    z_in=z_in_FTW,
    z_out=z_out_FTW,
    N_tubes=N_tubes_FTW,
    N_passes=N_passes_FTW,
    N_cv=N_cv_FTW,
    Delta_x=Delta_x_FTW) if FTWisCooled
                     annotation (Placement(transformation(
        extent={{-14,-5},{14,5}},
        rotation=90,
        origin={280,0})));
  ClaRa.Basics.ControlVolumes.SolidVolumes.CylindricalThinWall_L4 wall_FTW(
    redeclare model Material = Material_FTW,
    N_ax=N_cv_FTW,
    Delta_x=Delta_x_FTW,
    diameter_o=diameter_o_FTW,
    diameter_i=diameter_i_FTW,
    length=length_FTW*N_passes_FTW,
    N_tubes=N_tubes_FTW,
    T_start={TILMedia.VLEFluid.Functions.temperature_phxi(
        pipeFlow_TB.medium,
        p_start_FTW[i],
        h_start_FTW[i]) + 5 for i in 1:N_cv_FTW},
    suppressChattering="False") if FTWisCooled
    annotation (Placement(transformation(
        extent={{-14,5},{14,-5}},
        rotation=270,
        origin={240,0})));
  ClaRa.Components.Adapters.Scalar2VectorHeatPort adapt_FTW(N=N_cv_FTW) if FTWisCooled
                                                                    annotation (Placement(transformation(
        extent={{10,-10},{-10,10}},
        rotation=180,
        origin={188,0})));

  ClaRa.Basics.Interfaces.FluidPortOut outlet_FTW(Medium=medium_FTW) if FTWisCooled
                                                                 "Outlet port"               annotation (Placement(transformation(extent={{270,90},{290,110}}), iconTransformation(extent={{270,90},{290,110}})));
  ClaRa.Basics.Interfaces.FluidPortIn inlet_FTW(Medium=medium_FTW) if FTWisCooled
                                                               "Inlet port"               annotation (Placement(transformation(extent={{270,-110},{290,-90}}), iconTransformation(extent={{270,-110},{290,-90}})));
  ClaRa.Components.Furnace.FlameRoom.FlameRoomWithTubeBundle_L2_Dynamic flameRoom(
    fuelModel=fuelModel,
    slagType=slagType,
    flueGas=flueGas,
    slagTemperature_calculationType=slagTemperature_calculationType,
    T_slag=T_slag,
    redeclare model HeatTransfer_Wall = GasHeatTransfer_Wall,
    redeclare model HeatTransfer_Top = GasHeatTransfer_Top,
    redeclare model ParticleMigration = ParticleMigration,
    redeclare model Burning_time = Burning_time,
    redeclare model ReactionZone = ReactionZone,
    redeclare model Geometry = ClaRa.Basics.ControlVolumes.Fundamentals.Geometry.HollowBlockWithTubesAndCarrierTubes (
        CF_geo=fill(1, 3),
        N_inlet=1,
        N_outlet=1,
        z_in={z_in_furnace},
        z_out={z_out_furnace},
        height_fill=-1,
        height=abs(z_in_furnace - z_out_furnace),
        width=width_furnace,
        length=length_furnace,
        diameter_t=diameter_o_TB,
        N_tubes=N_tubes_TB,
        N_passes=N_passes_TB,
        N_ct=N_tubes_CT,
        d_ct=diameter_o_CT,
        Delta_z_par=Delta_z_par,
        Delta_z_ort=Delta_z_ort,
        staggeredAlignment=staggeredAlignment,
        N_rows=N_rows_TB,
        tubeOrientation=tubeOrientation_TB,
        flowOrientation=flowOrientation_furnace),
    redeclare model PressureLoss = GasPressureLoss,
    m_flow_nom=m_flow_nom_gas,
    p_start_flueGas_out=p_start_flueGas_out,
    T_start_flueGas_out=T_start_flueGas_out,
    T_top_initial=T_top_initial,
    xi_start_flueGas_out=xi_start_flueGas_out,
    redeclare model HeatTransfer_CarrierTubes = GasHeatTransfer_CarrierTubes,
    redeclare model HeatTransfer_TubeBundle = GasHeatTransfer_TubeBundle) annotation (Placement(transformation(extent={{-52,-10},{8,10}})));
  ClaRa.Basics.Interfaces.HeatPort_a heat_bottom annotation (Placement(transformation(extent={{-108,-110},{-88,-90}})));
  ClaRa.Basics.Interfaces.FuelSlagFlueGas_inlet inletFG(
    fuelModel=fuelModel,
    slagType=slagType,
    flueGas(Medium=flueGas))                            annotation (Placement(transformation(extent={{-210,-110},{-190,-90}})));
  ClaRa.Basics.Interfaces.FuelSlagFlueGas_outlet outletFG(
    fuelModel=fuelModel,
    slagType=slagType,
    flueGas(Medium=flueGas))                              annotation (Placement(transformation(extent={{-210,88},{-190,108}})));
  ClaRa.Basics.Interfaces.HeatPort_a heat_top annotation (Placement(transformation(extent={{-110,88},{-90,108}})));
  ClaRa.Basics.Interfaces.EyeOut eyeFTW if showData and FTWisCooled
                                        annotation (Placement(transformation(
        extent={{-5,-5},{5,5}},
        rotation=90,
        origin={240,104})));
  ClaRa.Basics.Interfaces.FluidPortOut outlet_CT(Medium=medium_CT) if CTisCooled
                                                                "Outlet port" annotation (Placement(transformation(extent={{10,-110},{30,-90}}), iconTransformation(extent={{-10,-110},{10,-90}})));
  ClaRa.Basics.Interfaces.FluidPortIn inlet_CT(Medium=medium_CT) if CTisCooled
                                                              "Inlet port" annotation (Placement(transformation(extent={{10,88},{30,108}}), iconTransformation(extent={{-10,90},{10,110}})));
  ClaRa.Basics.Interfaces.FluidPortIn inlet_TB(Medium=medium_TB) "Tub bundle inlet port"
                                                                                      annotation (Placement(transformation(extent={{-310,50},{-290,70}}), iconTransformation(extent={{-308,40},{-288,60}})));
  ClaRa.Basics.Interfaces.FluidPortOut outlet_TB(Medium=medium_TB) "Tube bundle outlet port"
                                                                                          annotation (Placement(transformation(extent={{-310,-60},{-290,-40}}), iconTransformation(extent={{-310,-60},{-290,-40}})));
  ClaRa.Components.Adapters.Scalar2VectorHeatPort adapt_TB(N=N_cv_TB)
                                                                   annotation (Placement(transformation(
        extent={{-10,-10},{10,10}},
        rotation=180,
        origin={-192,-20})));
  ClaRa.Basics.ControlVolumes.SolidVolumes.CylindricalThinWall_L4 wall_TB(
    redeclare model Material = Material_TB,
    N_ax=N_cv_TB,
    Delta_x=Delta_x_TB,
    diameter_o=diameter_o_TB,
    diameter_i=diameter_i_TB,
    length=length_TB*N_passes_TB,
    N_tubes=N_tubes_TB,
    T_start={TILMedia.VLEFluid.Functions.temperature_phxi(
        pipeFlow_TB.medium,
        p_start_TB[i],
        h_start_TB[i]) + 5 for i in 1:wall_TB.N_ax},
    suppressChattering="False") annotation (Placement(transformation(
        extent={{-14,-5},{14,5}},
        rotation=270,
        origin={-228,-20})));
  ClaRa.Components.VolumesValvesFittings.Pipes.PipeFlowVLE_L4_Simple pipeFlow_TB(
    medium=medium_TB,
    frictionAtInlet=frictionAtInlet_TB,
    frictionAtOutlet=frictionAtOutlet_TB,
    redeclare model PressureLoss = PressureLoss_TB,
    redeclare model HeatTransfer = HeatTransfer_TB,
    redeclare model MechanicalEquilibrium = MechanicalEquilibrium_TB,
    p_nom=p_nom_TB,
    h_nom=h_nom_TB,
    m_flow_nom=m_flow_nom_TB,
    Delta_p_nom=Delta_p_nom_TB,
    initOption=initOption_TB,
    h_start=h_start_TB,
    p_start=p_start_TB,
    xi_start=xi_start_TB,
    showExpertSummary=showExpertSummary,
    showData=true,
    length=length_TB,
    diameter_i=diameter_i_TB,
    z_in=z_in_TB,
    z_out=z_out_TB,
    N_tubes=N_tubes_TB,
    N_passes=N_passes_TB,
    N_cv=N_cv_TB,
    Delta_x=Delta_x_TB)
                     annotation (Placement(transformation(
        extent={{14,5},{-14,-5}},
        rotation=90,
        origin={-262,-20})));
  ClaRa.Basics.Interfaces.EyeOut eyeTB if showData
                                       annotation (Placement(transformation(extent={{-299,-85},{-309,-75}})));
  ClaRa.Basics.Interfaces.EyeOutGas eyeFG(medium=flueGas) if showData
                                          annotation (Placement(transformation(
        extent={{-6,-6},{6,6}},
        rotation=90,
        origin={-240,106})));
  ClaRa.Components.VolumesValvesFittings.Pipes.PipeFlowVLE_L2_Simple pipeFlow_CT(
    medium=medium_CT,
    frictionAtInlet=frictionAtInlet_CT,
    frictionAtOutlet=frictionAtOutlet_CT,
    redeclare model PressureLoss = PressureLoss_CT,
    redeclare model HeatTransfer = HeatTransfer_CT,
    redeclare model MechanicalEquilibrium = MechanicalEquilibrium_CT,
    p_nom={p_nom_CT},
    h_nom={h_nom},
    m_flow_nom=m_flow_nom_CT,
    Delta_p_nom=Delta_p_nom_CT,
    initOption=initOption_CT,
    h_start={h_start_CT},
    p_start={p_start_CT},
    xi_start=xi_start_CT,
    showExpertSummary=showExpertSummary,
    showData=true,
    length=length_CT,
    diameter_i=diameter_i_CT,
    z_in=z_in_CT,
    z_out=z_out_CT,
    N_tubes=N_tubes_CT,
    N_passes=N_passes_CT) if CTisCooled
                          annotation (Placement(transformation(
        extent={{-14,5},{14,-5}},
        rotation=270,
        origin={40,39})));
  ClaRa.Basics.ControlVolumes.SolidVolumes.CylindricalThinWall_L4 wall_CT(
    redeclare model Material = Material_CT,
    N_ax=1,
    diameter_o=diameter_o_CT,
    diameter_i=diameter_i_CT,
    length=length_CT*N_passes_CT,
    N_tubes=N_tubes_CT,
    T_start={TILMedia.VLEFluid.Functions.temperature_phxi(
        pipeFlow_TB.medium,
        p_start_CT,
        h_start_CT) + 5},
    suppressChattering="False") if CTisCooled
    annotation (Placement(transformation(
        extent={{-14,5},{14,-5}},
        rotation=270,
        origin={16,39})));

  ClaRa.Basics.Interfaces.EyeOut eyeCT if showData and CTisCooled
                                       annotation (Placement(transformation(
        extent={{-5,-5},{5,5}},
        rotation=270,
        origin={60,-104})));
equation
  connect(pipeFlow_FTW.heat, wall_FTW.innerPhase) annotation (Line(
      points={{276,0},{245,0}},
      color={167,25,48},
      thickness=0.5));
  connect(wall_FTW.outerPhase, adapt_FTW.heatVector) annotation (Line(
      points={{235,0},{235,-6.66134e-16},{198,-6.66134e-16}},
      color={167,25,48},
      thickness=0.5));
  connect(flameRoom.heat_bottom, heat_bottom) annotation (Line(
      points={{-20,-10},{-20,-60},{-98,-60},{-98,-100}},
      color={167,25,48},
      thickness=0.5));
  connect(heat_top, flameRoom.heat_top) annotation (Line(
      points={{-100,98},{-100,40},{-20,40},{-20,10}},
      color={167,25,48},
      thickness=0.5));
  connect(adapt_FTW.heatScalar, flameRoom.heat_wall) annotation (Line(
      points={{178,1.77636e-15},{94,1.77636e-15},{94,0},{8,0}},
      color={167,25,48},
      thickness=0.5));
  connect(pipeFlow_FTW.inlet, inlet_FTW) annotation (Line(
      points={{280,-14},{280,-100}},
      color={0,131,169},
      pattern=LinePattern.Solid,
      thickness=0.5));
  connect(pipeFlow_FTW.outlet, outlet_FTW) annotation (Line(
      points={{280,14},{280,100}},
      color={0,131,169},
      pattern=LinePattern.Solid,
      thickness=0.5));
  connect(pipeFlow_FTW.eye, eyeFTW) annotation (Line(points={{283.4,14.6},{283.4,46},{240,46},{240,104}},                   color={190,190,190}), Text(
      string="%second",
      index=1,
      extent={{-3,6},{-3,6}},
      horizontalAlignment=TextAlignment.Right));
  connect(pipeFlow_TB.inlet, inlet_TB) annotation (Line(
      points={{-262,-6},{-262,60},{-300,60}},
      color={0,131,169},
      pattern=LinePattern.Solid,
      thickness=0.5));
  connect(pipeFlow_TB.outlet, outlet_TB) annotation (Line(
      points={{-262,-34},{-262,-50},{-300,-50}},
      color={0,131,169},
      pattern=LinePattern.Solid,
      thickness=0.5));
  connect(pipeFlow_TB.eye, eyeTB) annotation (Line(points={{-265.4,-34.6},{-265.4,-40},{-256,-40},{-256,-80},{-304,-80}}, color={190,190,190}));
  connect(flameRoom.heat_TubeBundle, adapt_TB.heatScalar) annotation (Line(
      points={{-2,-10},{-2,-20},{-182,-20}},
      color={167,25,48},
      thickness=0.5));
  connect(wall_TB.outerPhase, adapt_TB.heatVector) annotation (Line(
      points={{-223,-20},{-202,-20}},
      color={167,25,48},
      thickness=0.5));
  connect(wall_TB.innerPhase, pipeFlow_TB.heat) annotation (Line(
      points={{-233,-20},{-258,-20}},
      color={167,25,48},
      thickness=0.5));
  connect(flameRoom.eyeOut, eyeFG) annotation (Line(points={{-52,8},{-240,8},{-240,106}},                   color={190,190,190}));
  connect(wall_CT.innerPhase, pipeFlow_CT.heat) annotation (Line(
      points={{21,39},{36,39}},
      color={167,25,48},
      thickness=0.5));
  connect(wall_CT.outerPhase[1], flameRoom.heat_CarrierTubes) annotation (Line(
      points={{11,39},{-2,39},{-2,10}},
      color={167,25,48},
      thickness=0.5));
  connect(pipeFlow_CT.inlet, inlet_CT) annotation (Line(
      points={{40,53},{40,80},{20,80},{20,98}},
      color={0,131,169},
      pattern=LinePattern.Solid,
      thickness=0.5));
  connect(pipeFlow_CT.outlet, outlet_CT) annotation (Line(
      points={{40,25},{40,-48},{20,-48},{20,-100}},
      color={0,131,169},
      pattern=LinePattern.Solid,
      thickness=0.5));
  connect(flameRoom.inlet, inletFG) annotation (Line(
      points={{-38,-10},{-38,-40},{-200,-40},{-200,-100}},
      color={118,106,98},
      thickness=0.5));
  connect(flameRoom.outlet, outletFG) annotation (Line(
      points={{-38,10},{-38,22},{-200,22},{-200,98}},
      color={118,106,98},
      thickness=0.5));
  connect(pipeFlow_CT.eye, eyeCT) annotation (Line(points={{43.4,24.4},{43.4,-39.8},{60,-39.8},{60,-104}}, color={190,190,190}));

                                          //not FTWisCooled and not CTisCooled, FTWisCooled and not CTisCooled
  annotation (Documentation(info="<html>
<p><b>For detailed model documentation please consult the html-documentation shipped with ClaRa.</b> </p>
<p>&nbsp;</p>
<p><br><b><span style=\"font-size: 10pt;\">Authorship and Copyright Statement for original (initial) Contribution</span></b></p>
<p><b>Author:</b> </p>
<p>Friedrich Gottelt, Forschungszentrum f&uuml;r Verbrennungsmotoren und Thermodynamik Rostock GmbH, Copyright &copy; 2019-2020</p>
<p><a href=\"http://www.fvtr.de\">www.fvtr.de</a>
<p><b>References:</b> </p>
<p> For references please consult the html-documentation shipped with ClaRa. </p>
<p><b>Remarks:</b> </p>
<p>This component was developed by Forschungszentrum f&uuml;r Verbrennungsmotoren und Thermodynamik Rostock GmbH for industry projects in cooperation with Lausitz Energie Kraftwerke AG, Cottbus.</p>
<b>Acknowledgements:</b>
<p>This model contribution is sponsored by Lausitz Energie Kraftwerke AG.</p>

<p><a href=\"http://www.leag.de\">www.leag.de</a> </p>
<p><b>CLA:</b> </p>
<p>The author(s) have agreed to ClaRa CLA, version 1.0. See <a href=\"https://claralib.com/pdf/CLA.pdf\">https://claralib.com/pdf/CLA.pdf</a></p>
<p>By agreeing to ClaRa CLA, version 1.0 the author has granted the ClaRa development team a permanent right to use and modify his initial contribution as well as to publish it or its modified versions under the 3-clause BSD License.</p>
<p>The ClaRa development team consists of the following partners:</p>
<p>TLK-Thermo GmbH (Braunschweig, Germany)</p>
<p>XRG Simulation GmbH (Hamburg, Germany).</p>
</html>",
        revisions="<html>
<body>
<table>
  <tr>
    <th style=\"text-align: left;\">Date</th>
    <th style=\"text-align: left;\">&nbsp;&nbsp;</th>
    <th style=\"text-align: left;\">Version</th>
    <th style=\"text-align: left;\">&nbsp;&nbsp;</th>
    <th style=\"text-align: left;\">Author</th>
    <th style=\"text-align: left;\">&nbsp;&nbsp;</th>
    <th style=\"text-align: left;\">Affiliation</th>
    <th style=\"text-align: left;\">&nbsp;&nbsp;</th>
    <th style=\"text-align: left;\">Changes</th>
  </tr>
  <tr>
    <td>2020-08-20</td>
    <td> </td>
    <td>ClaRa 1.6.0</td>
    <td> </td>
    <td>Friedrich Gottelt</td>
    <td> </td>
    <td>FVTR GmbH</td>
    <td> </td>
    <td>Initial version of model</td>
  </tr>
</table>
<p>Version means first ClaRa version where the applied change was published.</p>
</body>
</html>"),
         Icon(coordinateSystem(preserveAspectRatio=false), graphics={
        Bitmap(
          visible = FTWisCooled and CTisCooled,
          extent={{-298,-100},{302,100}},
          imageSource="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"
               + "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",
          fileName="modelica://ClaRa/Resources/Images/Components/FurnaceSlice.png"),
        Bitmap(
          visible = FTWisCooled and not CTisCooled,
          extent={{-300,-100},{300,100}},
          imageSource="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"
               + "61aW5bnBgsVLVVJaajSjR7cuRue7RSAQ0CbDbxc2TWlidD4QLCj0cLQo9QAAAAAAABASAoGAHp34tNakr7Ms0+fz6YIxoy3Lc4tZP88xnkGp96ut27eruLjY2PyIiAg1TEoyNh8IFhR6qA6UegAAAAAAAAh6gUBATz77or7/cbqluSeMGK6mTRpbmul0fr9fs+fOM5qRVL++kimaJElr1qQbnd80pYl8Pp/RDMDtKPRQXSj1AAAAAAAAEPReev1NfTblK0szo6OiNG7MeZZmusHS5Su0Z+9eoxndu3NK7zemT6Y2b5ZidD7gdhR6qE6UegAAAAAAAAhqr775jiZ98LHluReOOVfx8XGW5zrd1Gk/Gs/o2a2r8Qy3WLV2rdH5zVKaGp0PuBmFHqobpR4AAAAAAACC1hvvvKe3J022PLdpSopOO3mU5blOV1pWpp9mzTaaER4eru5dOaknSRUVFdqwMdNoRovmzYzOB9yKQg8mhNm9AAAAAAAAAFDd/H6/nnzuRX3x1TeWZ/u8Xo2/8TreGTuAmT/NVmlZmdGMbl3SFB0VZTTDLTLWb9D+/fuNzfd4PGrbKtXYfMCtKPRgCqUeAAAAAAAAgkpZWbnufegRzZ4735b8MeeepTatKToOZOq0H4xnDOjbx3iGW6xem250fpNGDRUTU8toBuA2o04YQaEHYyj1AAAAAAAAEDQKCgt12133acWq1bbkp7ZorvPOPtOWbKfbnpWtpctXGM3wer3q17un0Qw3WZ2+zuj8Nm1aG50PuM2oE0boxmuuotCDMZR6AAAAAAAACArrMtbrzgceVnZ2ji354eHhuv3mGxQWxkduB/Lpl1+psrLSaEbH9u0UHx9nNMMtAoGAlixbbjSjfVtKPeA3FHqwAn/DAAAAAAAAgOt9+/0PmvD0cyovL7dth3FjRqt5s6a25TtZWVm5pn4/zXjOwH5cvfmbzM1blJeXbzSjXWtKPUCi0IN1KPUAAAAAAADgWqVlZXrsiWc0bfoMW/fo0K6tzjz9VFt3cLLvf5yhwsIioxler1eDBw4wmuEmi5cuMzo/IiJCLZo3M5oBuAGFHqxEqQcAAAAAAABXWpueoYcee1yZW7baukdCfLzuu+NW+Xw+W/dwqkAgoM+nfGU8p3vXNNVNTDCe4xaLDV+92Tq1Jb/nEfIo9GA1Sj0AAAAAAAC4it/v1wcff6ZX3nxbfr/f1l3CwsJ0zx23KjEx0dY9nGzBwsXK2LDReM5xxwwznuEWfr9fS5evMJrRtnUro/MBp6PQgx0o9QAAAAAAAOAaGzZm6qEJE5WxfoPdq0iSrr/q/9S5Q3u713C0dyZ/aDwjOipK/fvwnt5vMtZvNH7daeeOHYzOB5yMQg92odQDAAAAAACA45WUlurt9ybrg08+s/103m9OP/VknXj8CLvXcLTVa9O1bMVK4zlDBg5QZGQN4zlusXiZ2ff0vF6v0jp1NJoBOBWFHuxEqQcAAAAAAADHCgQCmjFrtp576RXt3JVr9zq/69YlTVdcOs7uNRzvnUmTLckZccxQS3Lc4pd5C4zOT23ZXDExtYxmAE5EoQe7UeoBAAAAAADAkdamZ+ip51/UqjVr7V7lT5o2aax7b79FPq/P7lUcLT0jQ3MMl0uS1DSliTpxBervCgoLtXLVaqMZ3dLSjM4HnIhCD05AqQcAAAAAAABH2bx1m959/wNNmz7TMVdt/qZhcpImPvqgYmNj7V7F8V5+7S0FAgHjOaedfCIfsv/B3PkL5a+sNJrRtXMno/MBp6HQg1NQ6gEAAAAAAMARduzcpbffn6xv/v2d8VLiSNSrm6iJjz6oxIQEu1dxvGUrV2nB4iXGc6Jq1tQxQ4cYz3GTOXPnGZ3v83rVoUM7oxmAk1DowUko9QAAAAAAAGCrrOxsTfrwE3099TvHncz7Td3EBD0z4VEl1a9v9yqu8NKrb1iSc/yxwxQdFWVJlhv4/X7NX7TYaEb7dm0UVbOm0QzAKSj04DSUegAAAAAAALBFekaGPv7sS037cYYjT+b9Jq5ObT3+8ANKSmpg9yquMPPn2Vq5eo3xHI/Ho5NPHGk8x02WLF+hoqJioxld0zobnQ84BYUenIhSDwAAAAAAAJbx+/2a88s8Tf7kM61YtdrudQ4pPq6Onnj0ITVNaWL3Kq5QXl6uF15+3ZKsHt268L/LX8yZO994RreuXYxnAHaLi6tDoQdHotQDAAAAAACAcbm7d+u7adP1xVffKHvHDrvXqZLkpCRNeOg+NWqYbPcqrvH+R58qKyfHkqzRZ/7Tkhy3qKys1MxZPxvNiI2NVYd2bYxmAE6Q3KABhR4ciVIPAAAAAAAARvgr/Zq3YLGmfP2N5i5Y5Nj38g6kfds2euT+u1U7NtbuVVxjV+5uvf/hx5ZktW3dSmmdO1qS5RbLV67SrtzdRjP69Oohn9dnNANwAgo9OBWlHgAAAAAAAKpVekaGpn7/o6b/NEt5efl2r3PY+vfppbtuu0WRkTXsXsVVnn7+JZWWlVmSdd7ZZ1qS4ybTps80ntGvdy/jGQCAv0epBwAAAAAAgKO2LmO9Zv48R9N+nOGa6zUP5KSRx+u6Ky+Xz8dppMMxZ94Czfx5tiVZTZs0Vt/ePS3Jcgu/368Zs8x+/cPCwtSjG+/pAYCdKPUAAAAAAABw2PyVfi1dtkKzfpmr2XPmasfOXXavdFR8Xq8uGTdWZ5/xD7tXcZ3i4mJNeOpZy/JGn32mvF6vZXluMHfBIhUUFBjN6Nq5o6KjooxmAAAOjlIPAAAAAAAAVbIrd7fmL1qkhYuXav7CRSosLLJ7pWqREB+vu8ffrLROvNF2JF549Q3l5uZaktU0JUXDBg+0JMtNfvhxhvGMPr17G88AABwcpR4AAAAAAAAOqLCwSCtWr9GiJUu1YNFiZW7eYvdK1S6tU0fdM/4WxcfH2b2KKy1cslRTvplqWd7F57c4VSMAACAASURBVI/matS/KC0r0+y584zn9O3dw3gGAODgKPUAAAAAAAAg6deTeCtXr9GylSu1bPkKbcrcosrKSrvXMsLj8ejsM/6hiy8cI5+XkuhIFBUV65EJTygQCFiS17pVS/Xv18eSLDeZ/tMslZaVGc1IbdFcSfXrG80AABwapR4AAAAAAEAIyt6xQxnrNyh93Xqlr1+vjIz1yt+z1+61LFE7Nla3XH+N+vflOsGjMeHpZ7VzlzXXbkrSpeMukMfjsSzPLaZ8bf6k5PChg41nAAAOjVIPAAAAAAAgiJWWlWnbtu3aum271m3YoHXr1mvd+g0qKCy0ezVbDOzXV9dffYXi4+rYvYqrTf3uB/044yfL8rqmdVb3LmmW5bnFxk2ZWrVmrdEMj8ejwQP7G80AAFQNpR4AAAAAAIDL7du3Tzk7dmrr9ixt3bZNW7dt17b//OddubvtXs8RYmNjde2Vl2n44EF2r+J6W7dt15PPvWBZns/n05WXXWxZnpt8+W/zp/Q6dWjP1ZsA4BCUegAAAAAAAA7kr/SroKBIBYWFKiws1N6CQuXvydeuXbuVuztXubn52pm7U3l5+SFzbeaR6t+3t268+krFx8fZvYrrlZaV6fZ7H1BJaallmSePPF4tmzezLM8tysvL9f0P043nDB9CEQ4ATmG01FuXsV4XX3GtyQgAAAAACHp7CwuMzn/j3Un69IuvjGYAOLTyigqVV5SrvKxM5eUVKi4psXsl10uIj9cVl47T8CGD7V4laEx85nllbt5iWV5MTC1dMOZcy/LcZPrMn1VYWGQ0w+f1amD/fkYzAABVZ7TUKyktVXpGhskIAAAAAMBRysnZoZycHXavAQDVxufz6ZRRI3XR2PMUHRVl9zpB49Mvv9K33/9gaea4889T7dhYSzPd4otv/m08o0f3boqrU9t4DgCgarh+EwAAAAAAAEGje5c0Xf1/l6ppShO7Vwkqi5Ys1TPPv2RpZsvmzXTyyOMtzXSLNenrtGr1GuM5xwzl6k0AcBJKPQAAAAAAALhe40YNdeVlF6lPz552rxJ0srNzdM+Dj8pfWWlZptfr1Q1XXyGfz2dZppu8/+HHxjOio6LUv28f4zkAgKqj1AMAAAAAAIBr1aubqPPPOVsnHH+MfF4KoOpWWFikG2+/S3sLzL7v+lenn3KS2rdra2mmW2Tl5GjWnLnGc44ZOkQ1IyON5wAAqo5SDwAAAAAAAK5Tp3ZtnXX6aTr91JMUERFh9zpBqaKiQuPvvk9bt223NLdB/XoaN/Y8SzPd5KNPPpPf7zeec+LxxxrPAAAcHko9AAAAAAAAuEad2rV1zpmn69RRI1WjRg271wlafr9f9z8yQctWrrI8+8ZrruKE2N8oLCzSN99OM57TulVLtUptaTwHQNWUV1TYvQIcglIPAAAAAAAAjpeU1EBnnHKSRh4/gsLHAk88+4Jm/jzb8twTjj1GPbt3tTzXLb74+huVlpUZzznp+OONZwComvw9e/XLvAV2rwGHoNQDAAAAAACAY7VOTdXpp4zS8KGD5fPxZp4VXnnjbX359b8tz01u0EBXX3Gp5bluUVpWpo8+/dx4Ts3ISA0dMtB4DoBDy9+zV9fefJuKiorsXgUOQakHAAAAAAAAR/H5fBrQt7f++Y9T1aFdW7vXCSlvvPOe3nn/A8tzvV6vbr3xWkXVrGl5tlt8PuVr5e/Zazxn+LDBio6KMp4D4OB+K/Q2ZW62exU4CKUeAAAAAAAAHCExMVHHDhusU0edqPr16tq9TsiZ/PGneuOdSbZkn3f2P5XWqaMt2W5QWlamyR99YkkWV28C9qPQw9+h1AMAAAAAAIBtvF6vunbupFEjj9fAfn24YtMmkz74WC++9oYt2a1btdTY0efYku0WVp3S69yhvVq3amk8B8Dfo9DDwVDqAQAAAAAAwHJtW7fSMUMHa+iggYqPj7N7nZBmZ6FXq1a07rv9NsrcgygrK7fslN6Zp59mSQ6AA6PQw6FQ6gEAAAAAAMASSfXra8jgARo54lg1btTQ7nVCXiAQ0PMvv6YPPvnMlnyPx6Nbb7hWSUkNbMl3i8+/suaUXsPkJPXt3dN4DoADo9BDVVDqAQAAAAAAwJgWzZuqf5/eGtivr1JbtrB7HfyH3+/X408/p6/+/a1tO5x1+qka2K+vbfluUFRUrHcnf2RJ1pmnnyqv12tJFoA/o9BDVVHqAQAAAAAAoNr4fD61a9Na/Xr3Uv9+vdWkUSO7V8JflJaV6b6HHtXsufNt26F9u7a65MKxtuW7xTvvT1ZBQYHxnNiYGI0YPsx4DoD/RaF3YNnZOZr1y1wtXrJUO3flKm/PHsVER6tOnTpqndpS/fv0VscO7ULu+mZKPQAAAAAAAByVBvXrqXvXLurRrau6dems2JgYu1fC38jfs1e33Hm31qZn2LZDYmKi7r+Td/QOJSsnRx9/PsWSrFNGjVTNyEhLsgD8F4Xe/9q8dZtef+sdzZg1W4FA4E//LC8vX5u3btOyFSv14aefq17dRJ1/ztk64bhjQubPFEo9AAAAAAAAHJbY2Fh16tBO3dPS1L17F07juUTmlq269c57lZWdbdsOkZE19PA9dyoxIcG2HdzilTfe1r59+4znhIeH69STRhrPAfBnFHr/a/LHn+qlV9+Qv7KySr9+565cPfbUM/ph5k+6745bFRsba3hD+1HqAQAAAAAA4KDi4+qoTatUderQQd26dlZqixa8veUy8xcu1j0PPaKiomLbdvB4PLr5+mvUulVL23ZwizXp6/TjjJ8syRo54lglxMdbkgXgVxR6fxYIBDTh6Wc15eupR/TvL166TJdcdb0mPvqAkhs0qObtnIVSDwAAAAAAAL+Liamlpk2aqFVqS7VJbanWrVoqpUkTeTweu1fDEZry9VRNfOa5Kp98MOWC887V8MGDbN3BDQKBgJ598ZX/uXbOhLCwMJ171hnGcwD8F4Xe/3p70gdHXOj9Jis7W7fecY9eeHqioqOiqmkz56HUAwAAAAAACFEJ8fFqndpCrVulqlVqqpqnNFFSUnD/hHsoKSsr14SnntF3P0y3exUNHzJY5597lt1ruMJ3037UilWrLck68bgRql+vriVZAKS8/D269ubxytxMofebpctW6PW3362WWZlbtuqp51/U+Buvr5Z5TkSpBwAAAAAAEMQiIiLUqGGyGiUnq3GjhmrUKFmNGzZSSpNGqh0Cb8+Eqm3bs3Tn/Q9qw8ZMu1dRl7ROuu3GazntWQXFJSV68bU3Lcny+Xw6+8x/WJIF4NcTetfdQqH3R4FAQM+98mq1nkz+9vsf9Y+TR6l1amq1zXQSSj0AAAAAAAAXi4iIUGJCguomxKtu3bpKTIhXUlJ9NWrYUI0aJqt+3bq8fxdiZsyarUcff1LFJSV2r6LWqal6+N67FB4ebvcqrvDSq29qd16eJVmjjj9OSfXrW5IFhDqu3DyweQsWKn3d+mqdGQgE9O77H+n+u8ZX61ynoNQDAAAAAACwWc3ISIWFhymyRg2FhYerZmSkwsPCFRtTSzGxMYqtFaPY2BjFxNRS7ZhY1akTq/r16ikhPk6xnLbDf1RUVOiFV9/Qp19MseQ9tkNJTkrSow/co6iaNe1exRXSMzI05d9H96ZUVXFKD7AOhd7fmzlrjpG5cxcsVFlZuSIjaxiZbyejpV5ap456esIjJiMAAAAAwLH8fr8envCEI94yMm3ooAG687ab5PP67F4FAELSho2ZuvfhR5W5eYvdq0iSEhMSNPGR+xUfV8fuVVzB7/drwlPPyu/3W5J34vEjOKUHWIBC7+DmLVxkZG55ebmWLV+hXj27G5lvJ07qAQAAAIAhPp9Pt914nSQFdbFHoQcA9vH7/frkiyl6+fW3VFFRYfc6kqT4uDp64l8PKTkpye5VXOOLKd9U+xV0f6dmZKTOP/csS7KAUEahd3Dl5eXK3b3b2Pxt2TnqZWy6fSj1AAAAAMCgYC/2KPQAwD7btmfpX088raXLV9i9yu9qx8bqiUcfUkrjRnav4hpZOTl66fU3Lcs764zTlJiQYFkeEIoo9A5td36+2fkGC0M7UeoBAAAAgGHBWuwNGdifQg8AbOCv9OuDjz7Ta2+/q3379tm9zu9q1YrWhIfvU7OmKXav4hqVlZV6ZMKTKi0rsyQvPq6Ozjz9NEuygFBFoVc1Xo/X6Hyfz+x8u1DqAQAAAIAFgq3YGzKwv+4afzOFHgBYLH3dej0y8Qlt2Jhp9yp/Ehsbq8cfvk+tU1PtXsVVPv1iiqUnLS+6YIyiata0LA8INRR6VZeYEC+Px6NAIGBofnCeSKbUAwAAAACLBEuxR6EHANYrLy/XpA8/1tuTPpDf77d7nT+Jj6ujxx9+UC2aN7V7FVfJysnRK2+8bVle05QmOv7Y4ZblAaFmz14KvcMRFhamBvXqKXvHDiPzGzVMNjLXbsF5/hAAAAAAHOq3Yu/YYUPsXuWIUOgBgPVmz52vsZdeoTfemeS4Qq9+vbp6duJjFHqHqbKyUg//a6Jl125K0pWXXMSf34AhvxZ64yn0DlPf3j2MzI2Ojlbnjh2MzLYbJ/UAAAAAwGJuPbFHoQcA1tq8dZuee+lVzZ2/wO5VDii5QQM98eiDSkpqYPcqrvPeBx9p2cpVluX17N5VPXt0sywPCCW/FXobN2XavYrrDBrQX5988VW1z+3ft7fCwoKz/grO/1YAAAAA4HBuK/Yo9ADAOgWFhXrjnUn6/Muv5K+stHudA2rRvKkee+BeJSYm2r2K66xJX6fX337PsrywsDBddfmlluUBoYRC7+ikdeqo7l3StHDJ0mqb6fP5dP45Z1XbPKfh+k0AAAAAsIlbruKk0AMAa/j9fk35eqrOveASffL5l44t9Hp07aJnJz5GoXcEiouLde+Dj1p6jeo5/zxdKY0bWZYHhAoKvepx2UUXyOervv+f8Y+TRwXte3oSJ/UAAAAAwFZOP7FHoQcA5gUCAc3+ZZ5efPUNbdm2ze51DurE40fohmuu4M+FI/T4U88pKyfHsrz69erq3LPOsCwPCBUUetWnVWpL3XTdVXpkwpNHPatDu7a6dNzYo57jZJR6AAAAAGAzpxZ7FHoAYFYgENCceQv0xjvvaV3GervXOSiPx6Nx54/WmCC+0sy0Kd98q2kzZlqaee2V/6eakZGWZgLBjkKv+p1w7DHauTNXb7zzngKBwBHNaN2qpR66906Fh4dX83bOQqkHAAAAAA7gtGKPQg8AzFq4ZKlefv1NrU3PsHuVQ6oZGanxN1+vQf372b2Ka63fuEnPvPiSpZkD+vVRv949Lc0Egh2FnjljR5+tJo0a6l9PPK2S0tLD+ndHDB+qG6+5UjVq1DC0nXNQ6gEAAACAQzil2KPQAwBzFi5Zqldef0tr0tfZvUqV1E1M0EP33qnWqal2r+JaBYWFuv3eB1RWVm5ZZmRkDV1x6UWW5QGhgELPvKGDB6pLWme9+c4kfTX1W+3bt++gv7592za6dNxYpXXqaM2CDkCpBwAAAAAOYnexR6EHANWvsrJSc+bO17uTP9Tqtel2r1NlnTq00/133aG4OrXtXsW1/H6/7nngEWVnW/eOniRdOGa0khs0sDQTCGYUetaJq1Nb1111uS4Zd77mLVikRUuXKndXnvL25KlmZE3VTUxUyxbNNKBvHzVqmGz3upaj1AMAAAAAh7Gr2KPQA4DqVVFRoekzZ+m9Dz5S5patdq9zWEaNPE7XXXG5wsL4+PBovPz6m1q4ZKmlme3btdUZp55saSYQzCj07BEdFaWhgwZo6KABdq/iKPypDAAAAAAOZHWxR6EHANUnLy9fX3z9jT754isVFBTYvc5hiapZUzdfd7WGDh5o9yqu99PsOZr88WeWZkZG1tD4G6+Tz8ef50B1cEqhl9SgvrJzdti6A5yBUg8AAAAAHOq3Ym97do5WrV5jLKdN61YUegBQDTI2bNTnX32tb7//URUVFXavc9hSGjfS/XeNV9OUFLtXcb30jAw9+OjjCgQCluZePHaMGjdqaGkmEKycUugNHtBPUdHRyp76na17wBko9QAAAADAwXw+n0487lijpd7JJ55AoQcAR6i8vFxz5i3QlK//bfk1i9VpxPChuv7qK1QzMtLuVVwvNzdXt939gErLyizN7dCurU47eZSlmUCw2u/f75hC7+7bb9HjTz1n6x5wDko9AAAAAHA4r9drdr7HY3Q+AASj9IwMffnNVP3w40yVlJbavc4Rq1UrWjdefSXXbVaTktJS3XTH3crNzbU0NzKyhsbfdD3XbgLVJHPzVpVZXMz/1W+FHj98hz+i1AMAAAAAAACqoKioWNNnztLnX32tjA0b7V7nqHVJ66Q7br5RdRMT7F4lKPj9ft1574PasDHT8uzLLrpQjRomW54LBCsKPTgVpR4AAAAAAADwN8rKyjX7l7n6YeZPmrdgkfbt22f3SketZmSkLr1wrE45aaTx0+ChZOIzz2nB4iWW5/bv00unjhppeS4AMyj0cDCUegAAAAAAAMAf+P1+LVm+Qt9+/4Nmzf7F1ddr/lVap466+bqrOdVVzV598x1N+eZby3MTExN1yw3XysNV2kBQoNDDoVDqAQAAAAAAIOT5/X4tXrZc02fM0szZs1VYWGT3StUqJqaWrrz0Yh13zDAKoGr26edT9PakyZbner1e3XHz9aodG2t5NoDqR6GHqqDUAwAAAAAAQEgqLSvT4qXLNeOnWZo9d56KiortXsmIvr166IZrruLtPAO+nfajnnrhJVuyzzv7n+qa1tmWbADVi0IPVUWpBwAAAAAAgJCRvWOHFi5cotlz52r+oiXav3+/3SsZEx9XR5dfdKFGHDPM7lWC0s9z5uqRCU8oEAhYnt22dSuNHX2O5bkAqh+FHg4HpR4AAAAAAACCVklpqZYsW6GFi5do0eIlytyy1e6VjPN6vTr+mOG6/JILFRsTY/c6QWn+oiW6+8FH5K+stDw7NjZW995xm3w+CgDA7Sj0cLgo9QAAAAAAABA0/H6/1m/cqEWLl2nh4iVaumJlUJ/G+6u2rVvp6ssvUft2be1eJWjNX7REt99zv/bt22d5ts/n01233KgG9etZng2gelHo4UhQ6gEAAAAAAMC1ysvLtSZ9nZavXKUVq9doxYpVKikttXstyyXVr69LL7pAQwb2l8fjsXudoLVg8a+FXnl5uS35l1w4Vj17dLMlG0D1odDDkaLUAwAAAAAAgGvk5e/R6jVrfy/x1qavk9/vt3st29SMjNTos/+pM/9xqiIiIuxeJ6gtWLxE4++2r9Ab2K+vzjr9VFuyAVQfCj0cDUo9AAAAAAAAOFJxSYk2bMzUuowMpWesV3rGem3eslWBQMDu1Wzn8Xg0eEA//d8lF6l+vbp2rxP0frty065Cr3mzprr9lhs4hQm43OAB/XT3eAo9HDlKPQAAAAAAANiqsrJSOTt2aMPGTG3M3KyNmzYpPWODsrKz7V7NkbqmddblF12o1q1a2r1KSPj5l3m658FHVFFRYUt+TEwtPXj3HaoZGWlLPoDq8Xuh56PQw5Gj1AMAAAAAAIAl/JV+5eTs1JZtW7Vl63Zt3rJVGzZuUubmLSotK7N7PcdL69xR48aMVueOHexeJWR8/e13mvDks7Zd8erz+XTP7beqYXKSLfkAqgeFHqoLpR4AAAAAAACqzZ69e7Vz5y7t2LlLO3ftUs6OncrKydGWLVu1PTtH+/fvt3tF1+ncsYPGjRmttM4d7V4lpHzy+Zd6+oWXbb3u9ZorLlWPrl1sywdw9Cj0UJ0o9QAAAAAAAHBIZWXlKigs1N69e5W/Z6/y8vKVs3Ondu7cpZ27dmrHzl3K2bnLtjfHglGnDu104ZjR6prW2e5VQs6kDz7Wi6+9YesO5555hk45caStOwA4OhR6qG5GS73du/M05eupJiMAAAAAIOitXLPG6Pyly1do/z5OzgChap9/v8pKy1RcUqKKigqVlJWqpKRUhYWF2rO3QHv37tXeggKVlVHWWaVDu7a68PzR6t4lze5VQo6/0q+Jzzxv+2eagwf008UXjLF1BwBHh0IPJhgt9bZu367HnnrGZAQAAAAA4Cj9+7tp+vd30+xeAwBCmtfrVe8e3XT6aadQ5tmkpLRUdz3wsOYvWGTrHq1btdT4m26Q1+u1dQ8AR45CD6Zw/SYAAAAAAABgk5qRkRo+bLD+edqpSmncyO51Qlb2jh265Y57lLl5i617JCU10L8euE+RkTVs3QPAkRvUn0IP5lDqAQAAAAAAABaLj4/TySOP1z9OOUmxMTF2rxPS1qZn6Na77lFe/h5b94irU1sTHrxPcXVq27oHgCM3qH8/3XM7hR7ModQDAAAAAAAALNK6VUud+Y9TNWTgAD70dYAfps/Uo088ZfubkdHR0Zrw8ANq3KihrXsAOHKDB/TT3bffIp+X7+0wh1IPAAAAAAAAMCg6KkpDhwzUiGHD1KlDO7vXgSS/369X3nxb73/4iQKBgK27REbW0KP336PUFs1t3QPAkfv9yk0KPRhGqQcAAAAAAAAY0Do1VSedcJyGDxusmpGRdq+D/ygoLNR9Dz2q+YuW2L2KwsLC9MCdt1P2Ai7GlZuwEqUeAAAAAAAAUE1iYmpp8MD+Om3UKLVo3tTudfAXmzI367a771dWdrbdq8jn8+mu225Wzx7d7F4FwBGi0IPVKPUAAAAAAACAoxAREaHePbrrmGGD1a93L4WF8ZGbE333w3Q9/tSzKi0rs3sV+Xw+3XrDtRo8oJ/dqwA4QhR6sAN/wwAAAAAAAAAOk8/nU7s2rXXc8GEaOmSgoqOi7F4Jf6OiokIvvPqGPvn8S7tXkfTfQm/E8KF2rwLgCFHowS6UegAAAAAAAEAVeL1etW/bRkMG9tfQwYMUH1fH7pVwCNu2Z+mu+x/S+o2b7F5FEoUeEAwo9GAnSj0AAAAAAADgb/i8XnXu1FH9+/TSoAH9VTcxwe6VUEXTZszU408+q+KSErtXkfRroTf+put0zNAhdq8C4AhR6MFulHoAAAAAAADAH9SMjFSXzh01ZOAA9e3dSzExtexeCYehuLhYE595Qd//ON3uVX7n8/l0240UeoCbUejBCSj1AOD/27v36L7rOs/j7/xC+OXSJE3S0qRJ07RNWnqjVwpIsVxX5DYMiFtQxAFlFWVxR1kcFhnRFWeV2dXdPTNHZxkZGF10Pa4zzjLqgDAgjOUi9Iq1pS1taYHSK7ZJaJPuHxyZcYCQpvnl8/0lj8c5OfmjPZ/fKyf9o83z9/0UAACAEW98Y2OccvKiWHzKSTH3hNl+aFukVq95Nr74X/40tm3fnnrKG4455pi45abPxJlLTks9BRggQY+sEPUAAAAAGHEqystj5vTjY+H8ebFg/pyY1tGRehJHoae3J+7+9r1x93e+Gz09PannvKGivDy+eOt/ikUL56eeAgyQoEeWiHoAAAAADHulpaXRPnlyLJg/JxbOnxdzZ8+KY47xo7HhYN365+LLd/y3WL9hY+opv6Omujq+8qXbYsbx01JPAQZI0CNr/M0FAAAAgGGnvDwfx0+dGifMmhknzJwRs2fPjIry8tSzGEQ9PT3x3e//3/hff3VPHDp0KPWc31FfXxd/evt/jimT21JPAQZI0COLRD0AAAAAil5VVVVMn9oRC+fPi1kzZ8T0aR1RVlaWehYFsvbX6+P2O/5rbNz0fOopb9LSPD7u+PIXY3xjY+opwADVVFcLemSSqAcAAABAUamsqIipHe1x/NSOmNbRHsdPmxrN45tSz2IIdHV1x11//Z347vd/ED29vannvMnsmTPi9ts+F7U1NamnAEehtaVZ0COTRD0AAAAAMqumujomT26LKZMmvR7xpnZEa0tz5HK51NMYYo8teyK+/j//PLa/9FLqKW9pyeJT45abPh35fD71FOBolZSkXgBvSdQDAAAAILljjz022ia2xqS2iTG5bWJMmTQpJk+aGGMaGlJPI7FXdu6Mb9x5V/zk/p+lnvK2Lr34orj+Yx8VmwEoKFEPAAAAgCFRUlISY8c0xISWlmhtaYnWCc0xoaUlJkxojnFjxwoi/I6enp74/g//Nr5197fjQGdn6jlvqTSXiz+8/hNx4fnnpp4CwAgg6gEAAAAwKMrKymLMmIZobmyMhob6aGioj+bGpmga3xhj6uti3LhxUVFennomReDxJ38Z/+PPvxHPb9maesrbqq4eFX9882dj0YJ5qacAMEKIegAAAAC8rYry8qgbPTpqa2uitrb29c/V1VFbWxOja0dHXd3oGHfc2Bg3dkzU1NSknkuRe2Hb9vjmX94VDz7889RT+tTa0hJfuu1zMXFCS+opAIwgBY16E5qbY+n7LinkSwAAAGRWb29v/NPjT8T+A/uP6pxdu/fElq0vDNKqN5vQ0hz1daOP6oyqyqo4ZdGJrs6DIlNRWR6lpcdEaWkuKisro6K8PPL5fFRVVERFZUVUVVZGWVlZ6pmMAPv374977v1e/J8f/E0cPHgw9Zw+nbb4XXHLf/y0p04BGHIFjXoNDfXukwYAAEa0+fPmxA03/lG8snNn6ilva8vWF44qGtbXjY6vfeX2aJs4cRBXATAS9PT0xH0//oe48+57YtfuPann9KmkpCQuf/+lce0fXOVNLAAk4fpNAACAAprQ0hxf/+qXMx/2BkrQA2AgDh8+HA/848PxF9+6O7ZvfzH1nHc0alRV3Hzjp2PxKSelngLACCbqAQAAFNhwDXuCHgAD8fgTT8Vf3HVPrF23LvWUfpnW0RFfuOWz0dTUmHoKACOcqAcAADAEhlvYqxtdK+gBcERWrFoTd959Tzz9zIrUU/rtPWefGZ+54ZORz+dTTwEAUQ8AAGCoTGhpjq999fb41I03F3XYqxtdG1//6pcFPQD6ZfWzv4p7vnNvPLbsidRT+q2yoiI+8x+uj7NPX5J6CgC8QdQDt9/jpgAAEBVJREFUAAAYQq0tLUUd9gQ9APrrmRUr457vfDee+OXTqacckfbJk+K2W/4oJrQ0p54CAL9D1AMAABhixRr2BD0A+mPFqjXxrXu+HU89/UzqKUekpKQkLvm9C+O6j14dZWVlqecAwJuIegAAAAkUW9gT9ADoS09PTzzy2C/ir+/9Xvx63frUc47Y2DENcfONfxgL5s1NPQUA3paoBwAAkEixhD1BD4C3c/DgwfjZQw/HPf/7e7F569bUcwZkyeJT48ZPfTJqampSTwGAPol6AAAACWU97Al6ALyVXbv3xA/+5kfxw7+7L/bt25d6zoCMGlUVN1z3sXjP2WemngIA/SLqAQAAJJbVsCfoAfCvrV23Lv72vh/HT+9/MLq7u1PPGbCTFi2MG2+4Po4bOyb1FADoN1EPAAAgA7IW9gQ9AH7rQGdnPPCzf4wf/t3/i3XPbUg956iMGlUVH7/m6rjw/HNTTwHot56entQTyAhRDwAAICOyEvYEPQAiIp7fsjX+/qf/ED+678fx6qu/ST3nqJ2yaFHc+KlPxJgxns4DikdnV1cse+Kp1DPICFEPAAAgQ1KHPUEPYGQ7ePBgPPTIz+OHP7ovVq5ek3rOoKivr4vr/91H46wzlqSeAnBEOru64qZbPh+7du9OPYWMEPUAAAAyJlXYE/QARq7nNmyKn9z/QPz9T++Pvfv2pZ4zKHK5XFx03nvj2quvilGjqlLPATgivw16z6xYmXoKGSLqAQAAZNBQhz1BD2DkeWXnznjo4Z/HT+5/MNauW5d6zqCaMrktPvPvPxkzZ0xPPQXgiAl6vB1RDwAAIKN+G/Y+dv2n4zf7C/d/GVVVVgh6ACNEZ1dXPPzIo/HTBx6MXy5fET09PaknDaqK8vK4+qoPxvt+/6IozZWmngNwxAQ9+iLqAQAAZFhrS0tcsfTS+Oadf1Ww17ji/ZcJegDDWG9vb6xa86v4yQMPxP0PPBSdXV2pJw26kpKSOP20U+O6az8S444bm3oOwIAIerwTUQ8AACDjGurqC3r+mDENBT0fgKHX09MTy1euikcefSweeuSx2LlrV+pJBTNtanvccN3HYparNoEiJujRH6IeAAAAAAwDr732WqxYvSYe/adl8eDDj8SuXbtTTyqompqa+PAHlsbvX3RBlJa6ahMoXoIe/SXqAQAAAECROtDZGb9Y9kQ8/Ohj8YvHn4wDnZ2pJxVcPp+PSy++MK5c+v6oqqpKPQfgqAh6HAlRDwAAAACKyI5XdsayJ56MRx79RTz59NNx8ODB1JOGRC6Xi3PPPiuuvuqDcdzYMannABw1QY8jJeoBAAAAQIb19PbE6jVr47Fly+KpXy6PX69fH4cPH049a0gtnDc3Pn7tNdExZXLqKQCDQtBjIEQ9AAAAAMiYF7Ztj2VPPhWPP/FkPL18ZXR2daWelMTMGdPj2g9/KObNPSH1FIBBI+gxUKIeAAAAACS2e8/eWL5yVTyzfEUse/KpeGHb9tSTkprW0RHXXPWBOHnRiamnAAwqQY+jIeoBAAAAwBDbtXtPLF+5KlasWh2rVj87Iq/UfCttEyfGH1x5RZx+2qlRUlKSeg7AoDrQ2Rk33nxrrFy9JvUUipSoBwAAAAAF9uJLL8fylati+YpV8czKlbH1hW2pJ2XKlMltceXlS+P0006NXC6Xeg7AoOvs6orP3nqboMdREfUAAAAAYBB1dnXFuvUb4tfr1sXadetjxcrVsf2ll1LPyqSOKZPjg5f/W0/mAcNaZ1dX3PS5z8czy125ydER9QAAAABggHp6emLDpufj2V+tjTXPro01a9fG5i1bo7e3N/W0TJs7Z3Z86IqlsXDe3NRTAApK0GMwiXoAAAAA0A+vvvqbWP/chnhu48ZY/9zGWL9xY2zYuCkOHTqUelpRyOVysfiUk+Pyyy6JmTOmp54DUHCCHoNN1AMAAACAf+G1116LzVtfiM2bt8SGTc/Hcxs2xHMbN8WLL72celpRKi/Px3vPOSfef+nF0Ty+KfUcgCEh6FEIoh4AAAAAI9K+ffti89YXYtPmzbFly9bY9PzW2LRlc7z00suuzxwE9fV1cclFF8TFF5wXNTU1qecADBlBj0IR9QAAAAAYlg4dOhQv79gRW7e9GNu3b49t2198/ePF1z/v378/9cRhp6SkJOacMCsuvuD8WLL4XVFaWpp6EsCQEvQoJFEPAAAAgKLS29sbe/e9Gnv37o09e/fGzp274uVXdsbLO3bEy6+8Ejt27IgdO3bGrt274/Dhw6nnjghVVVXxnnPOjIsvOD/aWiekngOQhKBHoYl6AAAAAAy5nt6eOLC/M/YfOBBd3d3R3d0Vr/5mf3R3vxZdnZ2xZ9+rsW/f3tizZ1/s3rM7du/ZG3v37Yu9e/fG3n2vuh4zI6Z2tMfFF5wXZ52xJCrKy1PPAUhG0GMolDS3z/B2JQAAAACgX8rKymLxKSfFhee/NxbOm5t6DjBCnXne78WhQ4cKcvasGdPjz752R79/v6DHUPGkHgAAAADwjiZPaovz3nNOnHvOWVFTXZ16DkAmCHoMJVEPAAAAAHhLDfX1cfq7F8cZ7z4tTpg1I/UcgEwR9Bhqoh4AAAAA8IZ8Ph/vOunE+DdnnxUnL1oQpbnS1JMAMkfQIwVRDwAAAABGuFwuFzOnHx/nnn1WnHXmkqisqEg9CSCzBD1SEfUAAAAAYISa1tERZ5+5JM4+Y0k01NenngOQeYIeKYl6AAAAADBC5HK56JgyJd518olxzplnREvz+NSTAIqGoEdqoh4AAAAADGOlpaUx4/hpcca7F8cZS07zRB7AAAh6ZIGoBwAAAADDTD6fjwVzT4gz3n1anHrKyTFqVFXqSQBFS9AjK0Q9AAAAABgGxh03NhaduCBOXrgwTlwwP8rL86knARQ9QY8sEfUAAAAAoAiV5nIxY/rxcerJJ8WC+XNiant7lJSUpJ4FMGx0dXVnIujl8/no7u5OuoFsEPUAAAAAoEg0NTXGwvlzY+G8eXHSiQuisqIi9SSAYamrqztuujV90Jsze1bs3L07tm59IekOskHUAwAAAICMqq+vi7mzZ8XcObPjpAULoqmpMfUkgGGvt/dw3HTr5+PpZ1Yk3TFn9qz4ypdui4984oakO8gOUQ8AAAAAMqK+vi7mzJoZs2fOiNmzZrhSEyCB57dsjv37DyTd8NugV1FennQH2SLqAQAAAEAiDfX1ccKsGbFw3ryYPWt6tE2cmHoSwIgn6JFVoh4AAAAADIGK8vJonzI5pna0x/Ed7TF79swY3+g6TQD+maBHX0Q9AAAAABhkpaWlMaGlOaZ1tMe0jvaY2tERM6dPi9LS0tTTAMgoQY93IuoBAAAAwFE49thjY9LEidExZXK0T54UU6ZMiqkd7X4oC0C/CXr0h6gHAAAAAP1UXT0q2lpb37hCs62tNSa3tUVZWVnqaQAUKUGP/hL1AAAAAOBfaaivj0kTW6NpfGNMam2NtraJMWliazTU16eeBsAwIuhxJEQ9AAAAAEackpKSGNNQH+ObmmJ8U2OMb2qKlvFN0do6IVpbmiOfz6eeCMAwJ+hxpEQ9AAAAAIad0tLSqKuri8bjxsZxY8fEmIaGGN/YGE3jx8X4pvExvnGcKzMBSEbQYyBEPQAAAACKQi6Xi9qa6qitrY3ampoYXVsb9XV1UVtbHaNramN03egYO6YhGseNi/r60VGaK009GQDeRNBjoAoa9SorKmJCS0shXwIAACCzDh8+HN3d3alnDJl8Ph8lJSWpZwBZV3I4qqtGRURErjQXVZWVUVlRGeXlx0a+vDyqR42K8nw+8vl81NRUR21NzesftbVRW1MduVwu8RcAAAOXxaA3btxxUVbqGbDBsGPnzoL+G7Cg36WpHe3x3+/4k0K+BAAAQKbd+/0fxJ99887UMwruIx++Mj50xdLUMwAAILOyGPQiIq66fGk0NNSnnjEsfOMv74oNGzcV7HxvbQIAACigpe+7JK679prUMwpK0AMAgL5lNehRXEQ9AACAAhvOYU/QAwCAvs2ZNVPQY1CIegAAAENgOIY9QQ8AAPo2Z9bM+MrtXxD0GBSiHgAAwBAZTmFP0AMAgL4Jegw2UQ8AAGAIDYewJ+gBAEDfBD0KQdQDAAAYYsUc9gQ9AADom6BHoYh6AAAACRRj2BP0AACgb4IehSTqAQAAJFJMYU/QAwCAvgl6FJqoBwAAkFAxhD1BDwAA+jZv7gnx1du/KOhRUKIeAABAYlkOe4IeAAD0bc6smfEnX/jjKC/Pp57CMCfqAQAAZMDS910SH/9otsKeoAcAAH1z5SZDSdQDAADIiMsvy07YE/QAAKBvgh5DTdQDAADIkCyEPUEPAAD6JuiRgqgHAACQMSnDnqAHAAB9E/RIRdQDAADIoBRhT9ADAIC+CXqkJOoBAABk1FCGPUEPAAD6JuiRmqgHAACQYUMR9gQ9AADom6BHFoh6AAAAGVfIsCfoAQBA3wQ9skLUAwAAKAKFCHuCHgAA9E3QI0tEPQAAgCIxmGFP0AMAgL4JemSNqAcAAFBEBiPsCXoAANA3QY8sEvUAAACKzNGEPUEPAAD6VlVZKeiRSaIeAABAERpI2BP0AADgnU1snSDokUmiHgAAQJE6krAn6AEAQP/kctIJ2eRPJgAAQBHrT9gT9AAAAIqfqAcAAFDkXg97V7/lr11zlaAHAAAwHIh6AAAAw8Dll136prB3zVVXxlUfEPQAAACGA1EPAABgmPiXYU/QAwAAGF6OST0AAACAwXP5ZZfGrBnTY/bMGamnAAAAMIg8qQcAADDMCHoAAADDj6gHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaVNLfPOFyow6uqqqJ9UluhjgcAAAAAAEagFavXxOHDhckbVVVVMX1qR0HOHoiVz/4quru6Cnb+xNYJcWxZWcHOH0m2btsWnZ2F+14VNOoBAAAAAAAAR8/1mwAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABknKgHAAAAAAAAGSfqAQAAAAAAQMaJegAAAAAAAJBxoh4AAAAAAABk3P8HXsQsVumGjLkAAAAASUVORK5CYII=",
          fileName="modelica://ClaRa/Resources/Images/Components/FurnaceSliceUncooledcarrierTubes.png"),
        Bitmap(
          visible = not FTWisCooled and not CTisCooled,
          extent={{-300,-100},{300,100}},
          imageSource="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"
               + "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",
          fileName="modelica://ClaRa/Resources/Images/Components/FurnaceSliceUncooledWallsAndarrierTubes.png"),
        Rectangle(
          extent={{264,-74},{292,-92}},
          lineColor={28,108,200},
          pattern=LinePattern.None,
          fillColor={0,131,169},
          fillPattern=FillPattern.Solid,
          visible=frictionAtInlet_FTW and FTWisCooled),
        Rectangle(
          extent={{264,92},{292,74}},
          lineColor={28,108,200},
          fillColor={0,131,169},
          fillPattern=FillPattern.Solid,
          pattern=LinePattern.None,
          visible=frictionAtOutlet_FTW and FTWisCooled),
        Rectangle(
          extent={{-4,74},{4,90}},
          lineColor={28,108,200},
          pattern=LinePattern.None,
          fillColor={0,131,169},
          fillPattern=FillPattern.Solid,
          visible=frictionAtInlet_CT and CTisCooled),
        Rectangle(
          extent={{-4,-74},{4,-92}},
          lineColor={28,108,200},
          pattern=LinePattern.None,
          fillColor={0,131,169},
          fillPattern=FillPattern.Solid,
          visible=frictionAtOutlet_CT and CTisCooled),
        Rectangle(
          extent={{-274,54},{-294,46}},
          lineColor={28,108,200},
          pattern=LinePattern.None,
          fillColor={0,131,169},
          fillPattern=FillPattern.Solid,
          visible=frictionAtInlet_TB),
        Rectangle(
          extent={{-278,-46},{-294,-54}},
          lineColor={28,108,200},
          pattern=LinePattern.None,
          fillColor={0,131,169},
          fillPattern=FillPattern.Solid,
          visible=frictionAtOutlet_TB)}), Diagram(graphics,
                                                  coordinateSystem(preserveAspectRatio=false)));
end ConvectiveSlice_L4;
