﻿within ClaRa.Components.VolumesValvesFittings.Valves;
model ThreeWayValveGas_L1_simple "Three way valve for flue gas with controllable mass fraction"
//__________________________________________________________________________//
// Component of the ClaRa library, version: 1.9.0                           //
//                                                                          //
// Licensed by the ClaRa development team under the 3-clause BSD License.   //
// Copyright  2013-2024, ClaRa development team.                            //
//                                                                          //
// The ClaRa development team consists of the following partners:           //
// TLK-Thermo GmbH (Braunschweig, Germany),                                 //
// XRG Simulation GmbH (Hamburg, Germany).                                  //
//__________________________________________________________________________//
// Contents published in ClaRa have been contributed by different authors   //
// and institutions. Please see model documentation for detailed information//
// on original authorship and copyrights.                                   //
//__________________________________________________________________________//

//  extends ClaRa.Basics.Icons.ComplexityLevel(complexity="1");

  extends ClaRa.Basics.Interfaces.DataInterfaceVectorGas(final N_sets=2,eye(
                                                                           each medium =       medium));

  parameter TILMedia.Gas.Types.BaseGas medium=simCenter.flueGasModel "Medium in the component"
    annotation (choicesAllMatching, Dialog(group="Fundamental Definitions"));

 model FlangeGas "A summary record for gas flanges"
    extends ClaRa.Basics.Icons.RecordIcon;
    input ClaRa.Basics.Units.MassFlowRate m_flow "Mass flow rate" annotation (Dialog);
    input ClaRa.Basics.Units.Temperature T "Temperature" annotation (Dialog);
    input ClaRa.Basics.Units.Pressure p "Pressure" annotation (Dialog);
 end FlangeGas;

 inner model Summary
   FlangeGas inlet;
   FlangeGas outlet;
   FlangeGas outlet2;
 end Summary;

  outer ClaRa.SimCenter simCenter;

  parameter Boolean splitRatio_input=false "= true, if split ratio is defined by input";
  parameter Real splitRatio_fixed = 0.5 annotation(Dialog(enable=not splitRatio_input));
  Real splitRatio;

  Basics.Interfaces.GasPortIn              inlet(Medium=
                       medium) "Inlet port"
    annotation (Placement(transformation(extent={{-110,-10},{-90,10}}),
        iconTransformation(extent={{-110,-10},{-90,10}})));
  Basics.Interfaces.GasPortOut        outlet(Medium=
                       medium) "Outlet port"
    annotation (Placement(transformation(extent={{90,-10},{110,10}}),
        iconTransformation(extent={{90,-10},{110,10}})));

  Basics.Interfaces.GasPortOut       outlet2(Medium=
                       medium) annotation (Placement(transformation(extent={{-10,
            -110},{10,-90}}), iconTransformation(extent={{-10,-110},{10,-90}})));
  Modelica.Blocks.Interfaces.RealInput splitRatio_external(min=0,max=1)=splitRatio if splitRatio_input "Controls mass fraction m2/m1"
    annotation (Placement(transformation(
        extent={{20,-20},{-20,20}},
        rotation=90,
        origin={0,31}), iconTransformation(
        extent={{20,-20},{-20,20}},
        rotation=90,
        origin={0,90})));

  inner Summary    summary(inlet(m_flow=inlet.m_flow,  T=actualStream(inlet.T_outflow), p=inlet.p),
                           outlet(m_flow=-outlet.m_flow,  T=actualStream(outlet.T_outflow), p=outlet.p),
                           outlet2(m_flow=-outlet2.m_flow,  T=actualStream(outlet2.T_outflow), p=outlet2.p))
  annotation (Placement(transformation(extent={{-60,-102},{-40,-82}})));

equation
  // Pressure drop in design flow direction
  outlet2.p = inlet.p;

  // Isothermal state transformation. Isenthalpic behaviour can not be modeled using ideal gas substance properties.
  inlet.T_outflow *(-inlet.m_flow)= (inStream(outlet.T_outflow) *outlet.m_flow + inStream(outlet2.T_outflow)*outlet2.m_flow);// Please note: This valve is not designed for back flows. Please consider this as dummy value!
  outlet.T_outflow = inStream(inlet.T_outflow);
  outlet2.T_outflow = inStream(inlet.T_outflow);

  if (not splitRatio_input) then
    splitRatio = splitRatio_fixed;
  end if;

  // mass balance (no storage)
  inlet.m_flow+outlet.m_flow+outlet2.m_flow=0;
  -outlet.m_flow=splitRatio*inlet.m_flow;

// No chemical reaction taking place:
  zeros(medium.nc-1) =inlet.xi_outflow  *inlet.m_flow +(inStream(outlet.xi_outflow)*outlet.m_flow + inStream(outlet2.xi_outflow)*outlet2.m_flow);
  outlet.xi_outflow   = inStream(inlet.xi_outflow);
  outlet2.xi_outflow   = inStream(inlet.xi_outflow);

 for i in 1:2 loop

    eye[i].T= inlet.T_outflow-273.15;
    eye[i].s=-1;
    eye[i].p=inlet.p/1e5;
    eye[i].h=-1;
    eye[i].xi=outlet.xi_outflow;
  end for;
    eye[1].m_flow=-outlet.m_flow;
    eye[2].m_flow=-outlet2.m_flow;
annotation (
  Icon(coordinateSystem(
        preserveAspectRatio=false,
        extent={{-100,-100},{100,80}},
        grid={2,2}), graphics={Bitmap(
          extent={{-100,-100},{100,70}},                                    visible = not splitRatio_input,
          imageSource=
              "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",
          fileName="modelica://ClaRa/Resources/Images/Components/ThreeWayValve.png"),
                           Text(
          extent={{-100,-58},{100,-100}},
          lineColor={27,36,42},
          fillColor={0,0,0},
          fillPattern=FillPattern.Solid,
          textString="L1")}),
  Diagram(coordinateSystem(
        preserveAspectRatio=false,
        extent={{-100,-100},{100,80}},
        grid={2,2}), graphics),
    Documentation(info="<html>
<p><b>For detailed model documentation please consult the html-documentation shipped with ClaRa.</b> </p>
<p>&nbsp;</p>
<p><br><b><span style=\"font-size: 10pt;\">Authorship and Copyright Statement for original (initial) Contribution</span></b></p>
<p><b>Author:</b> </p>
DYNCAP/DYNSTART development team, Copyright &copy; 2011-2024.</p>
<p><b>References:</b> </p>
<p> For references please consult the html-documentation shipped with ClaRa. </p>
<p><b>Remarks:</b> </p>
<p>This component was developed by ClaRa development team under the 3-clause BSD License.</p>
<b>Acknowledgements:</b>
<p>ClaRa originated from the collaborative research projects DYNCAP and DYNSTART. Both research projects were supported by the German Federal Ministry for Economic Affairs and Energy (FKZ 03ET2009 and FKZ 03ET7060).</p>
<p><b>CLA:</b> </p>
<p>The author(s) have agreed to ClaRa CLA, version 1.0. See <a href=\"https://claralib.com/pdf/CLA.pdf\">https://claralib.com/pdf/CLA.pdf</a></p>
<p>By agreeing to ClaRa CLA, version 1.0 the author has granted the ClaRa development team a permanent right to use and modify his initial contribution as well as to publish it or its modified versions under the 3-clause BSD License.</p>
<p>The ClaRa development team consists of the following partners:</p>
<p>TLK-Thermo GmbH (Braunschweig, Germany)</p>
<p>XRG Simulation GmbH (Hamburg, Germany).</p>
</html>",
revisions=
        "<html>
<body>
<p>For revisions please consult the html-documentation shipped with ClaRa.</p>
</body>
</html>"));
end ThreeWayValveGas_L1_simple;
