﻿within ClaRa.Components.Electrical;
model SimpleGenerator "A simple active power generator"
// This core of this model is taken over from Francesco Casella's Modelica library ThermoPower.
// The ClaRa developer's team appreciates this input

outer ClaRa.SimCenter simCenter;

  ClaRa.Basics.Interfaces.Connected2SimCenter connected2SimCenter(
    powerIn = 0,
    powerOut_elMech = summary.P_el,
    powerOut_th = 0,
    powerAux = 0) if contributeToCycleSummary;

  parameter Boolean contributeToCycleSummary = false "True if component shall contribute to automatic efficiency calculation" annotation(Dialog(tab="Summary and Visualisation"));

  parameter Real eta(min=0) = 0.98 "Conversion efficiency (electrical + mechanical losses)" annotation(Dialog(group="Fundamental Definitions"));
  parameter Integer N_pole_pairs(min=1) = 1 "Number of electrical pole pairs" annotation(Dialog(group="Fundamental Definitions"));

  parameter Boolean hasInertia = false "Model accounts for rotational inertia" annotation(Dialog(group="Mechanics"));
  parameter ClaRa.Basics.Units.MomentOfInertia J=1500 "Moment of inertia" annotation (Dialog(group="Mechanics", enable=hasInertia));

  parameter ClaRa.Basics.Units.Frequency f_start=50 "Initial grid frequency" annotation (Dialog(enable=hasInertia, group="Initialisation"));

protected
  ClaRa.Basics.Units.Energy E_rot "Rotational kinetic energy";
  ClaRa.Basics.Units.AngularVelocity omega "Shaft angular velocity";
  ClaRa.Basics.Units.RPM rpm "Shaft rotational speed";
  ClaRa.Basics.Units.Frequency f(start=f_start) "Electrical frequency";

public
  record Summary
    extends ClaRa.Basics.Icons.RecordIcon;
    ClaRa.Basics.Units.Power P_shaft "Shaft power";
    ClaRa.Basics.Units.Power P_el "Electrical power";
    ClaRa.Basics.Units.RPM rpm "Rotational speed of shaft";
    ClaRa.Basics.Units.Frequency f "Electrical frequency";
  end Summary;

  ClaRa.Basics.Interfaces.ElectricPortOut powerConnection "Electric connection" annotation (Placement(
        transformation(extent={{72,-14},{100,14}}, rotation=0),
        iconTransformation(extent={{90,-10},{110,10}})));
  Modelica.Mechanics.Rotational.Interfaces.Flange_a shaft "Mechanical shaft"
    annotation (Placement(transformation(extent={{-100,-14},{-72,14}}, rotation=
           0), iconTransformation(extent={{-110,-11},{-90,9}})));
  Summary summary(P_shaft= shaft.tau*omega,
                  P_el=-powerConnection.P,
                  rpm=rpm, f=f);

equation
  omega=der(shaft.phi) "Mechanical boundary condition";
  f = N_pole_pairs*omega/(2*Modelica.Constants.pi) "Electrical frequency";
  rpm = (30/Modelica.Constants.pi)*omega "Rotational speed in rpm";
  powerConnection.f = f "Electrical boundary condition";
  if hasInertia then
    E_rot = 1/2*J*omega^2 "Kinetic energy";
    der(E_rot) = omega*shaft.tau + powerConnection.P/eta "Energy balance";
  else
    E_rot = 0 "Kinetic energy";
    0 = omega*shaft.tau+powerConnection.P/eta "Energy balance";
  end if annotation (Diagram(graphics));
    annotation (Documentation(info="<html>
<p><b>For detailed model documentation please consult the html-documentation shipped with ClaRa.</b> </p>
<p>&nbsp;</p>
<p><br><b><span style=\"font-size: 10pt;\">Authorship and Copyright Statement for original (initial) Contribution</span></b></p>
<p><b>Author:</b> </p>
DYNCAP/DYNSTART development team, Copyright &copy; 2011-2024.</p>
<p><b>References:</b> </p>
<p> For references please consult the html-documentation shipped with ClaRa. </p>
<p><b>Remarks:</b> </p>
<p>This component was developed by ClaRa development team under the 3-clause BSD License.</p>
<b>Acknowledgements:</b>
<p>ClaRa originated from the collaborative research projects DYNCAP and DYNSTART. Both research projects were supported by the German Federal Ministry for Economic Affairs and Energy (FKZ 03ET2009 and FKZ 03ET7060).</p>
<p><b>CLA:</b> </p>
<p>The author(s) have agreed to ClaRa CLA, version 1.0. See <a href=\"https://claralib.com/pdf/CLA.pdf\">https://claralib.com/pdf/CLA.pdf</a></p>
<p>By agreeing to ClaRa CLA, version 1.0 the author has granted the ClaRa development team a permanent right to use and modify his initial contribution as well as to publish it or its modified versions under the 3-clause BSD License.</p>
<p>The ClaRa development team consists of the following partners:</p>
<p>TLK-Thermo GmbH (Braunschweig, Germany)</p>
<p>XRG Simulation GmbH (Hamburg, Germany).</p>
</html>",
  revisions="<html>
<body>
<p>For revisions please consult the html-documentation shipped with ClaRa.</p>
</body>
</html>"),
   Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
            -100},{100,100}}), graphics={Bitmap(
          extent={{-100,-100},{100,100}},
          imageSource="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"
               + "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",
          fileName="modelica://ClaRa/Resources/Images/Components/Generator.png")}),
                       Diagram(coordinateSystem(preserveAspectRatio=false,
          extent={{-100,-100},{100,100}}), graphics));
end SimpleGenerator;
