﻿within ClaRa.Basics.Functions;
function InitialiseCombustionGas "A simple function to find feasible start values for combustion product gases"
 //___________________________________________________________________________//
 // Component of the ClaRa library, version: 1.9.0                            //
 //                                                                           //
 // Licensed by the DYNCAP/DYNSTART research team under the 3-clause BSD License.   //
 // Copyright  2013-2021, DYNCAP/DYNSTART research team.                      //
 //___________________________________________________________________________//
 // DYNCAP and DYNSTART are research projects supported by the German Federal //
 // Ministry of Economic Affairs and Energy (FKZ 03ET2009/FKZ 03ET7060).      //
 // The research team consists of the following project partners:             //
 // Institute of Energy Systems (Hamburg University of Technology),           //
 // Institute of Thermo-Fluid Dynamics (Hamburg University of Technology),    //
 // TLK-Thermo GmbH (Braunschweig, Germany),                                  //
 // XRG Simulation GmbH (Hamburg, Germany).                                   //
 //___________________________________________________________________________//
  import M_i = ClaRa.Basics.Constants;

  input Real[6] y_fuel_in "elemental mole (or mass) fraction of fuel";
  input Real m_flow_fuel "massflow of fuel";
  input Real[9] xi_combustionGas_in "mass fraction of combustion gas (e.g. air)";
  input Real m_flow_combustionGas "massflow of combustion gas";
  input Boolean moleFraction "choose type of fuel composition. true = molar / false = mass based";
  output Real[9] xi_exhaustGas;

protected
  Real assertion = 0;
  Real xi_combustionGas[10] "full size (nc) array of components mass fraction";
  Real y_fuel[7] "full size (nc) array of elemental composition in fuel";
  Real M_i_fuel[7] = {
  M_i.M_C,
  M_i.M_H,
  M_i.M_O,
  M_i.M_N,
  M_i.M_S,
  M_i.M_Ash,
  M_i.M_H2O} "molar masses of fuel components C, H, O, N, S, Ash, H2O";
//   Real M_i_combustionGas[10] = {
//   M_i.M_Ash,
//   M_i.M_C + M_i.M_O,
//   M_i.M_C + M_i.M_O2,
//   M_i.M_S + M_i.M_O2,
//   2*M_i.M_N,
//   M_i.M_O2,
//   M_i.M_N + M_i.M_O2,
//   M_i.M_H2O,
//   M_i.M_N + 3*M_i.M_H,
//   0.039948} "molar masses of combustion gas components Ash, CO, CO2, SO2, N2, O2, NO2, H2O, NH3, Ar"; //falsch, NO nicht NO2!!!

  Real M_i_combustionGas[10] = {
  M_i.M_Ash,
  M_i.M_C + M_i.M_O,
  M_i.M_C + M_i.M_O2,
  M_i.M_S + M_i.M_O2,
  2*M_i.M_N,
  M_i.M_O2,
  M_i.M_N + M_i.M_O,
  M_i.M_H2O,
  M_i.M_N + 3*M_i.M_H,
  0.039948} "molar masses of combustion gas components Ash, CO, CO2, SO2, N2, O2, NO2, H2O, NH3, Ar";

  //stoichiometry of fuel components      Ash,CO,CO2,SO2,N2,O2,NO2,H2O,NH3,Ar //falsch, NO nicht NO2!!!
  // NH3 is treated inert -> not regarded in molar balance (see 9th column) And Ar too!!!
//   Real[7,10] stoichiometry_combustionGas = [0, 1, 1, 0, 0, 0, 0, 0, 0, 0;
//                                             0, 0, 0, 0, 0, 0, 0, 2, 0, 0;
//                                             0, 1, 2, 2, 0, 2, 2, 1, 0, 0;
//                                             0, 0, 0, 0, 2, 0, 1, 0, 0, 0;
//                                             0, 0, 0, 1, 0, 0, 0, 0, 0, 0;
//                                             1, 0, 0, 0, 0, 0, 0, 0, 0, 0;
//                                             0, 0, 0, 0, 0, 0, 0, 0, 0, 1];

  Real[7,10] stoichiometry_combustionGas = [0, 1, 1, 0, 0, 0, 0, 0, 0, 0;
                                            0, 0, 0, 0, 0, 0, 0, 2, 0, 0;
                                            0, 1, 2, 2, 0, 2, 1, 1, 0, 0;
                                            0, 0, 0, 0, 2, 0, 1, 0, 0, 0;
                                            0, 0, 0, 1, 0, 0, 0, 0, 0, 0;
                                            1, 0, 0, 0, 0, 0, 0, 0, 0, 0;
                                            0, 0, 0, 0, 0, 0, 0, 0, 0, 0];

  //stoichiometry of combustion gas components
                                 //C  H  O  N  S Ash H2O
  Real[7,7] stoichiometry_fuel = [ 1, 0, 0, 0, 0, 0, 0;
                                   0, 1, 0, 0, 0, 0, 2;
                                   0, 0, 1, 0, 0, 0, 1;
                                   0, 0, 0, 1, 0, 0, 0;
                                   0, 0, 0, 0, 1, 0, 0;
                                   0, 0, 0, 0, 0, 1, 0;
                                   0, 0, 0, 0, 0, 0, 0];

  //notice the vector of elemental composition's last entry after multiplikation with the stoichiometric matrix is Ar, NOT water!

  Real[7] molarFlowRate "molar flowrate of elements introduced in combustion chamber";
  Real[10] y_exh "molefraction of exhaust gas' components";
  Real[10] molarFlowRate_exhaustGas "molar flowrate of components in exhaust gas";

algorithm
  //write last entry to combustionGas composition vector (assuming its norm is 1)
  for i in 1:9 loop
    xi_combustionGas[i] :=xi_combustionGas_in[i];
  end for;
    xi_combustionGas[10] :=1 - sum(xi_combustionGas_in);

  //write last entry to fuel composition vector (assuming its norm is 1)
  for i in 1:6 loop
    y_fuel[i] := y_fuel_in[i];
  end for;
    y_fuel[7] := 1 - sum(y_fuel_in);

  //calculate the elemental molar flowrate fed into the combustion chamber either from molar, or from mass-based composition vector
    if moleFraction then
     molarFlowRate :=m_flow_fuel/(y_fuel*M_i_fuel)*stoichiometry_fuel*y_fuel +
       m_flow_combustionGas * stoichiometry_combustionGas*(xi_combustionGas ./ M_i_combustionGas);
    else
     molarFlowRate := m_flow_fuel * stoichiometry_fuel*(y_fuel ./ M_i_fuel) +
       m_flow_combustionGas * stoichiometry_combustionGas*(xi_combustionGas ./ M_i_combustionGas);
    end if;

  // molar balances
  // notice ammonia is treated as inert component -> its molar flowrate is calculated directly from feed flow. (9th component) //Combustion gas inlet flow not regarded??? Argon not correct (here set to water)
   molarFlowRate_exhaustGas[1] := molarFlowRate[6];                                       //Asche
   molarFlowRate_exhaustGas[2] := 0;                                                      //CO
   molarFlowRate_exhaustGas[3] := molarFlowRate[1];                                       //CO2
   molarFlowRate_exhaustGas[4] := molarFlowRate[5];                                       //SO2
   molarFlowRate_exhaustGas[7] := 0;                                                      //NO2
   molarFlowRate_exhaustGas[5] := 0.5*(molarFlowRate[4]-molarFlowRate_exhaustGas[7]);     //N2
   molarFlowRate_exhaustGas[9] := xi_combustionGas[9]/M_i_combustionGas[9]*m_flow_combustionGas;//NH3
   molarFlowRate_exhaustGas[8] := 0.5*(molarFlowRate[2]); //H2O
   molarFlowRate_exhaustGas[6] := 0.5*(molarFlowRate[3] - molarFlowRate_exhaustGas[2] - 2*molarFlowRate_exhaustGas[3]- 2*molarFlowRate_exhaustGas[4] - molarFlowRate_exhaustGas[7] - molarFlowRate_exhaustGas[8]); //O2
   molarFlowRate_exhaustGas[10] := (1-sum(xi_combustionGas))/M_i_combustionGas[10]*m_flow_combustionGas;//molarFlowRate[7];                                      //Ar

//   // molar balances
//   // notice ammonia is treated as inert component -> its molar flowrate is calculated directly from feed flow. (9th component) //Combustion gas inlet flow not regarded??? Argon not correct (here set to water)
//   molarFlowRate_exhaustGas[1] := molarFlowRate[6] + xi_combustionGas[1]/M_i_combustionGas[1]*m_flow_combustionGas;                                       //Asche
//   molarFlowRate_exhaustGas[2] := 0;                                                      //CO
//   molarFlowRate_exhaustGas[3] := molarFlowRate[1] + xi_combustionGas[3]/M_i_combustionGas[3]*m_flow_combustionGas;                                       //CO2
//   molarFlowRate_exhaustGas[4] := molarFlowRate[5] + xi_combustionGas[4]/M_i_combustionGas[4]*m_flow_combustionGas;                                       //SO2
//   molarFlowRate_exhaustGas[7] := 0;                                                      //NOx
//   molarFlowRate_exhaustGas[5] := 0.5*(molarFlowRate[4]-molarFlowRate_exhaustGas[7]) + xi_combustionGas[5]/M_i_combustionGas[5]*m_flow_combustionGas;     //N2
//   molarFlowRate_exhaustGas[9] := xi_combustionGas[9]/M_i_combustionGas[9]*m_flow_combustionGas;//NH3
//   molarFlowRate_exhaustGas[8] := 0.5*(molarFlowRate[2]) + xi_combustionGas[8]/M_i_combustionGas[8]*m_flow_combustionGas; //H2O
//   molarFlowRate_exhaustGas[6] := xi_combustionGas[6]/M_i_combustionGas[6]*m_flow_combustionGas - 0.5*(molarFlowRate[3] - molarFlowRate_exhaustGas[2] - 2*molarFlowRate_exhaustGas[3]- 2*molarFlowRate_exhaustGas[4] - 2*molarFlowRate_exhaustGas[7] - molarFlowRate_exhaustGas[8]); //O2
//   molarFlowRate_exhaustGas[10] := (1-sum(xi_combustionGas))/M_i_combustionGas[10]*m_flow_combustionGas;                                     //Ar

  // compute full size (nc) vector of composition in outlet
  for i in 1:10 loop
    y_exh[i] :=molarFlowRate_exhaustGas[i]/sum(molarFlowRate_exhaustGas);
    if y_exh[i] < 0 then
      assertion :=assertion + 10^i;
    end if;
  end for;

  assert(assertion<= 0, "At least one of your initial mass fractions is negative. Probably lambda is less than 1 in instance " + getInstanceName() +". The value of assertation is " + String(assertion));

  // pass part-size (nc-1) vector back
  for i in 1:9 loop
  xi_exhaustGas[i] := y_exh[i] .* M_i_combustionGas[i] ./ sum(y_exh.*M_i_combustionGas);
  end for;
    annotation (Documentation(info="<html>
<p><b>For detailed model documentation please consult the html-documentation shipped with ClaRa.</b> </p>
<p>&nbsp;</p>
<p><br><b><span style=\"font-size: 10pt;\">Authorship and Copyright Statement for original (initial) Contribution</span></b></p>
<p><b>Author:</b> </p>
DYNCAP/DYNSTART development team, Copyright &copy; 2011-2024.</p>
<p><b>References:</b> </p>
<p> For references please consult the html-documentation shipped with ClaRa. </p>
<p><b>Remarks:</b> </p>
<p>This component was developed by ClaRa development team under the 3-clause BSD License.</p>
<b>Acknowledgements:</b>
<p>ClaRa originated from the collaborative research projects DYNCAP and DYNSTART. Both research projects were supported by the German Federal Ministry for Economic Affairs and Energy (FKZ 03ET2009 and FKZ 03ET7060).</p>
<p><b>CLA:</b> </p>
<p>The author(s) have agreed to ClaRa CLA, version 1.0. See <a href=\"https://claralib.com/pdf/CLA.pdf\">https://claralib.com/pdf/CLA.pdf</a></p>
<p>By agreeing to ClaRa CLA, version 1.0 the author has granted the ClaRa development team a permanent right to use and modify his initial contribution as well as to publish it or its modified versions under the 3-clause BSD License.</p>
<p>The ClaRa development team consists of the following partners:</p>
<p>TLK-Thermo GmbH (Braunschweig, Germany)</p>
<p>XRG Simulation GmbH (Hamburg, Germany).</p>
</html>",
  revisions="<html>
<body>
<p>For revisions please consult the html-documentation shipped with ClaRa.</p>
</body>
</html>"),
   Icon(graphics={Bitmap(
          extent={{-100,-100},{100,100}},
          imageSource=
              "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"
               +
              "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",
          fileName="modelica://ClaRa/Resources/Images/Components/Function.png")}));
end InitialiseCombustionGas;
