﻿within ClaRa.Basics.ControlVolumes.Fundamentals.Geometry;
model HollowCylinderWithTubes "Cylindric shape || Shell with tubes"
//__________________________________________________________________________//
// Component of the ClaRa library, version: 1.9.0                           //
//                                                                          //
// Licensed by the ClaRa development team under the 3-clause BSD License.   //
// Copyright  2013-2024, ClaRa development team.                            //
//                                                                          //
// The ClaRa development team consists of the following partners:           //
// TLK-Thermo GmbH (Braunschweig, Germany),                                 //
// XRG Simulation GmbH (Hamburg, Germany).                                  //
//__________________________________________________________________________//
// Contents published in ClaRa have been contributed by different authors   //
// and institutions. Please see model documentation for detailed information//
// on original authorship and copyrights.                                   //
//__________________________________________________________________________//

  extends ClaRa.Basics.ControlVolumes.Fundamentals.Geometry.ShellWithTubes;
  extends ClaRa.Basics.ControlVolumes.Fundamentals.Geometry.GenericGeometry(
    final volume = pi/4*diameter^2*length - N_tubes*N_passes*pi/4*diameter_t^2*length_tubes,
    final N_heat=2,
    final A_heat={
          pi*diameter*length + 2* diameter^2/4*pi,
          pi*diameter_t*N_tubes*N_passes*length_tubes},
    final A_cross = if parallelTubes then A_front - N_passes*N_tubes*pi/4*diameter_t^2 else psi*A_front,
    final A_front = if N_baffle > 0 then diameter*length/(N_baffle + 1) else if flowOrientation==orientation then pi/4*diameter^2 else diameter*length,
    final A_hor=if orientation == ClaRa.Basics.Choices.GeometryOrientation.horizontal then diameter*length*interior else pi/4*diameter^2*interior,
    final height_fill=if orientation == ClaRa.Basics.Choices.GeometryOrientation.horizontal then diameter else length,
    final shape=if orientation == ClaRa.Basics.Choices.GeometryOrientation.vertical then [0, 1; 1, 1]
                else [0.0005, 0.02981;
                      0.0245, 0.20716;
                      0.1245, 0.45248;
                      0.2245, 0.58733;
                      0.3245, 0.68065;
                      0.4245, 0.74791;
                      0.5245, 0.7954;
                      0.6245, 0.8261;
                      0.7245, 0.84114;
                      0.8245, 0.84015;
                      0.9245, 0.82031;
                      1, 0.7854]);

  parameter ClaRa.Basics.Choices.GeometryOrientation orientation=ClaRa.Basics.Choices.GeometryOrientation.horizontal "Orientation of the component"  annotation(Dialog(group="Essential Geometry Definition"));

  parameter ClaRa.Basics.Choices.GeometryOrientation flowOrientation=ClaRa.Basics.Choices.GeometryOrientation.vertical "Orientation of the mass flow" annotation(Dialog(group="Essential Geometry Definition"));

  parameter Units.Length diameter=1 "Diameter of the component" annotation (Dialog(
      tab="General",
      group="Essential Geometry Definition",
      showStartAttribute=false,
      groupImage="modelica://ClaRa/Resources/Images/ParameterDialog/HollowCylinderWithTubes.png",
      connectorSizing=false));

  parameter Units.Length length=1 "Length of the component" annotation (Dialog(group="Essential Geometry Definition"));

  parameter Units.Length diameter_t=0.1 "Outer diameter of internal tubes" annotation (Dialog(
      tab="General",
      group="Interior Equipment",
      showStartAttribute=false,
      groupImage="modelica://ClaRa/Resources/Images/ParameterDialog/HEX_ParameterDialogTubes.png"));
  parameter Units.Length length_tubes=1 "Length of the internal tubes (single pass)" annotation (Dialog(group="Interior Equipment"));

  parameter Integer N_tubes=1 "Number of internal tubes" annotation(Dialog(group="Interior Equipment"));
  parameter Integer N_passes=1 "Number of passes of the internal tubes" annotation(Dialog(group="Interior Equipment"));
  parameter Boolean parallelTubes=false "True, if tubes are parallel to main flow orientation, else false" annotation(Dialog(group="Interior Equipment"));

  parameter Integer N_baffle=0 "Number of baffles on shell side"
                                                                annotation(Dialog(group="Interior Equipment"));

  final parameter Real interior(
    min=1e-6,
    max=1) = volume/(pi/4*diameter^2*length) "Volume fraction of interior equipment" annotation(Dialog(group="Interior Equipment"));

  constant Real MIN=1e-5 "Limiter";
  parameter Modelica.Units.SI.Length Delta_z_ort=2*diameter_t "Distance between tubes orthogonal to flow direction (center to center)" annotation (Dialog(group="Interior Equipment"));
  parameter Modelica.Units.SI.Length Delta_z_par=2*diameter_t "Distance between tubes parallel to flow direction (center to center)" annotation (Dialog(group="Interior Equipment"));
  final parameter Real a=Delta_z_ort/diameter_t "Lateral alignment ratio" annotation(Dialog(group="Interior Equipment"));
  final parameter Real b=Delta_z_par/diameter_t "Vertical alignment ratio" annotation(Dialog(group="Interior Equipment"));
  final parameter Real psi=if b >= 1 or b<=0 then 1 - Modelica.Constants.pi/4/a else 1 - Modelica.Constants.pi/4/a/b "Void ratio" annotation(Dialog(group="Interior Equipment"));
  parameter Boolean staggeredAlignment=true "True, if the tubes are aligned staggeredly, false otherwise";
  final parameter Real fa=if staggeredAlignment then (1 + (if b>0 then 2/3/b else 0)) else (1 + (if b>0 then 0.7/max(MIN,psi)^1.5*(b/a - 0.3)/(b/a + 0.7)^2 else 0)) "Alignment factor";

  parameter Integer N_rows(
    min=N_passes,
    max=N_tubes) = integer(ceil(sqrt(N_tubes))*N_passes) "Number of pipe rows in flow direction (minimum = N_passes)" annotation(Dialog(group="Interior Equipment"));

equation
   for i in 1:N_inlet loop
     assert(if integer(height_fill) <> -1 then z_in[i]<=height_fill else true, "Position of inlet flange no. " +String(i)+ "("+String(z_in[i], significantDigits=3)+" m) must be below max. fill height of "+ String(height_fill, significantDigits=3) + " m in component " +  getInstanceName() + ".");
   end for;
   for i in 1:N_outlet loop
     assert(if integer(height_fill) <> -1 then z_out[i]<=height_fill else true, "Position of outlet flange no. " +String(i)+ "("+String(z_out[i], significantDigits=3)+" m) must be below max. fill height of "+ String(height_fill, significantDigits=3) + " m in component " +  getInstanceName() + ".");
   end for;
  for i in 1:N_inlet loop
    assert(z_in[i]>=0, "Position of inlet flange no. " +String(i)+ "("+String(z_in[i], significantDigits=3)+" m) must be positive in component " +  getInstanceName() + ".");
  end for;
  for i in 1:N_outlet loop
    assert(z_out[i]>=0, "Position of outlet flange no. " +String(i)+ "("+String(z_out[i], significantDigits=3)+" m) must be positive in component " +  getInstanceName() + ".");
  end for;
  assert(psi>0, "Negative or zero psi leads to invalid Reynolds numbers. Check geometry values for pipe arrangement. Delta_z_par*delta_z_ort must be > pi*diameter_o^2/4.");
  assert(A_cross>0, "The cross section of the shell side must be > 0 but is "+String(A_cross, significantDigits=3) + " in instance" + getInstanceName() + ".");
  assert(volume>0, "The volume of the shell side must be > 0 but is "+String(volume, significantDigits=3) + " in instance" + getInstanceName() + ".");
    annotation (Documentation(info="<html>
<p><b>For detailed model documentation please consult the html-documentation shipped with ClaRa.</b> </p>
<p>&nbsp;</p>
<p><br><b><span style=\"font-size: 10pt;\">Authorship and Copyright Statement for original (initial) Contribution</span></b></p>
<p><b>Author:</b> </p>
DYNCAP/DYNSTART development team, Copyright &copy; 2011-2024.</p>
<p><b>References:</b> </p>
<p> For references please consult the html-documentation shipped with ClaRa. </p>
<p><b>Remarks:</b> </p>
<p>This component was developed by ClaRa development team under the 3-clause BSD License.</p>
<b>Acknowledgements:</b>
<p>ClaRa originated from the collaborative research projects DYNCAP and DYNSTART. Both research projects were supported by the German Federal Ministry for Economic Affairs and Energy (FKZ 03ET2009 and FKZ 03ET7060).</p>
<p><b>CLA:</b> </p>
<p>The author(s) have agreed to ClaRa CLA, version 1.0. See <a href=\"https://claralib.com/pdf/CLA.pdf\">https://claralib.com/pdf/CLA.pdf</a></p>
<p>By agreeing to ClaRa CLA, version 1.0 the author has granted the ClaRa development team a permanent right to use and modify his initial contribution as well as to publish it or its modified versions under the 3-clause BSD License.</p>
<p>The ClaRa development team consists of the following partners:</p>
<p>TLK-Thermo GmbH (Braunschweig, Germany)</p>
<p>XRG Simulation GmbH (Hamburg, Germany).</p>
</html>",
revisions=
        "<html>
<body>
<p>For revisions please consult the html-documentation shipped with ClaRa.</p>
</body>
</html>"),
 Icon(graphics={Bitmap(
          extent={{-100,-100},{100,100}},
          imageSource=
            "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"
             +
            "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"
             +
            "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",
          fileName="modelica://ClaRa/Resources/Images/Components/HollowCylinderWithTubes.png")}));
end HollowCylinderWithTubes;
