within Buildings.Electrical.AC.OnePhase.Sources;
model Generator "Model of a generator"
  extends Buildings.Electrical.Interfaces.Source(
    redeclare package PhaseSystem = PhaseSystems.OnePhase,
    redeclare replaceable Interfaces.Terminal_p terminal);
  parameter Modelica.Units.SI.Frequency f(start=60) "Frequency of the source";
  parameter Modelica.Units.SI.Angle phiGen(displayUnit="deg") = 0
    "Phase shift of the source";
  Modelica.Units.SI.Current I "value of current";
  Modelica.Blocks.Interfaces.RealInput P
    "Variable power generated by the source"
     annotation (Placement(transformation(
        extent={{-20,-20},{20,20}},
        origin={-100,0}),iconTransformation(
        extent={{-20,-20},{20,20}},
        origin={-100,0})));
protected
  Modelica.Units.SI.Angle thetaRel
    "Absolute angle of rotating system as offset to thetaRef";
equation
  if Connections.isRoot(terminal.theta) then
      PhaseSystem.thetaRef(terminal.theta) =  2*Modelica.Constants.pi*f*time;
  end if;

  thetaRel = PhaseSystem.thetaRel(terminal.theta);

  terminal.i = PhaseSystem.phaseCurrents(I, thetaRel+phiGen);
  0 = PhaseSystem.activePower(terminal.v, terminal.i) + P;

  annotation (
      defaultComponentName="gen",
      Icon(coordinateSystem(preserveAspectRatio=false, extent={{-100,
            -100},{100,100}}),
                         graphics={Rectangle(extent={{-100,100},{100,-100}},
            lineColor={255,255,255}),
        Rectangle(
          extent={{-100,100},{100,-100}},
          lineColor={0,0,0},
          fillColor={255,255,255},
          fillPattern=FillPattern.Solid),
        Polygon(
          points={{-52,48},{-52,-52},{60,-16},{60,12},{-52,48}},
          lineColor={0,0,0},
          smooth=Smooth.None,
          fillColor={255,255,255},
          fillPattern=FillPattern.Solid),
        Text(
          extent={{-120,130},{120,100}},
          textColor={0,0,0},
          textString="%name"),
        Line(visible = definiteReference == true,
          points={{-20,-128},{20,-128}},
          color={0,120,120},
          smooth=Smooth.None,
          thickness=0.5),
        Line(visible = definiteReference == true,
          points={{-20,-128},{6,-102}},
          color={0,120,120},
          smooth=Smooth.None,
          thickness=0.5),
        Line(visible = definiteReference == true,
          points={{2,-104},{14,-112},{18,-130}},
          color={0,120,120},
          smooth=Smooth.Bezier)}),
    Documentation(info="<html>
<p>
Model of a generator that produces real power
<code>P</code> with a given phase shift <code>phiGen</code>.
</p>
<p>
This model must be used with
<a href=\"modelica://Buildings.Electrical.AC.OnePhase.Sources.Grid\">
Buildings.Electrical.AC.OnePhase.Sources.Grid</a>
or with a voltage source from the package
<a href=\"modelica://Modelica.Electrical.QuasiStatic.SinglePhase.Sources\">
Modelica.Electrical.QuasiStatic.SinglePhase.Sources</a>.
Otherwise, there will be no equation that defines the phase
angle of the voltage.
</p>
<p>
The boolean flags <code>potentialReference</code> and
<code>definiteReference</code> allows to select the phase angle of the
generator as reference for the models connected to it. More info about
these flags are available in
<a href=\"modelica://Buildings.Electrical.AC.UsersGuide\">
Buildings.Electrical.AC.UsersGuide</a>.
</p>
</html>", revisions="<html>
<ul>
<li>
January 29, 2019, by Michael Wetter:<br/>
Added <code>replaceable</code> for terminal.
</li>
<li>
September 4, 2014, by Michael Wetter:<br/>
Revised model.
</li>
<li>
August 5, 2014, by Marco Bonvini:<br/>
Revised documentation.
</li>
<li>
January 4, 2012, by Michael Wetter:<br/>
First implementation.
</li>
</ul>
</html>"));
end Generator;
