within Buildings.Templates.Components.Interfaces;
partial model PartialChiller
  "Interface class for chiller models"
  extends Buildings.Fluid.Interfaces.PartialFourPortInterface(
    redeclare final package Medium1=MediumCon,
    redeclare final package Medium2=MediumChiWat,
    final m1_flow_nominal=mCon_flow_nominal,
    final m2_flow_nominal=mChiWat_flow_nominal);
  replaceable package MediumChiWat=Buildings.Media.Water
    constrainedby Modelica.Media.Interfaces.PartialMedium
    "CHW medium"
    annotation (__ctrlFlow(enable=false));
  replaceable package MediumCon=Buildings.Media.Water
    constrainedby Modelica.Media.Interfaces.PartialMedium
    "Medium model for condenser cooling fluid"
    annotation (__ctrlFlow(enable=false));
  parameter Buildings.Templates.Components.Types.Chiller typ
    "Type of chiller"
    annotation (Evaluate=true,
    Dialog(group="Configuration",
      enable=false));
  parameter Boolean have_switchover=false
    "Set to true for heat recovery chiller with built-in switchover"
    annotation (Evaluate=true,
    Dialog(group="Configuration", enable=false));
  parameter Boolean use_TChiWatSupForCtl=true
    "Set to true for CHW supply temperature control, false for CHW return temperature control"
    annotation (Evaluate=true,
    Dialog(group="Configuration", enable=false));
  parameter Buildings.Templates.Components.Data.Chiller dat(
    typ=typ)
    "Design and operating parameters"
    annotation (Placement(transformation(extent={{70,80},{90,100}})),
    __ctrlFlow(enable=false));
  final parameter Modelica.Units.SI.MassFlowRate mChiWat_flow_nominal=dat.mChiWat_flow_nominal
    "Design CHW mass flow rate";
  final parameter Modelica.Units.SI.MassFlowRate mCon_flow_nominal=dat.mCon_flow_nominal
    "Design condenser cooling fluid mass flow rate";
  final parameter Modelica.Units.SI.HeatFlowRate cap_nominal=dat.cap_nominal
    "Design cooling capacity";
  final parameter Modelica.Units.SI.HeatFlowRate QChiWat_flow_nominal=-abs(cap_nominal)
    "Design cooling heat flow rate";
  final parameter Modelica.Units.SI.PressureDifference dpChiWat_nominal=dat.dpChiWat_nominal
    "Design CHW pressure drop";
  final parameter Modelica.Units.SI.PressureDifference dpCon_nominal=dat.dpCon_nominal
    "Design condenser cooling fluid pressure drop";
  final parameter Modelica.Units.SI.Temperature TChiWatSup_nominal=dat.TChiWatSup_nominal
    "Design CHW supply temperature";
  final parameter Modelica.Units.SI.Temperature TChiWatRet_nominal=
    TChiWatSup_nominal - QChiWat_flow_nominal / cpChiWat_default / mChiWat_flow_nominal
    "Design CHW return temperature";
  // Derived classes must provide bindings for the condenser variables and COP at design.
  parameter Modelica.Units.SI.HeatFlowRate QCon_flow_nominal
    "Design condenser heat flow rate";
  parameter Modelica.Units.SI.Temperature TConEnt_nominal
    "Design condenser entering fluid temperature";
  parameter Modelica.Units.SI.Temperature TConLvg_nominal
    "Design condenser leaving fluid temperature";
  parameter Modelica.Units.SI.Efficiency COP_nominal
    "Coefficient of performance at design cooling conditions";
  parameter Boolean have_dpChiWat=true
    "Set to true for CHW pressure drop computed by this model, false for external computation"
    annotation (Evaluate=true,
    Dialog(tab="Assumptions"));
  parameter Boolean have_dpCon=true
    "Set to true for condenser cooling fluid pressure drop computed by this model, false for external computation"
    annotation (Evaluate=true,
    Dialog(tab="Assumptions",
      enable=typ==Buildings.Templates.Components.Types.Chiller.WaterCooled));
  parameter Modelica.Units.SI.Time tau=30
    "Time constant at nominal flow, used if energy or mass balance is dynamic"
    annotation (Dialog(tab="Dynamics",group="Nominal condition",
      enable=energyDynamics<>Modelica.Fluid.Types.Dynamics.SteadyState),
    __ctrlFlow(enable=false));
  parameter Modelica.Fluid.Types.Dynamics energyDynamics=Modelica.Fluid.Types.Dynamics.DynamicFreeInitial
    "Type of energy balance: dynamic (3 initialization options) or steady state"
    annotation (Evaluate=true,
    Dialog(tab="Dynamics",group="Conservation equations"),
    __ctrlFlow(enable=false));
  final parameter MediumChiWat.SpecificHeatCapacity cpChiWat_default=
    MediumChiWat.specificHeatCapacityCp(staChiWat_default)
    "CHW default specific heat capacity";
  final parameter MediumChiWat.ThermodynamicState staChiWat_default=MediumChiWat.setState_pTX(
    T=TChiWatSup_nominal,
    p=MediumChiWat.p_default,
    X=MediumChiWat.X_default)
    "CHW default state";
  final parameter MediumCon.SpecificHeatCapacity cpCon_default=MediumCon.specificHeatCapacityCp(staCon_default)
    "Condenser cooling fluid default specific heat capacity";
  final parameter MediumCon.ThermodynamicState staCon_default=MediumCon.setState_pTX(
    T=dat.TCon_nominal,
    p=MediumCon.p_default,
    X=MediumCon.X_default)
    "Condenser cooling fluid default state";
  Buildings.Templates.Components.Interfaces.Bus bus
    "Control bus"
    annotation (Placement(transformation(extent={{-20,80},{20,120}}),
      iconTransformation(extent={{-20,80},{20,120}})));
  annotation (
    Documentation(
      info="<html>
<p>
This partial class provides a standard interface for chiller models.
</p>
</html>",
      revisions="<html>
<ul>
<li>
April 17, 2025, by Antoine Gautier:<br/>
Refactored for load-dependent 2D table data chiller model.
</li>
<li>
November 18, 2022, by Antoine Gautier:<br/>
First implementation.
</li>
</ul>
</html>"),
    Icon(
      graphics={
        Rectangle(
          extent={{100,60},{-100,-60}},
          lineColor={0,0,0},
          lineThickness=1),
        Bitmap(
          extent={{-20,60},{20,100}},
          fileName=
            "modelica://Buildings/Resources/Images/Templates/Components/Boilers/ControllerOnboard.svg"),
        Text(
          extent={{-60,20},{60,-20}},
          textColor={0,0,0},
          textString="CHI")}));
end PartialChiller;
