package PowerPlants "Models of thermoelectrical power plants components" 
  import SI = Modelica.SIunits;
  import ThermoPower.Choices.Init.Options;
annotation (uses(ThermoPower(version="2"), Modelica(version="3.0-development")),
      Documentation(revisions="<html>
</html>"));
  
  package GasTurbine 
    "Models and tests of the gas turbine and its main components" 
    package Interfaces "Interface definitions" 
      
      partial model GasTurbine "Base class for Gas Turbine" 
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package AirMedium = ThermoPower.Media.Air extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package FuelMedium = ThermoPower.Media.NaturalGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        ThermoPower.Gas.FlangeA FuelInlet(redeclare package Medium = FuelMedium)
          annotation (extent=[-20,180; 20,220]);
        ThermoPower.Gas.FlangeA AirInlet(redeclare package Medium = AirMedium) 
          annotation (extent=[-220,140; -180,180]);
        ThermoPower.Gas.FlangeB FlueGasOutlet(redeclare package Medium = 
              FlueGasMedium) 
          annotation (extent=[180,140; 220,180]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        annotation (Diagram,                 Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-200,12; 200,-12],style(
                color=76,
                rgbcolor={170,170,255},
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Ellipse(extent=[-30,150; 30,90],style(
                color=9,
                rgbcolor={175,175,175},
                gradient=3,
                fillColor=1,
                rgbfillColor={255,0,0})),
            Polygon(points=[-30,30; -40,30; -40,120; -30,120; -30,30],
                                                                     style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[30,30; 40,30; 40,120; 30,120; 30,30],
                                                                style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[-130,112; -120,106; -120,166; -200,166; -200,154;
                  -130,154; -130,112],
                style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[130,112; 120,108; 120,166; 200,166; 200,154; 130,
                  154; 130,112],                                                style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[-130,120; -30,40; -30,-40; -130,-120; -130,120],
                                                                       style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[130,120; 30,40; 30,-40; 130,-120; 130,-120; 130,120],
                                                                          style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[4,150; -4,150; -8,200; 8,200; 4,150],    style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255}))),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        ThermoPower.PowerPlants.Buses.Sensors SensorsBus 
                                 annotation (extent=[180,-120; 220,-80]);
        ThermoPower.PowerPlants.Buses.Actuators ActuatorsBus 
                                     annotation (extent=[220,-180; 180,-140]);
      end GasTurbine;
      
      partial model GasTurbineSimplified 
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas 
          extends Modelica.Media.Interfaces.PartialMedium;
        ThermoPower.Gas.FlangeB flueGasOut(redeclare package Medium = 
              FlueGasMedium)               annotation (extent=[90,70; 110,90]);
        annotation (Diagram, Icon(
                            Rectangle(extent=[-100,100; 100,-100], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-44,8; 38,-6],    style(
                color=76,
                rgbcolor={170,170,255},
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Ellipse(extent=[-20,80; 20,40], style(
                color=9,
                rgbcolor={175,175,175},
                gradient=3,
                fillColor=1,
                rgbfillColor={255,0,0})),
            Polygon(points=[-20,18; -24,24; -24,64; -20,64; -20,18], style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[20,18; 24,24; 24,64; 20,64; 20,18], style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[80,60; 76,56; 76,82; 100,82; 100,78; 80,78; 80,60], style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[-80,70; -20,30; -20,-30; -80,-70; -80,70], style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255})),
            Polygon(points=[80,70; 20,30; 20,-30; 80,-70; 80,-70; 80,70], style(
                color=10,
                rgbcolor={135,135,135},
                gradient=3,
                fillColor=76,
                rgbfillColor={170,170,255}))));
        Modelica.Blocks.Interfaces.RealInput GTLoad "GT unit load in p.u." 
          annotation (extent=[-112,-12; -88,12],
                                               rotation=0);
      end GasTurbineSimplified;
    end Interfaces;
    
    package Examples "Example implementations" 
      
      model GasTurbineSimplified 
        extends 
          ThermoPower.PowerPlants.GasTurbine.Interfaces.GasTurbineSimplified;
        parameter Modelica.SIunits.Power maxPower=235e6;
        parameter Modelica.SIunits.MassFlowRate flueGasNomFlowRate=614 
          "Nominal flue gas flow rate";
        parameter Modelica.SIunits.MassFlowRate flueGasMinFlowRate=454 
          "Minimum flue gas flow rate";
        parameter Modelica.SIunits.MassFlowRate flueGasOffFlowRate=
            flueGasMinFlowRate/100 "Flue gas flow rate with GT switched off";
        parameter Modelica.SIunits.MassFlowRate fuelNomFlowRate=12.1 
          "Nominal fuel flow rate";
        parameter Modelica.SIunits.MassFlowRate fuelIntFlowRate=7.08 
          "Intermediate fuel flow rate";
        parameter Modelica.SIunits.MassFlowRate fuelMinFlowRate=4.58 
          "Minimum fuel flow rate";
        parameter Modelica.SIunits.MassFlowRate fuelOffFlowRate=0.1 
          "Flue gas flow rate with GT switched off";
        parameter Real constTempLoad = 0.60 
          "Fraction of load from which the temperature is kept constant";
        parameter Real intLoad = 0.42 
          "Intermediate load for fuel consumption computations";
        parameter Modelica.SIunits.Temperature flueGasNomTemp=843 
          "Maximum flue gas temperature";
        parameter Modelica.SIunits.Temperature flueGasMinTemp=548 
          "Minimum flue gas temperature (zero electrical load)";
        parameter Modelica.SIunits.Temperature flueGasOffTemp=363.15 
          "Flue gas temperature with GT switched off";
        parameter Modelica.SIunits.SpecificEnthalpy fuel_LHV=49e6 
          "Fuel Lower Heating Value";
        parameter Modelica.SIunits.SpecificEnthalpy fuel_HHV=55e6 
          "Fuel Higher Heating Value";
        FlueGasMedium.BaseProperties gas;
        Modelica.SIunits.MassFlowRate w;
        Modelica.SIunits.Power P_el=noEvent(if GTLoad > 0 then GTLoad*maxPower else 
                  0) "Electrical power output";
        Modelica.SIunits.MassFlowRate fuelFlowRate "Fuel flow rate";
      equation 
        gas.p = flueGasOut.p;
        gas.Xi = FlueGasMedium.reference_X[1:FlueGasMedium.nXi];
        gas.T = noEvent(
          if GTLoad > constTempLoad then flueGasNomTemp else 
          if GTLoad > 0 then flueGasMinTemp +
            GTLoad/constTempLoad*(flueGasNomTemp-flueGasMinTemp) else 
          flueGasMinTemp * (1+GTLoad) - flueGasOffTemp * GTLoad);
        w = noEvent(
          if GTLoad > constTempLoad then flueGasMinFlowRate + (GTLoad-constTempLoad)/
            (1-constTempLoad)*(flueGasNomFlowRate-flueGasMinFlowRate) else 
          if GTLoad > 0 then flueGasMinFlowRate else 
          flueGasMinFlowRate * (1+GTLoad) - flueGasOffFlowRate * GTLoad);
        fuelFlowRate = noEvent(
          if GTLoad > intLoad then fuelIntFlowRate + (GTLoad-intLoad)/
            (1-intLoad)*(fuelNomFlowRate-fuelIntFlowRate) else 
          if GTLoad > 0 then fuelMinFlowRate + GTLoad/intLoad *
            (fuelIntFlowRate-fuelMinFlowRate) else 
          fuelMinFlowRate * (1+GTLoad) - fuelOffFlowRate * GTLoad);
        
        flueGasOut.w = -w;
        flueGasOut.hBA = gas.h;
        flueGasOut.XBA = gas.Xi;
      end GasTurbineSimplified;
    end Examples;
    
    package Tests "Test cases" 
      model TestGasTurbine 
        
        Examples.GasTurbineSimplified gasTurbine 
          annotation (extent=[-40,-40; 0,0]);
        ThermoPower.Gas.SinkP sinkP(redeclare package Medium = 
              ThermoPower.Media.FlueGas) 
          annotation (extent=[62,-14; 82,6]);
        annotation (Diagram);
        Modelica.Blocks.Sources.Constant const 
          annotation (extent=[-90,-30; -70,-10]);
        ThermoPower.PowerPlants.HRSG.Components.StateReader_gas stateReader_gas(
            redeclare package Medium = ThermoPower.Media.FlueGas) 
          annotation (extent=[20,-14; 40,6]);
      equation 
        connect(const.y, gasTurbine.GTLoad) annotation (points=[-69,-20; -40,-20],
                         style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateReader_gas.inlet, gasTurbine.flueGasOut) annotation (points=
              [24,-4; 0,-4], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(stateReader_gas.outlet, sinkP.flange) annotation (points=[36,-4;
              62,-4], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
      end TestGasTurbine;
    end Tests;
    annotation (Documentation(revisions="<html>
<ul>
<li><i>15 Apr 2008</i>
    by <a>Luca Savoldelli</a>:<br>
       First release.</li>
</ul>
</html>"));
  end GasTurbine;
  
  package HRSG "Models and tests of the HRSG and its main components" 
    package Interfaces "Interface definitions" 
      partial model HeatExchanger "Base class for heat exchanger fluid - gas" 
        
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium "Flue gas model";
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance "Fluid model";
        
        parameter Integer N_G=2 "Number of node of the gas side";
        parameter Integer N_F=2 "Number of node of the fluid side";
        
        //Nominal parameter
        parameter SI.MassFlowRate gasNomFlowRate 
          "Nominal flow rate through the gas side";
        parameter SI.MassFlowRate fluidNomFlowRate 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure gasNomPressure 
          "Nominal pressure in the gas side inlet";
        parameter SI.Pressure fluidNomPressure 
          "Nominal pressure in the fluid side inlet";
        
        //Physical Parameter
        parameter SI.Area exchSurface_G 
          "Exchange surface between gas - metal tube";
        parameter SI.Area exchSurface_F 
          "Exchange surface between metal tube - fluid";
        parameter SI.Area extSurfaceTub "Total external surface of the tubes";
        parameter SI.Volume gasVol "Gas volume";
        parameter SI.Volume fluidVol "Fluid volume";
        parameter SI.Volume metalVol "Volume of the metal part in the tubes";
        parameter Real rhomcm "Metal heat capacity per unit volume [J/m^3.K]";
        parameter SI.ThermalConductivity lambda 
          "Thermal conductivity of the metal (density by specific heat capacity)";
        
        //Start value
        parameter Boolean use_T = true 
          "Select: -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        
        parameter SI.Temperature Tstart_G_In 
          "Inlet gas temperature start value"            annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_G_Out 
          "Outlet gas temperature start value"           annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_G[N_G]=linspace(Tstart_G_In,Tstart_G_Out,N_G) 
          "Start value of gas temperature vector (initialized by default)" annotation(Dialog(tab = "Initialization"));
        parameter SI.Pressure pstart_G=gasNomPressure 
          "Pressure start value, gas side" 
                                          annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M_In 
          "Inlet metal wall temperature start value"       annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M_Out 
          "Outlet metal wall temperature start value"      annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M[N_F]=linspace(Tstart_M_In,Tstart_M_Out,N_F) 
          "Start value of metal wall temperature vector (initialized by default)"
                                                                                  annotation(Dialog(tab = "Initialization"));
        
        parameter SI.Temperature Tstart_F_In=FluidMedium.temperature_ph(fluidNomPressure,hstart_F_In) 
          "Inlet fluid temperature start value"             annotation(Dialog(enable = use_T,
                                                                              tab = "Initialization"));
        parameter SI.Temperature Tstart_F_Out=FluidMedium.temperature_ph(fluidNomPressure,hstart_F_Out) 
          "Outlet fluid temperature start value"            annotation(Dialog(enable = use_T,
                                                                              tab = "Initialization"));
        parameter SI.Temperature Tstart_F[N_F]=linspace(Tstart_F_In,Tstart_F_Out,N_F) 
          "Start value of fluid temperature vector (initialized by default)" 
                                                                          annotation(Dialog(tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_In) 
          "Inlet fluid specific enthalpy start value"                   annotation(Dialog(enable = not use_T,
                                                                                          tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value"                  annotation(Dialog(enable = not use_T,
                                                                                          tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_F[N_F]=linspace(hstart_F_In,hstart_F_Out,N_F) 
          "Start value of fluid enthalpy vector (initialized by default)" annotation(Dialog(tab = "Initialization"));
        parameter SI.Pressure pstartin_F=fluidNomPressure 
          "Inlet fluid pressure start value"                annotation(Dialog(tab = "Initialization"));
        parameter SI.Pressure pstartout_F=fluidNomPressure 
          "Outlet fluid pressure start value"                annotation(Dialog(tab = "Initialization"));
        
        annotation (Diagram, Icon(Rectangle(extent=[-100,100; 100,-100],
                                                                      style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})), Line(points=[0,-80; 0,-40; 40,-20; -40,
                  20; 0,40; 0,80],           style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillPattern=1)),
            Text(
              extent=[-100,-115; 100,-145],
              string="%name",
              style(color=71, rgbcolor={85,170,255}))));
        Gas.FlangeA gasIn(redeclare package Medium = FlueGasMedium) 
                            annotation (extent=[-120,-20; -80,20]);
        Gas.FlangeB gasOut(redeclare package Medium = FlueGasMedium) 
                            annotation (extent=[80,-20; 120,20]);
        Water.FlangeA waterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[-20,80; 20,120]);
        Water.FlangeB waterOut(redeclare package Medium = FluidMedium) 
          annotation (extent=[-20,-120; 20,-80]);
        
      end HeatExchanger;
      
      partial model ParallelHE 
        "Base class for Parallel Heat Exchanger (two fluid with one gas)" 
        
        annotation (Diagram, Icon(
            Rectangle(extent=[-100,100; 100,-100],
                                               style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[-40,80; -40,60; -10,60; -10,42; -42,20; 20,-20; -10,
                  -40; -10,-60; -40,-60; -40,-80],  style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillPattern=1)),
            Line(points=[40,80; 40,60; 10,60; 10,40; -20,20; 42,-20; 10,-40; 10,-60;
                  40,-60; 40,-80],          style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillPattern=1)),
            Text(
              extent=[-100,-115; 100,-145],
              string="%name",
              style(color=71, rgbcolor={85,170,255}))));
        ThermoPower.Gas.FlangeA gasIn(redeclare package Medium = FlueGasMedium)
                            annotation (extent=[-120,-20; -80,20]);
        ThermoPower.Gas.FlangeB gasOut(redeclare package Medium = FlueGasMedium)
                            annotation (extent=[80,-20; 120,20]);
        ThermoPower.Water.FlangeA waterInA(redeclare package Medium = 
              FluidMedium) "water/steam first inlet" 
          annotation (extent=[-60,80; -20,120]);
        ThermoPower.Water.FlangeB waterOutA(redeclare package Medium = 
              FluidMedium) "water/steam first outlet" 
          annotation (extent=[-60,-120; -20,-80]);
        ThermoPower.Water.FlangeA waterInB(redeclare package Medium = 
              FluidMedium) "water/steam second inlet" 
          annotation (extent=[20,80; 60,120]);
        ThermoPower.Water.FlangeB waterOutB(redeclare package Medium = 
              FluidMedium) "water/steam second outlet" 
          annotation (extent=[20,-120; 60,-80]);
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        parameter Integer N_G=2 "Number of node of the gas side";
        parameter Integer N_F_A=2 "Number of node of the fluid side"   annotation (Dialog(group = "side A"));
        parameter Integer N_F_B=2 "Number of node of the fluid side"   annotation (Dialog(group = "side B"));
        
        //Nominal parameter
        parameter SI.MassFlowRate gasNomFlowRate 
          "Nominal flow rate through the gas side";
        parameter SI.MassFlowRate fluidNomFlowRate_A 
          "Nominal flow rate through the fluid side"     annotation (Dialog(group = "side A"));
        parameter SI.MassFlowRate fluidNomFlowRate_B 
          "Nominal flow rate through the fluid side"     annotation (Dialog(group = "side B"));
        parameter SI.Pressure gasNomPressure 
          "Nominal pressure in the gas side inlet";
        parameter SI.Pressure fluidNomPressure_A 
          "Nominal pressure in the fluid side inlet"     annotation (Dialog(group = "side A"));
        parameter SI.Pressure fluidNomPressure_B 
          "Nominal pressure in the fluid side inlet"     annotation (Dialog(group = "side B"));
        
        //Physical Parameter
        parameter SI.Area exchSurface_G_A 
          "Exchange surface between gas - metal tube,"     annotation (Dialog(group = "side A"));
        parameter SI.Area exchSurface_G_B 
          "Exchange surface between gas - metal tube,"     annotation (Dialog(group = "side B"));
        parameter SI.Area exchSurface_F_A 
          "Exchange surface between metal tube - fluid,"     annotation (Dialog(group = "side A"));
        parameter SI.Area exchSurface_F_B 
          "Exchange surface between metal tube - fluid,"     annotation (Dialog(group = "side B"));
        parameter SI.Area extSurfaceTub_A "Total external surface of the tubes"
                                                             annotation (Dialog(group = "side A"));
        parameter SI.Area extSurfaceTub_B "Total external surface of the tubes"
                                                             annotation (Dialog(group = "side B"));
        parameter SI.Volume gasVol "Gas volume";
        parameter SI.Volume fluidVol_A "Fluid volume"     annotation (Dialog(group = "side A"));
        parameter SI.Volume fluidVol_B "Fluid volume"     annotation (Dialog(group = "side B"));
        parameter SI.Volume metalVol_A "Volume of the metal part in the tube"     annotation (Dialog(group = "side A"));
        parameter SI.Volume metalVol_B "Volume of the metal part in the tube"     annotation (Dialog(group = "side B"));
        parameter Real rhomcm_A "Metal heat capacity per unit volume [J/m^3.K]"
                                                                                annotation (Dialog(group = "side A"));
        parameter Real rhomcm_B "Metal heat capacity per unit volume [J/m^3.K]"
                                                                                annotation (Dialog(group = "side B"));
        parameter SI.ThermalConductivity lambda 
          "Thermal conductivity of the metal (density by specific heat capacity)";
        
        //Initialization conditions
        parameter Boolean use_T = true 
          "Select: -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy"
             annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        //Start value
        parameter SI.Temperature Tstart_G_In 
          "Inlet gas temperature start value" 
            annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_G_Out 
          "Outlet gas temperature start value" 
            annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_G[N_G]=linspace(
                  Tstart_G_In,
                  Tstart_G_Out,
                  N_G) 
          "Start value of gas temperature vector (initialized by default)" annotation(Dialog(tab = "Initialization"));
        parameter SI.Pressure pstart_G=gasNomPressure 
          "Pressure start value, gas side" 
            annotation(Dialog(tab = "Initialization"));
        //A
        parameter SI.Temperature Tstart_M_A_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization",
                              group = "side A"));
        parameter SI.Temperature Tstart_M_A_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization",
                              group = "side A"));
        parameter SI.Temperature Tstart_M_A[N_F_A]=linspace(
                  Tstart_M_A_In,
                  Tstart_M_A_Out,
                  N_F_A) 
          "Start value of metal wall temperature vector (initialized by default)"
            annotation(Dialog(tab = "Initialization",
                              group = "side A"));
        parameter SI.Temperature Tstart_F_A_In=FluidMedium.temperature_ph(
            fluidNomPressure_A, hstart_F_A_In) 
          "Inlet fluid temperature start value"        annotation(Dialog(enable = use_T,
                                                                         tab = "Initialization",
                                                                         group = "side A"));
        parameter SI.Temperature Tstart_F_A_Out=FluidMedium.temperature_ph(
            fluidNomPressure_A, hstart_F_A_Out) 
          "Outlet fluid temperature start value"       annotation(Dialog(enable = use_T,
                                                                         tab = "Initialization",
                                                                         group = "side A"));
        parameter SI.Temperature Tstart_F_A[N_F_A]=linspace(
                  Tstart_F_A_In,
                  Tstart_F_A_Out,
                  N_F_A) 
          "Start value of fluid temperature vector (initialized by default)"     annotation(Dialog(tab = "Initialization",
                                                                                                   group = "side A"));
        parameter SI.SpecificEnthalpy hstart_F_A_In=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_A, Tstart_F_A_In) 
          "Inlet fluid specific enthalpy start value"        annotation(Dialog(enable = not use_T,
                                                                               tab = "Initialization",
                                                                               group = "side A"));
        parameter SI.SpecificEnthalpy hstart_F_A_Out=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_A, Tstart_F_A_Out) 
          "Outlet fluid specific enthalpy start value"        annotation(Dialog(enable = not use_T,
                                                                                tab = "Initialization",
                                                                                group = "side A"));
        parameter SI.SpecificEnthalpy hstart_F_A[N_F_A]=linspace(
                  hstart_F_A_In,
                  hstart_F_A_Out,
                  N_F_A) 
          "Start value of fluid enthalpy vector (initialized by default)"     annotation(Dialog(tab = "Initialization",
                                                                                                group = "side A"));
        parameter SI.Pressure pstartin_F_A=fluidNomPressure_A 
          "Inlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization",
                              group = "side A"));
        parameter SI.Pressure pstartout_F_A=fluidNomPressure_A 
          "Outlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization",
                              group = "side A"));
        //B
        parameter SI.Temperature Tstart_M_B_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization",
                              group = "side B"));
        parameter SI.Temperature Tstart_M_B_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization",
                              group = "side B"));
        parameter SI.Temperature Tstart_M_B[N_F_B]=linspace(
                  Tstart_M_B_In,
                  Tstart_M_B_Out,
                  N_F_B) 
          "Start value of metal wall temperature vector (initialized by default)"
            annotation(Dialog(tab = "Initialization",
                              group = "side B"));
        parameter SI.Temperature Tstart_F_B_In=FluidMedium.temperature_ph(
            fluidNomPressure_B, hstart_F_B_In) 
          "Inlet fluid temperature start value"        annotation(Dialog(enable = use_T,
                                                                         tab = "Initialization",
                                                                         group = "side B"));
        parameter SI.Temperature Tstart_F_B_Out=FluidMedium.temperature_ph(
            fluidNomPressure_B, hstart_F_B_Out) 
          "Outlet fluid temperature start value"        annotation(Dialog(enable = use_T,
                                                                          tab = "Initialization",
                                                                          group = "side B"));
        parameter SI.Temperature Tstart_F_B[N_F_B]=linspace(
                  Tstart_F_B_In,
                  Tstart_F_B_Out,
                  N_F_B) 
          "Start value of fluid temperature vector (initialized by default)"     annotation(Dialog(tab = "Initialization",
                                                                                                   group = "side B"));
        parameter SI.SpecificEnthalpy hstart_F_B_In=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_B, Tstart_F_B_In) 
          "Inlet fluid specific enthalpy start value"        annotation(Dialog(enable = not use_T,
                                                                               tab = "Initialization",
                                                                               group = "side B"));
        parameter SI.SpecificEnthalpy hstart_F_B_Out=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_B, Tstart_F_B_Out) 
          "Outlet fluid specific enthalpy start value"        annotation(Dialog(enable = not use_T,
                                                                                tab = "Initialization",
                                                                                group = "side B"));
        parameter SI.SpecificEnthalpy hstart_F_B[N_F_B]=linspace(
                  hstart_F_B_In,
                  hstart_F_B_Out,
                  N_F_B) 
          "Start value of fluid enthalpy vector (initialized by default)"     annotation(Dialog(tab = "Initialization",
                                                                                                group = "side B"));
        parameter SI.Pressure pstartin_F_B=fluidNomPressure_B 
          "Inlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization",
                              group = "side B"));
        parameter SI.Pressure pstartout_F_B=fluidNomPressure_B 
          "Outlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization",
                              group = "side B"));
        
      end ParallelHE;
      
      partial model ParallelHE_Des 
        "Base class for parallel Heat Exchanger (two fluid with one gas) with desuperheater" 
        
        annotation (Diagram, Icon(
            Rectangle(extent=[-100,100; 100,-100],
                                               style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[-40,80; -40,60; -10,60; -10,42; -42,20; 20,-20; -10,-40; -10,
                  -60; -40,-60; -40,-80],           style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillPattern=1)),
            Line(points=[40,80; 40,60; 10,60; 10,40; -20,20; 42,-20; 10,-40; 10,-60;
                  40,-60; 40,-80],          style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillPattern=1)),
            Text(
              extent=[-100,-115; 100,-145],
              string="%name",
              style(color=71, rgbcolor={85,170,255})),
            Polygon(points=[-58,-40; -68,-32; -46,-26; -56,-18; -58,-40],
                                                                   style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255})),
            Polygon(points=[-42,-54; -52,-46; -30,-40; -40,-32; -42,-54],
                                                                   style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255})),
            Line(points=[-48,-50; -68,-70; -96,-70; -98,-70], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230},
                fillPattern=1)),
            Line(points=[-62,-36; -68,-42; -100,-42; -102,-42], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230},
                fillPattern=1))));
        ThermoPower.Gas.FlangeA gasIn(redeclare package Medium = FlueGasMedium)
                            annotation (extent=[-120,-20; -80,20]);
        ThermoPower.Gas.FlangeB gasOut(redeclare package Medium = FlueGasMedium)
                            annotation (extent=[80,-20; 120,20]);
        ThermoPower.Water.FlangeA waterInA(redeclare package Medium = 
              FluidMedium) "water/steam first inlet" 
          annotation (extent=[-60,80; -20,120]);
        ThermoPower.Water.FlangeB waterOutA(redeclare package Medium = 
              FluidMedium) "water/steam first outlet" 
          annotation (extent=[-60,-120; -20,-80]);
        ThermoPower.Water.FlangeA waterInB(redeclare package Medium = 
              FluidMedium) "water/steam second inlet " 
          annotation (extent=[20,80; 60,120]);
        ThermoPower.Water.FlangeB waterOutB(redeclare package Medium = 
              FluidMedium) "water/steam second outlet" 
          annotation (extent=[20,-120; 60,-80]);
        ThermoPower.Water.FlangeA LiquidWaterIn_A(redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-110,-52; -90,-32]);
        Modelica.Blocks.Interfaces.RealOutput T_intermedA 
          "Intermediate temperature of the HE_A" 
          annotation (extent=[94,52; 108,66]);
        Modelica.Blocks.Interfaces.RealOutput T_intermedB 
          "Intermediate temperature of the HE_B" 
          annotation (extent=[94,32; 108,46]);
        Modelica.Blocks.Interfaces.RealInput theta_valveA 
          annotation (extent=[108,-50; 94,-34]);
        Modelica.Blocks.Interfaces.RealInput theta_valveB 
          annotation (extent=[108,-70; 94,-56]);
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Nominal parameter
        parameter SI.MassFlowRate gasNomFlowRate 
          "Nominal flow rate through the gas side";
        parameter SI.MassFlowRate fluidNomFlowRate_A 
          "Nominal flow rate through the fluid side "    annotation (Dialog(group = "side A"));
        parameter SI.MassFlowRate fluidNomFlowRate_B 
          "Nominal flow rate through the fluid side"     annotation (Dialog(group = "side B"));
        parameter SI.Pressure gasNomPressure 
          "Nominal pressure in the gas side inlet";
        parameter SI.Pressure fluidNomPressure_A 
          "Nominal pressure in the fluid side inlet"     annotation (Dialog(group = "side A"));
        parameter SI.Pressure fluidNomPressure_B 
          "Nominal pressure in the fluid side inlet"     annotation (Dialog(group = "side B"));
        
        //Other common parameter
        parameter SI.ThermalConductivity lambda 
          "Thermal conductivity of the metal (density by specific heat capacity)";
        
        //Parameter for first parallel heat exchangers
        parameter Integer N_G_p1=2 "Number of node of the gas side" 
                                                                  annotation (Dialog(tab = "pHE-1"));
        parameter Integer N_F_A_p1=2 "Number of node of the fluid side"   annotation (Dialog(tab = "pHE-1",
                                                                                             group = "side A"));
        parameter Integer N_F_B_p1=2 "Number of node of the fluid side"   annotation (Dialog(tab = "pHE-1",
                                                                                             group = "side B"));
        //Physical Parameter
        parameter SI.Area exchSurface_G_A_p1 
          "Exchange surface between gas - metal tube,"     annotation (Dialog(tab = "pHE-1",
                                                                              group = "side A"));
        parameter SI.Area exchSurface_G_B_p1 
          "Exchange surface between gas - metal tube,"     annotation (Dialog(tab = "pHE-1",
                                                                              group = "side B"));
        parameter SI.Area exchSurface_F_A_p1 
          "Exchange surface between metal tube - fluid,"     annotation (Dialog(tab = "pHE-1",
                                                                                group = "side A"));
        parameter SI.Area exchSurface_F_B_p1 
          "Exchange surface between metal tube - fluid,"     annotation (Dialog(tab = "pHE-1",
                                                                                group = "side B"));
        parameter SI.Area extSurfaceTub_A_p1 
          "Total external surface of the tubes"                                        annotation (Dialog(tab = "pHE-1",
                                                                                                    group = "side A"));
        parameter SI.Area extSurfaceTub_B_p1 
          "Total external surface of the tubes"                                        annotation (Dialog(tab = "pHE-1",
                                                                                                    group = "side B"));
        parameter SI.Volume gasVol_p1 "Gas volume" annotation (Dialog(tab = "pHE-1"));
        parameter SI.Volume fluidVol_A_p1 "Fluid volume"     annotation (Dialog(tab = "pHE-1",
                                                                                group = "side A"));
        parameter SI.Volume fluidVol_B_p1 "Fluid volume"     annotation (Dialog(tab = "pHE-1",
                                                                                group = "side B"));
        parameter SI.Volume metalVol_A_p1 
          "Volume of the metal part in the tube"                                     annotation (Dialog(tab = "pHE-1",
                                                                                                    group = "side A"));
        parameter SI.Volume metalVol_B_p1 
          "Volume of the metal part in the tube"                                     annotation (Dialog(tab = "pHE-1",
                                                                                                    group = "side B"));
        parameter Real rhomcm_A_p1 
          "Metal heat capacity per unit volume [J/m^3.K]"                          annotation (Dialog(tab = "pHE-1",
                                                                                                    group = "side A"));
        parameter Real rhomcm_B_p1 
          "Metal heat capacity per unit volume [J/m^3.K]"                          annotation (Dialog(tab = "pHE-1",
                                                                                                    group = "side B"));
        
        //Parameter for second parallel heat exchangers
        parameter Integer N_G_p2=2 "Number of node of the gas side" 
                                                                  annotation (Dialog(tab = "pHE-2"));
        parameter Integer N_F_A_p2=2 "Number of node of the fluid side"   annotation (Dialog(tab = "pHE-2",
                                                                                             group = "side A"));
        parameter Integer N_F_B_p2=2 "Number of node of the fluid side"   annotation (Dialog(tab = "pHE-2",
                                                                                             group = "side B"));
        //Physical Parameter
        parameter SI.Area exchSurface_G_A_p2 
          "Exchange surface between gas - metal tube,"     annotation (Dialog(tab = "pHE-2",
                                                                              group = "side A"));
        parameter SI.Area exchSurface_G_B_p2 
          "Exchange surface between gas - metal tube,"     annotation (Dialog(tab = "pHE-2",
                                                                              group = "side B"));
        parameter SI.Area exchSurface_F_A_p2 
          "Exchange surface between metal tube - fluid,"     annotation (Dialog(tab = "pHE-2",
                                                                                group = "side A"));
        parameter SI.Area exchSurface_F_B_p2 
          "Exchange surface between metal tube - fluid,"     annotation (Dialog(tab = "pHE-2",
                                                                                group = "side B"));
        parameter SI.Area extSurfaceTub_A_p2 
          "Total external surface of the tubes"                                        annotation (Dialog(tab = "pHE-2",
                                                                                                    group = "side A"));
        parameter SI.Area extSurfaceTub_B_p2 
          "Total external surface of the tubes"                                        annotation (Dialog(tab = "pHE-2",
                                                                                                    group = "side B"));
        parameter SI.Volume gasVol_p2 "Gas volume" annotation (Dialog(tab = "pHE-2"));
        parameter SI.Volume fluidVol_A_p2 "Fluid volume"     annotation (Dialog(tab = "pHE-2",
                                                                                group = "side A"));
        parameter SI.Volume fluidVol_B_p2 "Fluid volume"     annotation (Dialog(tab = "pHE-2",
                                                                                group = "side B"));
        parameter SI.Volume metalVol_A_p2 
          "Volume of the metal part in the tube"                                     annotation (Dialog(tab = "pHE-2",
                                                                                                    group = "side A"));
        parameter SI.Volume metalVol_B_p2 
          "Volume of the metal part in the tube"                                     annotation (Dialog(tab = "pHE-2",
                                                                                                    group = "side B"));
        parameter Real rhomcm_A_p2 
          "Metal heat capacity per unit volume [J/m^3.K]"                          annotation (Dialog(tab = "pHE-2",
                                                                                                    group = "side A"));
        parameter Real rhomcm_B_p2 
          "Metal heat capacity per unit volume [J/m^3.K]"                          annotation (Dialog(tab = "pHE-2",
                                                                                                    group = "side B"));
        
        //Initialization conditions
        parameter Boolean use_T = true 
          "Select: -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        //Start value pHE1
        parameter SI.Temperature Tstart_G_In_p1 
          "Inlet gas temperature start value"    annotation(Dialog(tab = "Initialization (pHE-1)"));
        parameter SI.Temperature Tstart_G_Out_p1 
          "Outlet gas temperature start value"    annotation(Dialog(tab = "Initialization (pHE-1)"));
        parameter SI.Temperature Tstart_G_p1[N_G_p1]=linspace(
                  Tstart_G_In_p1,
                  Tstart_G_Out_p1,
                  N_G_p1) 
          "Start value of gas temperature vector (initialized by default)" annotation(Dialog(tab = "Initialization (pHE-1)"));
        parameter SI.Pressure pstart_G_p1=gasNomPressure 
          "Pressure start value, gas side" annotation(Dialog(tab = "Initialization (pHE-1)"));
        //A
        parameter SI.Temperature Tstart_M_A_In_p1 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side A"));
        parameter SI.Temperature Tstart_M_A_Out_p1 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side A"));
        parameter SI.Temperature Tstart_M_A_p1[N_F_A_p1]=linspace(
                  Tstart_M_A_In_p1,
                  Tstart_M_A_Out_p1,
                  N_F_A_p1) 
          "Start value of metal wall temperature vector (initialized by default)"
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side A"));
        parameter SI.Temperature Tstart_F_A_In_p1=FluidMedium.temperature_ph(fluidNomPressure_A, hstart_F_A_In_p1) 
          "Inlet fluid temperature start value"        annotation(Dialog(enable = use_T,
                                                                         tab = "Initialization (pHE-1)",
                                                                         group = "side A"));
        parameter SI.Temperature Tstart_F_A_Out_p1=FluidMedium.temperature_ph(fluidNomPressure_A, hstart_F_A_Out_p1) 
          "Outlet fluid temperature start value"       annotation(Dialog(enable = use_T,
                                                                         tab = "Initialization (pHE-1)",
                                                                         group = "side A"));
        parameter SI.Temperature Tstart_F_A_p1[N_F_A_p1]=linspace(
                  Tstart_F_A_In_p1,
                  Tstart_F_A_Out_p1,
                  N_F_A_p1) 
          "Start value of fluid temperature vector (initialized by default)"     annotation(Dialog(tab = "Initialization (pHE-1)",
                                                                                                   group = "side A"));
        parameter SI.SpecificEnthalpy hstart_F_A_In_p1=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,Tstart_F_A_In_p1) 
          "Inlet fluid specific enthalpy start value"        annotation(Dialog(enable = not use_T,
                                                                               tab = "Initialization (pHE-1)",
                                                                               group = "side A"));
        parameter SI.SpecificEnthalpy hstart_F_A_Out_p1=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,Tstart_F_A_Out_p1) 
          "Outlet fluid specific enthalpy start value"        annotation(Dialog(enable = not use_T,
                                                                                tab = "Initialization (pHE-1)",
                                                                                group = "side A"));
        parameter SI.SpecificEnthalpy hstart_F_A_p1[N_F_A_p1]=linspace(
                  hstart_F_A_In_p1,
                  hstart_F_A_Out_p1,
                  N_F_A_p1) 
          "Start value of fluid enthalpy vector (initialized by default)"     annotation(Dialog(tab = "Initialization (pHE-1)",
                                                                                                group = "side A"));
        parameter SI.Pressure pstartin_F_A_p1=fluidNomPressure_A 
          "Inlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side A"));
        parameter SI.Pressure pstartout_F_A_p1=fluidNomPressure_A 
          "Outlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side A"));
        //B
        parameter SI.Temperature Tstart_M_B_In_p1 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side B"));
        parameter SI.Temperature Tstart_M_B_Out_p1 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side B"));
        parameter SI.Temperature Tstart_M_B_p1[N_F_B_p1]=linspace(
                  Tstart_M_B_In_p1,
                  Tstart_M_B_Out_p1,
                  N_F_B_p1) 
          "Start value of metal wall temperature vector (initialized by default)"
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side B"));
        parameter SI.Temperature Tstart_F_B_In_p1=FluidMedium.temperature_ph(
            fluidNomPressure_B, hstart_F_B_In_p1) 
          "Inlet fluid temperature start value"        annotation(Dialog(enable = use_T,
                                                                         tab = "Initialization (pHE-1)",
                                                                         group = "side B"));
        parameter SI.Temperature Tstart_F_B_Out_p1=FluidMedium.temperature_ph(
            fluidNomPressure_B, hstart_F_B_Out_p1) 
          "Outlet fluid temperature start value"        annotation(Dialog(enable = use_T,
                                                                          tab = "Initialization (pHE-1)",
                                                                          group = "side B"));
        parameter SI.Temperature Tstart_F_B_p1[N_F_B_p1]=linspace(
                  Tstart_F_B_In_p1,
                  Tstart_F_B_Out_p1,
                  N_F_B_p1) 
          "Start value of fluid temperature vector (initialized by default)"     annotation(Dialog(tab = "Initialization (pHE-1)",
                                                                                                   group = "side B"));
        parameter SI.SpecificEnthalpy hstart_F_B_In_p1=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,
            Tstart_F_B_In_p1) "Inlet fluid specific enthalpy start value" 
                                                             annotation(Dialog(enable = not use_T,
                                                                               tab = "Initialization (pHE-1)",
                                                                               group = "side B"));
        parameter SI.SpecificEnthalpy hstart_F_B_Out_p1=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,
            Tstart_F_B_Out_p1) "Outlet fluid specific enthalpy start value" 
                                                              annotation(Dialog(enable = not use_T,
                                                                                tab = "Initialization (pHE-1)",
                                                                                group = "side B"));
        parameter SI.SpecificEnthalpy hstart_F_B_p1[N_F_B_p1]=linspace(
                  hstart_F_B_In_p1,
                  hstart_F_B_Out_p1,
                  N_F_B_p1) 
          "Start value of fluid enthalpy vector (initialized by default)"     annotation(Dialog(tab = "Initialization (pHE-1)",
                                                                                                group = "side B"));
        parameter SI.Pressure pstartin_F_B_p1=fluidNomPressure_B 
          "Inlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side B"));
        parameter SI.Pressure pstartout_F_B_p1=fluidNomPressure_B 
          "Outlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization (pHE-1)",
                              group = "side B"));
        
        //Start value pHE2
        parameter SI.Temperature Tstart_G_In_p2 
          "Inlet gas temperature start value"    annotation(Dialog(tab = "Initialization (pHE-2)"));
        parameter SI.Temperature Tstart_G_Out_p2 
          "Outlet gas temperature start value"    annotation(Dialog(tab = "Initialization (pHE-2)"));
        parameter SI.Temperature Tstart_G_p2[N_G_p2]=linspace(
                  Tstart_G_In_p2,
                  Tstart_G_Out_p2,
                  N_G_p2) 
          "Start value of gas temperature vector (initialized by default)" annotation(Dialog(tab = "Initialization (pHE-2)"));
        parameter SI.Pressure pstart_G_p2=gasNomPressure 
          "Pressure start value, gas side" annotation(Dialog(tab = "Initialization (pHE-2)"));
        //A
        parameter SI.Temperature Tstart_M_A_In_p2 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side A"));
        parameter SI.Temperature Tstart_M_A_Out_p2 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side A"));
        parameter SI.Temperature Tstart_M_A_p2[N_F_A_p2]=linspace(
                  Tstart_M_A_In_p2,
                  Tstart_M_A_Out_p2,
                  N_F_A_p2) 
          "Start value of metal wall temperature vector (initialized by default)"
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side A"));
        parameter SI.Temperature Tstart_F_A_In_p2=FluidMedium.temperature_ph(
            fluidNomPressure_A, hstart_F_A_In_p1) 
          "Inlet fluid temperature start value"        annotation(Dialog(enable = use_T,
                                                                         tab = "Initialization (pHE-2)",
                                                                         group = "side A"));
        parameter SI.Temperature Tstart_F_A_Out_p2=FluidMedium.temperature_ph(
            fluidNomPressure_A, hstart_F_A_Out_p2) 
          "Outlet fluid temperature start value"       annotation(Dialog(enable = use_T,
                                                                         tab = "Initialization (pHE-2)",
                                                                         group = "side A"));
        parameter SI.Temperature Tstart_F_A_p2[N_F_A_p2]=linspace(
                  Tstart_F_A_In_p2,
                  Tstart_F_A_Out_p2,
                  N_F_A_p2) 
          "Start value of fluid temperature vector (initialized by default)"     annotation(Dialog(tab = "Initialization (pHE-2)",
                                                                                                   group = "side A"));
        parameter SI.SpecificEnthalpy hstart_F_A_In_p2=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,
            Tstart_F_A_In_p2) "Inlet fluid specific enthalpy start value" 
                                                             annotation(Dialog(enable = not use_T,
                                                                               tab = "Initialization (pHE-2)",
                                                                               group = "side A"));
        parameter SI.SpecificEnthalpy hstart_F_A_Out_p2=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,
            Tstart_F_A_Out_p2) "Outlet fluid specific enthalpy start value" 
                                                              annotation(Dialog(enable = not use_T,
                                                                                tab = "Initialization (pHE-2)",
                                                                                group = "side A"));
        parameter SI.SpecificEnthalpy hstart_F_A_p2[N_F_A_p2]=linspace(
                  hstart_F_A_In_p2,
                  hstart_F_A_Out_p2,
                  N_F_A_p2) 
          "Start value of fluid enthalpy vector (initialized by default)"     annotation(Dialog(tab = "Initialization (pHE-2)",
                                                                                                group = "side A"));
        parameter SI.Pressure pstartin_F_A_p2=fluidNomPressure_A 
          "Inlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side A"));
        parameter SI.Pressure pstartout_F_A_p2=fluidNomPressure_A 
          "Outlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side A"));
        //B
        parameter SI.Temperature Tstart_M_B_In_p2 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side B"));
        parameter SI.Temperature Tstart_M_B_Out_p2 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side B"));
        parameter SI.Temperature Tstart_M_B_p2[N_F_B_p2]=linspace(
                  Tstart_M_B_In_p2,
                  Tstart_M_B_Out_p2,
                  N_F_B_p2) 
          "Start value of metal wall temperature vector (initialized by default)"
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side B"));
        parameter SI.Temperature Tstart_F_B_In_p2=FluidMedium.temperature_ph(
            fluidNomPressure_B, hstart_F_B_In_p2) 
          "Inlet fluid temperature start value"        annotation(Dialog(enable = use_T,
                                                                         tab = "Initialization (pHE-2)",
                                                                         group = "side B"));
        parameter SI.Temperature Tstart_F_B_Out_p2=FluidMedium.temperature_ph(
            fluidNomPressure_B, hstart_F_B_Out_p2) 
          "Outlet fluid temperature start value"        annotation(Dialog(enable = use_T,
                                                                          tab = "Initialization (pHE-2)",
                                                                          group = "side B"));
        parameter SI.Temperature Tstart_F_B_p2[N_F_B_p2]=linspace(
                  Tstart_F_B_In_p2,
                  Tstart_F_B_Out_p2,
                  N_F_B_p2) 
          "Start value of fluid temperature vector (initialized by default)"     annotation(Dialog(tab = "Initialization (pHE-2)",
                                                                                                   group = "side B"));
        parameter SI.SpecificEnthalpy hstart_F_B_In_p2=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,
            Tstart_F_B_In_p2) "Inlet fluid specific enthalpy start value" 
                                                             annotation(Dialog(enable = not use_T,
                                                                               tab = "Initialization (pHE-2)",
                                                                               group = "side B"));
        parameter SI.SpecificEnthalpy hstart_F_B_Out_p2=
            FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,
            Tstart_F_B_Out_p2) "Outlet fluid specific enthalpy start value" 
                                                              annotation(Dialog(enable = not use_T,
                                                                                tab = "Initialization (pHE-2)",
                                                                                group = "side B"));
        parameter SI.SpecificEnthalpy hstart_F_B_p2[N_F_B_p2]=linspace(
                  hstart_F_B_In_p2,
                  hstart_F_B_Out_p2,
                  N_F_B_p2) 
          "Start value of fluid enthalpy vector (initialized by default)"     annotation(Dialog(tab = "Initialization (pHE-2)",
                                                                                                group = "side B"));
        parameter SI.Pressure pstartin_F_B_p2=fluidNomPressure_B 
          "Inlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side B"));
        parameter SI.Pressure pstartout_F_B_p2=fluidNomPressure_B 
          "Outlet fluid pressure start value" 
            annotation(Dialog(tab = "Initialization (pHE-2)",
                              group = "side B"));
        
        ThermoPower.Water.FlangeA LiquidWaterIn_B(redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-110,-80; -90,-60]);
      end ParallelHE_Des;
      
      partial model HEG_2L 
        "Base class for Heat Exchangers Group with two pressure levels and desuperheater" 
        
        annotation (Icon(
            Rectangle(extent=[-400,200; 400,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[-120,200; -120,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-160,-100; -156,-110; -144,-120; -136,-120; -124,-110;
                  -120,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-160,200; -160,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-200,200; -200,-102], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-240,-100; -236,-110; -224,-120; -216,-120; -204,-110;
                  -200,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-240,200; -240,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-200,-200; -200,-140; -280,-140; -280,204; -278,200],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-400,80; 400,80], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,60; 400,60], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,40; 400,40], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,100; 400,100], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,0; 400,0], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-20; 400,-20], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-40; 400,-40], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,20; 400,20], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-80; 400,-80], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-100; 400,-100], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-60; 400,-60], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[140,-200; 140,-140; 60,-140; 60,204; 62,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[100,-100; 104,-110; 116,-120; 124,-120; 136,-110; 140,
                  -100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[100,200; 100,-102], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[140,200; 140,-102], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[180,200; 180,-102], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[220,200; 220,-102], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[180,-100; 184,-110; 196,-120; 204,-120; 216,-110; 220,
                  -100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-126,138; -114,138; -120,122; -126,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-166,144; -154,144; -160,160; -166,144],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-246,144; -234,144; -240,160; -246,144],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-206,138; -194,138; -200,122; -206,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[94,144; 106,144; 100,160; 94,144],      style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[134,138; 146,138; 140,122; 134,138],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[174,144; 186,144; 180,160; 174,144],    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[214,138; 226,138; 220,122; 214,138],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[54,138; 66,138; 60,122; 54,138],         style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-286,138; -274,138; -280,122; -286,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
                             Diagram,
          Coordsys(extent=[-400,-200; 400,200], scale=0.1));
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Common Parameter
        parameter SI.MassFlowRate gasNomFlowRate 
          "Nominal flow rate through the gas side";
        parameter SI.Pressure gasNomPressure 
          "Nominal pressure in the gas side inlet";
        
        //Nominal parameter
        parameter SI.MassFlowRate fluidHPNomFlowRate_Sh "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Pressure fluidHPNomPressure_Sh "Nominal pressure" annotation (Dialog(tab = "HP", group= "Sh"));
        parameter SI.MassFlowRate fluidHPNomFlowRate_Ev "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Pressure fluidHPNomPressure_Ev "Nominal pressure" annotation (Dialog(tab = "HP", group= "Ev"));
        parameter SI.MassFlowRate fluidHPNomFlowRate_Ec "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Pressure fluidHPNomPressure_Ec "Nominal pressure" annotation (Dialog(tab = "HP", group= "Ec"));
        parameter SI.MassFlowRate fluidLPNomFlowRate_Sh "Nominal mass flowrate"
                                                                                annotation (Dialog(tab = "LP", group= "Sh"));
        parameter SI.Pressure fluidLPNomPressure_Sh "Nominal pressure" annotation (Dialog(tab = "LP", group= "Sh"));
        parameter SI.MassFlowRate fluidLPNomFlowRate_Ev "Nominal mass flowrate"
                                                                                annotation (Dialog(tab = "LP", group= "Ev"));
        parameter SI.Pressure fluidLPNomPressure_Ev "Nominal pressure" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.MassFlowRate fluidLPNomFlowRate_Ec "Nominal mass flowrate"
                                                                                annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Pressure fluidLPNomPressure_Ec "Nominal pressure" annotation (Dialog(tab = "LP", group = "Ec"));
        
        //Physical Parameter
        //Sh2_HP
        parameter Integer Sh2_HP_N_G=2 "Number of node of the gas side, Sh2" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter Integer Sh2_HP_N_F=2 "Number of node of the fluid side, Sh2" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh2_HP_exchSurface_G 
          "Exchange surface between gas - metal tube, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh2_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh2_HP_extSurfaceTub 
          "Total external surface of the tubes, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh2_HP_gasVol "Gas volume, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh2_HP_fluidVol "Fluid volume, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh2_HP_metalVol 
          "Volume of the metal part in the tube, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        //Sh1_HP
        parameter Integer Sh1_HP_N_G=2 "Number of node of the gas side, Sh1" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter Integer Sh1_HP_N_F=2 "Number of node of the fluid side, Sh1" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh1_HP_exchSurface_G 
          "Exchange surface between gas - metal tube, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh1_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh1_HP_extSurfaceTub 
          "Total external surface of the tubes, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh1_HP_gasVol "Gas volume, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh1_HP_fluidVol "Fluid volume, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh1_HP_metalVol 
          "Volume of the metal part in the tube, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        //Ev_HP
        parameter Integer Ev_HP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "HP",group = "Ev"));
        parameter Integer Ev_HP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Area Ev_HP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Area Ev_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Area Ev_HP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Volume Ev_HP_gasVol "Gas volume" annotation (Dialog(tab = "HP", group= "Ev"));
        parameter SI.Volume Ev_HP_fluidVol "Fluid volume" annotation (Dialog(tab = "HP", group= "Ev"));
        parameter SI.Volume Ev_HP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab= "HP", group= "Ev"));
        //Ec_HP
        parameter Integer Ec_HP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab= "HP", group= "Ec"));
        parameter Integer Ec_HP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Area Ec_HP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Area Ec_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Area Ec_HP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Volume Ec_HP_gasVol "Gas volume" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Volume Ec_HP_fluidVol "Fluid volume" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Volume Ec_HP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab= "HP", group= "Ec"));
        //Sh_LP
        parameter Integer Sh_LP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "LP", group = "Sh"));
        parameter Integer Sh_LP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Area Sh_LP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Area Sh_LP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Area Sh_LP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Volume Sh_LP_gasVol "Gas volume" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Volume Sh_LP_fluidVol "Fluid volume" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Volume Sh_LP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "LP", group = "Sh"));
        //Ev_LP
        parameter Integer Ev_LP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "LP", group = "Ev"));
        parameter Integer Ev_LP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Area Ev_LP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Area Ev_LP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Area Ev_LP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Volume Ev_LP_gasVol "Gas volume" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Volume Ev_LP_fluidVol "Fluid volume" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Volume Ev_LP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "LP", group = "Ev"));
        //Ec_LP
        parameter Integer Ec_LP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "LP", group = "Ec"));
        parameter Integer Ec_LP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Area Ec_LP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Area Ec_LP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Area Ec_LP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Volume Ec_LP_gasVol "Gas volume" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Volume Ec_LP_fluidVol "Fluid volume" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Volume Ec_LP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "LP", group = "Ec"));
        
        //Initialization conditions
        parameter Boolean use_T_Sh = true 
          "Select -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the Superheaters (Reheater)"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean use_T_Ev = true 
          "Select -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the Evaporators"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean use_T_Ec = true 
          "Select -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the Economizers"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        
        //Start values
        //Sh2_HP
        parameter SI.Temperature Sh2_HP_Tstart_G_In 
          "Intlet gas temperature start value of Sh2" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_G_Out 
          "Outlet gas temperature start value of Sh2" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_M_In 
          "Inlet metal wall temperature start value of Sh2" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Sh2" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh2_HP_hstart_F_In) 
          "Inlet fluid temperature start value of Sh2" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh2_HP_hstart_F_Out) 
          "Outlet fluid temperature start value of Sh2" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh2_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh2_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Sh2" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh2_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh2_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Sh2" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (HP)", group = "Sh"));
        //Sh1_HP
        parameter SI.Temperature Sh1_HP_Tstart_G_In 
          "Intlet gas temperature start value of Sh1" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_G_Out 
          "Outlet gas temperature start value of Sh1" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_M_In 
          "Inlet metal wall temperature start value of Sh1" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Sh1" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh1_HP_hstart_F_In) 
          "Inlet fluid temperature start value of Sh1" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh1_HP_hstart_F_Out) 
          "Outlet fluid temperature start value of Sh1" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh1_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh1_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Sh1" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh1_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh1_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Sh1" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (HP)", group = "Sh"));
        //Ev_HP
        parameter SI.Temperature Ev_HP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Ev,Ev_HP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                           tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Ev,Ev_HP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                            tab = "Initialization (HP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ev,Ev_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                 tab = "Initialization (HP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ev,Ev_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                  tab = "Initialization (HP)", group = "Ev"));
        
        //Ec_HP
        parameter SI.Temperature Ec_HP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Ec,Ec_HP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                           tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Ec,Ec_HP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                            tab = "Initialization (HP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ec,Ec_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                 tab = "Initialization (HP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ec,Ec_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                  tab = "Initialization (HP)", group = "Ec"));
        
        //Sh_LP
        parameter SI.Temperature Sh_LP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_F_In=FluidMedium.temperature_ph(fluidLPNomPressure_Sh,Sh_LP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Sh,
                                                           tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_F_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Sh,Sh_LP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Sh,
                                                            tab = "Initialization (LP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh_LP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Sh,Sh_LP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Sh,
                                                                 tab = "Initialization (LP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh_LP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Sh,Sh_LP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Sh,
                                                                  tab = "Initialization (LP)", group = "Sh"));
        //Ev_LP
        parameter SI.Temperature Ev_LP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_F_In=FluidMedium.temperature_ph(fluidLPNomPressure_Ev,Ev_LP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                                 tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_F_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Ev,Ev_LP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                            tab = "Initialization (LP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_LP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ev,Ev_LP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                 tab = "Initialization (LP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_LP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ev,Ev_LP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                  tab = "Initialization (LP)", group = "Ev"));
        //Ec_LP
        parameter SI.Temperature Ec_LP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_F_In=FluidMedium.temperature_ph(fluidLPNomPressure_Ec,Ec_LP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                           tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_F_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Ec,Ec_LP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                            tab = "Initialization (LP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_LP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ec,Ec_LP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                 tab = "Initialization (LP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_LP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ec,Ec_LP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                  tab = "Initialization (LP)", group = "Ec"));
        
        Water.FlangeA Sh_HP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-294,186; -266,214]);
        Water.FlangeB Ev_HP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-254,186; -226,214]);
        Water.FlangeA Ev_HP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-214,186; -186,214]);
        Water.FlangeB Ec_HP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-174,186; -146,214]);
        Water.FlangeA Ec_HP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-134,186; -106,214]);
        Water.FlangeB Sh_HP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-214,-214; -186,-186]);
        Gas.FlangeA GasIn(redeclare package Medium = FlueGasMedium) 
                          annotation (extent=[-414,-14; -386,14]);
        Gas.FlangeB GasOut(redeclare package Medium = FlueGasMedium) 
                           annotation (extent=[386,-14; 414,14]);
        Water.FlangeA Sh_LP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[46,186; 74,214]);
        Water.FlangeB Ev_LP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[86,186; 114,214]);
        Water.FlangeA Ev_LP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[126,186; 154,214]);
        Water.FlangeB Ec_LP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[166,186; 194,214]);
        Water.FlangeA Ec_LP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[206,186; 234,214]);
        Water.FlangeB Sh_LP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[126,-214; 154,-186]);
        Buses.Sensors SensorsBus 
                              annotation (extent=[380,120; 420,160]);
        Buses.Actuators ActuatorsBus 
                                  annotation (extent=[420,60; 380,100]);
      end HEG_2L;
      
      partial model HEG_2LRh 
        "Base class for Heat Exchangers Group with two pressure levels" 
        
        annotation (Icon(
            Rectangle(extent=[-400,200; 400,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[-120,200; -120,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-160,-100; -156,-110; -144,-120; -136,-120; -124,-110;
                  -120,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-160,200; -160,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-200,200; -200,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-240,-100; -236,-110; -224,-120; -216,-120; -204,-110;
                  -200,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-240,200; -240,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-200,-200; -200,-140; -280,-140; -280,204; -278,200],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-400,80; 400,80], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,60; 400,60], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,40; 400,40], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,100; 400,100], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,0; 400,0], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-20; 400,-20], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-40; 400,-40], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,20; 400,20], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-80; 400,-80], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-100; 400,-100], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-400,-60; 400,-60], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-60,100; -50,114; -34,120; -26,120; -10,114; 0,100],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-60,100; -60,-198], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[0,100; 0,-198], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[140,-200; 140,-140; 60,-140; 60,204; 62,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[100,-100; 104,-110; 116,-120; 124,-120; 136,-110; 140,
                  -100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[100,200; 100,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[140,200; 140,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[180,202; 180,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[220,202; 220,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[180,-100; 184,-110; 196,-120; 204,-120; 216,-110; 220,
                  -100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-286,138; -274,138; -280,122; -286,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-246,144; -234,144; -240,160; -246,144],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-206,138; -194,138; -200,122; -206,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-166,144; -154,144; -160,160; -166,144],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-126,138; -114,138; -120,122; -126,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[54,138; 66,138; 60,122; 54,138],         style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[94,144; 106,144; 100,160; 94,144],      style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[134,138; 146,138; 140,122; 134,138],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[174,144; 186,144; 180,160; 174,144],    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[214,138; 226,138; 220,122; 214,138],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-66,-142; -54,-142; -60,-158; -66,-142], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-6,-136; 6,-136; 0,-120; -6,-136],      style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Text(
              extent=[-78,-132; 22,-152],
              style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillPattern=1),
              string="Rh")), Diagram,
          Coordsys(extent=[-400,-200; 400,200], scale=0.1));
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Common Parameter
        parameter SI.MassFlowRate gasNomFlowRate 
          "Nominal flow rate through the gas side";
        parameter SI.Pressure gasNomPressure 
          "Nominal pressure in the gas side inlet";
        
        //Nominal parameter
        parameter SI.MassFlowRate fluidHPNomFlowRate_Sh "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Pressure fluidHPNomPressure_Sh "Nominal pressure" annotation (Dialog(tab = "HP", group= "Sh"));
        parameter SI.MassFlowRate fluidHPNomFlowRate_Ev "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Pressure fluidHPNomPressure_Ev "Nominal pressure" annotation (Dialog(tab = "HP", group= "Ev"));
        parameter SI.MassFlowRate fluidHPNomFlowRate_Ec "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Pressure fluidHPNomPressure_Ec "Nominal pressure" annotation (Dialog(tab = "HP", group= "Ec"));
        parameter SI.MassFlowRate fluidLPNomFlowRate_Sh "Nominal mass flowrate"
                                                                                annotation (Dialog(tab = "LP", group= "Sh"));
        parameter SI.Pressure fluidLPNomPressure_Sh "Nominal pressure" annotation (Dialog(tab = "LP", group= "Sh"));
        parameter SI.MassFlowRate fluidLPNomFlowRate_Ev "Nominal mass flowrate"
                                                                                annotation (Dialog(tab = "LP", group= "Ev"));
        parameter SI.Pressure fluidLPNomPressure_Ev "Nominal pressure" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.MassFlowRate fluidLPNomFlowRate_Ec "Nominal mass flowrate"
                                                                                annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Pressure fluidLPNomPressure_Ec "Nominal pressure" annotation (Dialog(tab = "LP", group = "Ec"));
        
        //Physical Parameter
        //Sh2_HP
        parameter Integer Sh2_HP_N_G=2 "Number of node of the gas side, Sh2" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter Integer Sh2_HP_N_F=2 "Number of node of the fluid side, Sh2" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh2_HP_exchSurface_G 
          "Exchange surface between gas - metal tube, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh2_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh2_HP_extSurfaceTub 
          "Total external surface of the tubes, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh2_HP_gasVol "Gas volume, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh2_HP_fluidVol "Fluid volume, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh2_HP_metalVol 
          "Volume of the metal part in the tube, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        //Sh1_HP
        parameter Integer Sh1_HP_N_G=2 "Number of node of the gas side, Sh1" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter Integer Sh1_HP_N_F=2 "Number of node of the fluid side, Sh1" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh1_HP_exchSurface_G 
          "Exchange surface between gas - metal tube, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh1_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh1_HP_extSurfaceTub 
          "Total external surface of the tubes, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh1_HP_gasVol "Gas volume, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh1_HP_fluidVol "Fluid volume, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh1_HP_metalVol 
          "Volume of the metal part in the tube, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        //Ev_HP
        parameter Integer Ev_HP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "HP",group = "Ev"));
        parameter Integer Ev_HP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Area Ev_HP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Area Ev_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Area Ev_HP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Volume Ev_HP_gasVol "Gas volume" annotation (Dialog(tab = "HP", group= "Ev"));
        parameter SI.Volume Ev_HP_fluidVol "Fluid volume" annotation (Dialog(tab = "HP", group= "Ev"));
        parameter SI.Volume Ev_HP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab= "HP", group= "Ev"));
        //Ec_HP
        parameter Integer Ec_HP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab= "HP", group= "Ec"));
        parameter Integer Ec_HP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Area Ec_HP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Area Ec_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Area Ec_HP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Volume Ec_HP_gasVol "Gas volume" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Volume Ec_HP_fluidVol "Fluid volume" annotation (Dialog(tab= "HP", group= "Ec"));
        parameter SI.Volume Ec_HP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab= "HP", group= "Ec"));
        //Rh2_IP
        parameter Integer Rh2_IP_N_G=2 "Number of node of the gas side, Rh2" 
            annotation (Dialog(tab = "IP", group = "Rh"));
        parameter Integer Rh2_IP_N_F=2 "Number of node of the fluid side, Rh2" 
            annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh2_IP_exchSurface_G 
          "Exchange surface between gas - metal tube, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh2_IP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh2_IP_extSurfaceTub 
          "Total external surface of the tubes, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh2_IP_gasVol "Gas volume, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh2_IP_fluidVol "Fluid volume, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh2_IP_metalVol 
          "Volume of the metal part in the tube, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        //Rh1_IP
        parameter Integer Rh1_IP_N_G=2 "Number of node of the gas side, Rh1"  annotation (Dialog(tab = "IP", group = "Rh"));
        parameter Integer Rh1_IP_N_F=2 "Number of node of the fluid side, Rh1"  annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh1_IP_exchSurface_G 
          "Exchange surface between gas - metal tube, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh1_IP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh1_IP_extSurfaceTub 
          "Total external surface of the tubes, Rh1"                                      annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh1_IP_gasVol "Gas volume, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh1_IP_fluidVol "Fluid volume, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh1_IP_metalVol 
          "Volume of the metal part in the tube, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        //Sh_LP
        parameter Integer Sh_LP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "LP", group = "Sh"));
        parameter Integer Sh_LP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Area Sh_LP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Area Sh_LP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Area Sh_LP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Volume Sh_LP_gasVol "Gas volume" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Volume Sh_LP_fluidVol "Fluid volume" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Volume Sh_LP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "LP", group = "Sh"));
        //Ev_LP
        parameter Integer Ev_LP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "LP", group = "Ev"));
        parameter Integer Ev_LP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Area Ev_LP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Area Ev_LP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Area Ev_LP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Volume Ev_LP_gasVol "Gas volume" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Volume Ev_LP_fluidVol "Fluid volume" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Volume Ev_LP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "LP", group = "Ev"));
        //Ec_LP
        parameter Integer Ec_LP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "LP", group = "Ec"));
        parameter Integer Ec_LP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Area Ec_LP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Area Ec_LP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Area Ec_LP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Volume Ec_LP_gasVol "Gas volume" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Volume Ec_LP_fluidVol "Fluid volume" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Volume Ec_LP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "LP", group = "Ec"));
        
        //Initialization conditions
        parameter Boolean use_T_Sh = true 
          "Select -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the Superheaters (Reheater)"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean use_T_Ev = true 
          "Select -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the Evaporators"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean use_T_Ec = true 
          "Select -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the Economizers"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        
        //Start values
        //Sh2_HP
        parameter SI.Temperature Sh2_HP_Tstart_G_In 
          "Intlet gas temperature start value of Sh2" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_G_Out 
          "Outlet gas temperature start value of Sh2" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_M_In 
          "Inlet metal wall temperature start value of Sh2" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Sh2" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh2_HP_hstart_F_In) 
          "Inlet fluid temperature start value of Sh2" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh2_HP_hstart_F_Out) 
          "Outlet fluid temperature start value of Sh2" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh2_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh2_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Sh2" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh2_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh2_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Sh2" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (HP)", group = "Sh"));
        //Sh1_HP
        parameter SI.Temperature Sh1_HP_Tstart_G_In 
          "Intlet gas temperature start value of Sh1" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_G_Out 
          "Outlet gas temperature start value of Sh1" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_M_In 
          "Inlet metal wall temperature start value of Sh1" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Sh1" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh1_HP_hstart_F_In) 
          "Inlet fluid temperature start value of Sh1" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh1_HP_hstart_F_Out) 
          "Outlet fluid temperature start value of Sh1" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh1_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh1_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Sh1" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh1_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh1_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Sh1" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (HP)", group = "Sh"));
        //Ev_HP
        parameter SI.Temperature Ev_HP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Ev,Ev_HP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                           tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Ev,Ev_HP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                            tab = "Initialization (HP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ev,Ev_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                 tab = "Initialization (HP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ev,Ev_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                  tab = "Initialization (HP)", group = "Ev"));
        
        //Ec_HP
        parameter SI.Temperature Ec_HP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Ec,Ec_HP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                           tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Ec,Ec_HP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                            tab = "Initialization (HP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ec,Ec_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                 tab = "Initialization (HP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ec,Ec_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                  tab = "Initialization (HP)", group = "Ec"));
        //Rh2_IP
        parameter SI.Temperature Rh2_IP_Tstart_G_In 
          "Intlet gas temperature start value of Rh2" annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_G_Out 
          "Outlet gas temperature start value of Rh2" annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_M_In 
          "Inlet metal wall temperature start value of Rh2" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Rh2" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Rh2_IP_hstart_F_In) 
          "Inlet fluid temperature start value of Rh2" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Rh2_IP_hstart_F_Out) 
          "Outlet fluid temperature start value of Rh2" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (IP)", group = "Rh"));
        parameter SI.SpecificEnthalpy Rh2_IP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Rh2_IP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Rh2" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (IP)", group = "Rh"));
        parameter SI.SpecificEnthalpy Rh2_IP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Rh2_IP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Rh2" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (IP)", group = "Rh"));
        //Rh1_HP
        parameter SI.Temperature Rh1_IP_Tstart_G_In 
          "Intlet gas temperature start value of Rh1" annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_G_Out 
          "Outlet gas temperature start value of Rh1" annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_M_In 
          "Inlet metal wall temperature start value of Rh1" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Rh1" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Rh1_IP_hstart_F_In) 
          "Inlet fluid temperature start value of Rh1" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Rh1_IP_hstart_F_Out) 
          "Outlet fluid temperature start value of Rh1" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (IP)", group = "Rh"));
        parameter SI.SpecificEnthalpy Rh1_IP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Rh1_IP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Rh1" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (IP)", group = "Rh"));
        parameter SI.SpecificEnthalpy Rh1_IP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Rh1_IP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Rh1" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (IP)", group = "Rh"));
        
        //Sh_LP
        parameter SI.Temperature Sh_LP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_F_In=FluidMedium.temperature_ph(fluidLPNomPressure_Sh,Sh_LP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Sh,
                                                           tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_F_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Sh,Sh_LP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Sh,
                                                            tab = "Initialization (LP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh_LP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Sh,Sh_LP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Sh,
                                                                 tab = "Initialization (LP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh_LP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Sh,Sh_LP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Sh,
                                                                  tab = "Initialization (LP)", group = "Sh"));
        //Ev_LP
        parameter SI.Temperature Ev_LP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_F_In=FluidMedium.temperature_ph(fluidLPNomPressure_Ev,Ev_LP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                                 tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_F_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Ev,Ev_LP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                            tab = "Initialization (LP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_LP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ev,Ev_LP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                 tab = "Initialization (LP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_LP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ev,Ev_LP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                  tab = "Initialization (LP)", group = "Ev"));
        //Ec_LP
        parameter SI.Temperature Ec_LP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_F_In=FluidMedium.temperature_ph(fluidLPNomPressure_Ec,Ec_LP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                           tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_F_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Ec,Ec_LP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                            tab = "Initialization (LP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_LP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ec,Ec_LP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                 tab = "Initialization (LP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_LP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ec,Ec_LP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                  tab = "Initialization (LP)", group = "Ec"));
        
        Water.FlangeA Sh_HP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-294,186; -266,214]);
        Water.FlangeB Ev_HP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-254,186; -226,214]);
        Water.FlangeA Ev_HP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-214,186; -186,214]);
        Water.FlangeB Ec_HP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-174,186; -146,214]);
        Water.FlangeA Ec_HP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-134,186; -106,214]);
        Water.FlangeB Sh_HP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-214,-214; -186,-186]);
        Gas.FlangeA GasIn(redeclare package Medium = FlueGasMedium) 
                          annotation (extent=[-414,-14; -386,14]);
        Gas.FlangeB GasOut(redeclare package Medium = FlueGasMedium) 
                           annotation (extent=[386,-14; 414,14]);
        Water.FlangeA Sh_LP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[46,186; 74,214]);
        Water.FlangeB Ev_LP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[86,186; 114,214]);
        Water.FlangeA Ev_LP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[126,186; 154,214]);
        Water.FlangeB Ec_LP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[166,186; 194,214]);
        Water.FlangeA Ec_LP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[206,186; 234,214]);
        Water.FlangeA Rh_IP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-14,-214; 14,-186]);
        Water.FlangeB Rh_IP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-74,-214; -46,-186]);
        Water.FlangeB Sh_LP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[126,-214; 154,-186]);
        Buses.Sensors SensorsBus 
                              annotation (extent=[380,120; 420,160]);
        Buses.Actuators ActuatorsBus 
                                  annotation (extent=[420,60; 380,100]);
      end HEG_2LRh;
      
      partial model HEG_3LRh 
        "Base class for Heat Exchangers Group with three pressure levels and reheat" 
        
        Water.FlangeA Sh_HP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-394,186; -366,214]);
        Water.FlangeB Ev_HP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-354,186; -326,214]);
        Water.FlangeA Ev_HP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-314,186; -286,214]);
        Water.FlangeB Ec_HP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-274,186; -246,214]);
        Water.FlangeA Ec_HP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-234,186; -206,214]);
        Water.FlangeA Sh_IP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-94,186; -66,214]);
        Water.FlangeB Ev_IP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-54,186; -26,214]);
        Water.FlangeA Ev_IP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-14,186; 14,214]);
        Water.FlangeB Ec_IP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[26,186; 54,214]);
        Water.FlangeA Ec_IP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[66,186; 94,214]);
        Water.FlangeB Sh_HP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-314,-214; -286,-186]);
        Gas.FlangeA GasIn(redeclare package Medium = FlueGasMedium) 
                          annotation (extent=[-514,-14; -486,14]);
        Gas.FlangeB GasOut(redeclare package Medium = FlueGasMedium) 
                           annotation (extent=[486,-14; 514,14]);
        annotation (Icon(
            Rectangle(extent=[-500,200; 500,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Text(
              extent=[-210,-140; -110,-160],
              style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillPattern=1),
              string="Rh"),
            Line(points=[380,202; 380,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[340,-100; 344,-110; 356,-120; 364,-120; 376,-110; 380,
                  -100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[340,202; 340,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[300,202; 300,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[260,-100; 264,-110; 276,-120; 284,-120; 296,-110; 300,
                  -100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[260,202; 260,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[80,200; 80,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[40,-100; 44,-110; 56,-120; 64,-120; 76,-110; 80,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[40,202; 40,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[0,200; 0,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,-100; -36,-110; -24,-120; -16,-120; -4,-110; 0,
                  -100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,200; -40,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-220,200; -220,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-260,-100; -256,-110; -244,-120; -236,-120; -224,-110;
                  -220,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-260,200; -260,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-300,202; -300,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-340,-100; -336,-110; -324,-120; -316,-120; -304,-110;
                  -300,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-340,200; -340,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-300,-200; -300,-140; -380,-140; -380,204; -378,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-130,-140; -80,-140; -80,-140; -80,204; -78,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[300,-200; 300,-140; 220,-140; 220,204; 222,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-192,100; -180,114; -164,120; -156,120; -140,114; -130,
                  100], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-130,100; -130,-198], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-190,100; -190,-198], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[-500,80; 500,80], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,60; 500,60], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,40; 500,40], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,100; 500,100], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,0; 500,0], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,-20; 500,-20], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,-40; 500,-40], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,20; 500,20], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,-80; 500,-80], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,-100; 500,-100], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Line(points=[-500,-60; 500,-60], style(
                color=76,
                rgbcolor={170,170,255},
                pattern=2,
                thickness=2,
                fillPattern=1)),
            Polygon(points=[-386,138; -374,138; -380,122; -386,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-346,144; -334,144; -340,160; -346,144],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-306,138; -294,138; -300,122; -306,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-266,144; -254,144; -260,160; -266,144],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-226,138; -214,138; -220,122; -226,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-86,138; -74,138; -80,122; -86,138],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,144; -34,144; -40,160; -46,144],    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-6,138; 6,138; 0,122; -6,138],           style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[34,144; 46,144; 40,160; 34,144],        style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,138; 86,138; 80,122; 74,138],         style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[214,138; 226,138; 220,122; 214,138],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[254,144; 266,144; 260,160; 254,144],    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[294,138; 306,138; 300,122; 294,138],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[334,144; 346,144; 340,160; 334,144],    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[374,138; 386,138; 380,122; 374,138],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-196,-144; -184,-144; -190,-160; -196,-144],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-136,-170; -124,-170; -130,-154; -136,-170],
                                                                    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-136,-124; -124,-124; -130,-108; -136,-124],
                                                                    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-98,-146; -98,-134; -114,-140; -98,-146],
                                                                    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[-134,-136; -126,-144],
                                           style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1))),
            Diagram,
          Coordsys(extent=[-500,-200; 500,200], scale=0.1));
        Water.FlangeA Sh_LP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[206,186; 234,214]);
        Water.FlangeB Ev_LP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[246,186; 274,214]);
        Water.FlangeA Ev_LP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[286,186; 314,214]);
        Water.FlangeB Ec_LP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[326,186; 354,214]);
        Water.FlangeA Ec_LP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[366,186; 394,214]);
        Water.FlangeA Rh_IP_In(redeclare package Medium = FluidMedium) 
          annotation (extent=[-144,-214; -116,-186]);
        Water.FlangeB Rh_IP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[-204,-214; -176,-186]);
        Water.FlangeB Sh_LP_Out(redeclare package Medium = FluidMedium) 
          annotation (extent=[286,-212; 314,-184]);
        Buses.Sensors SensorsBus 
                              annotation (extent=[480,120; 520,160]);
        Buses.Actuators ActuatorsBus 
                                  annotation (extent=[520,60; 480,100]);
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Common Parameter
        parameter SI.MassFlowRate gasNomFlowRate 
          "Nominal flow rate through the gas side";
        parameter SI.Pressure gasNomPressure 
          "Nominal pressure in the gas side inlet";
        
        //Nominal parameter
        parameter SI.MassFlowRate fluidHPNomFlowRate_Sh "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "HP", group = "Sh"));
        parameter SI.Pressure fluidHPNomPressure_Sh "Nominal pressure" annotation (Dialog(tab= "HP", group = "Sh"));
        parameter SI.MassFlowRate fluidHPNomFlowRate_Ev "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "HP", group = "Ev"));
        parameter SI.Pressure fluidHPNomPressure_Ev "Nominal pressure" annotation (Dialog(tab= "HP", group = "Ev"));
        parameter SI.MassFlowRate fluidHPNomFlowRate_Ec "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "HP", group = "Ec"));
        parameter SI.Pressure fluidHPNomPressure_Ec "Nominal pressure" annotation (Dialog(tab= "HP", group = "Ec"));
        parameter SI.MassFlowRate fluidIPNomFlowRate_Rh "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "IP", group = "Rh"));
        parameter SI.Pressure fluidIPNomPressure_Rh "Nominal pressure" annotation (Dialog(tab= "IP", group = "Rh"));
        parameter SI.MassFlowRate fluidIPNomFlowRate_Sh "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "IP", group = "Sh"));
        parameter SI.Pressure fluidIPNomPressure_Sh "Nominal pressure" annotation (Dialog(tab= "IP", group = "Sh"));
        parameter SI.MassFlowRate fluidIPNomFlowRate_Ev "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "IP", group = "Ev"));
        parameter SI.Pressure fluidIPNomPressure_Ev "Nominal pressure" annotation (Dialog(tab= "IP", group = "Ev"));
        parameter SI.MassFlowRate fluidIPNomFlowRate_Ec "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "IP", group = "Ec"));
        parameter SI.Pressure fluidIPNomPressure_Ec "Nominal pressure" annotation (Dialog(tab= "IP", group = "Ec"));
        parameter SI.MassFlowRate fluidLPNomFlowRate_Sh "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "LP", group = "Sh"));
        parameter SI.Pressure fluidLPNomPressure_Sh "Nominal pressure" annotation (Dialog(tab= "LP", group = "Sh"));
        parameter SI.MassFlowRate fluidLPNomFlowRate_Ev "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "LP", group = "Ev"));
        parameter SI.Pressure fluidLPNomPressure_Ev "Nominal pressure" annotation (Dialog(tab= "LP", group = "Ev"));
        parameter SI.MassFlowRate fluidLPNomFlowRate_Ec "Nominal mass flowrate"
                                                                                annotation (Dialog(tab= "LP", group = "Ec"));
        parameter SI.Pressure fluidLPNomPressure_Ec "Nominal pressure" annotation (Dialog(tab= "LP", group = "Ec"));
        
        //Physical Parameter
        //Sh2_HP
        parameter Integer Sh2_HP_N_G=2 "Number of node of the gas side, Sh2" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter Integer Sh2_HP_N_F=2 "Number of node of the fluid side, Sh2" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh2_HP_exchSurface_G 
          "Exchange surface between gas - metal tube, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh2_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh2_HP_extSurfaceTub 
          "Total external surface of the tubes, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh2_HP_gasVol "Gas volume, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh2_HP_fluidVol "Fluid volume, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh2_HP_metalVol 
          "Volume of the metal part in the tube, Sh2" annotation (Dialog(tab= "HP", group= "Sh"));
        //Sh1_HP
        parameter Integer Sh1_HP_N_G=2 "Number of node of the gas side, Sh1" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter Integer Sh1_HP_N_F=2 "Number of node of the fluid side, Sh1" 
            annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh1_HP_exchSurface_G 
          "Exchange surface between gas - metal tube, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh1_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Area Sh1_HP_extSurfaceTub 
          "Total external surface of the tubes, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh1_HP_gasVol "Gas volume, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh1_HP_fluidVol "Fluid volume, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        parameter SI.Volume Sh1_HP_metalVol 
          "Volume of the metal part in the tube, Sh1" annotation (Dialog(tab= "HP", group= "Sh"));
        //Ev_HP
        parameter Integer Ev_HP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "HP",group = "Ev"));
        parameter Integer Ev_HP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Area Ev_HP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Area Ev_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Area Ev_HP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab= "HP", group= "Ev"));
        parameter SI.Volume Ev_HP_gasVol "Gas volume" annotation (Dialog(tab = "HP", group= "Ev"));
        parameter SI.Volume Ev_HP_fluidVol "Fluid volume" annotation (Dialog(tab = "HP", group= "Ev"));
        parameter SI.Volume Ev_HP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab= "HP", group= "Ev"));
        //Ec2_HP
        parameter Integer Ec2_HP_N_G=2 "Number of node of the gas side, Ec2" 
            annotation (Dialog(tab = "HP", group = "Ec"));
        parameter Integer Ec2_HP_N_F=2 "Number of node of the fluid side, Ec2" 
            annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Area Ec2_HP_exchSurface_G 
          "Exchange surface between gas - metal tube, Ec2" annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Area Ec2_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Ec2" annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Area Ec2_HP_extSurfaceTub 
          "Total external surface of the tubes, Ec2" 
                                                annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Volume Ec2_HP_gasVol "Gas volume, Ec2" annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Volume Ec2_HP_fluidVol "Fluid volume, Ec2" annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Volume Ec2_HP_metalVol 
          "Volume of the metal part in the tube, Ec2" 
                                                 annotation (Dialog(tab = "HP", group = "Ec"));
        //Ec1_HP
        parameter Integer Ec1_HP_N_G=2 "Number of node of the gas side, Ec1" 
            annotation (Dialog(tab = "HP", group = "Ec"));
        parameter Integer Ec1_HP_N_F=2 "Number of node of the fluid side, Ec1" 
            annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Area Ec1_HP_exchSurface_G 
          "Exchange surface between gas - metal tube, Ec1" annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Area Ec1_HP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Ec1" annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Area Ec1_HP_extSurfaceTub 
          "Total external surface of the tubes, Ec1" 
                                                annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Volume Ec1_HP_gasVol "Gas volume, Ec1" annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Volume Ec1_HP_fluidVol "Fluid volume, Ec1" annotation (Dialog(tab = "HP", group = "Ec"));
        parameter SI.Volume Ec1_HP_metalVol 
          "Volume of the metal part in the tube, Ec1" 
                                                 annotation (Dialog(tab = "HP", group = "Ec"));
        //Rh2_IP
        parameter Integer Rh2_IP_N_G=2 "Number of node of the gas side, Rh2" 
            annotation (Dialog(tab = "IP", group = "Rh"));
        parameter Integer Rh2_IP_N_F=2 "Number of node of the fluid side, Rh2" 
            annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh2_IP_exchSurface_G 
          "Exchange surface between gas - metal tube, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh2_IP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh2_IP_extSurfaceTub 
          "Total external surface of the tubes, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh2_IP_gasVol "Gas volume, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh2_IP_fluidVol "Fluid volume, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh2_IP_metalVol 
          "Volume of the metal part in the tube, Rh2" annotation (Dialog(tab = "IP", group = "Rh"));
        //Rh1_IP
        parameter Integer Rh1_IP_N_G=2 "Number of node of the gas side, Rh1"  annotation (Dialog(tab = "IP", group = "Rh"));
        parameter Integer Rh1_IP_N_F=2 "Number of node of the fluid side, Rh1"  annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh1_IP_exchSurface_G 
          "Exchange surface between gas - metal tube, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh1_IP_exchSurface_F 
          "Exchange surface between metal tube - fluid, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Area Rh1_IP_extSurfaceTub 
          "Total external surface of the tubes, Rh1"                                      annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh1_IP_gasVol "Gas volume, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh1_IP_fluidVol "Fluid volume, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        parameter SI.Volume Rh1_IP_metalVol 
          "Volume of the metal part in the tube, Rh1" annotation (Dialog(tab = "IP", group = "Rh"));
        //Sh_IP
        parameter Integer Sh_IP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "IP", group = "Sh"));
        parameter Integer Sh_IP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "IP", group = "Sh"));
        parameter SI.Area Sh_IP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "IP", group = "Sh"));
        parameter SI.Area Sh_IP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "IP", group = "Sh"));
        parameter SI.Area Sh_IP_extSurfaceTub 
          "Total external surface of the tube" annotation (Dialog(tab = "IP", group = "Sh"));
        parameter SI.Volume Sh_IP_gasVol "Gas volume" annotation (Dialog(tab = "IP", group = "Sh"));
        parameter SI.Volume Sh_IP_fluidVol "Fluid volume" annotation (Dialog(tab = "IP", group = "Sh"));
        parameter SI.Volume Sh_IP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "IP", group = "Sh"));
        //Ev_IP
        parameter Integer Ev_IP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "IP", group = "Ev"));
        parameter Integer Ev_IP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "IP", group = "Ev"));
        parameter SI.Area Ev_IP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "IP", group = "Ev"));
        parameter SI.Area Ev_IP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "IP", group = "Ev"));
        parameter SI.Area Ev_IP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "IP", group = "Ev"));
        parameter SI.Volume Ev_IP_gasVol "Gas volume" annotation (Dialog(tab = "IP", group = "Ev"));
        parameter SI.Volume Ev_IP_fluidVol "Fluid volume" annotation (Dialog(tab = "IP", group = "Ev"));
        parameter SI.Volume Ev_IP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "IP", group = "Ev"));
        //Ec_IP
        parameter Integer Ec_IP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "IP", group = "Ec"));
        parameter Integer Ec_IP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "IP", group = "Ec"));
        parameter SI.Area Ec_IP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "IP", group = "Ec"));
        parameter SI.Area Ec_IP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "IP", group = "Ec"));
        parameter SI.Area Ec_IP_extSurfaceTub 
          "Total external surface of the tube" annotation (Dialog(tab = "IP", group = "Ec"));
        parameter SI.Volume Ec_IP_gasVol "Gas volume" annotation (Dialog(tab = "IP", group = "Ec"));
        parameter SI.Volume Ec_IP_fluidVol "Fluid volume" annotation (Dialog(tab = "IP", group = "Ec"));
        parameter SI.Volume Ec_IP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "IP", group = "Ec"));
        //Sh_LP
        parameter Integer Sh_LP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "LP", group = "Sh"));
        parameter Integer Sh_LP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Area Sh_LP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Area Sh_LP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Area Sh_LP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Volume Sh_LP_gasVol "Gas volume" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Volume Sh_LP_fluidVol "Fluid volume" annotation (Dialog(tab = "LP", group = "Sh"));
        parameter SI.Volume Sh_LP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "LP", group = "Sh"));
        //Ev_LP
        parameter Integer Ev_LP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "LP", group = "Ev"));
        parameter Integer Ev_LP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Area Ev_LP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Area Ev_LP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Area Ev_LP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Volume Ev_LP_gasVol "Gas volume" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Volume Ev_LP_fluidVol "Fluid volume" annotation (Dialog(tab = "LP", group = "Ev"));
        parameter SI.Volume Ev_LP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "LP", group = "Ev"));
        //Ec_LP
        parameter Integer Ec_LP_N_G=2 "Number of node of the gas side" 
            annotation (Dialog(tab = "LP", group = "Ec"));
        parameter Integer Ec_LP_N_F=2 "Number of node of the fluid side" 
            annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Area Ec_LP_exchSurface_G 
          "Exchange surface between gas - metal tube" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Area Ec_LP_exchSurface_F 
          "Exchange surface between metal tube - fluid" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Area Ec_LP_extSurfaceTub 
          "Total external surface of the tubes" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Volume Ec_LP_gasVol "Gas volume" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Volume Ec_LP_fluidVol "Fluid volume" annotation (Dialog(tab = "LP", group = "Ec"));
        parameter SI.Volume Ec_LP_metalVol 
          "Volume of the metal part in the tube" annotation (Dialog(tab = "LP", group = "Ec"));
        
        //Initialization conditions
        parameter Boolean use_T_Sh = true 
          "Select -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the Superheaters (Reheater)"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean use_T_Ev = true 
          "Select -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the Evaporators"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean use_T_Ec = true 
          "Select -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the Economizers"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        
        //Start values
        //Sh2_HP
        parameter SI.Temperature Sh2_HP_Tstart_G_In 
          "Intlet gas temperature start value of Sh2" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_G_Out 
          "Outlet gas temperature start value of Sh2" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_M_In 
          "Inlet metal wall temperature start value of Sh2" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Sh2" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh2_HP_hstart_F_In) 
          "Inlet fluid temperature start value of Sh2" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh2_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh2_HP_hstart_F_Out) 
          "Outlet fluid temperature start value of Sh2" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh2_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh2_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Sh2" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh2_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh2_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Sh2" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (HP)", group = "Sh"));
        //Sh1_HP
        parameter SI.Temperature Sh1_HP_Tstart_G_In 
          "Intlet gas temperature start value of Sh1" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_G_Out 
          "Outlet gas temperature start value of Sh1" annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_M_In 
          "Inlet metal wall temperature start value of Sh1" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Sh1" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh1_HP_hstart_F_In) 
          "Inlet fluid temperature start value of Sh1" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (HP)", group = "Sh"));
        parameter SI.Temperature Sh1_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Sh,Sh1_HP_hstart_F_Out) 
          "Outlet fluid temperature start value of Sh1" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh1_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh1_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Sh1" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (HP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh1_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Sh,Sh1_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Sh1" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (HP)", group = "Sh"));
        //Ev_HP
        parameter SI.Temperature Ev_HP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Ev,Ev_HP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                           tab = "Initialization (HP)", group = "Ev"));
        parameter SI.Temperature Ev_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Ev,Ev_HP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                            tab = "Initialization (HP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ev,Ev_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                 tab = "Initialization (HP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ev,Ev_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                  tab = "Initialization (HP)", group = "Ev"));
        //Ec2_HP
        parameter SI.Temperature Ec2_HP_Tstart_G_In 
          "Intlet gas temperature start value of Ec2" annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec2_HP_Tstart_G_Out 
          "Outlet gas temperature start value of Ec2" annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec2_HP_Tstart_M_In 
          "Inlet metal wall temperature start value of Ec2" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec2_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Ec2" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec2_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Ec,Ec2_HP_hstart_F_In) 
          "Inlet fluid temperature start value of Ec2" annotation(Dialog(enable = use_T_Ec,
                                                                  tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec2_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Ec,Ec2_HP_hstart_F_Out) 
          "Outlet fluid temperature start value of Ec2" annotation(Dialog(enable = use_T_Ec,
                                                                   tab = "Initialization (HP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec2_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ec,Ec2_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Ec2" annotation(Dialog(enable = not use_T_Ec,
                                                                        tab = "Initialization (HP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec2_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ec,Ec2_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Ec2" annotation(Dialog(enable = not use_T_Ec,
                                                                         tab = "Initialization (HP)", group = "Ec"));
        //Ec1_HP
        parameter SI.Temperature Ec1_HP_Tstart_G_In 
          "Intlet gas temperature start value of Ec1" annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec1_HP_Tstart_G_Out 
          "Outlet gas temperature start value of Ec1" annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec1_HP_Tstart_M_In 
          "Inlet metal wall temperature start value of Ec1" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec1_HP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Ec1" 
            annotation(Dialog(tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec1_HP_Tstart_F_In=FluidMedium.temperature_ph(fluidHPNomPressure_Ec,Ec1_HP_hstart_F_In) 
          "Inlet fluid temperature start value of Ec1" annotation(Dialog(enable = use_T_Ec,
                                                                  tab = "Initialization (HP)", group = "Ec"));
        parameter SI.Temperature Ec1_HP_Tstart_F_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Ec,Ec1_HP_hstart_F_Out) 
          "Outlet fluid temperature start value of Ec1" annotation(Dialog(enable = use_T_Ec,
                                                                   tab = "Initialization (HP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec1_HP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ec,Ec1_HP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Ec1" annotation(Dialog(enable = not use_T_Ec,
                                                                        tab = "Initialization (HP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec1_HP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidHPNomPressure_Ec,Ec1_HP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Ec1" annotation(Dialog(enable = not use_T_Ec,
                                                                         tab = "Initialization (HP)", group = "Ec"));
        //Rh2_IP
        parameter SI.Temperature Rh2_IP_Tstart_G_In 
          "Intlet gas temperature start value of Rh2" annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_G_Out 
          "Outlet gas temperature start value of Rh2" annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_M_In 
          "Inlet metal wall temperature start value of Rh2" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Rh2" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_F_In=FluidMedium.temperature_ph(fluidIPNomPressure_Rh,Rh2_IP_hstart_F_In) 
          "Inlet fluid temperature start value of Rh2" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh2_IP_Tstart_F_Out=FluidMedium.temperature_ph(fluidIPNomPressure_Rh,Rh2_IP_hstart_F_Out) 
          "Outlet fluid temperature start value of Rh2" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (IP)", group = "Rh"));
        parameter SI.SpecificEnthalpy Rh2_IP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Rh,Rh2_IP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Rh2" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (IP)", group = "Rh"));
        parameter SI.SpecificEnthalpy Rh2_IP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Rh,Rh2_IP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Rh2" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (IP)", group = "Rh"));
        //Rh1_HP
        parameter SI.Temperature Rh1_IP_Tstart_G_In 
          "Intlet gas temperature start value of Rh1" annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_G_Out 
          "Outlet gas temperature start value of Rh1" annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_M_In 
          "Inlet metal wall temperature start value of Rh1" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_M_Out 
          "Outlet metal wall temperature start value of Rh1" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_F_In=FluidMedium.temperature_ph(fluidIPNomPressure_Rh,Rh1_IP_hstart_F_In) 
          "Inlet fluid temperature start value of Rh1" annotation(Dialog(enable = use_T_Sh,
                                                                  tab = "Initialization (IP)", group = "Rh"));
        parameter SI.Temperature Rh1_IP_Tstart_F_Out=FluidMedium.temperature_ph(fluidIPNomPressure_Rh,Rh1_IP_hstart_F_Out) 
          "Outlet fluid temperature start value of Rh1" annotation(Dialog(enable = use_T_Sh,
                                                                   tab = "Initialization (IP)", group = "Rh"));
        parameter SI.SpecificEnthalpy Rh1_IP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Rh,Rh1_IP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value of Rh1" annotation(Dialog(enable = not use_T_Sh,
                                                                        tab = "Initialization (IP)", group = "Rh"));
        parameter SI.SpecificEnthalpy Rh1_IP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Rh,Rh1_IP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value of Rh1" annotation(Dialog(enable = not use_T_Sh,
                                                                         tab = "Initialization (IP)", group = "Rh"));
        //Sh_IP
        parameter SI.Temperature Sh_IP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (IP)", group = "Sh"));
        parameter SI.Temperature Sh_IP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (IP)", group = "Sh"));
        parameter SI.Temperature Sh_IP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Sh"));
        parameter SI.Temperature Sh_IP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Sh"));
        parameter SI.Temperature Sh_IP_Tstart_F_In=FluidMedium.temperature_ph(fluidIPNomPressure_Sh,Sh_IP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Sh,
                                                           tab = "Initialization (IP)", group = "Sh"));
        parameter SI.Temperature Sh_IP_Tstart_F_Out=FluidMedium.temperature_ph(fluidIPNomPressure_Sh,Sh_IP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Sh,
                                                            tab = "Initialization (IP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh_IP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Sh,Sh_IP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Sh,
                                                                 tab = "Initialization (IP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh_IP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Sh,Sh_IP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Sh,
                                                                  tab = "Initialization (IP)", group = "Sh"));
        //Ev_IP
        parameter SI.Temperature Ev_IP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (IP)", group = "Ev"));
        parameter SI.Temperature Ev_IP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (IP)", group = "Ev"));
        parameter SI.Temperature Ev_IP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Ev"));
        parameter SI.Temperature Ev_IP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Ev"));
        parameter SI.Temperature Ev_IP_Tstart_F_In=FluidMedium.temperature_ph(fluidIPNomPressure_Ev,Ev_IP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                                 tab = "Initialization (IP)", group = "Ev"));
        parameter SI.Temperature Ev_IP_Tstart_F_Out=FluidMedium.temperature_ph(fluidIPNomPressure_Ev,Ev_IP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                                  tab = "Initialization (IP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_IP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Ev,Ev_IP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                 tab = "Initialization (IP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_IP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Ev,Ev_IP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                  tab = "Initialization (IP)", group = "Ev"));
        //Ec_IP
        parameter SI.Temperature Ec_IP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (IP)", group = "Ec"));
        parameter SI.Temperature Ec_IP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (IP)", group = "Ec"));
        parameter SI.Temperature Ec_IP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Ec"));
        parameter SI.Temperature Ec_IP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (IP)", group = "Ec"));
        parameter SI.Temperature Ec_IP_Tstart_F_In=FluidMedium.temperature_ph(fluidIPNomPressure_Ec,Ec_IP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                           tab = "Initialization (IP)", group = "Ec"));
        parameter SI.Temperature Ec_IP_Tstart_F_Out=FluidMedium.temperature_ph(fluidIPNomPressure_Ec,Ec_IP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                            tab = "Initialization (IP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_IP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Ec,Ec_IP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                 tab = "Initialization (IP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_IP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidIPNomPressure_Ec,Ec_IP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                  tab = "Initialization (IP)", group = "Ec"));
        
        //Sh_LP
        parameter SI.Temperature Sh_LP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_F_In=FluidMedium.temperature_ph(fluidLPNomPressure_Sh,Sh_LP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Sh,
                                                           tab = "Initialization (LP)", group = "Sh"));
        parameter SI.Temperature Sh_LP_Tstart_F_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Sh,Sh_LP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Sh,
                                                            tab = "Initialization (LP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh_LP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Sh,Sh_LP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Sh,
                                                                 tab = "Initialization (LP)", group = "Sh"));
        parameter SI.SpecificEnthalpy Sh_LP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Sh,Sh_LP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Sh,
                                                                  tab = "Initialization (LP)", group = "Sh"));
        //Ev_LP
        parameter SI.Temperature Ev_LP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_F_In=FluidMedium.temperature_ph(fluidLPNomPressure_Ev,Ev_LP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                                 tab = "Initialization (LP)", group = "Ev"));
        parameter SI.Temperature Ev_LP_Tstart_F_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Ev,Ev_LP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ev,
                                                            tab = "Initialization (LP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_LP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ev,Ev_LP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                 tab = "Initialization (LP)", group = "Ev"));
        parameter SI.SpecificEnthalpy Ev_LP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ev,Ev_LP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ev,
                                                                  tab = "Initialization (LP)", group = "Ev"));
        //Ec_LP
        parameter SI.Temperature Ec_LP_Tstart_G_In 
          "Intlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_G_Out 
          "Outlet gas temperature start value" annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_M_In 
          "Inlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_M_Out 
          "Outlet metal wall temperature start value" 
            annotation(Dialog(tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_F_In=FluidMedium.temperature_ph(fluidLPNomPressure_Ec,Ec_LP_hstart_F_In) 
          "Inlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                           tab = "Initialization (LP)", group = "Ec"));
        parameter SI.Temperature Ec_LP_Tstart_F_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Ec,Ec_LP_hstart_F_Out) 
          "Outlet fluid temperature start value" annotation(Dialog(enable = use_T_Ec,
                                                            tab = "Initialization (LP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_LP_hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ec,Ec_LP_Tstart_F_In) 
          "Inlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                 tab = "Initialization (LP)", group = "Ec"));
        parameter SI.SpecificEnthalpy Ec_LP_hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidLPNomPressure_Ec,Ec_LP_Tstart_F_Out) 
          "Outlet fluid specific enthalpy start value" annotation(Dialog(enable = not use_T_Ec,
                                                                  tab = "Initialization (LP)", group = "Ec"));
        
      end HEG_3LRh;
      
      partial model DG_2L "Base class for Drums Group with two pressure levels" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Charateristics 
        //HP drum 
        parameter SI.Pressure fluidHPNomPressure "Nominal internal pressure" 
                                                     annotation (Dialog(group = "HP drum"));
        parameter SI.Length HPd_rint "Internal radius"                annotation (Dialog(group = "HP drum"));
        parameter SI.Length HPd_rext "External radius"                annotation (Dialog(group = "HP drum"));
        parameter SI.Length HPd_L "Length"                annotation (Dialog(group = "HP drum"));
        parameter SI.HeatCapacity HPd_Cm 
          "Total Heat Capacity of the metal wall"          annotation (Dialog(group = "HP drum"));
        //LP drum 
        parameter SI.Pressure fluidLPNomPressure "Nominal internal pressure" 
                                                     annotation (Dialog(group = "LP drum"));
        parameter SI.Length LPd_rint "Internal radius"                annotation (Dialog(group = "LP drum"));
        parameter SI.Length LPd_rext "External radius"                annotation (Dialog(group = "LP drum"));
        parameter SI.Length LPd_L "Length"                annotation (Dialog(group = "LP drum"));
        parameter SI.HeatCapacity LPd_Cm 
          "Total Heat Capacity of the metal wall"          annotation (Dialog(group = "LP drum"));
        
        //Initialization 
        parameter Boolean SSInit = false "Steady-state initialisation" annotation(Dialog(tab = "Initialisation"));
        //HP drum
        parameter SI.Pressure HPd_pstart=fluidHPNomPressure 
          "Pressure start value" 
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        parameter SI.SpecificEnthalpy HPd_hvstart "Vapour enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        parameter SI.SpecificEnthalpy HPd_hlstart "Liquid enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        parameter SI.Temperature HPd_Tmstart=300 
          "Metal wall temperature start value" 
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        //LP drum
        parameter SI.Pressure LPd_pstart=fluidLPNomPressure 
          "Pressure start value" 
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        parameter SI.SpecificEnthalpy LPd_hvstart "Vapour enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        parameter SI.SpecificEnthalpy LPd_hlstart "Liquid enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        parameter SI.Temperature LPd_Tmstart=300 
          "Metal wall temperature start value" 
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        
        constant Real g=Modelica.Constants.g_n;
        constant Real pi=Modelica.Constants.pi;
        
        annotation (Icon(
            Rectangle(extent=[-400,300; 400,-300], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[140,10; -120,10; -120,-300], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[80,80; 180,-20],    style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Line(points=[180,-300; 180,-20; 140,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[140,-300; 140,72], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[100,-296; 100,-6], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[60,-296; 60,-50; 60,-10; 60,80; 80,80; 130,30; 130,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Polygon(points=[80,30; 180,30; 180,38; 178,48; 172,58; 166,66; 158,
                  72; 148,78; 138,80; 130,80; 122,80; 112,78; 102,72; 94,66; 88,
                  58; 82,48; 80,38; 80,38; 80,30],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[-260,80; -160,-20], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Line(points=[-160,-290; -160,-20; -200,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-200,-300; -200,-2], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-240,-298; -240,-6], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-280,-298; -280,-50; -280,-10; -280,80; -260,80; -210,
                  30; -210,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Polygon(points=[-260,30; -160,30; -160,38; -162,48; -168,58; -174,
                  66; -182,72; -192,78; -202,80; -210,80; -218,80; -228,78;
                  -238,72; -246,66; -252,58; -258,48; -260,38; -260,38; -260,30],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[-60,30; -20,-10],
                                           style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[-20,10; -60,10; -30,28; -60,10; -32,-8],
                                                             style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Polygon(points=[-286,-204; -274,-204; -280,-220; -286,-204],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-246,-198; -234,-198; -240,-182; -246,-198],
                                                                    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-206,-204; -194,-204; -200,-220; -206,-204],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-166,-198; -154,-198; -160,-182; -166,-198],
                                                                    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-126,-204; -114,-204; -120,-220; -126,-204],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[54,-204; 66,-204; 60,-220; 54,-204],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[94,-198; 106,-198; 100,-182; 94,-198],  style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[134,-204; 146,-204; 140,-220; 134,-204], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[174,-198; 186,-198; 180,-182; 174,-198],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
                             Diagram,
          Coordsys(extent=[-400,-300; 400,300], scale=0.1));
        Water.FlangeB Steam_HP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-294,-314; -266,-286]);
        Water.FlangeA Riser_HP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-254,-314; -226,-286]);
        Water.FlangeB Downcomer_HP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-214,-314; -186,-286]);
        Water.FlangeA Feed_HP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-174,-314; -146,-286]);
        Water.FlangeB WaterForHP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-134,-314; -106,-286]);
        Water.FlangeB Steam_LP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[46,-314; 74,-286]);
        Water.FlangeA Riser_LP(redeclare package Medium = FluidMedium) 
          annotation (extent=[86,-314; 114,-286]);
        Water.FlangeB Downcomer_LP(redeclare package Medium = FluidMedium) 
          annotation (extent=[126,-314; 154,-286]);
        Water.FlangeA Feed_LP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[166,-314; 194,-286]);
        Buses.Sensors SensorsBus 
                              annotation (extent=[380,60; 420,100]);
        Buses.Actuators ActuatorsBus 
                                  annotation (extent=[420,0; 380,40]);
      end DG_2L;
      
      partial model DG_3L_s 
        "Base class for Drums Group with three pressure levels, series feedwater pumps" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Charateristics 
        //HP drum 
        parameter SI.Pressure fluidHPNomPressure "Nominal internal pressure" 
                                                     annotation (Dialog(group = "HP drum"));
        parameter SI.Length HPd_rint "Internal radius"                annotation (Dialog(group = "HP drum"));
        parameter SI.Length HPd_rext "External radius"                annotation (Dialog(group = "HP drum"));
        parameter SI.Length HPd_L "Length"                annotation (Dialog(group = "HP drum"));
        parameter SI.HeatCapacity HPd_Cm 
          "Total Heat Capacity of the metal wall"          annotation (Dialog(group = "HP drum"));
        //IP drum 
        parameter SI.Pressure fluidIPNomPressure "Nominal internal pressure" 
                                                     annotation (Dialog(group = "IP drum"));
        parameter SI.Length IPd_rint "Internal radius"                annotation (Dialog(group = "IP drum"));
        parameter SI.Length IPd_rext "External radius"                annotation (Dialog(group = "IP drum"));
        parameter SI.Length IPd_L "Length"                annotation (Dialog(group = "IP drum"));
        parameter SI.HeatCapacity IPd_Cm 
          "Total Heat Capacity of the metal wall"          annotation (Dialog(group = "IP drum"));
        //LP drum 
        parameter SI.Pressure fluidLPNomPressure "Nominal internal pressure" 
                                                     annotation (Dialog(group = "LP drum"));
        parameter SI.Length LPd_rint "Internal radius"                annotation (Dialog(group = "LP drum"));
        parameter SI.Length LPd_rext "External radius"                annotation (Dialog(group = "LP drum"));
        parameter SI.Length LPd_L "Length"                annotation (Dialog(group = "LP drum"));
        parameter SI.HeatCapacity LPd_Cm 
          "Total Heat Capacity of the metal wall"          annotation (Dialog(group = "LP drum"));
        
        //Initialization 
        parameter Boolean SSInit = false "Steady-state initialisation" annotation(Dialog(tab = "Initialisation"));
        //HP drum
        parameter SI.Pressure HPd_pstart=fluidHPNomPressure 
          "Pressure start value" 
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        parameter SI.SpecificEnthalpy HPd_hvstart "Vapour enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        parameter SI.SpecificEnthalpy HPd_hlstart "Liquid enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        parameter SI.Temperature HPd_Tmstart=300 
          "Metal wall temperature start value" 
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        //IP drum
        parameter SI.Pressure IPd_pstart=fluidIPNomPressure 
          "Pressure start value" 
          annotation(Dialog(tab = "Initialisation", group = "IP drum"));
        parameter SI.SpecificEnthalpy IPd_hvstart "Vapour enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "IP drum"));
        parameter SI.SpecificEnthalpy IPd_hlstart "Liquid enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "IP drum"));
        parameter SI.Temperature IPd_Tmstart=300 
          "Metal wall temperature start value" 
          annotation(Dialog(tab = "Initialisation", group = "IP drum"));
        //LP drum
        parameter SI.Pressure LPd_pstart=fluidLPNomPressure 
          "Pressure start value" 
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        parameter SI.SpecificEnthalpy LPd_hvstart "Vapour enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        parameter SI.SpecificEnthalpy LPd_hlstart "Liquid enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        parameter SI.Temperature LPd_Tmstart=300 
          "Metal wall temperature start value" 
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        
        constant Real g=Modelica.Constants.g_n;
        constant Real pi=Modelica.Constants.pi;
        
        annotation (Icon(
            Rectangle(extent=[-500,300; 500,-300], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[240,80; 340,-20],   style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Line(points=[340,-302; 340,-20; 300,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[300,-300; 300,-2], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[260,-300; 260,-6], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[220,-300; 220,-50; 220,-10; 220,80; 240,80; 290,30;
                  290,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Polygon(points=[240,30; 340,30; 340,38; 338,48; 332,58; 326,66; 318,
                  72; 308,78; 298,80; 290,80; 282,80; 272,78; 262,72; 254,66;
                  248,58; 242,48; 240,38; 240,38; 240,30],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[300,8; 80,8; 80,-300], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[80,-120; -220,-120; -220,-302], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[-60,80; 40,-20],    style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Line(points=[40,-300; 40,-20; 0,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[0,-300; 0,72], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-40,-296; -40,-6], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-80,-296; -80,-50; -80,-10; -80,80; -60,80; -10,30;
                  -10,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Polygon(points=[-60,30; 40,30; 40,38; 38,48; 32,58; 26,66; 18,72; 8,
                  78; -2,80; -10,80; -18,80; -28,78; -38,72; -46,66; -52,58;
                  -58,48; -60,38; -60,38; -60,30],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[-360,80; -260,-20], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Line(points=[-260,-290; -260,-20; -300,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-300,-300; -300,-2], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-340,-298; -340,-6], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-380,-298; -380,-50; -380,-10; -380,80; -360,80; -310,
                  30; -310,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Polygon(points=[-360,30; -260,30; -260,38; -262,48; -268,58; -274,
                  66; -282,72; -292,78; -302,80; -310,80; -318,80; -328,78;
                  -338,72; -346,66; -352,58; -358,48; -360,38; -360,38; -360,30],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[132,28; 172,-12],
                                           style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[172,8; 132,8; 162,26; 132,8; 160,-10],
                                                             style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Ellipse(extent=[-180,-100; -140,-140],
                                           style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[-140,-120; -180,-120; -150,-102; -180,-120; -152,-138],
                                                             style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Polygon(points=[-386,-234; -374,-234; -380,-250; -386,-234],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-346,-228; -334,-228; -340,-212; -346,-228],
                                                                    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-306,-234; -294,-234; -300,-250; -306,-234],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-266,-228; -254,-228; -260,-212; -266,-228],
                                                                    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-226,-234; -214,-234; -220,-250; -226,-234],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-86,-234; -74,-234; -80,-250; -86,-234], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,-228; -34,-228; -40,-212; -46,-228],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-6,-234; 6,-234; 0,-250; -6,-234],       style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[34,-228; 46,-228; 40,-212; 34,-228],    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,-234; 86,-234; 80,-250; 74,-234],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[214,-234; 226,-234; 220,-250; 214,-234], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[254,-228; 266,-228; 260,-212; 254,-228],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[294,-234; 306,-234; 300,-250; 294,-234], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[334,-228; 346,-228; 340,-212; 334,-228],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[76,-116; 84,-124],
                                           style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,-60; 86,-60; 80,-76; 74,-60],         style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
                             Diagram,
          Coordsys(extent=[-500,-300; 500,300], scale=0.1));
        Water.FlangeB Steam_HP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-394,-314; -366,-286]);
        Water.FlangeA Riser_HP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-354,-314; -326,-286]);
        Water.FlangeB Downcomer_HP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-314,-314; -286,-286]);
        Water.FlangeA Feed_HP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-274,-314; -246,-286]);
        Water.FlangeB WaterForHP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-234,-314; -206,-286]);
        Water.FlangeB Steam_IP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-94,-314; -66,-286]);
        Water.FlangeA Riser_IP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-54,-314; -26,-286]);
        Water.FlangeB Downcomer_IP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-14,-314; 14,-286]);
        Water.FlangeA Feed_IP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[26,-314; 54,-286]);
        Water.FlangeB WaterForIP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[66,-314; 94,-286]);
        Water.FlangeB Steam_LP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[206,-314; 234,-286]);
        Water.FlangeA Riser_LP(redeclare package Medium = FluidMedium) 
          annotation (extent=[246,-314; 274,-286]);
        Water.FlangeB Downcomer_LP(redeclare package Medium = FluidMedium) 
          annotation (extent=[286,-314; 314,-286]);
        Water.FlangeA Feed_LP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[326,-314; 354,-286]);
        Buses.Sensors SensorsBus 
                              annotation (extent=[480,60; 520,100]);
        Buses.Actuators ActuatorsBus 
                                  annotation (extent=[520,0; 480,40]);
      end DG_3L_s;
      
      partial model DG_3L_p 
        "Base class for Drums Group with three pressure levels, parallel feedwater pumps" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Charateristics 
        //HP drum 
        parameter SI.Pressure fluidHPNomPressure "Nominal internal pressure" 
                                                     annotation (Dialog(group = "HP drum"));
        parameter SI.Length HPd_rint "Internal radius"                annotation (Dialog(group = "HP drum"));
        parameter SI.Length HPd_rext "External radius"                annotation (Dialog(group = "HP drum"));
        parameter SI.Length HPd_L "Length"                annotation (Dialog(group = "HP drum"));
        parameter SI.HeatCapacity HPd_Cm 
          "Total Heat Capacity of the metal wall"          annotation (Dialog(group = "HP drum"));
        //IP drum 
        parameter SI.Pressure fluidIPNomPressure "Nominal internal pressure" 
                                                     annotation (Dialog(group = "IP drum"));
        parameter SI.Length IPd_rint "Internal radius"                annotation (Dialog(group = "IP drum"));
        parameter SI.Length IPd_rext "External radius"                annotation (Dialog(group = "IP drum"));
        parameter SI.Length IPd_L "Length"                annotation (Dialog(group = "IP drum"));
        parameter SI.HeatCapacity IPd_Cm 
          "Total Heat Capacity of the metal wall"          annotation (Dialog(group = "IP drum"));
        //LP drum 
        parameter SI.Pressure fluidLPNomPressure "Nominal internal pressure" 
                                                     annotation (Dialog(group = "LP drum"));
        parameter SI.Length LPd_rint "Internal radius"                annotation (Dialog(group = "LP drum"));
        parameter SI.Length LPd_rext "External radius"                annotation (Dialog(group = "LP drum"));
        parameter SI.Length LPd_L "Length"                annotation (Dialog(group = "LP drum"));
        parameter SI.HeatCapacity LPd_Cm 
          "Total Heat Capacity of the metal wall"          annotation (Dialog(group = "LP drum"));
        
        //Initialization 
        parameter Boolean SSInit = false "Steady-state initialisation" annotation(Dialog(tab = "Initialisation"));
        //HP drum
        parameter SI.Pressure HPd_pstart=fluidHPNomPressure 
          "Pressure start value" 
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        parameter SI.SpecificEnthalpy HPd_hvstart "Vapour enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        parameter SI.SpecificEnthalpy HPd_hlstart "Liquid enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        parameter SI.Temperature HPd_Tmstart=300 
          "Metal wall temperature start value" 
          annotation(Dialog(tab = "Initialisation", group = "HP drum"));
        //IP drum
        parameter SI.Pressure IPd_pstart=fluidIPNomPressure 
          "Pressure start value" 
          annotation(Dialog(tab = "Initialisation", group = "IP drum"));
        parameter SI.SpecificEnthalpy IPd_hvstart "Vapour enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "IP drum"));
        parameter SI.SpecificEnthalpy IPd_hlstart "Liquid enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "IP drum"));
        parameter SI.Temperature IPd_Tmstart=300 
          "Metal wall temperature start value" 
          annotation(Dialog(tab = "Initialisation", group = "IP drum"));
        //LP drum
        parameter SI.Pressure LPd_pstart=fluidLPNomPressure 
          "Pressure start value" 
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        parameter SI.SpecificEnthalpy LPd_hvstart "Vapour enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        parameter SI.SpecificEnthalpy LPd_hlstart "Liquid enthalpy start value"
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        parameter SI.Temperature LPd_Tmstart=300 
          "Metal wall temperature start value" 
          annotation(Dialog(tab = "Initialisation", group = "LP drum"));
        
        constant Real g=Modelica.Constants.g_n;
        constant Real pi=Modelica.Constants.pi;
        
        annotation (Icon(
            Rectangle(extent=[-500,300; 500,-300], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[240,80; 340,-20],   style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Line(points=[340,-302; 340,-20; 300,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[300,-300; 300,-2], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[260,-300; 260,-6], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[220,-300; 220,-50; 220,-10; 220,80; 240,80; 290,30;
                  290,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Polygon(points=[240,30; 340,30; 340,38; 338,48; 332,58; 326,66; 318,
                  72; 308,78; 298,80; 290,80; 282,80; 272,78; 262,72; 254,66;
                  248,58; 242,48; 240,38; 240,38; 240,30],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[-60,80; 40,-20],    style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Line(points=[40,-300; 40,-20; 0,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[0,-300; 0,72], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-40,-296; -40,-6], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-80,-296; -80,-50; -80,-10; -80,80; -60,80; -10,30;
                  -10,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Polygon(points=[-60,30; 40,30; 40,38; 38,48; 32,58; 26,66; 18,72; 8,
                  78; -2,80; -10,80; -18,80; -28,78; -38,72; -46,66; -52,58;
                  -58,48; -60,38; -60,38; -60,30],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[-360,80; -260,-20], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Line(points=[-260,-290; -260,-20; -300,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-300,-300; -300,-2], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-340,-298; -340,-6], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[-380,-298; -380,-50; -380,-10; -380,80; -360,80; -310,
                  30; -310,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Polygon(points=[-360,30; -260,30; -260,38; -262,48; -268,58; -274,
                  66; -282,72; -292,78; -302,80; -310,80; -318,80; -328,78;
                  -338,72; -346,66; -352,58; -358,48; -360,38; -360,38; -360,30],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[300,8; 80,8; 80,-300], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[80,-120; -220,-120; -220,-302], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[78,-118; 82,-122], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillPattern=1)),
            Ellipse(extent=[-240,-160; -200,-200],
                                           style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[-220,-160; -220,-200; -238,-172; -220,-200; -202,-172],
                                                             style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Ellipse(extent=[60,-160; 100,-200],
                                           style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[80,-160; 80,-200; 62,-172; 80,-200; 98,-172],
                                                             style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Polygon(points=[-386,-234; -374,-234; -380,-250; -386,-234],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-346,-228; -334,-228; -340,-212; -346,-228],
                                                                    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-306,-234; -294,-234; -300,-250; -306,-234],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-266,-228; -254,-228; -260,-212; -266,-228],
                                                                    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-226,-234; -214,-234; -220,-250; -226,-234],
                                                                     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-86,-234; -74,-234; -80,-250; -86,-234], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,-228; -34,-228; -40,-212; -46,-228],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-6,-234; 6,-234; 0,-250; -6,-234],       style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[34,-228; 46,-228; 40,-212; 34,-228],    style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,-234; 86,-234; 80,-250; 74,-234],     style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[214,-234; 226,-234; 220,-250; 214,-234], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[254,-228; 266,-228; 260,-212; 254,-228],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[294,-234; 306,-234; 300,-250; 294,-234], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[334,-228; 346,-228; 340,-212; 334,-228],style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,-52; 86,-52; 80,-68; 74,-52],         style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
                             Diagram,
          Coordsys(extent=[-500,-300; 500,300], scale=0.1));
        Water.FlangeB Steam_HP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-394,-314; -366,-286]);
        Water.FlangeA Riser_HP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-354,-314; -326,-286]);
        Water.FlangeB Downcomer_HP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-314,-314; -286,-286]);
        Water.FlangeA Feed_HP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-274,-314; -246,-286]);
        Water.FlangeB WaterForHP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-234,-314; -206,-286]);
        Water.FlangeB Steam_IP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-94,-314; -66,-286]);
        Water.FlangeA Riser_IP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-54,-314; -26,-286]);
        Water.FlangeB Downcomer_IP(redeclare package Medium = FluidMedium) 
          annotation (extent=[-14,-314; 14,-286]);
        Water.FlangeA Feed_IP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[26,-314; 54,-286]);
        Water.FlangeB WaterForIP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[66,-314; 94,-286]);
        Water.FlangeB Steam_LP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[206,-314; 234,-286]);
        Water.FlangeA Riser_LP(redeclare package Medium = FluidMedium) 
          annotation (extent=[246,-314; 274,-286]);
        Water.FlangeB Downcomer_LP(redeclare package Medium = FluidMedium) 
          annotation (extent=[286,-314; 314,-286]);
        Water.FlangeA Feed_LP(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[326,-314; 354,-286]);
        Buses.Sensors SensorsBus 
                              annotation (extent=[480,60; 520,100]);
        Buses.Actuators ActuatorsBus 
                                  annotation (extent=[520,0; 480,40]);
      end DG_3L_p;
      
      partial model HRSG_2L 
        "Base class for Heat Recovery Steam Generator with two pressure levels" 
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        Water.FlangeA WaterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[140,-220; 180,-180]);
        annotation (Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[-130,160; -70,100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Polygon(points=[-130,130; -70,130; -70,134; -70,134; -72,142; -74,
                  146; -78,152; -84,156; -88,158; -96,160; -100,160; -104,160;
                  -112,158; -116,156; -122,152; -126,146; -128,142; -130,134;
                  -130,130],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[70,160; 130,100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Polygon(points=[70,130; 130,130; 130,134; 130,134; 128,142; 126,146;
                  122,152; 116,156; 112,158; 104,160; 100,160; 96,160; 88,158;
                  84,156; 78,152; 74,146; 72,142; 70,134; 70,130], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Rectangle(extent=[-200,60; 200,-140], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={215,215,215})),
            Line(points=[-100,60; -100,20; -60,0; -140,-40; -60,-80; -100,-100;
                  -100,-140], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[100,60; 100,20; 140,0; 60,-40; 140,-80; 100,-100; 100,
                  -140], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1))), Diagram,
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        
        Water.FlangeB Sh_HP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-140,-220; -100,-180]);
        Gas.FlangeA GasIn(redeclare package Medium = FlueGasMedium) 
                          annotation (extent=[-220,-60; -180,-20]);
        Gas.FlangeB GasOut(redeclare package Medium = FlueGasMedium) 
                           annotation (extent=[180,-60; 220,-20]);
        Water.FlangeB Sh_LP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-20,-220; 20,-180]);
        Buses.Sensors SensorsBus 
                              annotation (extent=[180,140; 220,180]);
        Buses.Actuators ActuatorsBus 
                                  annotation (extent=[220,80; 180,120]);
      end HRSG_2L;
      
      partial model HRSG_2LRh 
        "Base class for Heat Recovery Steam Generator with two pressure levels and reheat" 
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        Water.FlangeA WaterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[140,-220; 180,-180]);
        annotation (Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[-130,160; -70,100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Polygon(points=[-130,130; -70,130; -70,134; -70,134; -72,142; -74,
                  146; -78,152; -84,156; -88,158; -96,160; -100,160; -104,160;
                  -112,158; -116,156; -122,152; -126,146; -128,142; -130,134;
                  -130,130],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[70,160; 130,100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Polygon(points=[70,130; 130,130; 130,134; 130,134; 128,142; 126,146;
                  122,152; 116,156; 112,158; 104,160; 100,160; 96,160; 88,158;
                  84,156; 78,152; 74,146; 72,142; 70,134; 70,130], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Rectangle(extent=[-200,60; 200,-140], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={215,215,215})),
            Line(points=[-100,60; -100,20; -60,0; -140,-40; -60,-80; -100,-100;
                  -100,-140], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[100,60; 100,20; 140,0; 60,-40; 140,-80; 100,-100; 100,
                  -140], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1))), Diagram,
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        
        Water.FlangeB Sh_HP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-180,-220; -140,-180]);
        Gas.FlangeA GasIn(redeclare package Medium = FlueGasMedium) 
                          annotation (extent=[-220,-60; -180,-20]);
        Gas.FlangeB GasOut(redeclare package Medium = FlueGasMedium) 
                           annotation (extent=[180,-60; 220,-20]);
        Water.FlangeA Rh_IP_In(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-120,-220; -80,-180]);
        Water.FlangeB Rh_IP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-60,-220; -20,-180]);
        Water.FlangeB Sh_LP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[60,-220; 100,-180]);
        Buses.Sensors SensorsBus 
                              annotation (extent=[180,140; 220,180]);
        Buses.Actuators ActuatorsBus 
                                  annotation (extent=[220,80; 180,120]);
      end HRSG_2LRh;
      
      partial model HRSG_3LRh 
        "Base class for Heat Recovery Steam Generator with three pressure levels and reheat" 
        replaceable package FlueGasMedium = ThermoPower.Media.FlueGas extends 
          Modelica.Media.Interfaces.PartialMedium;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        Water.FlangeA WaterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[140,-220; 180,-180]);
        annotation (Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[-140,160; -80,100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Polygon(points=[-140,130; -80,130; -80,134; -80,134; -82,142; -84,
                  146; -88,152; -94,156; -98,158; -106,160; -110,160; -114,160;
                  -122,158; -126,156; -132,152; -136,146; -138,142; -140,134;
                  -140,130],
                style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[-30,160; 30,100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Polygon(points=[-30,130; 30,130; 30,134; 30,134; 28,142; 26,146; 22,
                  152; 16,156; 12,158; 4,160; 0,160; -4,160; -12,158; -16,156;
                  -22,152; -26,146; -28,142; -30,134; -30,130], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Ellipse(extent=[80,160; 140,100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Polygon(points=[80,130; 140,130; 140,134; 140,134; 138,142; 136,146;
                  132,152; 126,156; 122,158; 114,160; 110,160; 106,160; 98,158;
                  94,156; 88,152; 84,146; 82,142; 80,134; 80,130], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Rectangle(extent=[-200,60; 200,-140], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={215,215,215})),
            Line(points=[-110,60; -110,20; -70,0; -150,-40; -70,-80; -110,-100;
                  -110,-140], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[0,60; 0,20; 40,0; -40,-40; 40,-80; 0,-100; 0,-140],
                style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1)),
            Line(points=[110,60; 110,20; 150,0; 70,-40; 150,-80; 110,-100; 110,
                  -140], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1))),
            Diagram,
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        
        Water.FlangeB Sh_HP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-180,-220; -140,-180]);
        Gas.FlangeA GasIn(redeclare package Medium = FlueGasMedium) 
                          annotation (extent=[-220,-60; -180,-20]);
        Gas.FlangeB GasOut(redeclare package Medium = FlueGasMedium) 
                           annotation (extent=[180,-60; 220,-20]);
        Water.FlangeA Rh_IP_In(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-120,-220; -80,-180]);
        Water.FlangeB Rh_IP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-60,-220; -20,-180]);
        Water.FlangeB Sh_LP_Out(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[60,-220; 100,-180]);
        Buses.Sensors SensorsBus 
                              annotation (extent=[180,140; 220,180]);
        Buses.Actuators ActuatorsBus 
                                  annotation (extent=[220,80; 180,120]);
      end HRSG_3LRh;
      
    end Interfaces;
    
    package Components "HRSG component models" 
      
      model HE "Heat Exchanger fluid - gas" 
        extends Interfaces.HeatExchanger(pstartout_F=fluidNomPressure-dpnom_F);
        
        parameter SI.CoefficientOfHeatTransfer gamma_G 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F 
          "Constant heat transfer coefficient in the fluid side";
        parameter Choices.Flow1D.FFtypes.Temp FFtype_F=ThermoPower.Choices.Flow1D.FFtypes.NoFriction 
          "Friction Factor Type, fluid side";
        parameter Real Kfnom_F=0 "Nominal hydraulic resistance coefficient";
        parameter SI.Pressure dpnom_F=0 
          "Nominal pressure drop fluid side (friction term only!)";
        parameter SI.Density rhonom_F=0 "Nominal inlet density fluid side";
        parameter Real Cfnom_F=0 "Nominal Fanning friction factor";
        parameter Choices.Flow1D.HCtypes.Temp HCtype_F=ThermoPower.Choices.Flow1D.HCtypes.Downstream 
          "Location of the hydraulic capacitance, fluid side";
        parameter Boolean counterCurrent=true "Counter-current flow";
        parameter Boolean gasQuasiStatic=false 
          "Quasi-static model of the flue gas (mass, energy and momentum static balances";
        constant Real pi=Modelica.Constants.pi;
        
      Water.Flow1D fluidFlow(
          Nt=1,
          N=N_F,
          wnom=fluidNomFlowRate,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          L=exchSurface_F^2/(fluidVol*pi*4),
          A=(fluidVol*4/exchSurface_F)^2/4*pi,
          omega=fluidVol*4/exchSurface_F*pi,
          Dhyd=fluidVol*4/exchSurface_F,
          pstartin=pstartin_F,
          pstartout=pstartout_F,
          FFtype=FFtype_F,
          HydraulicCapacitance=HCtype_F,
          hstart=hstart_F,
          hstartin=hstart_F_In,
          hstartout=hstart_F_Out,
          Kfnom=Kfnom_F,
          dpnom=dpnom_F,
          rhonom=rhonom_F,
          Cfnom=Cfnom_F)         annotation (extent=[-10,-60; 10,-40]);
        Thermal.ConvHT convHT(               N=N_F,
          Tstart11=Tstart_M_In,
          Tstart1N=Tstart_M_Out,
          Tstart21=Tstart_F_In,
          Tstart2N=Tstart_F_Out,
          gamma=gamma_F,
          Tstart1=Tstart_M,
          Tstart2=Tstart_F) 
          annotation (extent=[-10,-40; 10,-20]);
        Thermal.MetalTube metalTube(
          rhomcm=rhomcm,
          lambda=lambda,
          N=N_F,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          L=exchSurface_F^2/(fluidVol*pi*4),
          rint=fluidVol*4/exchSurface_F/2,
          Tstart1=Tstart_M_In,
          TstartN=Tstart_M_Out,
          WallRes=false,
          Tstart=Tstart_M,
          rext=(metalVol + fluidVol)*4/extSurfaceTub/2) 
                 annotation (extent=[-10,-6; 10,-26], rotation=0);
        Gas.Flow1D gasFlow(
          L=1,
          Dhyd=1,
          wnom=gasNomFlowRate,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          N=N_G,
          A=gasVol/1,
          omega=exchSurface_G/1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FlueGasMedium,
          QuasiStatic=gasQuasiStatic,
          Tstartin=Tstart_G_In,
          Tstartout=Tstart_G_Out,
          pstart=pstart_G,
          Tstart=Tstart_G)       annotation (extent=[-12,58; 12,38]);
        annotation (Diagram);
        Thermal.CounterCurrent cC(                                    N=N_F,
            counterCurrent=counterCurrent) 
          annotation (extent=[-10,-10; 10,10]);
        Thermal.HeatFlowDistribution heatFlowDistribution(
          N=N_F,
          A1=exchSurface_G,
          A2=extSurfaceTub) 
          annotation (extent=[-10,4; 10,24]);
        Thermal.ConvHT2N convHT2N(
          N1=N_G,
          N2=N_F,
          Tstart11=Tstart_G_In,
          Tstart1N=Tstart_G_Out,
          Tstart21=Tstart_M_In,
          Tstart2N=Tstart_M_Out,
          gamma=gamma_G,
          Tstart1=Tstart_G,
          Tstart2=Tstart_M)    annotation (extent=[-10,20; 10,40]);
        
      equation 
        connect(fluidFlow.wall, convHT.side2) 
                                           annotation (points=[0,-45; 0,-33.1],
            style(color=45, rgbcolor={255,127,0}));
        connect(gasFlow.infl, gasIn) annotation (points=[-12,48; -100,48; -100,
              0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(gasFlow.outfl, gasOut) annotation (points=[12,48; 100,48; 100,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(fluidFlow.outfl, waterOut) 
                                        annotation (points=[10,-50; 40,-50; 40,
              -100; 0,-100],
                           style(thickness=2));
        connect(fluidFlow.infl, waterIn) 
                                      annotation (points=[-10,-50; -40,-50; -40,
              100; 0,100],
                         style(thickness=2));
        connect(heatFlowDistribution.side2, cC.side1) annotation (points=[0,10.9;
              0,3],           style(
            color=45,
            rgbcolor={255,127,0},
            fillPattern=1));
        connect(convHT2N.side1, gasFlow.wall) annotation (points=[0,33; 0,43],
            style(color=45, rgbcolor={255,127,0}));
        connect(heatFlowDistribution.side1, convHT2N.side2) annotation (points=[0,17; 0,
              26.9],          style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.int, convHT.side1) annotation (points=[0,-19; 0,-27],
            style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.ext, cC.side2) annotation (points=[0,-12.9; 0,-3.1],
            style(color=45, rgbcolor={255,127,0}));
      end HE;
      
      model HE2ph "Heat Exchanger fluid - gas (fluid 2-phase)" 
        extends Interfaces.HeatExchanger(pstartout_F=fluidNomPressure-dpnom_F);
        
        parameter SI.CoefficientOfHeatTransfer gamma_G 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F 
          "Constant heat transfer coefficient in the fluid side";
        parameter Choices.Flow1D.FFtypes.Temp FFtype_F=ThermoPower.Choices.Flow1D.FFtypes.NoFriction 
          "Friction Factor Type, fluid side";
        parameter Real Kfnom_F=0 "Nominal hydraulic resistance coefficient";
        parameter SI.Pressure dpnom_F=0 
          "Nominal pressure drop fluid side (friction term only!)";
        parameter SI.Density rhonom_F=0 "Nominal inlet density fluid side";
        parameter Real Cfnom_F=0 "Nominal Fanning friction factor";
        parameter Choices.Flow1D.HCtypes.Temp HCtype_F=ThermoPower.Choices.Flow1D.HCtypes.Downstream 
          "Location of the hydraulic capacitance, fluid side";
        parameter Boolean counterCurrent=true "Counter-current flow";
        parameter Boolean gasQuasiStatic=false 
          "Quasi-static model of the flue gas (mass, energy and momentum static balances";
        constant Real pi=Modelica.Constants.pi;
        
      Water.Flow1D2ph fluidFlow(
          Nt=1,
          N=N_F,
          wnom=fluidNomFlowRate,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          L=exchSurface_F^2/(fluidVol*pi*4),
          A=(1/exchSurface_F*fluidVol*4)^2/4*pi,
          omega=1/exchSurface_F*fluidVol*4*pi,
          Dhyd=1/exchSurface_F*fluidVol*4,
          pstartin=pstartin_F,
          pstartout=pstartout_F,
          FFtype=FFtype_F,
          HydraulicCapacitance=HCtype_F,
          hstart=hstart_F,
          hstartin=hstart_F_In,
          hstartout=hstart_F_Out,
          Kfnom=Kfnom_F,
          dpnom=dpnom_F,
          rhonom=rhonom_F,
          Cfnom=Cfnom_F)         annotation (extent=[-10,-58; 10,-38]);
        Thermal.ConvHT convHT(gamma=gamma_F, N=N_F,
          Tstart11=Tstart_M_In,
          Tstart1N=Tstart_M_Out,
          Tstart21=Tstart_F_In,
          Tstart2N=Tstart_F_Out,
          Tstart1=Tstart_M,
          Tstart2=Tstart_F) 
          annotation (extent=[-10,-40; 10,-20]);
        Thermal.MetalTube metalTube(
          rhomcm=rhomcm,
          lambda=lambda,
          N=N_F,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          L=exchSurface_F^2/(fluidVol*pi*4),
          rint=1/exchSurface_F*fluidVol*4/2,
          Tstart1=Tstart_M_In,
          TstartN=Tstart_M_Out,
          WallRes=false,
          Tstart=Tstart_M,
          rext=1/extSurfaceTub*(metalVol + fluidVol)*4/2) 
                 annotation (extent=[-10,-8; 10,-28]);
        Gas.Flow1D gasFlow(
          L=1,
          Dhyd=1,
          wnom=gasNomFlowRate,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          N=N_G,
          A=gasVol/1,
          omega=exchSurface_G/1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FlueGasMedium,
          pstart=pstart_G,
          Tstartin=Tstart_G_In,
          Tstartout=Tstart_G_Out,
          Tstart=Tstart_G,
          QuasiStatic=gasQuasiStatic) 
                                 annotation (extent=[-12,58; 12,38]);
        annotation (Diagram);
        Thermal.CounterCurrent cC(     counterCurrent=counterCurrent, N=N_F) 
          annotation (extent=[-10,-10; 10,10]);
        Thermal.HeatFlowDistribution heatFlowDistribution(
          N=N_F,
          A1=exchSurface_G,
          A2=extSurfaceTub) 
          annotation (extent=[-10,6; 10,26]);
        Thermal.ConvHT2N convHT2N(
          N1=N_G,
          N2=N_F,
          Tstart11=Tstart_G_In,
          Tstart1N=Tstart_G_Out,
          Tstart21=Tstart_M_In,
          Tstart2N=Tstart_M_Out,
          gamma=gamma_G,
          Tstart1=Tstart_G,
          Tstart2=Tstart_M)    annotation (extent=[-10,22; 10,42]);
        
      equation 
        connect(fluidFlow.wall, convHT.side2) 
                                           annotation (points=[0,-43; 0,-33.1],
            style(color=45, rgbcolor={255,127,0}));
        connect(gasFlow.infl, gasIn) annotation (points=[-12,48; -100,48; -100,
              0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(gasFlow.outfl, gasOut) annotation (points=[12,48; 100,48; 100,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(fluidFlow.outfl, waterOut) 
                                        annotation (points=[10,-48; 40,-48; 40,
              -100; 0,-100],
                           style(thickness=2));
        connect(fluidFlow.infl, waterIn) 
                                      annotation (points=[-10,-48; -40,-48; -40,
              100; 0,100],
                         style(thickness=2));
        connect(heatFlowDistribution.side2,cC. side1) annotation (points=[0,12.9;
              0,3],           style(
            color=45,
            rgbcolor={255,127,0},
            fillPattern=1));
        connect(convHT2N.side1,gasFlow. wall) annotation (points=[0,35; 0,43],
            style(color=45, rgbcolor={255,127,0}));
        connect(convHT2N.side2, heatFlowDistribution.side1) annotation (points=[0,28.9;
              0,19],          style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.int, convHT.side1) annotation (points=[0,-21; 0,-27],
            style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.ext, cC.side2) annotation (points=[0,-14.9; 0,-3.1],
            style(color=45, rgbcolor={255,127,0}));
      end HE2ph;
      
      model ParHE "Parallel heat exchangers fluid - gas" 
        extends Interfaces.ParallelHE(pstartout_F_A=fluidNomPressure_A-dpnom_F_A,
                                      pstartout_F_B=fluidNomPressure_B-dpnom_F_B);
        
        parameter SI.CoefficientOfHeatTransfer gamma_G_A 
          "Constant heat transfer coefficient in the gas side"     annotation (Dialog(group = "side A"));
        parameter SI.CoefficientOfHeatTransfer gamma_G_B 
          "Constant heat transfer coefficient in the gas side"     annotation (Dialog(group = "side B"));
        parameter SI.CoefficientOfHeatTransfer gamma_F_A 
          "Constant heat transfer coefficient in the fluid side"     annotation (Dialog(group = "side A"));
        parameter SI.CoefficientOfHeatTransfer gamma_F_B 
          "Constant heat transfer coefficient in the fluid side"     annotation (Dialog(group = "side B"));
        parameter Choices.Flow1D.FFtypes.Temp FFtype_F_A=ThermoPower.Choices.Flow1D.FFtypes.NoFriction 
          "Friction Factor Type, fluid side"     annotation (Dialog(group = "side A"));
        parameter Real Kfnom_F_A=0 "Nominal hydraulic resistance coefficient" 
                                                                             annotation (Dialog(group = "side A"));
        parameter SI.Pressure dpnom_F_A=0 
          "Nominal pressure drop fluid side (friction term only!)"     annotation (Dialog(group = "side A"));
        parameter SI.Density rhonom_F_A=0 "Nominal inlet density fluid side"   annotation (Dialog(group = "side A"));
        parameter Real Cfnom_F_A=0 "Nominal Fanning friction factor" 
                                                                    annotation (Dialog(group = "side A"));
        parameter Choices.Flow1D.FFtypes.Temp FFtype_F_B=ThermoPower.Choices.Flow1D.FFtypes.NoFriction 
          "Friction Factor Type, fluid side"     annotation (Dialog(group = "side B"));
        parameter Real Kfnom_F_B=0 "Nominal hydraulic resistance coefficient" 
                                                                             annotation (Dialog(group = "side B"));
        parameter SI.Pressure dpnom_F_B=0 
          "Nominal pressure drop fluid side (friction term only!)"   annotation (Dialog(group = "side B"));
        parameter SI.Density rhonom_F_B=0 "Nominal inlet density fluid side"   annotation (Dialog(group = "side B"));
        parameter Real Cfnom_F_B=0 "Nominal Fanning friction factor" 
                                                                    annotation (Dialog(group = "side B"));
        parameter Choices.Flow1D.HCtypes.Temp HCtype_F_A=ThermoPower.Choices.Flow1D.HCtypes.Downstream 
          "Location of the hydraulic capacitance, fluid side"     annotation (Dialog(group = "side A"));
        parameter Choices.Flow1D.HCtypes.Temp HCtype_F_B=ThermoPower.Choices.Flow1D.HCtypes.Downstream 
          "Location of the hydraulic capacitance, fluid side"     annotation (Dialog(group = "side B"));
        parameter Boolean counterCurrent_A = true "Counter-current flow"     annotation (Dialog(group = "side A"));
        parameter Boolean counterCurrent_B = true "Counter-current flow"     annotation (Dialog(group = "side B"));
        
        parameter Boolean gasQuasiStatic=false 
          "Quasi-static model of the flue gas (mass, energy and momentum static balances";
        constant Real pi=Modelica.Constants.pi;
        
        ConvParallel convParallel(
          N=N_G,
          As=exchSurface_G_A + exchSurface_G_B,
          Aa=exchSurface_G_A,
          Ab=exchSurface_G_B) 
                             annotation (extent=[-10,12; 10,32]);
        annotation (Diagram);
      Water.Flow1D fluidAFlow(
          Nt=1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          N=N_F_A,
          L=exchSurface_F_A^2/(fluidVol_A*pi*4),
          A=(fluidVol_A*4/exchSurface_F_A)^2/4*pi,
          omega=fluidVol_A*4/exchSurface_F_A*pi,
          Dhyd=fluidVol_A*4/exchSurface_F_A,
          pstartin=pstartin_F_A,
          pstartout=pstartout_F_A,
          hstart=hstart_F_A,
          wnom=fluidNomFlowRate_A,
          FFtype=FFtype_F_A,
          HydraulicCapacitance=HCtype_F_A,
          hstartin=hstart_F_A_In,
          hstartout=hstart_F_A_Out,
          Kfnom=Kfnom_F_A,
          dpnom=dpnom_F_A,
          rhonom=rhonom_F_A,
          Cfnom=Cfnom_F_A)       annotation (extent=[-38,-56; -18,-36]);
        Thermal.ConvHT convHT_A(
          N=N_F_A,
          gamma=gamma_F_A,
          Tstart11=Tstart_M_A_In,
          Tstart1N=Tstart_M_A_Out,
          Tstart21=Tstart_F_A_In,
          Tstart2N=Tstart_F_A_Out,
          Tstart1=Tstart_M_A,
          Tstart2=Tstart_F_A) 
          annotation (extent=[-38,-38; -18,-18]);
        Thermal.MetalTube metalTube_A(
          lambda=lambda,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          N=N_F_A,
          L=exchSurface_F_A^2/(fluidVol_A*pi*4),
          rint=fluidVol_A*4/exchSurface_F_A/2,
          Tstart1=Tstart_M_A_In,
          TstartN=Tstart_M_A_Out,
          WallRes=false,
          Tstart=Tstart_M_A,
          rext=(metalVol_A + fluidVol_A)*4/extSurfaceTub_A/2,
          rhomcm=rhomcm_A) 
                 annotation (extent=[-38,-6; -18,-26]);
        Gas.Flow1D gasFlow(
          L=1,
          Dhyd=1,
          wnom=gasNomFlowRate,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          N=N_G,
          A=gasVol/1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FlueGasMedium,
          omega=(exchSurface_G_A + exchSurface_G_B)/1,
          QuasiStatic=gasQuasiStatic,
          pstart=pstart_G,
          Tstartin=Tstart_G_In,
          Tstartout=Tstart_G_Out,
          Tstart=Tstart_G)       annotation (extent=[-12,60; 12,40]);
      Water.Flow1D fluidBFlow(
          Nt=1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          N=N_F_B,
          L=exchSurface_F_B^2/(fluidVol_B*pi*4),
          A=(fluidVol_B*4/exchSurface_F_B)^2/4*pi,
          omega=fluidVol_B*4/exchSurface_F_B*pi,
          Dhyd=fluidVol_B*4/exchSurface_F_B,
          wnom=fluidNomFlowRate_B,
          pstartin=pstartin_F_B,
          pstartout=pstartout_F_B,
          hstart=hstart_F_B,
          FFtype=FFtype_F_B,
          HydraulicCapacitance=HCtype_F_B,
          hstartin=hstart_F_B_In,
          hstartout=hstart_F_B_Out,
          Kfnom=Kfnom_F_B,
          dpnom=dpnom_F_B,
          rhonom=rhonom_F_B,
          Cfnom=Cfnom_F_B)       annotation (extent=[38,-56; 18,-36]);
        Thermal.ConvHT convHT_B(
          N=N_F_B,
          gamma=gamma_F_B,
          Tstart11=Tstart_M_B_In,
          Tstart1N=Tstart_M_B_Out,
          Tstart21=Tstart_F_B_In,
          Tstart2N=Tstart_F_B_Out,
          Tstart1=Tstart_M_B,
          Tstart2=Tstart_F_B) 
          annotation (extent=[18,-38; 38,-18]);
        Thermal.MetalTube metalTube_B(
          lambda=lambda,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          L=exchSurface_F_B^2/(fluidVol_B*pi*4),
          rint=fluidVol_B*4/exchSurface_F_B/2,
          N=N_F_B,
          Tstart1=Tstart_M_B_In,
          TstartN=Tstart_M_B_Out,
          WallRes=false,
          Tstart=Tstart_M_B,
          rext=(metalVol_B + fluidVol_B)*4/extSurfaceTub_B/2,
          rhomcm=rhomcm_B) 
                 annotation (extent=[18,-6; 38,-26]);
        Thermal.CounterCurrent cC_A(N=N_F_A, counterCurrent=counterCurrent_A) 
          annotation (extent=[-38,-8; -18,12]);
        Thermal.HeatFlowDistribution heatFlowDistribution_A(
          N=N_F_A,
          A1=exchSurface_G_A,
          A2=extSurfaceTub_A) 
          annotation (extent=[-38,8; -18,28]);
        Thermal.CounterCurrent cC_B(N=N_F_B, counterCurrent=counterCurrent_B) 
          annotation (extent=[18,-8; 38,12]);
        Thermal.HeatFlowDistribution heatFlowDistribution_B(
          N=N_F_B,
          A1=exchSurface_G_B,
          A2=extSurfaceTub_B) 
          annotation (extent=[18,8; 38,28]);
        Thermal.ConvHT2N convHT2N_A(
          N1=N_G,
          N2=N_F_A,
          Tstart11=Tstart_G_In,
          Tstart1N=Tstart_G_Out,
          Tstart21=Tstart_M_A_In,
          Tstart2N=Tstart_M_A_Out,
          gamma=gamma_G_A,
          Tstart1=Tstart_G,
          Tstart2=Tstart_M_A)    annotation (extent=[-38,24; -18,44]);
        Thermal.ConvHT2N convHT2N_B(
          N1=N_G,
          N2=N_F_B,
          Tstart11=Tstart_G_In,
          Tstart1N=Tstart_G_Out,
          Tstart21=Tstart_M_B_In,
          Tstart2N=Tstart_M_B_Out,
          gamma=gamma_G_B,
          Tstart1=Tstart_G,
          Tstart2=Tstart_M_B)    annotation (extent=[18,24; 38,44]);
        
      equation 
        connect(heatFlowDistribution_A.side2, cC_A.side1) annotation (points=[-28,14.9;
              -28,5],           style(color=45, rgbcolor={255,127,0}));
        connect(convHT_A.side2, fluidAFlow.wall)    annotation (points=[-28,
              -31.1; -28,-41], style(color=45, rgbcolor={255,127,0}));
        connect(heatFlowDistribution_B.side2, cC_B.side1) annotation (points=[28,14.9;
              28,5],          style(color=45, rgbcolor={255,127,0}));
        connect(convHT_B.side2, fluidBFlow.wall)    annotation (points=[28,-31.1;
              28,-41],        style(color=45, rgbcolor={255,127,0}));
        connect(fluidBFlow.outfl, waterOutB)    annotation (points=[18,-46; 8,
              -46; 8,-100; 40,-100],
                                   style(thickness=2));
        connect(fluidAFlow.outfl, waterOutA)    annotation (points=[-18,-46; -8,
              -46; -8,-100; -40,-100],
                                     style(thickness=2));
        connect(fluidAFlow.infl, waterInA)    annotation (points=[-38,-46; -50,
              -46; -50,100; -40,100],
                                    style(thickness=2));
        connect(fluidBFlow.infl, waterInB)    annotation (points=[38,-46; 52,
              -46; 52,100; 40,100],
                                  style(thickness=2));
        connect(gasFlow.outfl, gasOut)   annotation (points=[12,50; 100,50; 100,
              0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(gasFlow.infl, gasIn)   annotation (points=[-12,50; -100,50;
              -100,0],
                  style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(convParallel.source, gasFlow.wall)   annotation (points=[0,29; 0,
              45],   style(color=45, rgbcolor={255,127,0}));
        connect(heatFlowDistribution_A.side1, convHT2N_A.side2) 
                                                              annotation (
            points=[-28,21; -28,30.9], style(color=45, rgbcolor={255,127,0}));
        connect(convHT2N_A.side1, convParallel.objectA) 
                                                      annotation (points=[-28,37;
              -28,42; -14,42; -14,4; -6,4; -6,19],     style(color=45, rgbcolor=
               {255,127,0}));
        connect(convHT2N_B.side2, heatFlowDistribution_B.side1) 
                                                               annotation (
            points=[28,30.9; 28,21], style(color=45, rgbcolor={255,127,0}));
        connect(convHT2N_B.side1, convParallel.objectB) 
                                                       annotation (points=[28,37;
              28,42; 16,42; 16,4; 6,4; 6,19],           style(color=45,
              rgbcolor={255,127,0}));
        connect(metalTube_B.ext, cC_B.side2) annotation (points=[28,-12.9; 28,
              -1.1], style(color=45, rgbcolor={255,127,0}));
        connect(metalTube_B.int, convHT_B.side1) annotation (points=[28,-19; 28,
              -25], style(color=45, rgbcolor={255,127,0}));
        connect(metalTube_A.int, convHT_A.side1) annotation (points=[-28,-19;
              -28,-25], style(color=45, rgbcolor={255,127,0}));
        connect(metalTube_A.ext, cC_A.side2) annotation (points=[-28,-12.9; -28,
              -1.1], style(color=45, rgbcolor={255,127,0}));
      end ParHE;
      
      model ParHE_Des 
        "Parallel Heat Exchanger (two fluid with one gas) with Desuperheater" 
        extends ThermoPower.PowerPlants.HRSG.Interfaces.ParallelHE_Des;
        
        parameter SI.CoefficientOfHeatTransfer gamma_G_A_p1 
          "Constant heat transfer coefficient in the gas side"            annotation (Dialog(tab = "pHE-1", group = "side A"));
        parameter SI.CoefficientOfHeatTransfer gamma_G_B_p1 
          "Constant heat transfer coefficient in the gas side"            annotation (Dialog(tab = "pHE-1", group = "side B"));
        parameter SI.CoefficientOfHeatTransfer gamma_F_A_p1 
          "Constant heat transfer coefficient in the fluid side"            annotation (Dialog(tab = "pHE-1", group = "side A"));
        parameter SI.CoefficientOfHeatTransfer gamma_F_B_p1 
          "Constant heat transfer coefficient in the fluid side"            annotation (Dialog(tab = "pHE-1", group = "side B"));
        parameter Choices.Flow1D.HCtypes.Temp HCtype_F_A_p1=ThermoPower.Choices.Flow1D.HCtypes.Downstream 
          "Location of the hydraulic capacitance, fluid side"            annotation (Dialog(tab = "pHE-1", group = "side A"));
        parameter Choices.Flow1D.HCtypes.Temp HCtype_F_B_p1=ThermoPower.Choices.Flow1D.HCtypes.Downstream 
          "Location of the hydraulic capacitance, fluid side"            annotation (Dialog(tab = "pHE-1", group = "side B"));
        parameter Boolean counterCurrent_A_p1 = true "Counter-current flow" 
                                            annotation (Dialog(tab = "pHE-1", group = "side A"));
        parameter Boolean counterCurrent_B_p1 = true "Counter-current flow" 
                                            annotation (Dialog(tab = "pHE-1", group = "side B"));
        parameter SI.CoefficientOfHeatTransfer gamma_G_A_p2 
          "Constant heat transfer coefficient in the gas side"            annotation (Dialog(tab = "pHE-2", group = "side A"));
        parameter SI.CoefficientOfHeatTransfer gamma_G_B_p2 
          "Constant heat transfer coefficient in the gas side"            annotation (Dialog(tab = "pHE-2", group = "side B"));
        parameter SI.CoefficientOfHeatTransfer gamma_F_A_p2 
          "Constant heat transfer coefficient in the fluid side"            annotation (Dialog(tab = "pHE-2", group = "side A"));
        parameter SI.CoefficientOfHeatTransfer gamma_F_B_p2 
          "Constant heat transfer coefficient in the fluid side"            annotation (Dialog(tab = "pHE-2", group = "side B"));
        parameter Choices.Flow1D.HCtypes.Temp HCtype_F_A_p2=ThermoPower.Choices.Flow1D.HCtypes.Downstream 
          "Location of the hydraulic capacitance, fluid side"            annotation (Dialog(tab = "pHE-2", group = "side A"));
        parameter Choices.Flow1D.HCtypes.Temp HCtype_F_B_p2=ThermoPower.Choices.Flow1D.HCtypes.Downstream 
          "Location of the hydraulic capacitance, fluid side"            annotation (Dialog(tab = "pHE-2", group = "side B"));
        parameter Boolean counterCurrent_A_p2 = true "Counter-current flow" 
                                            annotation (Dialog(tab = "pHE-2", group = "side A"));
        parameter Boolean counterCurrent_B_p2 = true "Counter-current flow" 
                                            annotation (Dialog(tab = "pHE-2", group = "side B"));
        parameter Boolean gasQuasiStatic=false 
          "Quasi-static model of the flue gas (mass, energy and momentum static balances";
        constant Real pi=Modelica.Constants.pi;
        
        //Valves parameters
        parameter Real Cv_valA "Cv (US) flow coefficient, valve A" annotation (Dialog(group = "Desuperheater A"));
        parameter SI.Pressure pnom_valA "Nominal inlet pressure, valve A" annotation (Dialog(group = "Desuperheater A"));
        parameter SI.Pressure dpnom_valA "Nominal pressure drop, valve A" annotation (Dialog(group = "Desuperheater A"));
        parameter SI.MassFlowRate wnom_valA "Nominal mass flowrate, valve A" annotation (Dialog(group = "Desuperheater A"));
        parameter SI.SpecificEnthalpy hstart_valA=1e5 
          "Specific enthalpy start value, valve A" annotation (Dialog(group = "Desuperheater A"));
        parameter Real Cv_valB "Cv (US) flow coefficient, valve B" annotation (Dialog(group = "Desuperheater B"));
        parameter SI.Pressure pnom_valB "Nominal inlet pressure, valve B" annotation (Dialog(group = "Desuperheater B"));
        parameter SI.Pressure dpnom_valB "Nominal pressure drop, valve B" annotation (Dialog(group = "Desuperheater B"));
        parameter SI.MassFlowRate wnom_valB "Nominal mass flowrate, valve B" annotation (Dialog(group = "Desuperheater B"));
        parameter SI.SpecificEnthalpy hstart_valB=1e5 
          "Specific enthalpy start value, valve B" annotation (Dialog(group = "Desuperheater B"));
        
        ParHE pHE1(
          redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=N_G_p1,
          N_F_A=N_F_A_p1,
          N_F_B=N_F_B_p1,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate_A=fluidNomFlowRate_A,
          fluidNomFlowRate_B=fluidNomFlowRate_B,
          gasNomPressure=gasNomPressure,
          fluidNomPressure_A=fluidNomPressure_A,
          fluidNomPressure_B=fluidNomPressure_B,
          exchSurface_G_A=exchSurface_G_A_p1,
          exchSurface_G_B=exchSurface_G_B_p1,
          exchSurface_F_A=exchSurface_F_A_p1,
          exchSurface_F_B=exchSurface_F_B_p1,
          gasVol=gasVol_p1,
          fluidVol_A=fluidVol_A_p1,
          fluidVol_B=fluidVol_B_p1,
          metalVol_A=metalVol_A_p1,
          metalVol_B=metalVol_B_p1,
          gamma_G_A=gamma_G_A_p1,
          gamma_G_B=gamma_G_B_p1,
          gamma_F_A=gamma_F_A_p1,
          gamma_F_B=gamma_F_B_p1,
          HCtype_F_A=HCtype_F_A_p1,
          HCtype_F_B=HCtype_F_B_p1,
          counterCurrent_A=counterCurrent_A_p1,
          counterCurrent_B=counterCurrent_B_p1,
          gasQuasiStatic=gasQuasiStatic,
          lambda=lambda,
          extSurfaceTub_A=extSurfaceTub_A_p1,
          extSurfaceTub_B=extSurfaceTub_B_p1,
          use_T=use_T,
          SSInit=SSInit,
          Tstart_G_In=Tstart_G_In_p1,
          Tstart_G_Out=Tstart_G_Out_p1,
          Tstart_M_A_In=Tstart_M_A_In_p1,
          Tstart_M_A_Out=Tstart_M_A_Out_p1,
          Tstart_M_A=Tstart_M_A_p1,
          Tstart_F_A_In=Tstart_F_A_In_p1,
          Tstart_F_A_Out=Tstart_F_A_Out_p1,
          Tstart_F_A=Tstart_F_A_p1,
          hstart_F_A_In=hstart_F_A_In_p1,
          hstart_F_A_Out=hstart_F_A_Out_p1,
          hstart_F_A=hstart_F_A_p1,
          Tstart_M_B_In=Tstart_M_B_In_p1,
          Tstart_M_B_Out=Tstart_M_B_Out_p1,
          Tstart_F_B_In=Tstart_F_B_In_p1,
          Tstart_F_B=Tstart_F_B_p1,
          hstart_F_B_In=hstart_F_B_In_p1,
          hstart_F_B_Out=hstart_F_B_Out_p1,
          hstart_F_B=hstart_F_B_p1,
          pstart_G=pstart_G_p1,
          pstartin_F_A=pstartin_F_A_p1,
          pstartout_F_A=pstartout_F_A_p1,
          pstartin_F_B=pstartin_F_B_p1,
          pstartout_F_B=pstartout_F_B_p1,
          rhomcm_A=rhomcm_A_p1,
          rhomcm_B=rhomcm_B_p1,
          Tstart_M_B=Tstart_M_B_p1,
          Tstart_F_B_Out=Tstart_F_B_Out_p1) 
          annotation (extent=[-46,28; -26,48]);
        annotation (Diagram);
        Water.ValveLiq valveA(
          redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=Cv_valA,
          pnom=pnom_valA,
          dpnom=dpnom_valA,
          wnom=wnom_valA,
          CheckValve=true,
          hstart=hstart_valA)  annotation (extent=[-18,-32; -34,-16],
                                                                  rotation=180);
        Water.ValveLiq valveB(
          redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=Cv_valB,
          pnom=pnom_valB,
          dpnom=dpnom_valB,
          wnom=wnom_valB,
          CheckValve=true,
          hstart=hstart_valB) annotation (extent=[-18,2; -34,18],rotation=180);
        Water.Mixer flowJoinA(
          redeclare package Medium = FluidMedium,
          V=3,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          pstart=pstartout_F_A_p1,
          hstart=hstart_F_A_Out_p1) 
                                annotation (extent=[-10,-18; 10,2],
                                                                  rotation=0);
        Water.Mixer flowJoinB(   redeclare package Medium = FluidMedium,
          V=3,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          pstart=pstartout_F_B_p1,
          hstart=hstart_F_B_Out_p1) 
          annotation (extent=[-10,6; 10,26], rotation=0);
        ParHE pHE2(
          redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=N_G_p2,
          N_F_A=N_F_A_p2,
          N_F_B=N_F_B_p2,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate_A=fluidNomFlowRate_A,
          fluidNomFlowRate_B=fluidNomFlowRate_B,
          gasNomPressure=gasNomPressure,
          fluidNomPressure_A=fluidNomPressure_A,
          fluidNomPressure_B=fluidNomPressure_B,
          exchSurface_G_A=exchSurface_G_A_p2,
          exchSurface_G_B=exchSurface_G_B_p2,
          exchSurface_F_A=exchSurface_F_A_p2,
          exchSurface_F_B=exchSurface_F_B_p2,
          gasVol=gasVol_p2,
          fluidVol_A=fluidVol_A_p2,
          fluidVol_B=fluidVol_B_p2,
          metalVol_A=metalVol_A_p2,
          metalVol_B=metalVol_B_p2,
          gamma_G_A=gamma_G_A_p2,
          gamma_G_B=gamma_G_B_p2,
          gamma_F_A=gamma_F_A_p2,
          gamma_F_B=gamma_F_B_p2,
          HCtype_F_A=HCtype_F_A_p2,
          HCtype_F_B=HCtype_F_B_p2,
          counterCurrent_A=counterCurrent_A_p2,
          counterCurrent_B=counterCurrent_B_p2,
          gasQuasiStatic=gasQuasiStatic,
          lambda=lambda,
          extSurfaceTub_A=extSurfaceTub_A_p2,
          extSurfaceTub_B=extSurfaceTub_B_p2,
          use_T=use_T,
          SSInit=SSInit,
          Tstart_G_In=Tstart_G_In_p2,
          Tstart_G_Out=Tstart_G_Out_p2,
          Tstart_M_A_In=Tstart_M_A_In_p2,
          Tstart_M_A_Out=Tstart_M_A_Out_p2,
          Tstart_M_A=Tstart_M_A_p2,
          Tstart_F_A_In=Tstart_F_A_In_p2,
          Tstart_F_A_Out=Tstart_F_A_Out_p2,
          Tstart_F_A=Tstart_F_A_p2,
          hstart_F_A_In=hstart_F_A_In_p2,
          hstart_F_A_Out=hstart_F_A_Out_p2,
          hstart_F_A=hstart_F_A_p2,
          Tstart_M_B_In=Tstart_M_B_In_p2,
          Tstart_M_B_Out=Tstart_M_B_Out_p2,
          Tstart_F_B_In=Tstart_F_B_In_p2,
          Tstart_F_B=Tstart_F_B_p2,
          hstart_F_B_In=hstart_F_B_In_p2,
          hstart_F_B_Out=hstart_F_B_Out_p2,
          hstart_F_B=hstart_F_B_p2,
          pstart_G=pstart_G_p2,
          pstartin_F_A=pstartin_F_A_p2,
          pstartout_F_A=pstartout_F_A_p2,
          pstartin_F_B=pstartin_F_B_p2,
          pstartout_F_B=pstartout_F_B_p2,
          rhomcm_A=rhomcm_A_p2,
          rhomcm_B=rhomcm_B_p2,
          Tstart_M_B=Tstart_M_B_p2,
          Tstart_F_B_Out=Tstart_F_B_Out_p2) 
          annotation (extent=[26,-36; 46,-16], style(thickness=2));
        Water.SensT intermediate_B(redeclare package Medium = FluidMedium) 
          annotation (extent=[16,12; 30,26]);
        Water.SensT intermediate_A(redeclare package Medium = FluidMedium) 
          annotation (extent=[16,-12; 30,2]);
      equation 
        connect(pHE1.waterInB, waterInB)  annotation (points=[-32,48; -32,54;
              12,54; 12,100; 40,100],
                                    style(thickness=2));
        connect(pHE1.waterInA, waterInA) 
          annotation (points=[-40,48; -40,100],style(thickness=2));
        connect(pHE1.gasIn, gasIn)  annotation (points=[-46,38; -100,38; -100,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(pHE2.gasOut, gasOut) annotation (points=[46,-26; 100,-26; 100,0],
                                                                                style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(pHE2.gasIn, pHE1.gasOut) annotation (points=[26,-26; 14,-26; 14,
              38; -26,38],
                   style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(valveA.outlet, flowJoinA.in2) annotation (points=[-18,-24; -14,
              -24; -14,-14; -7.9,-14],                                             style(
            thickness=2,
            fillColor=62,
            rgbfillColor={0,127,127},
            fillPattern=1));
        connect(flowJoinA.in1, pHE1.waterOutA) 
                                              annotation (points=[-8,-2; -14,-2;
              -14,16; -40,16; -40,28],
                       style(
            thickness=2,
            fillColor=62,
            rgbfillColor={0,127,127},
            fillPattern=1));
        connect(flowJoinB.in1, pHE1.waterOutB) annotation (points=[-8,22; -32,
              22; -32,28],            style(
            thickness=2,
            fillColor=62,
            rgbfillColor={0,127,127},
            fillPattern=1));
        connect(valveB.outlet,flowJoinB. in2) annotation (points=[-18,10; 
              -12.95,10; -12.95,10; -7.9,10],
                                   style(
            thickness=2,
            fillColor=62,
            rgbfillColor={0,127,127},
            fillPattern=1));
        connect(intermediate_A.outlet, pHE2.waterInA) 
          annotation (points=[27.2,-7.8; 32,-7.8; 32,-16], style(thickness=2));
        connect(intermediate_A.inlet, flowJoinA.out) 
          annotation (points=[18.8,-7.8; 18.8,-8; 10,-8],style(thickness=2));
        connect(intermediate_B.inlet,flowJoinB. out) annotation (points=[18.8,
              16.2; 19.4,16.2; 19.4,16; 10,16],style(thickness=2));
        connect(intermediate_B.outlet, pHE2.waterInB) annotation (points=[27.2,
              16.2; 27.2,16; 40,16; 40,-16], style(thickness=2));
        connect(waterOutB, pHE2.waterOutB) 
          annotation (points=[40,-100; 40,-36],style(thickness=2));
        connect(pHE2.waterOutA, waterOutA) annotation (points=[32,-36; 32,-50;
              -40,-50; -40,-100],style(thickness=2));
        connect(T_intermedB, intermediate_B.T) annotation (points=[101,39; 40,
              39; 40,23.2; 28.6,23.2],
                                   style(color=74, rgbcolor={0,0,127}));
        connect(intermediate_A.T, T_intermedA) annotation (points=[28.6,-0.8; 
              52,-0.8; 52,59; 101,59],style(color=74, rgbcolor={0,0,127}));
        connect(theta_valveA, valveA.theta)  annotation (points=[101,-42; -26,
              -42; -26,-30.4], style(color=74, rgbcolor={0,0,127}));
        connect(theta_valveB, valveB.theta) annotation (points=[101,-63; 60,-63; 
              60,-46; -36,-46; -36,-10; -26,-10; -26,3.6],  style(color=74,
              rgbcolor={0,0,127}));
        connect(LiquidWaterIn_A, valveA.inlet) annotation (points=[-100,-42;
              -60,-42; -60,-24; -34,-24], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(LiquidWaterIn_B, valveB.inlet) annotation (points=[-100,-70; 
              -70,-70; -70,10; -34,10], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end ParHE_Des;
      
      model HE_simp "Simplified Heat Exchanger fluid - gas" 
        extends Interfaces.HeatExchanger(extSurfaceTub = exchSurface_G);
        
        parameter SI.CoefficientOfHeatTransfer gamma_G 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F 
          "Constant heat transfer coefficient in the fluid side";
        constant Real pi=Modelica.Constants.pi;
        
      Water.Flow1D fluidFlow(
          Nt=1,
          N=N_F,
          wnom=fluidNomFlowRate,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          L=exchSurface_F^2/(fluidVol*pi*4),
          A=(fluidVol*4/exchSurface_F)^2/4*pi,
          omega=fluidVol*4/exchSurface_F*pi,
          Dhyd=fluidVol*4/exchSurface_F,
          pstartin=pstartin_F,
          pstartout=pstartout_F,
          hstart=hstart_F,
          hstartin=hstart_F_In,
          hstartout=hstart_F_Out) 
                                 annotation (extent=[-10,-50; 10,-30]);
        Thermal.ConvHT convHT(gamma=gamma_F, N=N_F,
          Tstart11=Tstart_M_In,
          Tstart1N=Tstart_M_Out,
          Tstart21=Tstart_F_In,
          Tstart2N=Tstart_F_Out,
          Tstart1=Tstart_M,
          Tstart2=Tstart_F) 
          annotation (extent=[-10,-30; 10,-10]);
        Thermal.MetalTube metalTube(
          rhomcm=rhomcm,
          lambda=lambda,
          N=N_F,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          L=exchSurface_F^2/(fluidVol*pi*4),
          rint=fluidVol*4/exchSurface_F/2,
          Tstart1=Tstart_M_In,
          TstartN=Tstart_M_Out,
          WallRes=false,
          Tstart=Tstart_M,
          rext=(metalVol + fluidVol)*4/extSurfaceTub/2) 
                 annotation (extent=[-10,10; 10,-10]);
        Gas.Flow1D gasFlow(
          L=1,
          Dhyd=1,
          wnom=gasNomFlowRate,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          QuasiStatic=true,
          N=N_G,
          A=gasVol/1,
          omega=exchSurface_G/1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FlueGasMedium,
          Tstartin=Tstart_G_In,
          Tstartout=Tstart_G_Out,
          pstart=pstart_G,
          Tstart=Tstart_G)       annotation (extent=[-12,50; 12,30]);
        annotation (Diagram);
        Thermal.ConvHT2N convHT2N(
          N1=N_G,
          N2=N_F,
          gamma=gamma_G,
          Tstart11=Tstart_G_In,
          Tstart1N=Tstart_G_Out,
          Tstart21=Tstart_M_In,
          Tstart2N=Tstart_M_Out,
          Tstart1=Tstart_G,
          Tstart2=Tstart_M)    annotation (extent=[-10,10; 10,30]);
      equation 
        connect(fluidFlow.wall, convHT.side2) 
                                           annotation (points=[0,-35; 0,-23.1],
            style(color=45, rgbcolor={255,127,0}));
        connect(gasFlow.infl, gasIn) annotation (points=[-12,40; -100,40; -100,
              0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(gasFlow.outfl, gasOut) annotation (points=[12,40; 100,40; 100,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(fluidFlow.outfl, waterOut) 
                                        annotation (points=[10,-40; 60,-40; 60,
              -100; 0,-100],
                           style(thickness=2));
        connect(fluidFlow.infl, waterIn) 
                                      annotation (points=[-10,-40; -40,-40; -40,
              100; 0,100],
                         style(thickness=2));
        connect(convHT2N.side1,gasFlow. wall) annotation (points=[0,23; 0,35],
            style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.int, convHT.side1) annotation (points=[0,-3; 0,-17],
            style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.ext, convHT2N.side2) annotation (points=[0,3.1; 0,
              16.9], style(color=45, rgbcolor={255,127,0}));
      end HE_simp;
      
      model HE2ph_simp "Simplified Heat Exchanger fluid - gas (fluid 2-phase)" 
        extends Interfaces.HeatExchanger(extSurfaceTub = exchSurface_G);
        
        parameter SI.CoefficientOfHeatTransfer gamma_G 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F 
          "Constant heat transfer coefficient in the fluid side";
        constant Real pi=Modelica.Constants.pi;
        
      Water.Flow1D2ph fluidFlow(
          Nt=1,
          N=N_F,
          wnom=fluidNomFlowRate,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          L=exchSurface_F^2/(fluidVol*pi*4),
          A=(1/exchSurface_F*fluidVol*4)^2/4*pi,
          omega=1/exchSurface_F*fluidVol*4*pi,
          Dhyd=1/exchSurface_F*fluidVol*4,
          pstartin=pstartin_F,
          pstartout=pstartout_F,
          hstart=hstart_F,
          hstartin=hstart_F_In,
          hstartout=hstart_F_Out) 
                                 annotation (extent=[-10,-50; 10,-30]);
        Thermal.ConvHT convHT(gamma=gamma_F, N=N_F,
          Tstart11=Tstart_M_In,
          Tstart1N=Tstart_M_Out,
          Tstart21=Tstart_F_In,
          Tstart2N=Tstart_F_Out,
          Tstart1=Tstart_M,
          Tstart2=Tstart_F) 
          annotation (extent=[-10,-30; 10,-10]);
        Thermal.MetalTube metalTube(
          rhomcm=rhomcm,
          lambda=lambda,
          N=N_F,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          L=exchSurface_F^2/(fluidVol*pi*4),
          rint=1/exchSurface_F*fluidVol*4/2,
          WallRes=false,
          Tstart1=Tstart_M_In,
          TstartN=Tstart_M_Out,
          Tstart=Tstart_M,
          rext=1/extSurfaceTub*(metalVol + fluidVol)*4/2) 
                 annotation (extent=[-10,10; 10,-10]);
        Gas.Flow1D gasFlow(
          L=1,
          Dhyd=1,
          wnom=gasNomFlowRate,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          QuasiStatic=true,
          N=N_G,
          A=gasVol/1,
          omega=exchSurface_G/1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FlueGasMedium,
          Tstartin=Tstart_G_In,
          Tstartout=Tstart_G_Out,
          pstart=pstart_G,
          Tstart=Tstart_G)       annotation (extent=[-12,50; 12,30]);
        annotation (Diagram);
        Thermal.ConvHT2N convHT2N(
          N1=N_G,
          N2=N_F,
          gamma=gamma_G,
          Tstart11=Tstart_G_In,
          Tstart1N=Tstart_G_Out,
          Tstart21=Tstart_M_In,
          Tstart2N=Tstart_M_Out,
          Tstart1=Tstart_G,
          Tstart2=Tstart_M)    annotation (extent=[-10,10; 10,30]);
      equation 
        connect(fluidFlow.wall, convHT.side2) 
                                           annotation (points=[0,-35; 0,-23.1],
            style(color=45, rgbcolor={255,127,0}));
        connect(gasFlow.infl, gasIn) annotation (points=[-12,40; -100,40; -100,
              0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(gasFlow.outfl, gasOut) annotation (points=[12,40; 100,40; 100,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(fluidFlow.outfl, waterOut) 
                                        annotation (points=[10,-40; 60,-40; 60,
              -100; 0,-100],
                           style(thickness=2));
        connect(fluidFlow.infl, waterIn) 
                                      annotation (points=[-10,-40; -40,-40; -40,
              100; 0,100],
                         style(thickness=2));
        connect(convHT2N.side1,gasFlow. wall) annotation (points=[0,23; 0,35],
            style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.int, convHT.side1) annotation (points=[0,-3; 0,-17],
            style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.ext, convHT2N.side2) annotation (points=[0,3.1; 0,
              16.9], style(color=45, rgbcolor={255,127,0}));
      end HE2ph_simp;
      
      model ParHE_simp "Simplified parallel heat exchangers fluid - gas" 
        extends Interfaces.ParallelHE(extSurfaceTub_A = exchSurface_G_A,
                                      extSurfaceTub_B = exchSurface_G_B);
        
        parameter SI.CoefficientOfHeatTransfer gamma_G_A 
          "Constant heat transfer coefficient in the gas side"     annotation (Dialog(group = "side A"));
        parameter SI.CoefficientOfHeatTransfer gamma_G_B 
          "Constant heat transfer coefficient in the gas side"     annotation (Dialog(group = "side B"));
        parameter SI.CoefficientOfHeatTransfer gamma_F_A 
          "Constant heat transfer coefficient in the fluid side"     annotation (Dialog(group = "side A"));
        parameter SI.CoefficientOfHeatTransfer gamma_F_B 
          "Constant heat transfer coefficient in the fluid side"     annotation (Dialog(group = "side B"));
        constant Real pi=Modelica.Constants.pi;
        
        annotation (Diagram);
        ConvParallel convParallel(
          N=N_G,
          As=exchSurface_G_A + exchSurface_G_B,
          Aa=exchSurface_G_A,
          Ab=exchSurface_G_B) 
                             annotation (extent=[-10,16; 10,36]);
      Water.Flow1D fluidAFlow(
          Nt=1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          N=N_F_A,
          L=exchSurface_F_A^2/(fluidVol_A*pi*4),
          A=(fluidVol_A*4/exchSurface_F_A)^2/4*pi,
          omega=fluidVol_A*4/exchSurface_F_A*pi,
          Dhyd=fluidVol_A*4/exchSurface_F_A,
          pstartin=pstartin_F_A,
          pstartout=pstartout_F_A,
          hstart=hstart_F_A,
          wnom=fluidNomFlowRate_A,
          hstartin=hstart_F_A_In,
          hstartout=hstart_F_A_Out) 
                                 annotation (extent=[-38,-52; -18,-32]);
        Thermal.ConvHT convHT_A(
          N=N_F_A,
          gamma=gamma_F_A,
          Tstart11=Tstart_M_A_In,
          Tstart1N=Tstart_M_A_Out,
          Tstart21=Tstart_F_A_In,
          Tstart2N=Tstart_F_A_Out,
          Tstart1=Tstart_M_A,
          Tstart2=Tstart_F_A) 
          annotation (extent=[-38,-30; -18,-10]);
        Thermal.MetalTube metalTube_A(
          lambda=lambda,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          N=N_F_A,
          L=exchSurface_F_A^2/(fluidVol_A*pi*4),
          rint=fluidVol_A*4/exchSurface_F_A/2,
          WallRes=false,
          Tstart1=Tstart_M_A_In,
          TstartN=Tstart_M_A_Out,
          Tstart=Tstart_M_A,
          rext=(metalVol_A + fluidVol_A)*4/extSurfaceTub_A/2,
          rhomcm=rhomcm_A) 
                 annotation (extent=[-38,6; -18,-14]);
        Gas.Flow1D gasFlow(
          L=1,
          Dhyd=1,
          wnom=gasNomFlowRate,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          QuasiStatic=true,
          N=N_G,
          A=gasVol/1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FlueGasMedium,
          omega=(exchSurface_G_A + exchSurface_G_B)/1,
          Tstartin=Tstart_G_In,
          Tstartout=Tstart_G_Out,
          pstart=pstart_G,
          Tstart=Tstart_G)       annotation (extent=[-12,64; 12,44]);
      Water.Flow1D fluidBFlow(
          Nt=1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          N=N_F_B,
          L=exchSurface_F_B^2/(fluidVol_B*pi*4),
          A=(fluidVol_B*4/exchSurface_F_B)^2/4*pi,
          omega=fluidVol_B*4/exchSurface_F_B*pi,
          Dhyd=fluidVol_B*4/exchSurface_F_B,
          wnom=fluidNomFlowRate_B,
          pstartin=pstartin_F_B,
          pstartout=pstartout_F_B,
          hstart=hstart_F_B,
          hstartin=hstart_F_B_In,
          hstartout=hstart_F_B_Out) 
                                 annotation (extent=[38,-52; 18,-32]);
        Thermal.ConvHT convHT_B(
          N=N_F_B,
          gamma=gamma_F_B,
          Tstart11=Tstart_M_B_In,
          Tstart1N=Tstart_M_B_Out,
          Tstart21=Tstart_F_B_In,
          Tstart2N=Tstart_F_B_Out,
          Tstart1=Tstart_M_B,
          Tstart2=Tstart_F_B) 
          annotation (extent=[18,-30; 38,-10]);
        Thermal.MetalTube metalTube_B(
          lambda=lambda,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          L=exchSurface_F_B^2/(fluidVol_B*pi*4),
          rint=fluidVol_B*4/exchSurface_F_B/2,
          N=N_F_B,
          WallRes=false,
          Tstart1=Tstart_M_B_In,
          TstartN=Tstart_M_B_Out,
          Tstart=Tstart_M_B,
          rext=(metalVol_B + fluidVol_B)*4/extSurfaceTub_B/2,
          rhomcm=rhomcm_B) 
                 annotation (extent=[18,6; 38,-14]);
        Thermal.ConvHT2N convHT2N_A(
          N1=N_G,
          N2=N_F_A,
          gamma=gamma_G_A,
          Tstart11=Tstart_G_In,
          Tstart1N=Tstart_G_Out,
          Tstart21=Tstart_M_A_In,
          Tstart2N=Tstart_M_A_Out,
          Tstart1=Tstart_G,
          Tstart2=Tstart_M_A)    annotation (extent=[-38,10; -18,30]);
        Thermal.ConvHT2N convHT2N_B(
          N1=N_G,
          N2=N_F_B,
          gamma=gamma_G_B,
          Tstart11=Tstart_G_In,
          Tstart1N=Tstart_G_Out,
          Tstart21=Tstart_M_B_In,
          Tstart2N=Tstart_M_B_Out,
          Tstart1=Tstart_G,
          Tstart2=Tstart_M_B)    annotation (extent=[18,10; 38,30]);
      equation 
        connect(convHT_A.side2, fluidAFlow.wall)    annotation (points=[-28,
              -23.1; -28,-37], style(color=45, rgbcolor={255,127,0}));
        connect(convHT_B.side2, fluidBFlow.wall)    annotation (points=[28,-23.1;
              28,-37],        style(color=45, rgbcolor={255,127,0}));
        connect(fluidBFlow.outfl, waterOutB)    annotation (points=[18,-42; 8,
              -42; 8,-100; 40,-100],
                                   style(thickness=2));
        connect(fluidAFlow.outfl, waterOutA)    annotation (points=[-18,-42; -8,
              -42; -8,-100; -40,-100],
                                     style(thickness=2));
        connect(fluidAFlow.infl, waterInA)    annotation (points=[-38,-42; -50,
              -42; -50,100; -40,100],
                                    style(thickness=2));
        connect(fluidBFlow.infl, waterInB)    annotation (points=[38,-42; 52,
              -42; 52,100; 40,100],
                                  style(thickness=2));
        connect(gasFlow.outfl, gasOut)   annotation (points=[12,54; 100,54; 100,
              0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(gasFlow.infl, gasIn)   annotation (points=[-12,54; -100,54;
              -100,0],
                  style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(convParallel.source, gasFlow.wall)   annotation (points=[0,33; 0,
              49],   style(color=45, rgbcolor={255,127,0}));
        connect(convHT2N_A.side1, convParallel.objectA) 
                                                      annotation (points=[-28,23;
              -28,36; -14,36; -14,6; -6,6; -6,23],     style(color=45, rgbcolor=
               {255,127,0}));
        connect(convParallel.objectB, convHT2N_B.side1) 
                                                       annotation (points=[6,23; 6,
              6; 14,6; 14,36; 28,36; 28,23],   style(color=45, rgbcolor={255,
                127,0}));
        connect(metalTube_A.int, convHT_A.side1) annotation (points=[-28,-7;
              -28,-17], style(color=45, rgbcolor={255,127,0}));
        connect(metalTube_A.ext, convHT2N_A.side2) 
                                                 annotation (points=[-28,-0.9;
              -28,16.9], style(color=45, rgbcolor={255,127,0}));
        connect(convHT_B.side1, metalTube_B.int) annotation (points=[28,-17; 28,
              -7], style(color=45, rgbcolor={255,127,0}));
        connect(metalTube_B.ext, convHT2N_B.side2) 
                                                  annotation (points=[28,-0.9;
              28,16.9], style(color=45, rgbcolor={255,127,0}));
      end ParHE_simp;
      
      model HEhtc_f 
        "Heat Exchanger fluid - gas, with variable heat transfer coefficient only fluid side" 
        extends Interfaces.HeatExchanger;
        
        parameter SI.CoefficientOfHeatTransfer gamma_G 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_nom_F 
          "Nominal h.t.c. coefficient fluid side";
        parameter Real kw_F 
          "Exponent of the mass flow rate in the h.t.c. correlation fluid side";
        parameter Boolean gasQuasiStatic=false 
          "Quasi-static model of the flue gas (mass, energy and momentum static balances";
        parameter Boolean counterCurrent=true "Counter-current flow";
        constant Real pi=Modelica.Constants.pi;
        
        Thermal.MetalTube metalTube(
          rhomcm=rhomcm,
          lambda=lambda,
          N=N_F,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          L=exchSurface_F^2/(fluidVol*pi*4),
          rint=fluidVol*4/exchSurface_F/2,
          WallRes=false,
          Tstart1=Tstart_M_In,
          TstartN=Tstart_M_Out,
          Tstart=Tstart_M,
          rext=(metalVol + fluidVol)*4/extSurfaceTub/2) 
                 annotation (extent=[-10,-4; 10,-24]);
        Gas.Flow1D gasFlow(
          L=1,
          Dhyd=1,
          wnom=gasNomFlowRate,
          FFtype=ThermoPower.Choices.Flow1D.FFtypes.NoFriction,
          N=N_G,
          A=gasVol/1,
          omega=exchSurface_G/1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FlueGasMedium,
          Tstartin=Tstart_G_In,
          Tstartout=Tstart_G_Out,
          pstart=pstart_G,
          QuasiStatic=gasQuasiStatic) 
                                 annotation (extent=[-12,60; 12,40]);
        annotation (Diagram);
        Thermal.CounterCurrent cC(counterCurrent=counterCurrent, N=N_F) 
          annotation (extent=[-10,-8; 10,12]);
        Thermal.HeatFlowDistribution heatFlowDistribution(
          N=N_F,
          A1=exchSurface_G,
          A2=extSurfaceTub) 
          annotation (extent=[-10,8; 10,28]);
        ThermoPower.PowerPlants.HRSG.Components.Flow1Dhtc fluidFlow_Extension(
                                                  Nt=1,
          N=N_F,
          wnom=fluidNomFlowRate,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          L=exchSurface_F^2/(fluidVol*pi*4),
          A=(fluidVol*4/exchSurface_F)^2/4*pi,
          omega=fluidVol*4/exchSurface_F*pi,
          Dhyd=fluidVol*4/exchSurface_F,
          pstartin=pstartin_F,
          pstartout=pstartout_F,
          hstart=hstart_F,
          hstartin=hstart_F_In,
          hstartout=hstart_F_Out,
          gamma_nom=gamma_nom_F,
          kw=kw_F) 
          annotation (extent=[-10,-56; 10,-36]);
        Thermal.ConvHT_htc convHT_htc(
          N=N_F,
          TstartF1=Tstart_M_In,
          TstartFN=Tstart_M_Out,
          TstartO1=Tstart_F_In,
          TstartON=Tstart_F_Out,
          TstartF=Tstart_M,
          TstartO=Tstart_F)    annotation (extent=[-10,-20; 10,-40], rotation=0);
        Thermal.ConvHT2N convHT2N(
          N1=N_G,
          N2=N_F,
          gamma=gamma_G,
          Tstart11=Tstart_G_In,
          Tstart1N=Tstart_G_Out,
          Tstart21=Tstart_M_In,
          Tstart2N=Tstart_M_Out,
          Tstart1=Tstart_G,
          Tstart2=Tstart_M)    annotation (extent=[-10,24; 10,44]);
      equation 
        connect(gasFlow.infl, gasIn) annotation (points=[-12,50; -100,50; -100,
              0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(gasFlow.outfl, gasOut) annotation (points=[12,50; 100,50; 100,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(heatFlowDistribution.side2,cC. side1) annotation (points=[0,14.9;
              0,5],           style(
            color=45,
            rgbcolor={255,127,0},
            fillPattern=1));
        connect(convHT_htc.fluidside, fluidFlow_Extension.wall)   annotation (
            points=[0,-33; 0,-41], style(color=3, rgbcolor={0,0,255}));
        connect(fluidFlow_Extension.infl, waterIn)   annotation (points=[-10,-46;
              -40,-46; -40,100; 0,100],    style(thickness=2));
        connect(fluidFlow_Extension.outfl, waterOut)   annotation (points=[10,-46;
              40,-46; 40,-100; 0,-100],    style(thickness=2));
        connect(convHT2N.side2, heatFlowDistribution.side1) annotation (points=[0,30.9;
              0,21],          style(color=45, rgbcolor={255,127,0}));
        connect(convHT2N.side1,gasFlow. wall) annotation (points=[0,37; 0,45],
            style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.int, convHT_htc.otherside) annotation (points=[0,-17;
              0,-27], style(color=45, rgbcolor={255,127,0}));
        connect(metalTube.ext, cC.side2) annotation (points=[0,-10.9; 0,-1.1],
            style(color=45, rgbcolor={255,127,0}));
      end HEhtc_f;
      
      model DrumSensors 
        extends Water.Drum;
        Modelica.Blocks.Interfaces.RealOutput p_out "Pressure sensor output" 
                                                 annotation (extent=[-26,64;
              -46,84]);
        Modelica.Blocks.Interfaces.RealOutput y_out "Level sensor output" 
                                            annotation (extent=[-58,30; -78,50]);
        annotation (Icon(
            Text(extent=[-76,102; -4,82],
              style(color=0, rgbcolor={0,0,0}),
              string="p"),
            Text(extent=[-106,70; -34,50],
              style(color=0, rgbcolor={0,0,0}),
              string="y")),
                          Diagram);
      equation 
        y_out = y;
        p_out = p;
      end DrumSensors;
      
      model Flow1Dhtc 
        "Extension of the Water.Flow1D for fluid flow model with variable heat transfer coefficient" 
        extends ThermoPower.Water.Flow1D(redeclare ThermoPower.Thermal.DHThtc 
            wall);
        parameter SI.CoefficientOfHeatTransfer gamma_nom 
          "Nominal h.t.c. coefficient";
        parameter Real kw 
          "Exponent of the mass flow rate in the h.t.c. correlation";
      equation 
        for j in 1:N loop
          wall.gamma[j] = gamma_nom*noEvent(abs(infl.w/wnom)^kw);
        end for;
        annotation (Diagram);
      end Flow1Dhtc;
      
      model Flow1D2phhtc 
        "Extension of the Water.Flow1D2ph for fluid flow model with variable heat transfer coefficient" 
        extends ThermoPower.Water.Flow1D2ph(redeclare 
            ThermoPower.Thermal.DHThtc wall);
        parameter SI.CoefficientOfHeatTransfer gamma_nom 
          "Nominal h.t.c. coefficient";
        parameter Real kw 
          "Exponent of the mass flow rate in the h.t.c. correlation";
      equation 
        for j in 1:N loop
          wall.gamma[j] = gamma_nom*noEvent(abs(infl.w/wnom)^kw);
        end for;
        annotation (Diagram);
      end Flow1D2phhtc;
      
      model FlowGas1Dhtc 
        "Extension of the Gas.Flow1D for fluid flow model with variable heat transfer coefficient" 
        extends Gas.Flow1D( redeclare ThermoPower.Thermal.DHThtc wall);
        parameter SI.CoefficientOfHeatTransfer gamma_nom 
          "Nominal h.t.c. coefficient";
        parameter Real kw 
          "Exponent of the mass flow rate in the h.t.c. correlation";
      equation 
        for j in 1:N loop
          wall.gamma[j] = gamma_nom*noEvent(abs(infl.w/wnom)^kw);
        end for;
        annotation (Diagram);
      end FlowGas1Dhtc;
      
      model ConvParallel 
        "Convective heat transfer between one source and two objects in parallel" 
        import Modelica.SIunits.*;
        parameter Integer N=2 "Number of Nodes";
        parameter Area As "Area of source";
        parameter Area Aa "Area of object a";
        parameter Area Ab "Area of object b";
        ThermoPower.Thermal.DHT source(N=N) 
                       annotation (extent=[-40,60; 40,80]);
        ThermoPower.Thermal.DHT objectA(N=N) 
                       annotation (extent=[-100,-40; -20,-20]);
        ThermoPower.Thermal.DHT objectB(N=N) 
                       annotation (extent=[20,-40; 100,-20]);
        annotation (Icon(
            Text(
              extent=[-100,-42; -20,-80],
              style(color=46),
              string="A"),
                         Rectangle(extent=[-100,60; 100,-20], style(
                color=0,
                fillColor=7,
                fillPattern=7)),
            Line(points=[-8,60; -60,-20], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2)),
            Line(points=[-56,-2; -60,-20; -44,-12], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2)),
            Line(points=[60,-2; 62,-20; 48,-8], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2)),
            Line(points=[10,60; 62,-20], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2)),
            Line(points=[26,50; 10,60; 12,42], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2)),
            Line(points=[-24,52; -8,60; -12,44], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2)),
            Text(
              extent=[-110,-74; 112,-100],
              string="%name",
              style(color=46)),
            Text(
              extent=[20,-40; 100,-80],
              style(color=46),
              string="B")),      Documentation(info="<HTML>
<p>The model connected to the <tt>source</tt> connector exchanges power \"in parallel\" with the models connected to the <tt>objectA</tt> and <tt>objectB</tt> connectors.
</HTML>",     revisions=""),
          Diagram);
        
      equation 
        source.phi*As + objectA.phi*Aa + objectB.phi*Ab = zeros(N) 
          "Energy balance";
        source.T = objectA.T;
        source.T = objectB.T;
      end ConvParallel;
      
      model BaseReader_water 
        "Base reader for the visualization of the state in the simulation (water)" 
        replaceable package Medium = Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance "Medium model";
        ThermoPower.Water.FlangeA inlet(redeclare package Medium = Medium) 
                                        annotation (extent=[-80,-20; -40,20]);
        ThermoPower.Water.FlangeB outlet(redeclare package Medium = Medium) 
                                         annotation (extent=[40,-20; 80,20]);
        annotation (
          Diagram,
          Icon(Polygon(points=[-80,0; 0,32; 80,0; 0,-32; -80,0], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)), Text(
              extent=[-80,20; 80,-20],
              string="S",
              style(
                color=7,
                rgbcolor={255,255,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1))),
          Documentation(info="<HTML>
<p>This component can be inserted in a hydraulic circuit to measure the temperature of the fluid flowing through it.
<p>Flow reversal is supported.
</HTML>",   revisions="<html>
<ul>
<li><i>16 Dec 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Standard medium definition added.</li>
<li><i>1 Jul 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Adapted to Modelica.Media.</li>
<li><i>1 Oct 2003</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       First release.</li>
</ul>
</html>"));
      equation 
        inlet.w + outlet.w = 0 "Mass balance";
        inlet.p = outlet.p "No pressure drop";
        
        // Boundary conditions
        inlet.hAB = outlet.hAB;
        inlet.hBA = outlet.hBA;
      end BaseReader_water;
      
      model StateReader_water 
        "State reader for the visualization of the state in the simulation (water)" 
        extends ThermoPower.PowerPlants.HRSG.Components.BaseReader_water;
        SI.Temperature T "Temperature";
        SI.Pressure p "Pressure";
        SI.SpecificEnthalpy h "Specific enthalpy";
        SI.MassFlowRate w "Mass flow rate";
        Medium.BaseProperties fluid;
      equation 
        // Set fluid properties
        fluid.p=inlet.p;
        fluid.h = if inlet.w >= 0 then inlet.hBA else inlet.hAB;
        
        T=fluid.T;
        p=fluid.p;
        h=fluid.h;
        w=inlet.w;
      end StateReader_water;
      
      model BaseReader_gas 
        "Base reader for the visualization of the state in the simulation (gas)" 
        replaceable package Medium = Modelica.Media.Interfaces.PartialMedium;
        Gas.FlangeA inlet(redeclare package Medium = Medium) 
          annotation (extent=[-80,-20; -40,20]);
        Gas.FlangeB outlet(redeclare package Medium = Medium) 
          annotation (extent=[40,-20; 80,20]);
        annotation (Diagram,
          Icon(Polygon(points=[-80,0; 0,30; 80,0; 0,-30; -80,0], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=76,
                rgbfillColor={170,170,255})),
                                 Text(
              extent=[-80,20; 80,-20],
              string="S",
              style(
                color=7,
                rgbcolor={255,255,255},
                fillColor=76,
                rgbfillColor={170,170,255}))));
      equation 
        inlet.w + outlet.w=0 "Mass balance";
        inlet.p = outlet.p "Momentum balance";
        // Energy balance
        inlet.hAB = outlet.hAB;
        inlet.hBA = outlet.hBA;
        // Independent composition mass balances
        inlet.XAB = outlet.XAB;
        inlet.XBA = outlet.XBA;
      end BaseReader_gas;
      
      model StateReader_gas 
        "State reader for the visualization of the state in the simulation (gas)" 
        extends BaseReader_gas;
        Medium.BaseProperties gas;
        SI.Temperature T "Temperature";
        SI.Pressure p "Pressure";
        SI.SpecificEnthalpy h "Specific enthalpy";
        SI.MassFlowRate w "Mass flow rate";
        
      equation 
        // Set gas properties
        inlet.p=gas.p;
        if inlet.w >= 0 then
          gas.h = inlet.hBA;
          gas.Xi = inlet.XBA;
        else
          gas.h = inlet.hAB;
          gas.Xi = inlet.XAB;
        end if;
        
        T=gas.T;
        p=gas.p;
        h=gas.h;
        w=inlet.w;
      end StateReader_gas;
      
      model PrescribedSpeedPump "Prescribed speed pump" 
        replaceable package WaterMedium = 
            Modelica.Media.Interfaces.PartialTwoPhaseMedium;
        parameter SI.VolumeFlowRate q_nom[3] "Nominal volume flow rates";
        parameter SI.Height head_nom[3] "Nominal heads";
        parameter SI.Density rho_nom "Nominal density";
        parameter SI.Conversions.NonSIunits.AngularVelocity_rpm n0 
          "Nominal rpm";
        parameter SI.Pressure nominalSteamPressure 
          "Nominal live steam pressure";
        parameter SI.Pressure nominalCondensationPressure 
          "Nominal condensation pressure";
        parameter SI.MassFlowRate nominalSteamFlow 
          "Nominal steam mass flow rate";
        parameter SI.SpecificEnthalpy hstart 
          "Fluid Specific Enthalpy Start Value";
        
        function flowCharacteristic = 
            ThermoPower.Functions.PumpCharacteristics.quadraticFlow (
              q_nom = q_nom, head_nom = head_nom);
        
        annotation (Icon(
            Rectangle(extent=[-100,100; 100,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Text(
              extent=[-100,-118; 100,-144],
              style(color=3, rgbcolor={0,0,255}),
              string="%name"),
            Ellipse(extent=[-62,62; 58,-58],   style(gradient=3)),
            Polygon(points=[-32,34; -32,-26; 46,2; -32,34],     style(
                pattern=0,
                gradient=2,
                fillColor=7))), Diagram);
        ThermoPower.Water.FlangeA inlet(
          redeclare package Medium = WaterMedium) 
          annotation (extent=[-120,-20; -80,20]);
        ThermoPower.Water.FlangeB outlet(
          redeclare package Medium = WaterMedium) 
          annotation (extent=[80,-20; 120,20]);
        Water.Pump feedWaterPump(     redeclare package Medium = 
              WaterMedium,
          redeclare function flowCharacteristic = flowCharacteristic,
          pin_start=nominalCondensationPressure,
          pout_start=nominalSteamPressure,
          wstart=nominalSteamFlow,
          n0=n0,
          hstart=hstart) 
          annotation (extent=[-40,-24; 0,16]);
        Modelica.Blocks.Interfaces.RealInput pumpSpeed_rpm 
                                         annotation (extent=[-128,40; -88,80]);
      equation 
        connect(inlet, feedWaterPump.infl) 
          annotation (points=[-100,0; -67,0; -67,0.4; -36,0.4], style(thickness=
               2));
        connect(pumpSpeed_rpm, feedWaterPump.in_n) annotation (points=[-108,60;
              -25.2,60; -25.2,12], style(color=74, rgbcolor={0,0,127}));
        connect(outlet, feedWaterPump.outfl) annotation (points=[100,0; 10,0;
              10,10.4; -8,10.4], style(thickness=2));
      end PrescribedSpeedPump;
      
      model PumpControlledFlowRate "Pump with flow rate control" 
        import ThermoPower;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialTwoPhaseMedium;
        
        parameter SI.VolumeFlowRate q_nom[3] "Nominal volume flow rates";
        parameter SI.Height head_nom[3] "Nominal heads";
        parameter SI.Density rho_nom "Nominal density";
        parameter SI.Conversions.NonSIunits.AngularVelocity_rpm n0 
          "Nominal rpm";
        parameter SI.Pressure nominalWaterPressure 
          "Nominal live steam pressure";
        parameter SI.Pressure nominalCondensationPressure 
          "Nominal condensation pressure";
        parameter SI.MassFlowRate nominalWaterFlowRate 
          "Nominal steam mass flow rate";
        parameter SI.SpecificEnthalpy hstart 
          "Fluid Specific Enthalpy Start Value";
        
        //PID for flow rate control
        parameter Real Kp=4 "Proportional gain (normalised units)" 
          annotation (Dialog(tab="PID"));
        parameter SI.Time Ti=200 "Integral time" 
          annotation (Dialog(tab="PID"));
        parameter SI.Time Td=0 "Derivative time" 
          annotation (Dialog(tab="PID"));
        parameter Real PVmin=-1 "Minimum value of process variable for scaling"
          annotation (Dialog(tab="PID"));
        parameter Real PVmax=1 "Maximum value of process variable for scaling" 
          annotation (Dialog(tab="PID"));
        parameter Real CSmin=500 "Minimum value of control signal for scaling" 
          annotation (Dialog(tab="PID"));
        parameter Real CSmax=2500 "Maximum value of control signal for scaling"
          annotation (Dialog(tab="PID"));
        parameter Real PVstart=0.5 "Start value of PV (scaled)" 
          annotation (Dialog(tab="PID"));
        parameter Real CSstart=0.5 "Start value of CS (scaled)" 
          annotation (Dialog(tab="PID"));
        parameter SI.Time T_filter=1 "Time Constant of the filter" 
          annotation (Dialog(tab="PID"));
        
        annotation (Icon(
            Rectangle(extent=[-100,100; 100,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={240,240,240})),
            Text(
              extent=[-100,-118; 100,-144],
              style(color=3, rgbcolor={0,0,255}),
              string="%name"),
            Ellipse(extent=[-62,60; 58,-60],   style(gradient=3)),
            Polygon(points=[-32,32; -32,-28; 46,0; -32,32],     style(
                pattern=0,
                gradient=2,
                fillColor=7)),
            Rectangle(extent=[-88,90; -48,50], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Text(
              extent=[-86,88; -50,52],
              style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1),
              string="C"),
            Line(points=[-48,70; -2,70; -2,60], style(
                color=74,
                rgbcolor={0,0,127},
                fillColor=30,
                rgbfillColor={240,240,240},
                fillPattern=1)),
            Line(points=[-62,0; -80,0], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=30,
                rgbfillColor={240,240,240},
                fillPattern=1)),
            Line(points=[-70,0; -70,50], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={240,240,240},
                fillPattern=1)),
            Polygon(points=[-70,46; -72,40; -68,40; -70,46], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-30,72; -30,68; -24,70; -30,72], style(
                color=74,
                rgbcolor={0,0,127},
                fillColor=74,
                rgbfillColor={0,0,127},
                fillPattern=1))),
                                Diagram);
      public 
        ThermoPower.PowerPlants.HRSG.Components.PrescribedSpeedPump pump(
          redeclare package WaterMedium = FluidMedium,
          q_nom=q_nom,
          head_nom=head_nom,
          rho_nom=rho_nom,
          n0=n0,
          nominalSteamPressure=nominalWaterPressure,
          nominalCondensationPressure=nominalCondensationPressure,
          nominalSteamFlow=nominalWaterFlowRate,
          hstart=hstart)        annotation (extent=[-8,16; -40,-16], rotation=180);
        ThermoPower.Water.SensW feed_w(redeclare package Medium = FluidMedium) 
          annotation (extent=[40,14; 20,-6],      rotation=180);
        ThermoPower.Water.FlangeA inlet(redeclare package Medium = FluidMedium)
          annotation (extent=[-120,-20; -80,20]);
        ThermoPower.Water.FlangeB outlet(redeclare package Medium = FluidMedium)
          annotation (extent=[80,-20; 120,20]);
        Modelica.Blocks.Interfaces.RealInput setpoint_FlowRate 
                                         annotation (extent=[-120,54; -88,86]);
        ThermoPower.PowerPlants.Control.PID pID(
          Kp=Kp,
          Ti=Ti,
          Td=Td,
          PVmin=PVmin,
          PVmax=PVmax,
          CSmin=CSmin,
          CSmax=CSmax,
          PVstart=PVstart,
          CSstart=CSstart) annotation (extent=[-10,60; 10,40],  rotation=180);
        Modelica.Blocks.Continuous.FirstOrder firstOrder(T=T_filter, y_start=n0) 
          annotation (extent=[-26,40; -46,60], rotation=0);
        
      equation 
        connect(pump.inlet, inlet)          annotation (points=[-40,
              1.95937e-015; -67,1.95937e-015; -67,0; -100,0], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(pump.outlet, feed_w.inlet)          annotation (points=[-8,
              -1.95937e-015; 3,-1.95937e-015; 3,8.88178e-016; 24,8.88178e-016],
                                                                style(
            thickness=2,
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(feed_w.outlet, outlet) annotation (points=[36,-8.88178e-016;
              57.8,-8.88178e-016; 57.8,0; 100,0],
                              style(
            thickness=2,
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(pID.PV, feed_w.w) annotation (points=[10,46; 60,46; 60,10; 38,
              10],        style(color=74, rgbcolor={0,0,127}));
        connect(pID.SP, setpoint_FlowRate) annotation (points=[10,54; 20,54; 20,
              70; -104,70],    style(color=74, rgbcolor={0,0,127}));
        connect(firstOrder.u, pID.CS) annotation (points=[-24,50; -10.6,50],
            style(color=74, rgbcolor={0,0,127}));
        connect(firstOrder.y, pump.pumpSpeed_rpm)          annotation (points=[-47,50;
              -60,50; -60,9.6; -41.28,9.6],         style(color=74, rgbcolor={0,
                0,127}));
      end PumpControlledFlowRate;
      
      model PumpFilter "Pump with filter" 
        import ThermoPower;
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialTwoPhaseMedium;
        
        parameter SI.VolumeFlowRate q_nom[3] "Nominal volume flow rates";
        parameter SI.Height head_nom[3] "Nominal heads";
        parameter SI.Density rho_nom "Nominal density";
        parameter SI.Conversions.NonSIunits.AngularVelocity_rpm n0 
          "Nominal rpm";
        parameter SI.Pressure nominalWaterPressure 
          "Nominal live steam pressure";
        parameter SI.Pressure nominalCondensationPressure 
          "Nominal condensation pressure";
        parameter SI.MassFlowRate nominalWaterFlowRate 
          "Nominal steam mass flow rate";
        parameter SI.SpecificEnthalpy hstart 
          "Fluid Specific Enthalpy Start Value";
        parameter SI.Time T_filter=1 "Time Constant of the filter";
        
        annotation (Icon(
            Rectangle(extent=[-100,100; 100,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={240,240,240})),
            Text(
              extent=[-100,-118; 100,-144],
              style(color=3, rgbcolor={0,0,255}),
              string="%name"),
            Ellipse(extent=[-62,60; 58,-60],   style(gradient=3)),
            Polygon(points=[-32,32; -32,-28; 46,0; -32,32],     style(
                pattern=0,
                gradient=2,
                fillColor=7)),
            Rectangle(extent=[-88,90; -48,50], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Text(
              extent=[-86,88; -50,52],
              style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=68,
                rgbfillColor={170,213,255},
                fillPattern=1),
              string="F"),
            Line(points=[-48,70; -2,70; -2,60], style(
                color=74,
                rgbcolor={0,0,127},
                fillColor=30,
                rgbfillColor={240,240,240},
                fillPattern=1)),
            Line(points=[-62,0; -80,0], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=30,
                rgbfillColor={240,240,240},
                fillPattern=1)),
            Polygon(points=[-30,72; -30,68; -24,70; -30,72], style(
                color=74,
                rgbcolor={0,0,127},
                fillColor=74,
                rgbfillColor={0,0,127},
                fillPattern=1))),
                                Diagram);
      public 
        ThermoPower.PowerPlants.HRSG.Components.PrescribedSpeedPump pump(
          redeclare package WaterMedium = FluidMedium,
          q_nom=q_nom,
          head_nom=head_nom,
          rho_nom=rho_nom,
          n0=n0,
          nominalSteamPressure=nominalWaterPressure,
          nominalCondensationPressure=nominalCondensationPressure,
          nominalSteamFlow=nominalWaterFlowRate,
          hstart=hstart)        annotation (extent=[20,20; -20,-20], rotation=180);
        ThermoPower.Water.FlangeA inlet(redeclare package Medium = FluidMedium)
          annotation (extent=[-120,-20; -80,20]);
        ThermoPower.Water.FlangeB outlet(redeclare package Medium = FluidMedium)
          annotation (extent=[80,-20; 120,20]);
        Modelica.Blocks.Interfaces.RealInput setpoint_FlowRate 
                                         annotation (extent=[-120,54; -88,86]);
        Modelica.Blocks.Continuous.FirstOrder firstOrder(T=T_filter, y_start=n0) 
          annotation (extent=[-10,40; -30,60], rotation=0);
      equation 
        connect(pump.inlet, inlet)          annotation (points=[-20,
              2.44921e-015; -67,2.44921e-015; -67,0; -100,0], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(firstOrder.y, pump.pumpSpeed_rpm)          annotation (points=[-31,50;
              -60,50; -60,12; -21.6,12],            style(color=74, rgbcolor={0,
                0,127}));
        connect(pump.outlet, outlet)          annotation (points=[20,
              -2.44921e-015; 40,-2.44921e-015; 40,0; 100,0], style(thickness=2));
        connect(firstOrder.u, setpoint_FlowRate) annotation (points=[-8,50; 20,
              50; 20,70; -104,70],style(color=74, rgbcolor={0,0,127}));
      end PumpFilter;
    end Components;
    
    package Control 
      partial model InterfacesControl "Base of class for control" 
        annotation (Icon(Rectangle(extent=[-100,100; 100,-100], style(
                color=0,
                rgbcolor={0,0,0},
                fillColor=30,
                rgbfillColor={240,240,240})), Text(
              extent=[-80,80; 80,-80],
              style(color=0, rgbcolor={0,0,0}),
              string="C")), Diagram);
        Buses.Sensors SensorsBus annotation (extent=[-120,-20; -80,20]);
        Buses.Actuators ActuatorsBus annotation (extent=[80,-20; 120,20]);
      end InterfacesControl;
      
      model levelsControl "PID controllers for levels control" 
        extends InterfacesControl;
        
        //PID for level HP control
        parameter Real Kp_levelHP=3 "Proportional gain (normalised units)" 
          annotation (Dialog(tab="PID - level HP"));
        parameter SI.Time Ti_levelHP=400 "Integral time" 
          annotation (Dialog(tab="PID - level HP"));
        parameter SI.Time Td_levelHP=0 "Derivative time" 
          annotation (Dialog(tab="PID - level HP"));
        parameter Real PVmin_levelHP=-1 
          "Minimum value of process variable for scaling" 
          annotation (Dialog(tab="PID - level HP"));
        parameter Real PVmax_levelHP=1 
          "Maximum value of process variable for scaling" 
          annotation (Dialog(tab="PID - level HP"));
        parameter Real CSmin_levelHP=30 
          "Minimum value of control signal for scaling" 
          annotation (Dialog(tab="PID - level HP"));
        parameter Real CSmax_levelHP=100 
          "Maximum value of control signal for scaling" 
          annotation (Dialog(tab="PID - level HP"));
        parameter Real PVstart_levelHP=0.5 "Start value of PV (scaled)" 
          annotation (Dialog(tab="PID - level HP"));
        parameter Real CSstart_levelHP=0.5 "Start value of CS (scaled)" 
          annotation (Dialog(tab="PID - level HP"));
        parameter SI.Length setPoint_levelHP=0 "Set point level of HP drum";
        
        //PID for level IP control
        parameter Real Kp_levelIP=3 "Proportional gain (normalised units)" 
          annotation (Dialog(tab="PID - level IP"));
        parameter SI.Time Ti_levelIP=600 "Integral time" 
          annotation (Dialog(tab="PID - level IP"));
        parameter SI.Time Td_levelIP=0 "Derivative time" 
          annotation (Dialog(tab="PID - level IP"));
        parameter Real PVmin_levelIP=-1 
          "Minimum value of process variable for scaling" 
          annotation (Dialog(tab="PID - level IP"));
        parameter Real PVmax_levelIP=1 
          "Maximum value of process variable for scaling" 
          annotation (Dialog(tab="PID - level IP"));
        parameter Real CSmin_levelIP=8 
          "Minimum value of control signal for scaling" 
          annotation (Dialog(tab="PID - level IP"));
        parameter Real CSmax_levelIP=20 
          "Maximum value of control signal for scaling" 
          annotation (Dialog(tab="PID - level IP"));
        parameter Real PVstart_levelIP=0.5 "Start value of PV (scaled)" 
          annotation (Dialog(tab="PID - level IP"));
        parameter Real CSstart_levelIP=0.5 "Start value of CS (scaled)" 
          annotation (Dialog(tab="PID - level IP"));
        parameter SI.Length setPoint_levelIP=0 "Set point level of IP drum";
        
        //PID for level LP control
        parameter Real Kp_levelLP=3 "Proportional gain (normalised units)" 
          annotation (Dialog(tab="PID - level LP"));
        parameter SI.Time Ti_levelLP=800 "Integral time" 
          annotation (Dialog(tab="PID - level LP"));
        parameter SI.Time Td_levelLP=0 "Derivative time" 
          annotation (Dialog(tab="PID - level LP"));
        parameter Real PVmin_levelLP=-1 
          "Minimum value of process variable for scaling" 
          annotation (Dialog(tab="PID - level LP"));
        parameter Real PVmax_levelLP=1 
          "Maximum value of process variable for scaling" 
          annotation (Dialog(tab="PID - level LP"));
        parameter Real CSmin_levelLP=500 
          "Minimum value of control signal for scaling" 
          annotation (Dialog(tab="PID - level LP"));
        parameter Real CSmax_levelLP=2500 
          "Maximum value of control signal for scaling" 
          annotation (Dialog(tab="PID - level LP"));
        parameter Real PVstart_levelLP=0.5 "Start value of PV (scaled)" 
          annotation (Dialog(tab="PID - level LP"));
        parameter Real CSstart_levelLP=0.5 "Start value of CS (scaled)" 
          annotation (Dialog(tab="PID - level LP"));
        parameter SI.Length setPoint_levelLP=0 "Set point level of LP drum";
        
         PowerPlants.Control.PID Level_HP(
          Kp=Kp_levelHP,
          Td=Td_levelHP,
          Ti=Ti_levelHP,
          PVmin=PVmin_levelHP,
          PVmax=PVmax_levelHP,
          CSmin=CSmin_levelHP,
          CSmax=CSmax_levelHP,
          PVstart=PVstart_levelHP,
          CSstart=CSstart_levelHP) 
                      annotation (extent=[20,40; 40,60]);
         PowerPlants.Control.PID Level_IP(
          Kp=Kp_levelIP,
          Td=Td_levelIP,
          Ti=Ti_levelIP,
          PVmin=PVmin_levelIP,
          PVmax=PVmax_levelIP,
          CSmin=CSmin_levelIP,
          CSmax=CSmax_levelIP,
          PVstart=PVstart_levelIP,
          CSstart=CSstart_levelIP) 
                     annotation (extent=[20,-10; 40,10]);
         PowerPlants.Control.PID Level_LP(
          Kp=Kp_levelLP,
          Td=Td_levelLP,
          Ti=Ti_levelLP,
          PVmin=PVmin_levelLP,
          PVmax=PVmax_levelLP,
          CSmin=CSmin_levelLP,
          CSmax=CSmax_levelLP,
          PVstart=PVstart_levelLP,
          CSstart=CSstart_levelLP) 
                     annotation (extent=[20,-54; 40,-34]);
        annotation (Diagram);
      public 
        Modelica.Blocks.Sources.Constant levelHP_SP(k=setPoint_levelHP) 
          annotation (extent=[-20,64; -40,44], rotation=180);
      public 
        Modelica.Blocks.Sources.Constant levelIP_SP(k=setPoint_levelIP) 
          annotation (extent=[-20,14; -40,-6],  rotation=180);
      public 
        Modelica.Blocks.Sources.Constant levelLP_SP(k=setPoint_levelLP) 
          annotation (extent=[-20,-30; -40,-50], rotation=180);
      equation 
        connect(SensorsBus.y_drumHP, Level_HP.PV) annotation (points=[-98,0;
              -60,0; -60,30; 0,30; 0,46; 20,46], style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(SensorsBus.y_drumIP, Level_IP.PV) annotation (points=[-100,0;
              -60,0; -60,-20; 0,-20; 0,-4; 20,-4], style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(SensorsBus.y_drumLP, Level_LP.PV) annotation (points=[-98,0;
              -60,0; -60,-66; 0,-66; 0,-48; 20,-48], style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(ActuatorsBus.flowRate_feedIP, Level_IP.CS) annotation (points=[
              100,0; 40.6,0], style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(ActuatorsBus.flowRate_feedHP, Level_HP.CS) annotation (points=[
              100,0; 60,0; 60,50; 40.6,50], style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(levelHP_SP.y, Level_HP.SP) annotation (points=[-19,54; 20,54],
            style(color=74, rgbcolor={0,0,127}));
        connect(levelIP_SP.y, Level_IP.SP) annotation (points=[-19,4; 0.5,4;
              0.5,4; 20,4],                                   style(color=74,
              rgbcolor={0,0,127}));
        connect(levelLP_SP.y, Level_LP.SP) annotation (points=[-19,-40; 20,-40],
                              style(color=74, rgbcolor={0,0,127}));
        connect(ActuatorsBus.nPump_feedLP, Level_LP.CS) annotation (points=[100,
              0; 60,0; 60,-44; 40,-44], style(color=52, rgbcolor={213,255,170}));
      end levelsControl;
      
    end Control;
    
    package Examples "Example implementations" 
      
      model HEG_2L "Heat Exchangers Group with two pressure level" 
        extends ThermoPower.PowerPlants.HRSG.Interfaces.HEG_2L;
        
        parameter Real rhomcm 
          "Metal heat capacity per unit volume (density by specific heat capacity)[J/m^3.K]";
        parameter SI.ThermalConductivity lambda 
          "Thermal conductivity of the metal";
        constant Real pi=Modelica.Constants.pi;
        
        //Valve parameter
        parameter Real Cv_attShHP 
          "Cv (US) flow coefficient, valve for Sh_HP attemperation" annotation (Dialog(group = "Attemperation"));
        parameter SI.Pressure pnom_attShHP 
          "Nominal inlet pressure, valve for Sh_HP attemperation" annotation (Dialog(group = "Attemperation"));
        parameter SI.Pressure dpnom_attShHP 
          "Nominal pressure drop, valve for Sh_HP attemperation" annotation (Dialog(group = "Attemperation"));
        parameter SI.MassFlowRate wnom_attShHP 
          "Nominal mass flowrate, valve for Sh_HP attemperation" annotation (Dialog(group = "Attemperation"));
        parameter SI.SpecificEnthalpy valShHP_hstart=1e5 
          "Start value of specific enthalpy, valve for Sh_HP attemperation" annotation (Dialog(group = "Attemperation"));
        
        replaceable Components.HE Sh2_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh2_HP_N_G,
          N_F=Sh2_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Sh,
          fluidNomPressure=fluidHPNomPressure_Sh,
          exchSurface_G=Sh2_HP_exchSurface_G,
          exchSurface_F=Sh2_HP_exchSurface_F,
          extSurfaceTub=Sh2_HP_extSurfaceTub,
          gasVol=Sh2_HP_gasVol,
          fluidVol=Sh2_HP_fluidVol,
          metalVol=Sh2_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh2_HP_Tstart_G_In,
          Tstart_G_Out=Sh2_HP_Tstart_G_Out,
          Tstart_M_In=Sh2_HP_Tstart_M_In,
          Tstart_M_Out=Sh2_HP_Tstart_M_Out,
          Tstart_F_In=Sh2_HP_Tstart_F_In,
          Tstart_F_Out=Sh2_HP_Tstart_F_Out,
          hstart_F_In=Sh2_HP_hstart_F_In,
          hstart_F_Out=Sh2_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Sh) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh2_HP_N_G,
          N_F=Sh2_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidHPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidHPNomPressure_Sh,
          exchSurface_G=Sh2_HP_exchSurface_G,
          exchSurface_F=Sh2_HP_exchSurface_F,
          extSurfaceTub=Sh2_HP_extSurfaceTub,
          gasVol=Sh2_HP_gasVol,
          fluidVol=Sh2_HP_fluidVol,
          metalVol=Sh2_HP_metalVol,
          use_T=use_T_Sh,
          SSInit=SSInit,
          Tstart_G_In=Sh2_HP_Tstart_G_In,
          Tstart_G_Out=Sh2_HP_Tstart_G_Out,
          Tstart_M_In=Sh2_HP_Tstart_M_In,
          Tstart_M_Out=Sh2_HP_Tstart_M_Out,
          Tstart_F_In=Sh2_HP_Tstart_F_In,
          Tstart_F_Out=Sh2_HP_Tstart_F_Out,
          hstart_F_In=Sh2_HP_hstart_F_In,
          hstart_F_Out=Sh2_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda) 
          annotation (extent=[-350,-10; -330,10]);
        replaceable Components.HE Sh1_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh1_HP_N_G,
          N_F=Sh1_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidHPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidHPNomPressure_Sh,
          exchSurface_G=Sh1_HP_exchSurface_G,
          exchSurface_F=Sh1_HP_exchSurface_F,
          extSurfaceTub=Sh1_HP_extSurfaceTub,
          gasVol=Sh1_HP_gasVol,
          fluidVol=Sh1_HP_fluidVol,
          metalVol=Sh1_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh1_HP_Tstart_G_In,
          Tstart_G_Out=Sh1_HP_Tstart_G_Out,
          Tstart_M_In=Sh1_HP_Tstart_M_In,
          Tstart_M_Out=Sh1_HP_Tstart_M_Out,
          Tstart_F_In=Sh1_HP_Tstart_F_In,
          Tstart_F_Out=Sh1_HP_Tstart_F_Out,
          hstart_F_In=Sh1_HP_hstart_F_In,
          hstart_F_Out=Sh1_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Sh) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh1_HP_N_G,
          N_F=Sh1_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidHPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidHPNomPressure_Sh,
          exchSurface_G=Sh1_HP_exchSurface_G,
          exchSurface_F=Sh1_HP_exchSurface_F,
          extSurfaceTub=Sh1_HP_extSurfaceTub,
          gasVol=Sh1_HP_gasVol,
          fluidVol=Sh1_HP_fluidVol,
          metalVol=Sh1_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh1_HP_Tstart_G_In,
          Tstart_G_Out=Sh1_HP_Tstart_G_Out,
          Tstart_M_In=Sh1_HP_Tstart_M_In,
          Tstart_M_Out=Sh1_HP_Tstart_M_Out,
          Tstart_F_In=Sh1_HP_Tstart_F_In,
          Tstart_F_Out=Sh1_HP_Tstart_F_Out,
          hstart_F_In=Sh1_HP_hstart_F_In,
          hstart_F_Out=Sh1_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Sh) 
          annotation (extent=[-230,-10; -210,10]);
        replaceable Components.HE2ph Ev_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_HP_N_G,
          N_F=Ev_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidHPNomFlowRate_Ev,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidHPNomPressure_Ev,
          exchSurface_G=Ev_HP_exchSurface_G,
          exchSurface_F=Ev_HP_exchSurface_F,
          extSurfaceTub=Ev_HP_extSurfaceTub,
          gasVol=Ev_HP_gasVol,
          fluidVol=Ev_HP_fluidVol,
          metalVol=Ev_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_HP_Tstart_G_In,
          Tstart_G_Out=Ev_HP_Tstart_G_Out,
          Tstart_M_In=Ev_HP_Tstart_M_In,
          Tstart_M_Out=Ev_HP_Tstart_M_Out,
          Tstart_F_In=Ev_HP_Tstart_F_In,
          Tstart_F_Out=Ev_HP_Tstart_F_Out,
          hstart_F_In=Ev_HP_hstart_F_In,
          hstart_F_Out=Ev_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_HP_N_G,
          N_F=Ev_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidHPNomFlowRate_Ev,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidHPNomPressure_Ev,
          exchSurface_G=Ev_HP_exchSurface_G,
          exchSurface_F=Ev_HP_exchSurface_F,
          extSurfaceTub=Ev_HP_extSurfaceTub,
          gasVol=Ev_HP_gasVol,
          fluidVol=Ev_HP_fluidVol,
          metalVol=Ev_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_HP_Tstart_G_In,
          Tstart_G_Out=Ev_HP_Tstart_G_Out,
          Tstart_M_In=Ev_HP_Tstart_M_In,
          Tstart_M_Out=Ev_HP_Tstart_M_Out,
          Tstart_F_In=Ev_HP_Tstart_F_In,
          Tstart_F_Out=Ev_HP_Tstart_F_Out,
          hstart_F_In=Ev_HP_hstart_F_In,
          hstart_F_Out=Ev_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev) 
          annotation (extent=[-110,-10; -90,10]);
        replaceable Components.HE Ec_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_HP_N_G,
          N_F=Ec_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidHPNomFlowRate_Ec,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidHPNomPressure_Ec,
          exchSurface_G=Ec_HP_exchSurface_G,
          exchSurface_F=Ec_HP_exchSurface_F,
          extSurfaceTub=Ec_HP_extSurfaceTub,
          gasVol=Ec_HP_gasVol,
          fluidVol=Ec_HP_fluidVol,
          metalVol=Ec_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_HP_Tstart_G_In,
          Tstart_G_Out=Ec_HP_Tstart_G_Out,
          Tstart_M_In=Ec_HP_Tstart_M_In,
          Tstart_M_Out=Ec_HP_Tstart_M_Out,
          Tstart_F_In=Ec_HP_Tstart_F_In,
          Tstart_F_Out=Ec_HP_Tstart_F_Out,
          hstart_F_In=Ec_HP_hstart_F_In,
          hstart_F_Out=Ec_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_HP_N_G,
          N_F=Ec_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidHPNomFlowRate_Ec,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidHPNomPressure_Ec,
          exchSurface_G=Ec_HP_exchSurface_G,
          exchSurface_F=Ec_HP_exchSurface_F,
          extSurfaceTub=Ec_HP_extSurfaceTub,
          gasVol=Ec_HP_gasVol,
          fluidVol=Ec_HP_fluidVol,
          metalVol=Ec_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_HP_Tstart_G_In,
          Tstart_G_Out=Ec_HP_Tstart_G_Out,
          Tstart_M_In=Ec_HP_Tstart_M_In,
          Tstart_M_Out=Ec_HP_Tstart_M_Out,
          Tstart_F_In=Ec_HP_Tstart_F_In,
          Tstart_F_Out=Ec_HP_Tstart_F_Out,
          hstart_F_In=Ec_HP_hstart_F_In,
          hstart_F_Out=Ec_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec) 
          annotation (extent=[30,-10; 50,10]);
        replaceable Components.HE Sh_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_LP_N_G,
          N_F=Sh_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidLPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Sh,
          exchSurface_G=Sh_LP_exchSurface_G,
          exchSurface_F=Sh_LP_exchSurface_F,
          extSurfaceTub=Sh_LP_extSurfaceTub,
          gasVol=Sh_LP_gasVol,
          fluidVol=Sh_LP_fluidVol,
          metalVol=Sh_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh_LP_Tstart_G_In,
          Tstart_G_Out=Sh_LP_Tstart_G_Out,
          Tstart_M_In=Sh_LP_Tstart_M_In,
          Tstart_M_Out=Sh_LP_Tstart_M_Out,
          Tstart_F_In=Sh_LP_Tstart_F_In,
          Tstart_F_Out=Sh_LP_Tstart_F_Out,
          hstart_F_In=Sh_LP_hstart_F_In,
          hstart_F_Out=Sh_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Sh) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_LP_N_G,
          N_F=Sh_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidLPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Sh,
          exchSurface_G=Sh_LP_exchSurface_G,
          exchSurface_F=Sh_LP_exchSurface_F,
          extSurfaceTub=Sh_LP_extSurfaceTub,
          gasVol=Sh_LP_gasVol,
          fluidVol=Sh_LP_fluidVol,
          metalVol=Sh_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh_LP_Tstart_G_In,
          Tstart_G_Out=Sh_LP_Tstart_G_Out,
          Tstart_M_In=Sh_LP_Tstart_M_In,
          Tstart_M_Out=Sh_LP_Tstart_M_Out,
          Tstart_F_In=Sh_LP_Tstart_F_In,
          Tstart_F_Out=Sh_LP_Tstart_F_Out,
          hstart_F_In=Sh_LP_hstart_F_In,
          hstart_F_Out=Sh_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Sh) 
          annotation (extent=[-30,-10; -10,10]);
        replaceable Components.HE2ph Ev_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_LP_N_G,
          N_F=Ev_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidLPNomFlowRate_Ev,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Ev,
          exchSurface_G=Ev_LP_exchSurface_G,
          exchSurface_F=Ev_LP_exchSurface_F,
          extSurfaceTub=Ev_LP_extSurfaceTub,
          gasVol=Ev_LP_gasVol,
          fluidVol=Ev_LP_fluidVol,
          metalVol=Ev_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_LP_Tstart_G_In,
          Tstart_G_Out=Ev_LP_Tstart_G_Out,
          Tstart_M_In=Ev_LP_Tstart_M_In,
          Tstart_M_Out=Ev_LP_Tstart_M_Out,
          Tstart_F_In=Ev_LP_Tstart_F_In,
          Tstart_F_Out=Ev_LP_Tstart_F_Out,
          hstart_F_In=Ev_LP_hstart_F_In,
          hstart_F_Out=Ev_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev,
          FFtype_F=ThermoPower.Choices.Flow1D.FFtypes.NoFriction) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_LP_N_G,
          N_F=Ev_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidLPNomFlowRate_Ev,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Ev,
          exchSurface_G=Ev_LP_exchSurface_G,
          exchSurface_F=Ev_LP_exchSurface_F,
          extSurfaceTub=Ev_LP_extSurfaceTub,
          gasVol=Ev_LP_gasVol,
          fluidVol=Ev_LP_fluidVol,
          metalVol=Ev_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_LP_Tstart_G_In,
          Tstart_G_Out=Ev_LP_Tstart_G_Out,
          Tstart_M_In=Ev_LP_Tstart_M_In,
          Tstart_M_Out=Ev_LP_Tstart_M_Out,
          Tstart_F_In=Ev_LP_Tstart_F_In,
          Tstart_F_Out=Ev_LP_Tstart_F_Out,
          hstart_F_In=Ev_LP_hstart_F_In,
          hstart_F_Out=Ev_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev) 
                         annotation (extent=[130,-10; 150,10]);
        replaceable Components.HE Ec_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_LP_N_G,
          N_F=Ec_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Ec,
          fluidNomFlowRate=fluidLPNomFlowRate_Ec,
          exchSurface_G=Ec_LP_exchSurface_G,
          exchSurface_F=Ec_LP_exchSurface_F,
          extSurfaceTub=Ec_LP_extSurfaceTub,
          gasVol=Ec_LP_gasVol,
          fluidVol=Ec_LP_fluidVol,
          metalVol=Ec_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_LP_Tstart_G_In,
          Tstart_G_Out=Ec_LP_Tstart_G_Out,
          Tstart_M_In=Ec_LP_Tstart_M_In,
          Tstart_M_Out=Ec_LP_Tstart_M_Out,
          Tstart_F_In=Ec_LP_Tstart_F_In,
          Tstart_F_Out=Ec_LP_Tstart_F_Out,
          hstart_F_In=Ec_LP_hstart_F_In,
          hstart_F_Out=Ec_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_LP_N_G,
          N_F=Ec_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Ec,
          fluidNomFlowRate=fluidLPNomFlowRate_Ec,
          exchSurface_G=Ec_LP_exchSurface_G,
          exchSurface_F=Ec_LP_exchSurface_F,
          extSurfaceTub=Ec_LP_extSurfaceTub,
          gasVol=Ec_LP_gasVol,
          fluidVol=Ec_LP_fluidVol,
          metalVol=Ec_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_LP_Tstart_G_In,
          Tstart_G_Out=Ec_LP_Tstart_G_Out,
          Tstart_M_In=Ec_LP_Tstart_M_In,
          Tstart_M_Out=Ec_LP_Tstart_M_Out,
          Tstart_F_In=Ec_LP_Tstart_F_In,
          Tstart_F_Out=Ec_LP_Tstart_F_Out,
          hstart_F_In=Ec_LP_hstart_F_In,
          hstart_F_Out=Ec_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec)                annotation (extent=[210,-10; 230,10]);
        Water.Mixer mixAtt(
          redeclare package Medium = FluidMedium,
          V=1,
          hstart=Sh1_HP_hstart_F_Out,
          pstart=fluidHPNomPressure_Sh,
          initOpt=if SSInit then Options.steadyState else Options.noInit) 
          annotation (extent=[-350,82; -330,102],rotation=270);
        Water.ValveLiq valveAttSh(
          redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          CheckValve=true,
          pnom=pnom_attShHP,
          dpnom=dpnom_attShHP,
          wnom=wnom_attShHP,
          hstart=valShHP_hstart,
          Cv=Cv_attShHP) "valve for attemperation of the Sh" 
                                annotation (extent=[-354,122; -338,138],
            rotation=270);
        Water.SensT Sh2HPIn_T(redeclare package Medium = FluidMedium) 
          annotation (extent=[-326,50; -346,30], rotation=90);
        Water.SensT Sh2HPOut_T(redeclare package Medium = FluidMedium) 
          annotation (extent=[-326,-42; -346,-62],
                                                 rotation=90);
        annotation (Diagram);
        Water.FlowSplit flowSplit2(           redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false) 
          annotation (extent=[-130,156; -110,176],
                                               rotation=270);
      equation 
        connect(SensorsBus.T_Sh2HP_In, Sh2HPIn_T.T) annotation (points=[400,140;
              -148,140; -148,-80; -292,-80; -292,16; -330,16; -330,32],
                                                   style(
            color=84,
            rgbcolor={255,170,213},
            thickness=1));
        connect(SensorsBus.T_Sh2HP_Out, Sh2HPOut_T.T) annotation (points=[400,140;
              -148,140; -148,-80; -330,-80; -330,-60],
                                                 style(color=84, rgbcolor={255,
                170,213}));
        connect(ActuatorsBus.theta_attShHP, valveAttSh.theta) annotation (
            points=[400,80; -320,80; -320,130; -339.6,130],
                                                 style(
            color=52,
            rgbcolor={213,255,170},
            thickness=1));
        connect(mixAtt.in1, Sh1_HP.waterOut) annotation (points=[-334,100; -334,
              110; -300,110; -300,-40; -220,-40; -220,-10],
                                                 style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh2HPOut_T.inlet, Sh2_HP.waterOut) annotation (points=[-340,-46;
              -340,-10], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh2_HP.waterIn, Sh2HPIn_T.outlet) annotation (points=[-340,10;
              -340,34], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh2HPIn_T.inlet, mixAtt.out) annotation (points=[-340,46; -340,
              82], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh1_HP.waterIn, Sh_HP_In) annotation (points=[-220,10; -220,40;
              -280,40; -280,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_HP.waterOut, Ev_HP_Out) annotation (points=[-100,-10; -100,
              -40; -136,-40; -136,60; -240,60; -240,200],
                                                   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_HP.waterIn, Ev_HP_In) annotation (points=[-100,10; -100,70;
              -200,70; -200,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_HP.waterOut, Ec_HP_Out) annotation (points=[40,-10; 40,-40;
              20,-40; 20,100; -160,100; -160,200],
                                               style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_LP.waterIn, Sh_LP_In) annotation (points=[-20,10; -20,52; 60,
              52; 60,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_LP.waterOut, Sh_LP_Out) annotation (points=[-20,-10; -20,-80;
              140,-80; 140,-200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_HP_Out, Sh2HPOut_T.outlet) annotation (points=[-200,-200;
              -200,-140; -340,-140; -340,-58],         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_LP.waterIn, Ev_LP_In) annotation (points=[140,10; 140,200],
            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_LP.waterOut, Ev_LP_Out) annotation (points=[140,-10; 140,-40;
              100,-40; 100,200],
                               style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_LP.waterOut, Ec_LP_Out) annotation (points=[220,-10; 220,-40;
              180,-40; 180,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_LP.waterIn, Ec_LP_In) annotation (points=[220,10; 220,200],
                        style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_LP.gasOut, GasOut) annotation (points=[230,0; 185,0; 185,
              1.77636e-015; 400,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_LP.gasIn, Ev_LP.gasOut) annotation (points=[210,0; 150,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_LP.gasIn, Ec_HP.gasOut) annotation (points=[130,0; 50,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_HP.gasIn, Sh_LP.gasOut) annotation (points=[30,0; -10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_LP.gasIn, Ev_HP.gasOut) annotation (points=[-30,0; -90,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_HP.gasIn, Sh1_HP.gasOut) annotation (points=[-110,0; -210,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh1_HP.gasIn, Sh2_HP.gasOut) annotation (points=[-230,0; -330,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh2_HP.gasIn, GasIn) annotation (points=[-350,0; -355,0; -355,
              1.77636e-015; -400,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(mixAtt.in2, valveAttSh.outlet) annotation (points=[-346,99.9;
              -346,122], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(flowSplit2.in1, Ec_HP_In) 
          annotation (points=[-120,172; -120,200], style(thickness=2));
        connect(flowSplit2.out2, valveAttSh.inlet) annotation (points=[-124,160;
              -124,150; -346,150; -346,138], style(thickness=2));
        connect(flowSplit2.out1, Ec_HP.waterIn) annotation (points=[-115.8,160;
              -115.8,150; 40,150; 40,10], style(thickness=2));
      end HEG_2L;
      
      model HEG_2LRh "Heat Exchangers Group with two pressure level and reheat" 
        extends ThermoPower.PowerPlants.HRSG.Interfaces.HEG_2LRh(
                                    Rh1_IP_Tstart_G_In = Sh1_HP_Tstart_G_In,
                                    Rh1_IP_Tstart_G_Out = Sh1_HP_Tstart_G_Out,
                                    Rh2_IP_Tstart_G_In = Sh2_HP_Tstart_G_In,
                                    Rh2_IP_Tstart_G_Out = Sh2_HP_Tstart_G_Out,
                                    Rh2_IP_N_G = Sh2_HP_N_G,
                                    Rh1_IP_N_G = Sh1_HP_N_G,
                                    Rh2_IP_gasVol = Sh2_HP_gasVol,
                                    Rh1_IP_gasVol = Sh1_HP_gasVol);
        
        parameter Real rhomcm 
          "Metal heat capacity per unit volume (density by specific heat capacity) [J/m^3.K]";
        parameter SI.ThermalConductivity lambda 
          "Thermal conductivity of the metal";
        constant Real pi=Modelica.Constants.pi;
        
        //Valves parameters
        parameter Real Cv_attShHP 
          "Cv (US) flow coefficient, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter SI.Pressure pnom_attShHP 
          "Nominal inlet pressure, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter SI.Pressure dpnom_attShHP 
          "Nominal pressure drop, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter SI.MassFlowRate wnom_attShHP 
          "Nominal mass flowrate, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter SI.SpecificEnthalpy valShHP_hstart 
          "Specific enthalpy start value, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter Real Cv_attRhIP 
          "Cv (US) flow coefficient, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        parameter SI.Pressure pnom_attRhIP 
          "Nominal inlet pressure, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        parameter SI.Pressure dpnom_attRhIP 
          "Nominal pressure drop, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        parameter SI.MassFlowRate wnom_attRhIP 
          "Nominal mass flowrate, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        parameter SI.SpecificEnthalpy valRhIP_hstart 
          "Specific enthalpy start value, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        
        replaceable Components.HE2ph Ev_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_HP_N_G,
          N_F=Ev_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ev,
          fluidNomPressure=fluidHPNomPressure_Ev,
          exchSurface_G=Ev_HP_exchSurface_G,
          exchSurface_F=Ev_HP_exchSurface_F,
          extSurfaceTub=Ev_HP_extSurfaceTub,
          gasVol=Ev_HP_gasVol,
          fluidVol=Ev_HP_fluidVol,
          metalVol=Ev_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_HP_Tstart_G_In,
          Tstart_G_Out=Ev_HP_Tstart_G_Out,
          Tstart_M_In=Ev_HP_Tstart_M_In,
          Tstart_M_Out=Ev_HP_Tstart_M_Out,
          Tstart_F_In=Ev_HP_Tstart_F_In,
          Tstart_F_Out=Ev_HP_Tstart_F_Out,
          hstart_F_In=Ev_HP_hstart_F_In,
          hstart_F_Out=Ev_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_HP_N_G,
          N_F=Ev_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ev,
          fluidNomPressure=fluidHPNomPressure_Ev,
          exchSurface_G=Ev_HP_exchSurface_G,
          exchSurface_F=Ev_HP_exchSurface_F,
          extSurfaceTub=Ev_HP_extSurfaceTub,
          gasVol=Ev_HP_gasVol,
          fluidVol=Ev_HP_fluidVol,
          metalVol=Ev_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_HP_Tstart_G_In,
          Tstart_G_Out=Ev_HP_Tstart_G_Out,
          Tstart_M_In=Ev_HP_Tstart_M_In,
          Tstart_M_Out=Ev_HP_Tstart_M_Out,
          Tstart_F_In=Ev_HP_Tstart_F_In,
          Tstart_F_Out=Ev_HP_Tstart_F_Out,
          hstart_F_In=Ev_HP_hstart_F_In,
          hstart_F_Out=Ev_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev) 
          annotation (extent=[-210,-10; -190,10]);
        replaceable Components.HE Ec_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_HP_N_G,
          N_F=Ec_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ec,
          fluidNomPressure=fluidHPNomPressure_Ec,
          exchSurface_G=Ec_HP_exchSurface_G,
          exchSurface_F=Ec_HP_exchSurface_F,
          extSurfaceTub=Ec_HP_extSurfaceTub,
          gasVol=Ec_HP_gasVol,
          fluidVol=Ec_HP_fluidVol,
          metalVol=Ec_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_HP_Tstart_G_In,
          Tstart_G_Out=Ec_HP_Tstart_G_Out,
          Tstart_M_In=Ec_HP_Tstart_M_In,
          Tstart_M_Out=Ec_HP_Tstart_M_Out,
          Tstart_F_In=Ec_HP_Tstart_F_In,
          Tstart_F_Out=Ec_HP_Tstart_F_Out,
          hstart_F_In=Ec_HP_hstart_F_In,
          hstart_F_Out=Ec_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_HP_N_G,
          N_F=Ec_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ec,
          fluidNomPressure=fluidHPNomPressure_Ec,
          exchSurface_G=Ec_HP_exchSurface_G,
          exchSurface_F=Ec_HP_exchSurface_F,
          extSurfaceTub=Ec_HP_extSurfaceTub,
          gasVol=Ec_HP_gasVol,
          fluidVol=Ec_HP_fluidVol,
          metalVol=Ec_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_HP_Tstart_G_In,
          Tstart_G_Out=Ec_HP_Tstart_G_Out,
          Tstart_M_In=Ec_HP_Tstart_M_In,
          Tstart_M_Out=Ec_HP_Tstart_M_Out,
          Tstart_F_In=Ec_HP_Tstart_F_In,
          Tstart_F_Out=Ec_HP_Tstart_F_Out,
          hstart_F_In=Ec_HP_hstart_F_In,
          hstart_F_Out=Ec_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec)          annotation (extent=[40,-10; 60,10]);
        replaceable Components.HE Sh_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_LP_N_G,
          N_F=Sh_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidLPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Sh,
          exchSurface_G=Sh_LP_exchSurface_G,
          exchSurface_F=Sh_LP_exchSurface_F,
          gasVol=Sh_LP_gasVol,
          fluidVol=Sh_LP_fluidVol,
          metalVol=Sh_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh_LP_Tstart_G_In,
          Tstart_G_Out=Sh_LP_Tstart_G_Out,
          Tstart_M_In=Sh_LP_Tstart_M_In,
          Tstart_M_Out=Sh_LP_Tstart_M_Out,
          Tstart_F_In=Sh_LP_Tstart_F_In,
          Tstart_F_Out=Sh_LP_Tstart_F_Out,
          hstart_F_In=Sh_LP_hstart_F_In,
          hstart_F_Out=Sh_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Sh_LP_extSurfaceTub,
          use_T=use_T_Sh) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_LP_N_G,
          N_F=Sh_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidLPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Sh,
          exchSurface_G=Sh_LP_exchSurface_G,
          exchSurface_F=Sh_LP_exchSurface_F,
          extSurfaceTub=Sh_LP_extSurfaceTub,
          gasVol=Sh_LP_gasVol,
          fluidVol=Sh_LP_fluidVol,
          metalVol=Sh_LP_metalVol,
          use_T=use_T,
          SSInit=SSInit,
          Tstart_G_In=Sh_LP_Tstart_G_In,
          Tstart_G_Out=Sh_LP_Tstart_G_Out,
          Tstart_M_In=Sh_LP_Tstart_M_In,
          Tstart_M_Out=Sh_LP_Tstart_M_Out,
          Tstart_F_In=Sh_LP_Tstart_F_In,
          Tstart_F_Out=Sh_LP_Tstart_F_Out,
          hstart_F_In=Sh_LP_hstart_F_In,
          hstart_F_Out=Sh_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda)           annotation (extent=[-30,-10; -10,10]);
        replaceable Components.HE2ph Ev_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_LP_N_G,
          N_F=Ev_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidLPNomFlowRate_Ev,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Ev,
          exchSurface_G=Ev_LP_exchSurface_G,
          exchSurface_F=Ev_LP_exchSurface_F,
          gasVol=Ev_LP_gasVol,
          fluidVol=Ev_LP_fluidVol,
          metalVol=Ev_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_LP_Tstart_G_In,
          Tstart_G_Out=Ev_LP_Tstart_G_Out,
          Tstart_M_In=Ev_LP_Tstart_M_In,
          Tstart_M_Out=Ev_LP_Tstart_M_Out,
          Tstart_F_In=Ev_LP_Tstart_F_In,
          Tstart_F_Out=Ev_LP_Tstart_F_Out,
          hstart_F_In=Ev_LP_hstart_F_In,
          hstart_F_Out=Ev_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ev_LP_extSurfaceTub,
          use_T=use_T_Ev) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_LP_N_G,
          N_F=Ev_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidLPNomFlowRate_Ev,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Ev,
          exchSurface_G=Ev_LP_exchSurface_G,
          exchSurface_F=Ev_LP_exchSurface_F,
          extSurfaceTub=Ev_LP_extSurfaceTub,
          gasVol=Ev_LP_gasVol,
          fluidVol=Ev_LP_fluidVol,
          metalVol=Ev_LP_metalVol,
          use_T=use_T,
          SSInit=SSInit,
          Tstart_G_In=Ev_LP_Tstart_G_In,
          Tstart_G_Out=Ev_LP_Tstart_G_Out,
          Tstart_M_In=Ev_LP_Tstart_M_In,
          Tstart_M_Out=Ev_LP_Tstart_M_Out,
          Tstart_F_In=Ev_LP_Tstart_F_In,
          Tstart_F_Out=Ev_LP_Tstart_F_Out,
          hstart_F_In=Ev_LP_hstart_F_In,
          hstart_F_Out=Ev_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda) annotation (extent=[130,-10; 150,10]);
        replaceable Components.HE Ec_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_LP_N_G,
          N_F=Ec_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Ec,
          fluidNomFlowRate=fluidLPNomFlowRate_Ec,
          exchSurface_G=Ec_LP_exchSurface_G,
          exchSurface_F=Ec_LP_exchSurface_F,
          extSurfaceTub=Ec_LP_extSurfaceTub,
          gasVol=Ec_LP_gasVol,
          fluidVol=Ec_LP_fluidVol,
          metalVol=Ec_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_LP_Tstart_G_In,
          Tstart_G_Out=Ec_LP_Tstart_G_Out,
          Tstart_M_In=Ec_LP_Tstart_M_In,
          Tstart_M_Out=Ec_LP_Tstart_M_Out,
          Tstart_F_In=Ec_LP_Tstart_F_In,
          Tstart_F_Out=Ec_LP_Tstart_F_Out,
          hstart_F_In=Ec_LP_hstart_F_In,
          hstart_F_Out=Ec_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec) 
                         extends Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_LP_N_G,
          N_F=Ec_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidLPNomPressure_Ec,
          fluidNomFlowRate=fluidLPNomFlowRate_Ec,
          exchSurface_G=Ec_LP_exchSurface_G,
          exchSurface_F=Ec_LP_exchSurface_F,
          extSurfaceTub=Ec_LP_extSurfaceTub,
          gasVol=Ec_LP_gasVol,
          fluidVol=Ec_LP_fluidVol,
          metalVol=Ec_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_LP_Tstart_G_In,
          Tstart_G_Out=Ec_LP_Tstart_G_Out,
          Tstart_M_In=Ec_LP_Tstart_M_In,
          Tstart_M_Out=Ec_LP_Tstart_M_Out,
          Tstart_F_In=Ec_LP_Tstart_F_In,
          Tstart_F_Out=Ec_LP_Tstart_F_Out,
          hstart_F_In=Ec_LP_hstart_F_In,
          hstart_F_Out=Ec_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec)                annotation (extent=[210,-10; 230,10]);
        annotation (Diagram, Icon);
        replaceable Components.ParHE Sh1HP_Rh1IP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          SSInit=SSInit,
          lambda=lambda,
          N_G=Sh1_HP_N_G,
          N_F_A=Sh1_HP_N_F,
          N_F_B=Rh1_IP_N_F,
          exchSurface_G_A=Sh1_HP_exchSurface_G,
          exchSurface_G_B=Rh1_IP_exchSurface_G,
          exchSurface_F_A=Sh1_HP_exchSurface_F,
          exchSurface_F_B=Rh1_IP_exchSurface_F,
          extSurfaceTub_A=Sh1_HP_extSurfaceTub,
          extSurfaceTub_B=Rh1_IP_extSurfaceTub,
          gasVol=Sh1_HP_gasVol,
          fluidVol_A=Sh1_HP_fluidVol,
          fluidVol_B=Rh1_IP_fluidVol,
          metalVol_A=Sh1_HP_metalVol,
          metalVol_B=Rh1_IP_metalVol,
          use_T=use_T_Sh,
          Tstart_G_In=Sh1_HP_Tstart_G_In,
          Tstart_G_Out=Sh1_HP_Tstart_G_Out,
          Tstart_M_A_In=Sh1_HP_Tstart_M_In,
          Tstart_M_A_Out=Sh1_HP_Tstart_M_Out,
          Tstart_F_A_In=Sh1_HP_Tstart_F_In,
          Tstart_F_A_Out=Sh1_HP_Tstart_F_Out,
          hstart_F_A_In=Sh1_HP_hstart_F_In,
          hstart_F_A_Out=Sh1_HP_hstart_F_Out,
          Tstart_M_B_In=Rh1_IP_Tstart_M_In,
          Tstart_M_B_Out=Rh1_IP_Tstart_M_Out,
          Tstart_F_B_In=Rh1_IP_Tstart_F_In,
          Tstart_F_B_Out=Rh1_IP_Tstart_F_Out,
          hstart_F_B_In=Rh1_IP_hstart_F_In,
          hstart_F_B_Out=Rh1_IP_hstart_F_Out,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm,
          fluidNomFlowRate_B=fluidHPNomFlowRate_Sh,
          fluidNomPressure_B=fluidHPNomPressure_Sh) 
                         extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.ParallelHE(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          SSInit=SSInit,
          lambda=lambda,
          N_G=Sh1_HP_N_G,
          N_F_A=Sh1_HP_N_F,
          N_F_B=Rh1_IP_N_F,
          exchSurface_G_A=Sh1_HP_exchSurface_G,
          exchSurface_G_B=Rh1_IP_exchSurface_G,
          exchSurface_F_A=Sh1_HP_exchSurface_F,
          exchSurface_F_B=Rh1_IP_exchSurface_F,
          extSurfaceTub_A=Sh1_HP_extSurfaceTub,
          extSurfaceTub_B=Rh1_IP_extSurfaceTub,
          gasVol=Sh1_HP_gasVol,
          fluidVol_A=Sh1_HP_fluidVol,
          fluidVol_B=Rh1_IP_fluidVol,
          metalVol_A=Sh1_HP_metalVol,
          metalVol_B=Rh1_IP_metalVol,
          use_T=use_T_Sh,
          Tstart_G_In=Sh1_HP_Tstart_G_In,
          Tstart_G_Out=Sh1_HP_Tstart_G_Out,
          Tstart_M_A_In=Sh1_HP_Tstart_M_In,
          Tstart_M_A_Out=Sh1_HP_Tstart_M_Out,
          Tstart_F_A_In=Sh1_HP_Tstart_F_In,
          Tstart_F_A_Out=Sh1_HP_Tstart_F_Out,
          hstart_F_A_In=Sh1_HP_hstart_F_In,
          hstart_F_A_Out=Sh1_HP_hstart_F_Out,
          Tstart_M_B_In=Rh1_IP_Tstart_M_In,
          Tstart_M_B_Out=Rh1_IP_Tstart_M_Out,
          Tstart_F_B_In=Rh1_IP_Tstart_F_In,
          Tstart_F_B_Out=Rh1_IP_Tstart_F_Out,
          hstart_F_B_In=Rh1_IP_hstart_F_In,
          hstart_F_B_Out=Rh1_IP_hstart_F_Out,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Rh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          fluidNomPressure_B=fluidIPNomPressure_Rh,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) 
                         annotation (extent=[-288,-12; -264,12]);
        replaceable Components.ParHE Sh2HP_Rh2IP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          SSInit=SSInit,
          lambda=lambda,
          use_T=use_T_Ec,
          N_G=Sh2_HP_N_G,
          N_F_A=Sh2_HP_N_F,
          N_F_B=Rh2_IP_N_F,
          exchSurface_G_A=Sh2_HP_exchSurface_G,
          exchSurface_G_B=Rh2_IP_exchSurface_G,
          exchSurface_F_A=Sh2_HP_exchSurface_F,
          exchSurface_F_B=Rh2_IP_exchSurface_F,
          extSurfaceTub_A=Sh2_HP_extSurfaceTub,
          extSurfaceTub_B=Rh2_IP_extSurfaceTub,
          gasVol=Sh2_HP_gasVol,
          fluidVol_A=Sh2_HP_fluidVol,
          fluidVol_B=Rh2_IP_fluidVol,
          metalVol_A=Sh2_HP_metalVol,
          Tstart_G_In=Sh2_HP_Tstart_G_In,
          Tstart_G_Out=Sh2_HP_Tstart_G_Out,
          Tstart_M_A_In=Sh2_HP_Tstart_M_In,
          Tstart_M_A_Out=Sh2_HP_Tstart_M_Out,
          Tstart_F_A_In=Sh2_HP_Tstart_F_In,
          Tstart_F_A_Out=Sh2_HP_Tstart_F_Out,
          hstart_F_A_In=Sh2_HP_hstart_F_In,
          hstart_F_A_Out=Sh2_HP_hstart_F_Out,
          Tstart_M_B_In=Rh2_IP_Tstart_M_In,
          Tstart_M_B_Out=Rh2_IP_Tstart_M_Out,
          Tstart_F_B_In=Rh2_IP_Tstart_F_In,
          Tstart_F_B_Out=Rh2_IP_Tstart_F_Out,
          hstart_F_B_In=Rh2_IP_hstart_F_In,
          hstart_F_B_Out=Rh2_IP_hstart_F_Out,
          metalVol_B=Rh2_IP_metalVol,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm,
          fluidNomFlowRate_B=fluidHPNomPressure_Sh,
          fluidNomPressure_B=fluidHPNomPressure_Sh) 
                         extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.ParallelHE(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          SSInit=SSInit,
          lambda=lambda,
          use_T=use_T_Ec,
          N_G=Sh2_HP_N_G,
          N_F_A=Sh2_HP_N_F,
          N_F_B=Rh2_IP_N_F,
          exchSurface_G_A=Sh2_HP_exchSurface_G,
          exchSurface_G_B=Rh2_IP_exchSurface_G,
          exchSurface_F_A=Sh2_HP_exchSurface_F,
          exchSurface_F_B=Rh2_IP_exchSurface_F,
          extSurfaceTub_A=Sh2_HP_extSurfaceTub,
          extSurfaceTub_B=Rh2_IP_extSurfaceTub,
          gasVol=Sh2_HP_gasVol,
          fluidVol_A=Sh2_HP_fluidVol,
          fluidVol_B=Rh2_IP_fluidVol,
          metalVol_A=Sh2_HP_metalVol,
          Tstart_G_In=Sh2_HP_Tstart_G_In,
          Tstart_G_Out=Sh2_HP_Tstart_G_Out,
          Tstart_M_A_In=Sh2_HP_Tstart_M_In,
          Tstart_M_A_Out=Sh2_HP_Tstart_M_Out,
          Tstart_F_A_In=Sh2_HP_Tstart_F_In,
          Tstart_F_A_Out=Sh2_HP_Tstart_F_Out,
          hstart_F_A_In=Sh2_HP_hstart_F_In,
          hstart_F_A_Out=Sh2_HP_hstart_F_Out,
          Tstart_M_B_In=Rh2_IP_Tstart_M_In,
          Tstart_M_B_Out=Rh2_IP_Tstart_M_Out,
          Tstart_F_B_In=Rh2_IP_Tstart_F_In,
          Tstart_F_B_Out=Rh2_IP_Tstart_F_Out,
          hstart_F_B_In=Rh2_IP_hstart_F_In,
          hstart_F_B_Out=Rh2_IP_hstart_F_Out,
          metalVol_B=Rh2_IP_metalVol,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Rh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          fluidNomPressure_B=fluidIPNomPressure_Rh,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) 
                         annotation (extent=[-360,-12; -336,12]);
      equation 
        connect(Ec_LP.waterIn, Ec_LP_In) annotation (points=[220,10; 220,200],
            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_LP.waterOut, Ec_LP_Out) annotation (points=[220,-10; 220,-40;
              180,-40; 180,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_LP.waterOut, Ev_LP_Out) annotation (points=[140,-10; 140,-40;
              100,-40; 100,200],
                               style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_LP.waterIn, Ev_LP_In) annotation (points=[140,10; 140,200],
            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_LP.gasIn, Ev_LP.gasOut) annotation (points=[210,0; 150,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(GasOut, Ec_LP.gasOut) annotation (points=[400,1.77636e-015; 262,
              1.77636e-015; 262,0; 230,0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_LP.gasIn, Ec_HP.gasOut) annotation (points=[130,0; 60,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_HP.gasIn, Sh_LP.gasOut) annotation (points=[40,0; -10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_LP.gasIn, Ev_HP.gasOut) annotation (points=[-30,0; -190,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_HP.waterOut, Ev_HP_Out) annotation (points=[-200,-10; -200,
              -60; -240,-60; -240,200],            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_HP.waterIn, Ev_HP_In) annotation (points=[-200,10; -200,200],
                                            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_LP.waterIn, Sh_LP_In) annotation (points=[-20,10; -20,80; 60,
              80; 60,200],
                       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_HP.waterIn, Ec_HP_In) annotation (points=[50,10; 50,68; -120,
              68; -120,200],style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_HP.waterOut, Ec_HP_Out) annotation (points=[50,-10; 50,-40;
              -160,-40; -160,200],             style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_LP.waterOut, Sh_LP_Out) annotation (points=[-20,-10; -20,-60;
              140,-60; 140,-200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh2HP_Rh2IP.gasIn, GasIn) annotation (points=[-360,0; -384,0;
              -384,1.77636e-015; -400,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh1HP_Rh1IP.gasIn,Sh2HP_Rh2IP. gasOut) annotation (points=[-288,0;
              -336,0],    style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ev_HP.gasIn, Sh1HP_Rh1IP.gasOut) annotation (points=[-210,0;
              -264,0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillPattern=1));
        connect(Sh1HP_Rh1IP.waterInB, Rh_IP_In) annotation (points=[-271.2,12;
              -271.2,44; -252,44; -252,-94; 1.77636e-015,-94; 1.77636e-015,-200],
                                                              style(thickness=2));
        connect(Sh1HP_Rh1IP.waterOutB, Sh2HP_Rh2IP.waterInB) annotation (points=
             [-271.2,-12; -270,-12; -270,-58; -324,-58; -324,28; -343.2,28;
              -343.2,12], style(thickness=2));
        connect(Sh1HP_Rh1IP.waterOutA, Sh2HP_Rh2IP.waterInA) annotation (points=
             [-280.8,-12; -280.8,-40; -312,-40; -312,50; -352.8,50; -352.8,12],
            style(thickness=2));
        connect(Sh2HP_Rh2IP.waterOutA, Sh_HP_Out) annotation (points=[-352.8,
              -12; -352,-12; -352,-148; -200,-148; -200,-200], style(thickness=
                2));
        connect(Sh1HP_Rh1IP.waterInA, Sh_HP_In) annotation (points=[-280.8,12;
              -280.8,103; -280,103; -280,200], style(thickness=2));
        connect(Rh_IP_Out, Sh2HP_Rh2IP.waterOutB) annotation (points=[-60,-200;
              -60,-120; -343.2,-120; -343.2,-12], style(
            color=3,
            rgbcolor={0,0,255},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end HEG_2LRh;
      
      model HEG_3LRh 
        "Heat Exchangers Group with three pressure level and reheat" 
        extends ThermoPower.PowerPlants.HRSG.Interfaces.HEG_3LRh(
                                    Ec1_HP_Tstart_G_In = Ec_IP_Tstart_G_In,
                                    Ec1_HP_Tstart_G_Out = Ec_IP_Tstart_G_Out,
                                    Rh1_IP_Tstart_G_In = Sh1_HP_Tstart_G_In,
                                    Rh1_IP_Tstart_G_Out = Sh1_HP_Tstart_G_Out,
                                    Rh2_IP_Tstart_G_In = Sh2_HP_Tstart_G_In,
                                    Rh2_IP_Tstart_G_Out = Sh2_HP_Tstart_G_Out,
                                    Rh2_IP_N_G = Sh2_HP_N_G,
                                    Rh1_IP_N_G = Sh1_HP_N_G,
                                    Rh2_IP_gasVol = Sh2_HP_gasVol,
                                    Rh1_IP_gasVol = Sh1_HP_gasVol,
                                    Ec_IP_N_G = Ec1_HP_N_G,
                                    Ec_IP_gasVol = Ec1_HP_gasVol);
        
        parameter Real rhomcm 
          "Metal heat capacity per unit volume (density by specific heat capacity) [J/m^3.K]";
        parameter SI.ThermalConductivity lambda 
          "Thermal conductivity of the metal";
        constant Real pi=Modelica.Constants.pi;
        
        parameter SI.Volume mixIP_V=3 "Internal volume of the IP mixer";
        
        annotation (Diagram, Icon,
          DymolaStoredErrors);
        replaceable Components.ParHE Sh1HP_Rh1IP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          SSInit=SSInit,
          lambda=lambda,
          N_G=Sh1_HP_N_G,
          N_F_A=Sh1_HP_N_F,
          N_F_B=Rh1_IP_N_F,
          exchSurface_G_A=Sh1_HP_exchSurface_G,
          exchSurface_G_B=Rh1_IP_exchSurface_G,
          exchSurface_F_A=Sh1_HP_exchSurface_F,
          exchSurface_F_B=Rh1_IP_exchSurface_F,
          extSurfaceTub_A=Sh1_HP_extSurfaceTub,
          extSurfaceTub_B=Rh1_IP_extSurfaceTub,
          gasVol=Sh1_HP_gasVol,
          fluidVol_A=Sh1_HP_fluidVol,
          fluidVol_B=Rh1_IP_fluidVol,
          metalVol_A=Sh1_HP_metalVol,
          metalVol_B=Rh1_IP_metalVol,
          use_T=use_T_Sh,
          Tstart_G_In=Sh1_HP_Tstart_G_In,
          Tstart_G_Out=Sh1_HP_Tstart_G_Out,
          Tstart_M_A_In=Sh1_HP_Tstart_M_In,
          Tstart_M_A_Out=Sh1_HP_Tstart_M_Out,
          Tstart_F_A_In=Sh1_HP_Tstart_F_In,
          Tstart_F_A_Out=Sh1_HP_Tstart_F_Out,
          hstart_F_A_In=Sh1_HP_hstart_F_In,
          hstart_F_A_Out=Sh1_HP_hstart_F_Out,
          Tstart_M_B_In=Rh1_IP_Tstart_M_In,
          Tstart_M_B_Out=Rh1_IP_Tstart_M_Out,
          Tstart_F_B_In=Rh1_IP_Tstart_F_In,
          Tstart_F_B_Out=Rh1_IP_Tstart_F_Out,
          hstart_F_B_In=Rh1_IP_hstart_F_In,
          hstart_F_B_Out=Rh1_IP_hstart_F_Out,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Rh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          fluidNomPressure_B=fluidIPNomPressure_Rh,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) 
                         extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.ParallelHE(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          SSInit=SSInit,
          lambda=lambda,
          N_G=Sh1_HP_N_G,
          N_F_A=Sh1_HP_N_F,
          N_F_B=Rh1_IP_N_F,
          exchSurface_G_A=Sh1_HP_exchSurface_G,
          exchSurface_G_B=Rh1_IP_exchSurface_G,
          exchSurface_F_A=Sh1_HP_exchSurface_F,
          exchSurface_F_B=Rh1_IP_exchSurface_F,
          extSurfaceTub_A=Sh1_HP_extSurfaceTub,
          extSurfaceTub_B=Rh1_IP_extSurfaceTub,
          gasVol=Sh1_HP_gasVol,
          fluidVol_A=Sh1_HP_fluidVol,
          fluidVol_B=Rh1_IP_fluidVol,
          metalVol_A=Sh1_HP_metalVol,
          metalVol_B=Rh1_IP_metalVol,
          use_T=use_T_Sh,
          Tstart_G_In=Sh1_HP_Tstart_G_In,
          Tstart_G_Out=Sh1_HP_Tstart_G_Out,
          Tstart_M_A_In=Sh1_HP_Tstart_M_In,
          Tstart_M_A_Out=Sh1_HP_Tstart_M_Out,
          Tstart_F_A_In=Sh1_HP_Tstart_F_In,
          Tstart_F_A_Out=Sh1_HP_Tstart_F_Out,
          hstart_F_A_In=Sh1_HP_hstart_F_In,
          hstart_F_A_Out=Sh1_HP_hstart_F_Out,
          Tstart_M_B_In=Rh1_IP_Tstart_M_In,
          Tstart_M_B_Out=Rh1_IP_Tstart_M_Out,
          Tstart_F_B_In=Rh1_IP_Tstart_F_In,
          Tstart_F_B_Out=Rh1_IP_Tstart_F_Out,
          hstart_F_B_In=Rh1_IP_hstart_F_In,
          hstart_F_B_Out=Rh1_IP_hstart_F_Out,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Rh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          fluidNomPressure_B=fluidIPNomPressure_Rh,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) 
                         annotation (extent=[-388,-12; -364,12]);
        replaceable Components.ParHE Sh2HP_Rh2IP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          SSInit=SSInit,
          lambda=lambda,
          use_T=use_T_Ec,
          N_G=Sh2_HP_N_G,
          N_F_A=Sh2_HP_N_F,
          N_F_B=Rh2_IP_N_F,
          exchSurface_G_A=Sh2_HP_exchSurface_G,
          exchSurface_G_B=Rh2_IP_exchSurface_G,
          exchSurface_F_A=Sh2_HP_exchSurface_F,
          exchSurface_F_B=Rh2_IP_exchSurface_F,
          extSurfaceTub_A=Sh2_HP_extSurfaceTub,
          extSurfaceTub_B=Rh2_IP_extSurfaceTub,
          gasVol=Sh2_HP_gasVol,
          fluidVol_A=Sh2_HP_fluidVol,
          fluidVol_B=Rh2_IP_fluidVol,
          metalVol_A=Sh2_HP_metalVol,
          Tstart_G_In=Sh2_HP_Tstart_G_In,
          Tstart_G_Out=Sh2_HP_Tstart_G_Out,
          Tstart_M_A_In=Sh2_HP_Tstart_M_In,
          Tstart_M_A_Out=Sh2_HP_Tstart_M_Out,
          Tstart_F_A_In=Sh2_HP_Tstart_F_In,
          Tstart_F_A_Out=Sh2_HP_Tstart_F_Out,
          hstart_F_A_In=Sh2_HP_hstart_F_In,
          hstart_F_A_Out=Sh2_HP_hstart_F_Out,
          Tstart_M_B_In=Rh2_IP_Tstart_M_In,
          Tstart_M_B_Out=Rh2_IP_Tstart_M_Out,
          Tstart_F_B_In=Rh2_IP_Tstart_F_In,
          Tstart_F_B_Out=Rh2_IP_Tstart_F_Out,
          hstart_F_B_In=Rh2_IP_hstart_F_In,
          hstart_F_B_Out=Rh2_IP_hstart_F_Out,
          metalVol_B=Rh2_IP_metalVol,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Rh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          fluidNomPressure_B=fluidIPNomPressure_Rh,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) 
                         extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.ParallelHE(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          SSInit=SSInit,
          lambda=lambda,
          use_T=use_T_Ec,
          N_G=Sh2_HP_N_G,
          N_F_A=Sh2_HP_N_F,
          N_F_B=Rh2_IP_N_F,
          exchSurface_G_A=Sh2_HP_exchSurface_G,
          exchSurface_G_B=Rh2_IP_exchSurface_G,
          exchSurface_F_A=Sh2_HP_exchSurface_F,
          exchSurface_F_B=Rh2_IP_exchSurface_F,
          extSurfaceTub_A=Sh2_HP_extSurfaceTub,
          extSurfaceTub_B=Rh2_IP_extSurfaceTub,
          gasVol=Sh2_HP_gasVol,
          fluidVol_A=Sh2_HP_fluidVol,
          fluidVol_B=Rh2_IP_fluidVol,
          metalVol_A=Sh2_HP_metalVol,
          Tstart_G_In=Sh2_HP_Tstart_G_In,
          Tstart_G_Out=Sh2_HP_Tstart_G_Out,
          Tstart_M_A_In=Sh2_HP_Tstart_M_In,
          Tstart_M_A_Out=Sh2_HP_Tstart_M_Out,
          Tstart_F_A_In=Sh2_HP_Tstart_F_In,
          Tstart_F_A_Out=Sh2_HP_Tstart_F_Out,
          hstart_F_A_In=Sh2_HP_hstart_F_In,
          hstart_F_A_Out=Sh2_HP_hstart_F_Out,
          Tstart_M_B_In=Rh2_IP_Tstart_M_In,
          Tstart_M_B_Out=Rh2_IP_Tstart_M_Out,
          Tstart_F_B_In=Rh2_IP_Tstart_F_In,
          Tstart_F_B_Out=Rh2_IP_Tstart_F_Out,
          hstart_F_B_In=Rh2_IP_hstart_F_In,
          hstart_F_B_Out=Rh2_IP_hstart_F_Out,
          metalVol_B=Rh2_IP_metalVol,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Rh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          fluidNomPressure_B=fluidIPNomPressure_Rh,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) 
                         annotation (extent=[-460,-12; -436,12]);
        replaceable Components.BaseReader_water stateRh1IP_out(redeclare 
            package Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-410,-50; -390,-30],
                                                                  rotation=180);
        replaceable Components.BaseReader_water stateSh1HP_out(redeclare 
            package Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-442,30; -422,50], rotation=180);
        replaceable Components.BaseReader_water stateSh1HP_in(redeclare package
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-390,58; -370,38], rotation=90);
        replaceable Components.BaseReader_water stateRh1IP_in(redeclare package
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-380,58; -360,38], rotation=90);
        replaceable Components.BaseReader_water stateRh2IP_out(redeclare 
            package Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-454,-42; -434,-62],
                                                                  rotation=90);
        replaceable Components.BaseReader_water stateSh2HP_out(redeclare 
            package Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-464,-42; -444,-62],
                                                                  rotation=90);
        replaceable Components.HE Ec2_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec2_HP_N_G,
          N_F=Ec2_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ec,
          fluidNomPressure=fluidHPNomPressure_Ec,
          exchSurface_G=Ec2_HP_exchSurface_G,
          exchSurface_F=Ec2_HP_exchSurface_F,
          extSurfaceTub=Ec2_HP_extSurfaceTub,
          gasVol=Ec2_HP_gasVol,
          fluidVol=Ec2_HP_fluidVol,
          metalVol=Ec2_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec2_HP_Tstart_G_In,
          Tstart_G_Out=Ec2_HP_Tstart_G_Out,
          Tstart_M_In=Ec2_HP_Tstart_M_In,
          Tstart_M_Out=Ec2_HP_Tstart_M_Out,
          Tstart_F_In=Ec2_HP_Tstart_F_In,
          Tstart_F_Out=Ec2_HP_Tstart_F_Out,
          hstart_F_In=Ec2_HP_hstart_F_In,
          hstart_F_Out=Ec2_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec2_HP_N_G,
          N_F=Ec2_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ec,
          fluidNomPressure=fluidHPNomPressure_Ec,
          exchSurface_G=Ec2_HP_exchSurface_G,
          exchSurface_F=Ec2_HP_exchSurface_F,
          extSurfaceTub=Ec2_HP_extSurfaceTub,
          gasVol=Ec2_HP_gasVol,
          fluidVol=Ec2_HP_fluidVol,
          metalVol=Ec2_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec2_HP_Tstart_G_In,
          Tstart_G_Out=Ec2_HP_Tstart_G_Out,
          Tstart_M_In=Ec2_HP_Tstart_M_In,
          Tstart_M_Out=Ec2_HP_Tstart_M_Out,
          Tstart_F_In=Ec2_HP_Tstart_F_In,
          Tstart_F_Out=Ec2_HP_Tstart_F_Out,
          hstart_F_In=Ec2_HP_hstart_F_In,
          hstart_F_Out=Ec2_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec)          annotation (extent=[-92,12; -68,-12]);
        replaceable Components.ParHE Ec1HP_EcIP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec1_HP_N_G,
          N_F_A=Ec1_HP_N_F,
          N_F_B=Ec_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G_A=Ec1_HP_exchSurface_G,
          exchSurface_G_B=Ec_IP_exchSurface_G,
          exchSurface_F_A=Ec1_HP_exchSurface_F,
          exchSurface_F_B=Ec_IP_exchSurface_F,
          gasVol=Ec1_HP_gasVol,
          fluidVol_A=Ec1_HP_fluidVol,
          fluidVol_B=Ec_IP_fluidVol,
          metalVol_A=Ec1_HP_metalVol,
          metalVol_B=Ec_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec1_HP_Tstart_G_In,
          Tstart_G_Out=Ec1_HP_Tstart_G_Out,
          Tstart_M_A_In=Ec1_HP_Tstart_M_In,
          Tstart_M_A_Out=Ec1_HP_Tstart_M_Out,
          Tstart_F_A_In=Ec1_HP_Tstart_F_In,
          Tstart_F_A_Out=Ec1_HP_Tstart_F_Out,
          hstart_F_A_In=Ec1_HP_hstart_F_In,
          hstart_F_A_Out=Ec1_HP_hstart_F_Out,
          Tstart_M_B_In=Ec_IP_Tstart_M_In,
          Tstart_M_B_Out=Ec_IP_Tstart_M_Out,
          Tstart_F_B_In=Ec_IP_Tstart_F_In,
          Tstart_F_B_Out=Ec_IP_Tstart_F_Out,
          hstart_F_B_In=Ec_IP_hstart_F_In,
          hstart_F_B_Out=Ec_IP_hstart_F_Out,
          lambda=lambda,
          extSurfaceTub_A=Ec1_HP_extSurfaceTub,
          extSurfaceTub_B=Ec_IP_extSurfaceTub,
          use_T=use_T_Ec,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Ec,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Ec,
          fluidNomPressure_A=fluidHPNomPressure_Ec,
          fluidNomPressure_B=fluidIPNomPressure_Ec,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.ParallelHE(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec1_HP_N_G,
          N_F_A=Ec1_HP_N_F,
          N_F_B=Ec_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G_A=Ec1_HP_exchSurface_G,
          exchSurface_G_B=Ec_IP_exchSurface_G,
          exchSurface_F_A=Ec1_HP_exchSurface_F,
          exchSurface_F_B=Ec_IP_exchSurface_F,
          gasVol=Ec1_HP_gasVol,
          fluidVol_A=Ec1_HP_fluidVol,
          fluidVol_B=Ec_IP_fluidVol,
          metalVol_A=Ec1_HP_metalVol,
          metalVol_B=Ec_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec1_HP_Tstart_G_In,
          Tstart_G_Out=Ec1_HP_Tstart_G_Out,
          Tstart_M_A_In=Ec1_HP_Tstart_M_In,
          Tstart_M_A_Out=Ec1_HP_Tstart_M_Out,
          Tstart_F_A_In=Ec1_HP_Tstart_F_In,
          Tstart_F_A_Out=Ec1_HP_Tstart_F_Out,
          hstart_F_A_In=Ec1_HP_hstart_F_In,
          hstart_F_A_Out=Ec1_HP_hstart_F_Out,
          Tstart_M_B_In=Ec_IP_Tstart_M_In,
          Tstart_M_B_Out=Ec_IP_Tstart_M_Out,
          Tstart_F_B_In=Ec_IP_Tstart_F_In,
          Tstart_F_B_Out=Ec_IP_Tstart_F_Out,
          hstart_F_B_In=Ec_IP_hstart_F_In,
          hstart_F_B_Out=Ec_IP_hstart_F_Out,
          lambda=lambda,
          extSurfaceTub_A=Ec1_HP_extSurfaceTub,
          extSurfaceTub_B=Ec_IP_extSurfaceTub,
          use_T=use_T_Ec,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Ec,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Ec,
          fluidNomPressure_A=fluidHPNomPressure_Ec,
          fluidNomPressure_B=fluidIPNomPressure_Ec,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) 
                         annotation (extent=[178,-12; 202,12]);
        replaceable Components.HE Ec_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_LP_N_G,
          N_F=Ec_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ec_LP_exchSurface_G,
          exchSurface_F=Ec_LP_exchSurface_F,
          gasVol=Ec_LP_gasVol,
          fluidVol=Ec_LP_fluidVol,
          metalVol=Ec_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_LP_Tstart_G_In,
          Tstart_G_Out=Ec_LP_Tstart_G_Out,
          Tstart_M_In=Ec_LP_Tstart_M_In,
          Tstart_M_Out=Ec_LP_Tstart_M_Out,
          Tstart_F_In=Ec_LP_Tstart_F_In,
          Tstart_F_Out=Ec_LP_Tstart_F_Out,
          hstart_F_In=Ec_LP_hstart_F_In,
          hstart_F_Out=Ec_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ec_LP_extSurfaceTub,
          use_T=use_T_Ec,
          fluidNomFlowRate=fluidLPNomFlowRate_Ec,
          fluidNomPressure=fluidLPNomPressure_Ec) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_LP_N_G,
          N_F=Ec_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ec_LP_exchSurface_G,
          exchSurface_F=Ec_LP_exchSurface_F,
          gasVol=Ec_LP_gasVol,
          fluidVol=Ec_LP_fluidVol,
          metalVol=Ec_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_LP_Tstart_G_In,
          Tstart_G_Out=Ec_LP_Tstart_G_Out,
          Tstart_M_In=Ec_LP_Tstart_M_In,
          Tstart_M_Out=Ec_LP_Tstart_M_Out,
          Tstart_F_In=Ec_LP_Tstart_F_In,
          Tstart_F_Out=Ec_LP_Tstart_F_Out,
          hstart_F_In=Ec_LP_hstart_F_In,
          hstart_F_Out=Ec_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ec_LP_extSurfaceTub,
          use_T=use_T_Ec,
          fluidNomFlowRate=fluidLPNomFlowRate_Ec,
          fluidNomPressure=fluidLPNomPressure_Ec) 
                                         annotation (extent=[368,-12; 392,12]);
        replaceable Components.HE2ph Ev_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_LP_N_G,
          N_F=Ev_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ev_LP_exchSurface_G,
          exchSurface_F=Ev_LP_exchSurface_F,
          gasVol=Ev_LP_gasVol,
          fluidVol=Ev_LP_fluidVol,
          metalVol=Ev_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_LP_Tstart_G_In,
          Tstart_G_Out=Ev_LP_Tstart_G_Out,
          Tstart_M_In=Ev_LP_Tstart_M_In,
          Tstart_M_Out=Ev_LP_Tstart_M_Out,
          Tstart_F_In=Ev_LP_Tstart_F_In,
          Tstart_F_Out=Ev_LP_Tstart_F_Out,
          hstart_F_In=Ev_LP_hstart_F_In,
          hstart_F_Out=Ev_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ev_LP_extSurfaceTub,
          use_T=use_T_Ev,
          fluidNomFlowRate=fluidLPNomFlowRate_Ev,
          fluidNomPressure=fluidLPNomPressure_Ev) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_LP_N_G,
          N_F=Ev_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ev_LP_exchSurface_G,
          exchSurface_F=Ev_LP_exchSurface_F,
          gasVol=Ev_LP_gasVol,
          fluidVol=Ev_LP_fluidVol,
          metalVol=Ev_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_LP_Tstart_G_In,
          Tstart_G_Out=Ev_LP_Tstart_G_Out,
          Tstart_M_In=Ev_LP_Tstart_M_In,
          Tstart_M_Out=Ev_LP_Tstart_M_Out,
          Tstart_F_In=Ev_LP_Tstart_F_In,
          Tstart_F_Out=Ev_LP_Tstart_F_Out,
          hstart_F_In=Ev_LP_hstart_F_In,
          hstart_F_Out=Ev_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ev_LP_extSurfaceTub,
          use_T=use_T_Ev,
          fluidNomFlowRate=fluidLPNomFlowRate_Ev,
          fluidNomPressure=fluidLPNomPressure_Ev) 
                         annotation (extent=[288,-12; 312,12]);
        replaceable Components.HE2ph Ev_IP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_IP_N_G,
          N_F=Ev_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ev_IP_exchSurface_G,
          exchSurface_F=Ev_IP_exchSurface_F,
          gasVol=Ev_IP_gasVol,
          fluidVol=Ev_IP_fluidVol,
          metalVol=Ev_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_IP_Tstart_G_In,
          Tstart_G_Out=Ev_IP_Tstart_G_Out,
          Tstart_M_In=Ev_IP_Tstart_M_In,
          Tstart_M_Out=Ev_IP_Tstart_M_Out,
          Tstart_F_In=Ev_IP_Tstart_F_In,
          Tstart_F_Out=Ev_IP_Tstart_F_Out,
          hstart_F_In=Ev_IP_hstart_F_In,
          hstart_F_Out=Ev_IP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ev_IP_extSurfaceTub,
          use_T=use_T_Ev,
          fluidNomFlowRate=fluidIPNomFlowRate_Ev,
          fluidNomPressure=fluidIPNomPressure_Ev) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_IP_N_G,
          N_F=Ev_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ev_IP_exchSurface_G,
          exchSurface_F=Ev_IP_exchSurface_F,
          gasVol=Ev_IP_gasVol,
          fluidVol=Ev_IP_fluidVol,
          metalVol=Ev_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_IP_Tstart_G_In,
          Tstart_G_Out=Ev_IP_Tstart_G_Out,
          Tstart_M_In=Ev_IP_Tstart_M_In,
          Tstart_M_Out=Ev_IP_Tstart_M_Out,
          Tstart_F_In=Ev_IP_Tstart_F_In,
          Tstart_F_Out=Ev_IP_Tstart_F_Out,
          hstart_F_In=Ev_IP_hstart_F_In,
          hstart_F_Out=Ev_IP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ev_IP_extSurfaceTub,
          use_T=use_T_Ev,
          fluidNomFlowRate=fluidIPNomFlowRate_Ev,
          fluidNomPressure=fluidIPNomPressure_Ev) 
          annotation (extent=[-12,-12; 12,12]);
        replaceable Components.HE Sh_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_LP_N_G,
          N_F=Sh_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Sh_LP_exchSurface_G,
          exchSurface_F=Sh_LP_exchSurface_F,
          gasVol=Sh_LP_gasVol,
          fluidVol=Sh_LP_fluidVol,
          metalVol=Sh_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh_LP_Tstart_G_In,
          Tstart_G_Out=Sh_LP_Tstart_G_Out,
          Tstart_M_In=Sh_LP_Tstart_M_In,
          Tstart_M_Out=Sh_LP_Tstart_M_Out,
          Tstart_F_In=Sh_LP_Tstart_F_In,
          Tstart_F_Out=Sh_LP_Tstart_F_Out,
          hstart_F_In=Sh_LP_hstart_F_In,
          hstart_F_Out=Sh_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Sh_LP_extSurfaceTub,
          use_T=use_T_Sh,
          fluidNomFlowRate=fluidLPNomFlowRate_Sh,
          fluidNomPressure=fluidLPNomPressure_Sh) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_LP_N_G,
          N_F=Sh_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Sh_LP_exchSurface_G,
          exchSurface_F=Sh_LP_exchSurface_F,
          gasVol=Sh_LP_gasVol,
          fluidVol=Sh_LP_fluidVol,
          metalVol=Sh_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh_LP_Tstart_G_In,
          Tstart_G_Out=Sh_LP_Tstart_G_Out,
          Tstart_M_In=Sh_LP_Tstart_M_In,
          Tstart_M_Out=Sh_LP_Tstart_M_Out,
          Tstart_F_In=Sh_LP_Tstart_F_In,
          Tstart_F_Out=Sh_LP_Tstart_F_Out,
          hstart_F_In=Sh_LP_hstart_F_In,
          hstart_F_Out=Sh_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Sh_LP_extSurfaceTub,
          use_T=use_T_Sh,
          fluidNomFlowRate=fluidLPNomFlowRate_Sh,
          fluidNomPressure=fluidLPNomPressure_Sh) 
                                   annotation (extent=[108,-12; 132,12]);
        replaceable Components.HE2ph Ev_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_HP_N_G,
          N_F=Ev_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ev,
          fluidNomPressure=fluidHPNomPressure_Ev,
          exchSurface_G=Ev_HP_exchSurface_G,
          exchSurface_F=Ev_HP_exchSurface_F,
          extSurfaceTub=Ev_HP_extSurfaceTub,
          gasVol=Ev_HP_gasVol,
          fluidVol=Ev_HP_fluidVol,
          metalVol=Ev_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_HP_Tstart_G_In,
          Tstart_G_Out=Ev_HP_Tstart_G_Out,
          Tstart_M_In=Ev_HP_Tstart_M_In,
          Tstart_M_Out=Ev_HP_Tstart_M_Out,
          Tstart_F_In=Ev_HP_Tstart_F_In,
          Tstart_F_Out=Ev_HP_Tstart_F_Out,
          hstart_F_In=Ev_HP_hstart_F_In,
          hstart_F_Out=Ev_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_HP_N_G,
          N_F=Ev_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ev,
          fluidNomPressure=fluidHPNomPressure_Ev,
          exchSurface_G=Ev_HP_exchSurface_G,
          exchSurface_F=Ev_HP_exchSurface_F,
          extSurfaceTub=Ev_HP_extSurfaceTub,
          gasVol=Ev_HP_gasVol,
          fluidVol=Ev_HP_fluidVol,
          metalVol=Ev_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_HP_Tstart_G_In,
          Tstart_G_Out=Ev_HP_Tstart_G_Out,
          Tstart_M_In=Ev_HP_Tstart_M_In,
          Tstart_M_Out=Ev_HP_Tstart_M_Out,
          Tstart_F_In=Ev_HP_Tstart_F_In,
          Tstart_F_Out=Ev_HP_Tstart_F_Out,
          hstart_F_In=Ev_HP_hstart_F_In,
          hstart_F_Out=Ev_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev) 
          annotation (extent=[-312,-12; -288,12]);
        replaceable Components.HE Sh_IP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_IP_N_G,
          N_F=Sh_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidIPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Sh_IP_exchSurface_G,
          exchSurface_F=Sh_IP_exchSurface_F,
          extSurfaceTub=Sh_IP_extSurfaceTub,
          gasVol=Sh_IP_gasVol,
          fluidVol=Sh_IP_fluidVol,
          metalVol=Sh_IP_metalVol,
          rhomcm=rhomcm,
          use_T=use_T_Sh,
          SSInit=SSInit,
          Tstart_G_In=Sh_IP_Tstart_G_In,
          Tstart_G_Out=Sh_IP_Tstart_G_Out,
          Tstart_M_In=Sh_IP_Tstart_M_In,
          Tstart_M_Out=Sh_IP_Tstart_M_Out,
          Tstart_F_In=Sh_IP_Tstart_F_In,
          Tstart_F_Out=Sh_IP_Tstart_F_Out,
          hstart_F_In=Sh_IP_hstart_F_In,
          hstart_F_Out=Sh_IP_hstart_F_Out,
          fluidNomPressure=fluidIPNomPressure_Sh,
          lambda=lambda) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_IP_N_G,
          N_F=Sh_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidIPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidNomPressure_Sh,
          exchSurface_G=Sh_IP_exchSurface_G,
          exchSurface_F=Sh_IP_exchSurface_F,
          extSurfaceTub=Sh_IP_extSurfaceTub,
          gasVol=Sh_IP_gasVol,
          fluidVol=Sh_IP_fluidVol,
          metalVol=Sh_IP_metalVol,
          rhomcm=rhomcm,
          lambda=lamba,
          use_T=use_T_Sh,
          SSInit=SSInit,
          Tstart_G_In=Sh_IP_Tstart_G_In,
          Tstart_G_Out=Sh_IP_Tstart_G_Out,
          Tstart_M_In=Sh_IP_Tstart_M_In,
          Tstart_M_Out=Sh_IP_Tstart_M_Out,
          Tstart_F_In=Sh_IP_Tstart_F_In,
          Tstart_F_Out=Sh_IP_Tstart_F_Out,
          hstart_F_In=Sh_IP_hstart_F_In,
          hstart_F_Out=Sh_IP_hstart_F_Out) annotation (extent=[-218,-12; -194,
              12]);
        replaceable Components.BaseReader_water stateEvHP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-310,60; -290,40], rotation=90);
        replaceable Components.BaseReader_water stateShIP_in(
                                                  redeclare package Medium = 
              FluidMedium) extends Components.BaseReader_water( redeclare 
            package Medium = 
              FluidMedium) annotation (extent=[-216,60; -196,40], rotation=90);
        replaceable Components.BaseReader_water stateEc2HP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-90,-60; -70,-40], rotation=90);
        replaceable Components.BaseReader_water stateEvIP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-10,60; 10,40],    rotation=90);
        replaceable Components.BaseReader_water stateShLP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[110,40; 130,60],
                                                              rotation=270);
        replaceable Components.BaseReader_water stateEcIP_in(
                                           redeclare package Medium = 
              FluidMedium) extends Components.BaseReader_water( redeclare 
            package Medium = 
              FluidMedium) annotation (extent=[186,60; 206,40],   rotation=90);
        replaceable Components.BaseReader_water stateEvLP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[290,40; 310,60],
                                                              rotation=270);
        replaceable Components.BaseReader_water stateEcLP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[370,40; 390,60],
                                                              rotation=270);
        replaceable Components.BaseReader_water stateEvHP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = 
                     FlueGasMedium) 
                           annotation (extent=[-310,-60; -290,-40],
                                                                  rotation=270);
        replaceable Components.BaseReader_water stateShIP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-216,-40; -196,-60],
                                                                  rotation=90);
        replaceable Components.BaseReader_water stateEc2HP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-90,38; -70,18],   rotation=270);
        replaceable Components.BaseReader_water stateEvIP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-10,-60; 10,-40],  rotation=270);
        replaceable Components.BaseReader_water stateShLP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[110,-40; 130,-60], rotation=90);
        replaceable Components.BaseReader_water stateEcIP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[186,-60; 206,-40], rotation=270);
        replaceable Components.BaseReader_water stateEvLP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[310,-40; 290,-60],
                                                              rotation=90);
        replaceable Components.BaseReader_water stateEcLP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water(  redeclare package Medium = FluidMedium)
                           annotation (extent=[390,-40; 370,-60],
                                                              rotation=90);
        replaceable Components.BaseReader_water stateEc1HP_in(
                                           redeclare package Medium = 
              FluidMedium) extends Components.BaseReader_water( redeclare 
            package Medium = 
              FluidMedium) annotation (extent=[174,60; 194,40],   rotation=90);
        replaceable Components.BaseReader_water stateEc1HP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[174,-60; 194,-40], rotation=270);
        Water.Mixer mixIP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          V=mixIP_V,
          pstart=fluidIPNomPressure_Rh,
          hstart=Rh1_IP_hstart_F_In) 
                          annotation (extent=[-260,-120; -240,-100],
                                                             rotation=180);
      equation 
        connect(stateRh1IP_out.inlet, Sh1HP_Rh1IP.waterOutB) annotation (points=[-394,-40;
              -371.2,-40; -371.2,-12],      style(thickness=2));
        connect(Sh2HP_Rh2IP.waterInA, stateSh1HP_out.outlet) annotation (points=[-452.8,
              12; -452.8,40; -438,40],      style(thickness=2));
        connect(Sh2HP_Rh2IP.gasIn, GasIn) annotation (points=[-460,0; -442,0;
              -442,1.77636e-015; -500,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateRh2IP_out.inlet, Sh2HP_Rh2IP.waterOutB) annotation (points=[-444,-46;
              -444,-12; -443.2,-12],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateSh2HP_out.inlet, Sh2HP_Rh2IP.waterOutA) annotation (points=[-454,-46;
              -454,-12; -452.8,-12],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateRh1IP_in.outlet, Sh1HP_Rh1IP.waterInB) annotation (points=[-370,42;
              -370,12; -371.2,12],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateSh1HP_in.outlet, Sh1HP_Rh1IP.waterInA) annotation (points=[-380,42;
              -380,12; -380.8,12],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateSh1HP_in.inlet, Sh_HP_In) annotation (points=[-380,54;
              -380,200],                   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_HP_Out, stateSh2HP_out.outlet) annotation (points=[-300,-200;
              -300,-160; -454,-160; -454,-58], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Rh_IP_Out, stateRh2IP_out.outlet) annotation (points=[-190,-200;
              -190,-140; -444,-140; -444,-58],style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh1HP_Rh1IP.waterOutA, stateSh1HP_out.inlet) annotation (points=[-380.8,
              -12; -380.8,-30; -400,-30; -400,40; -426,40],        style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateRh1IP_out.outlet, Sh2HP_Rh2IP.waterInB) annotation (points=[-406,-40;
              -420,-40; -420,22; -443.2,22; -443.2,12],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh1HP_Rh1IP.gasIn, Sh2HP_Rh2IP.gasOut) annotation (points=[-388,0;
              -436,0],    style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(stateEcLP_in.outlet,Ec_LP. waterIn) annotation (points=[380,44;
              380,12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvLP_in.outlet,Ev_LP. waterIn) annotation (points=[300,44;
              300,12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_LP.waterIn,stateShLP_in. outlet) annotation (points=[120,12;
              120,44],    style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_IP.waterIn,stateEvIP_in. outlet) annotation (points=[0,12; 0,
              44; 3.67382e-016,44],
            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_IP.waterIn,stateShIP_in. outlet) annotation (points=[-206,12;
              -206,44],                   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_HP.waterIn,stateEvHP_in. outlet) annotation (points=[-300,12;
              -300,44], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvIP_out.inlet,Ev_IP. waterOut) annotation (points=[
              1.10215e-015,-44; 1.10215e-015,-36.5; 0,-36.5; 0,-12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShLP_out.inlet,Sh_LP. waterOut) annotation (points=[120,-44;
              120,-12],style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShIP_out.inlet,Sh_IP. waterOut) annotation (points=[-206,-44;
              -206,-12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcLP_out.inlet,Ec_LP. waterOut) annotation (points=[380,-44;
              380,-12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvLP_out.inlet,Ev_LP. waterOut) annotation (points=[300,-44;
              300,-12],                   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvHP_out.inlet,Ev_HP. waterOut) annotation (points=[-300,-44;
              -300,-12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec1HP_EcIP.waterInB,stateEcIP_in. outlet) annotation (points=[194.8,12;
              196,12; 196,44],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc2HP_out.inlet,Ec2_HP. waterOut) annotation (points=[-80,22;
              -80,12],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec2_HP.waterIn,stateEc2HP_in. outlet) annotation (points=[-80,-12;
              -80,-44],    style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc2HP_in.inlet,stateEc1HP_out. outlet) annotation (points=[-80,-56;
              -80,-90; 184,-90; 184,-56],                            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcIP_out.inlet,Ec1HP_EcIP. waterOutB) annotation (points=[196,-44;
              196,-12; 194.8,-12],          style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc1HP_out.inlet,Ec1HP_EcIP. waterOutA) annotation (points=[184,-44;
              184,-12; 185.2,-12],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc1HP_in.outlet,Ec1HP_EcIP. waterInA) annotation (points=[184,44;
              185.2,44; 185.2,12],         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcLP_in.inlet, Ec_LP_In) annotation (points=[380,56; 380,
              200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvLP_in.inlet, Ev_LP_In) annotation (points=[300,56; 300,
              200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShLP_in.inlet, Sh_LP_In) annotation (points=[120,56; 120,
              130; 220,130; 220,200],
                                 style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcIP_in.inlet, Ec_IP_In) annotation (points=[196,56; 196,
              120; 80,120; 80,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcIP_out.outlet, Ec_IP_Out) annotation (points=[196,-56;
              196,-70; 40,-70; 40,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShIP_in.inlet, Sh_IP_In) annotation (points=[-206,56; -206,
              90; -80,90; -80,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvIP_in.inlet, Ev_IP_In) annotation (points=[-3.67382e-016,
              56; -3.67382e-016,200; 1.77636e-015,200],
                                style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvIP_out.outlet, Ev_IP_Out) annotation (points=[
              -1.10215e-015,-56; -1.10215e-015,-64; 0,-70; -40,-70; -40,200],
                                                                  style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShLP_out.outlet, Sh_LP_Out) annotation (points=[120,-56;
              120,-110; 300,-110; 300,-198],
                                         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvHP_out.outlet, Ev_HP_Out) annotation (points=[-300,-56;
              -300,-70; -340,-70; -340,200],                     style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvLP_out.outlet, Ev_LP_Out) annotation (points=[300,-56;
              300,-70; 260,-70; 260,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcLP_out.outlet, Ec_LP_Out) annotation (points=[380,-56;
              380,-70; 340,-70; 340,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc2HP_out.outlet, Ec_HP_Out) annotation (points=[-80,34;
              -80,60; -260,60; -260,200],                       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(GasOut,Ec_LP. gasOut) annotation (points=[500,1.77636e-015; 426,
              1.77636e-015; 392,0],        style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_IP.gasIn,Ev_HP. gasOut) annotation (points=[-218,0; -288,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ec2_HP.gasIn,Sh_IP. gasOut) annotation (points=[-92,0; -194,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ev_IP.gasIn,Ec2_HP. gasOut) annotation (points=[-12,0; -68,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Sh_LP.gasIn,Ev_IP. gasOut) annotation (points=[108,0; 12,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ec1HP_EcIP.gasIn,Sh_LP. gasOut) annotation (points=[178,0; 132,
              0],              style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ev_LP.gasIn,Ec1HP_EcIP. gasOut) annotation (points=[288,0; 202,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ec_LP.gasIn,Ev_LP. gasOut) annotation (points=[368,0; 312,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ev_HP.gasIn, Sh1HP_Rh1IP.gasOut) annotation (points=[-312,0;
              -364,0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillPattern=1));
        connect(stateEvHP_in.inlet, Ev_HP_In) annotation (points=[-300,56; -300,
              200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec_HP_In, stateEc1HP_in.inlet) annotation (points=[-220,200;
              -220,100; 184,100; 184,56],
                                        style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(mixIP.in1, Rh_IP_In) annotation (points=[-242,-116; -130,-116;
              -130,-200], style(thickness=2));
        connect(mixIP.in2, stateShIP_out.outlet) annotation (points=[-242.1,
              -104; -206,-104; -206,-56], style(thickness=2));
        connect(mixIP.out, stateRh1IP_in.inlet) annotation (points=[-260,-110;
              -350,-110; -350,86; -370,86; -370,54], style(thickness=2));
      end HEG_3LRh;
      
      model HEG_3LRh_des 
        "Heat Exchangers Group with three pressure level, reheat and desuperheater" 
        extends ThermoPower.PowerPlants.HRSG.Interfaces.HEG_3LRh(
                                    Ec1_HP_Tstart_G_In = Ec_IP_Tstart_G_In,
                                    Ec1_HP_Tstart_G_Out = Ec_IP_Tstart_G_Out,
                                    Rh1_IP_Tstart_G_In = Sh1_HP_Tstart_G_In,
                                    Rh1_IP_Tstart_G_Out = Sh1_HP_Tstart_G_Out,
                                    Rh2_IP_Tstart_G_In = Sh2_HP_Tstart_G_In,
                                    Rh2_IP_Tstart_G_Out = Sh2_HP_Tstart_G_Out,
                                    Rh2_IP_N_G = Sh2_HP_N_G,
                                    Rh1_IP_N_G = Sh1_HP_N_G,
                                    Rh2_IP_gasVol = Sh2_HP_gasVol,
                                    Rh1_IP_gasVol = Sh1_HP_gasVol,
                                    Ec_IP_N_G = Ec1_HP_N_G,
                                    Ec_IP_gasVol = Ec1_HP_gasVol);
        
        parameter Real rhomcm 
          "Metal heat capacity per unit volume (density by specific heat capacity) [J/m^3.K]";
        parameter SI.ThermalConductivity lambda 
          "Thermal conductivity of the metal";
        constant Real pi=Modelica.Constants.pi;
        
        //Valves parameters
        parameter Real Cv_attShHP 
          "Cv (US) flow coefficient, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter SI.Pressure pnom_attShHP 
          "Nominal inlet pressure, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter SI.Pressure dpnom_attShHP 
          "Nominal pressure drop, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter SI.MassFlowRate wnom_attShHP 
          "Nominal mass flowrate, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter SI.SpecificEnthalpy valShHP_hstart 
          "Specific enthalpy start value, valve for Sh_HP attemperation" annotation (dialog(group = "Attemperation Sh"));
        parameter Real Cv_attRhIP 
          "Cv (US) flow coefficient, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        parameter SI.Pressure pnom_attRhIP 
          "Nominal inlet pressure, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        parameter SI.Pressure dpnom_attRhIP 
          "Nominal pressure drop, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        parameter SI.MassFlowRate wnom_attRhIP 
          "Nominal mass flowrate, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        parameter SI.SpecificEnthalpy valRhIP_hstart 
          "Specific enthalpy start value, valve for Rh_IP attemperation" annotation (dialog(group = "Attemperation Rh"));
        
        parameter SI.Volume mixIP_V=3 "Internal volume of the IP mixer";
        
        replaceable ThermoPower.PowerPlants.HRSG.Components.ParHE_Des ShHP_RhIP(
          Cv_valA=Cv_attShHP,
          pnom_valA=pnom_attShHP,
          dpnom_valA=dpnom_attShHP,
          wnom_valA=wnom_attShHP,
          hstart_valA=valShHP_hstart,
          Cv_valB=Cv_attRhIP,
          pnom_valB=pnom_attRhIP,
          dpnom_valB=dpnom_attRhIP,
          wnom_valB=wnom_attRhIP,
          hstart_valB=valRhIP_hstart,
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          fluidNomPressure_B=fluidIPNomPressure_Rh,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Rh,
          rhomcm_A_p1=rhomcm,
          rhomcm_B_p1=rhomcm,
          rhomcm_A_p2=rhomcm,
          rhomcm_B_p2=rhomcm,
          lambda=lambda,
          N_G_p1=Sh1_HP_N_G,
          N_F_A_p1=Sh1_HP_N_F,
          N_F_B_p1=Rh1_IP_N_F,
          exchSurface_G_A_p1=Sh1_HP_exchSurface_G,
          exchSurface_G_B_p1=Rh1_IP_exchSurface_G,
          exchSurface_F_A_p1=Sh1_HP_exchSurface_F,
          exchSurface_F_B_p1=Rh1_IP_exchSurface_F,
          extSurfaceTub_A_p1=Sh1_HP_extSurfaceTub,
          extSurfaceTub_B_p1=Rh1_IP_extSurfaceTub,
          gasVol_p1=Sh1_HP_gasVol,
          fluidVol_A_p1=Sh1_HP_fluidVol,
          fluidVol_B_p1=Rh1_IP_fluidVol,
          metalVol_A_p1=Sh1_HP_metalVol,
          metalVol_B_p1=Rh1_IP_metalVol,
          N_G_p2=Sh2_HP_N_G,
          N_F_A_p2=Sh2_HP_N_F,
          N_F_B_p2=Rh2_IP_N_F,
          exchSurface_G_A_p2=Sh2_HP_exchSurface_G,
          exchSurface_G_B_p2=Rh2_IP_exchSurface_G,
          exchSurface_F_A_p2=Sh2_HP_exchSurface_F,
          exchSurface_F_B_p2=Rh2_IP_exchSurface_F,
          extSurfaceTub_A_p2=Sh2_HP_extSurfaceTub,
          extSurfaceTub_B_p2=Rh2_IP_extSurfaceTub,
          gasVol_p2=Sh2_HP_gasVol,
          fluidVol_A_p2=Sh2_HP_fluidVol,
          fluidVol_B_p2=Rh2_IP_fluidVol,
          metalVol_A_p2=Sh2_HP_metalVol,
          metalVol_B_p2=Rh2_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In_p1=Sh1_HP_Tstart_G_In,
          Tstart_G_Out_p1=Sh1_HP_Tstart_G_Out,
          Tstart_M_A_In_p1=Sh1_HP_Tstart_M_In,
          Tstart_M_A_Out_p1=Sh1_HP_Tstart_M_Out,
          Tstart_F_A_In_p1=Sh1_HP_Tstart_F_In,
          Tstart_F_A_Out_p1=Sh1_HP_Tstart_F_Out,
          hstart_F_A_In_p1=Sh1_HP_hstart_F_In,
          hstart_F_A_Out_p1=Sh1_HP_hstart_F_Out,
          Tstart_M_B_In_p1=Rh1_IP_Tstart_M_In,
          Tstart_M_B_Out_p1=Rh1_IP_Tstart_M_Out,
          Tstart_F_B_In_p1=Rh1_IP_Tstart_F_In,
          Tstart_F_B_Out_p1=Rh1_IP_Tstart_F_Out,
          hstart_F_B_In_p1=Rh1_IP_hstart_F_In,
          hstart_F_B_Out_p1=Rh1_IP_hstart_F_Out,
          Tstart_G_In_p2=Sh2_HP_Tstart_G_In,
          Tstart_G_Out_p2=Sh2_HP_Tstart_G_Out,
          Tstart_M_A_In_p2=Sh2_HP_Tstart_M_In,
          Tstart_M_A_Out_p2=Sh2_HP_Tstart_M_Out,
          Tstart_F_A_In_p2=Sh2_HP_Tstart_F_In,
          Tstart_F_A_Out_p2=Sh2_HP_Tstart_F_Out,
          hstart_F_A_In_p2=Sh2_HP_hstart_F_In,
          hstart_F_A_Out_p2=Sh2_HP_hstart_F_Out,
          Tstart_M_B_In_p2=Rh2_IP_Tstart_M_In,
          Tstart_M_B_Out_p2=Rh2_IP_Tstart_M_Out,
          Tstart_F_B_In_p2=Rh2_IP_Tstart_F_In,
          Tstart_F_B_Out_p2=Rh2_IP_Tstart_F_Out,
          hstart_F_B_In_p2=Rh2_IP_hstart_F_In,
          hstart_F_B_Out_p2=Rh2_IP_hstart_F_Out,
          use_T=use_T_Sh) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.ParallelHE_Des(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Sh,
          fluidNomPressure_A=fluidHPNomPressure_Sh,
          fluidNomPressure_B=fluidIPNomPressure_Rh,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Rh,
          rhomcm_A_p1=rhomcm,
          rhomcm_B_p1=rhomcm,
          rhomcm_A_p2=rhomcm,
          rhomcm_B_p2=rhomcm,
          lambda=lambda,
          N_G_p1=Sh1_HP_N_G,
          N_F_A_p1=Sh1_HP_N_F,
          N_F_B_p1=Rh1_IP_N_F,
          exchSurface_G_A_p1=Sh1_HP_exchSurface_G,
          exchSurface_G_B_p1=Rh1_IP_exchSurface_G,
          exchSurface_F_A_p1=Sh1_HP_exchSurface_F,
          exchSurface_F_B_p1=Rh1_IP_exchSurface_F,
          extSurfaceTub_A_p1=Sh1_HP_extSurfaceTub,
          extSurfaceTub_B_p1=Rh1_IP_extSurfaceTub,
          gasVol_p1=Sh1_HP_gasVol,
          fluidVol_A_p1=Sh1_HP_fluidVol,
          fluidVol_B_p1=Rh1_IP_fluidVol,
          metalVol_A_p1=Sh1_HP_metalVol,
          metalVol_B_p1=Rh1_IP_metalVol,
          N_G_p2=Sh2_HP_N_G,
          N_F_A_p2=Sh2_HP_N_F,
          N_F_B_p2=Rh2_IP_N_F,
          exchSurface_G_A_p2=Sh2_HP_exchSurface_G,
          exchSurface_G_B_p2=Rh2_IP_exchSurface_G,
          exchSurface_F_A_p2=Sh2_HP_exchSurface_F,
          exchSurface_F_B_p2=Rh2_IP_exchSurface_F,
          extSurfaceTub_A_p2=Sh2_HP_extSurfaceTub,
          extSurfaceTub_B_p2=Rh2_IP_extSurfaceTub,
          gasVol_p2=Sh2_HP_gasVol,
          fluidVol_A_p2=Sh2_HP_fluidVol,
          fluidVol_B_p2=Rh2_IP_fluidVol,
          metalVol_A_p2=Sh2_HP_metalVol,
          metalVol_B_p2=Rh2_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In_p1=Sh1_HP_Tstart_G_In,
          Tstart_G_Out_p1=Sh1_HP_Tstart_G_Out,
          Tstart_M_A_In_p1=Sh1_HP_Tstart_M_In,
          Tstart_M_A_Out_p1=Sh1_HP_Tstart_M_Out,
          Tstart_F_A_In_p1=Sh1_HP_Tstart_F_In,
          Tstart_F_A_Out_p1=Sh1_HP_Tstart_F_Out,
          hstart_F_A_In_p1=Sh1_HP_hstart_F_In,
          hstart_F_A_Out_p1=Sh1_HP_hstart_F_Out,
          Tstart_M_B_In_p1=Rh1_IP_Tstart_M_In,
          Tstart_M_B_Out_p1=Rh1_IP_Tstart_M_Out,
          Tstart_F_B_In_p1=Rh1_IP_Tstart_F_In,
          Tstart_F_B_Out_p1=Rh1_IP_Tstart_F_Out,
          hstart_F_B_In_p1=Rh1_IP_hstart_F_In,
          hstart_F_B_Out_p1=Rh1_IP_hstart_F_Out,
          Tstart_G_In_p2=Sh2_HP_Tstart_G_In,
          Tstart_G_Out_p2=Sh2_HP_Tstart_G_Out,
          Tstart_M_A_In_p2=Sh2_HP_Tstart_M_In,
          Tstart_M_A_Out_p2=Sh2_HP_Tstart_M_Out,
          Tstart_F_A_In_p2=Sh2_HP_Tstart_F_In,
          Tstart_F_A_Out_p2=Sh2_HP_Tstart_F_Out,
          hstart_F_A_In_p2=Sh2_HP_hstart_F_In,
          hstart_F_A_Out_p2=Sh2_HP_hstart_F_Out,
          Tstart_M_B_In_p2=Rh2_IP_Tstart_M_In,
          Tstart_M_B_Out_p2=Rh2_IP_Tstart_M_Out,
          Tstart_F_B_In_p2=Rh2_IP_Tstart_F_In,
          Tstart_F_B_Out_p2=Rh2_IP_Tstart_F_Out,
          hstart_F_B_In_p2=Rh2_IP_hstart_F_In,
          hstart_F_B_Out_p2=Rh2_IP_hstart_F_Out,
          use_T=use_T_Sh)         annotation (extent=[-454,-24; -406,24]);
        annotation (Diagram, Icon);
        Water.SensT ShHPOut_T(redeclare package Medium = FluidMedium) 
          annotation (extent=[-360,-146; -380,-166],
                                                 rotation=180);
        Water.SensT RhIPOut_T(redeclare package Medium = FluidMedium) 
          annotation (extent=[-380,-140; -360,-120]);
        
        replaceable Components.HE Ec2_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec2_HP_N_G,
          N_F=Ec2_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ec,
          fluidNomPressure=fluidHPNomPressure_Ec,
          exchSurface_G=Ec2_HP_exchSurface_G,
          exchSurface_F=Ec2_HP_exchSurface_F,
          extSurfaceTub=Ec2_HP_extSurfaceTub,
          gasVol=Ec2_HP_gasVol,
          fluidVol=Ec2_HP_fluidVol,
          metalVol=Ec2_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec2_HP_Tstart_G_In,
          Tstart_G_Out=Ec2_HP_Tstart_G_Out,
          Tstart_M_In=Ec2_HP_Tstart_M_In,
          Tstart_M_Out=Ec2_HP_Tstart_M_Out,
          Tstart_F_In=Ec2_HP_Tstart_F_In,
          Tstart_F_Out=Ec2_HP_Tstart_F_Out,
          hstart_F_In=Ec2_HP_hstart_F_In,
          hstart_F_Out=Ec2_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec2_HP_N_G,
          N_F=Ec2_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ec,
          fluidNomPressure=fluidHPNomPressure_Ec,
          exchSurface_G=Ec2_HP_exchSurface_G,
          exchSurface_F=Ec2_HP_exchSurface_F,
          extSurfaceTub=Ec2_HP_extSurfaceTub,
          gasVol=Ec2_HP_gasVol,
          fluidVol=Ec2_HP_fluidVol,
          metalVol=Ec2_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec2_HP_Tstart_G_In,
          Tstart_G_Out=Ec2_HP_Tstart_G_Out,
          Tstart_M_In=Ec2_HP_Tstart_M_In,
          Tstart_M_Out=Ec2_HP_Tstart_M_Out,
          Tstart_F_In=Ec2_HP_Tstart_F_In,
          Tstart_F_Out=Ec2_HP_Tstart_F_Out,
          hstart_F_In=Ec2_HP_hstart_F_In,
          hstart_F_Out=Ec2_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ec)          annotation (extent=[-92,12; -68,-12]);
        replaceable Components.ParHE Ec1HP_EcIP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec1_HP_N_G,
          N_F_A=Ec1_HP_N_F,
          N_F_B=Ec_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G_A=Ec1_HP_exchSurface_G,
          exchSurface_G_B=Ec_IP_exchSurface_G,
          exchSurface_F_A=Ec1_HP_exchSurface_F,
          exchSurface_F_B=Ec_IP_exchSurface_F,
          gasVol=Ec1_HP_gasVol,
          fluidVol_A=Ec1_HP_fluidVol,
          fluidVol_B=Ec_IP_fluidVol,
          metalVol_A=Ec1_HP_metalVol,
          metalVol_B=Ec_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec1_HP_Tstart_G_In,
          Tstart_G_Out=Ec1_HP_Tstart_G_Out,
          Tstart_M_A_In=Ec1_HP_Tstart_M_In,
          Tstart_M_A_Out=Ec1_HP_Tstart_M_Out,
          Tstart_F_A_In=Ec1_HP_Tstart_F_In,
          Tstart_F_A_Out=Ec1_HP_Tstart_F_Out,
          hstart_F_A_In=Ec1_HP_hstart_F_In,
          hstart_F_A_Out=Ec1_HP_hstart_F_Out,
          Tstart_M_B_In=Ec_IP_Tstart_M_In,
          Tstart_M_B_Out=Ec_IP_Tstart_M_Out,
          Tstart_F_B_In=Ec_IP_Tstart_F_In,
          Tstart_F_B_Out=Ec_IP_Tstart_F_Out,
          hstart_F_B_In=Ec_IP_hstart_F_In,
          hstart_F_B_Out=Ec_IP_hstart_F_Out,
          lambda=lambda,
          extSurfaceTub_A=Ec1_HP_extSurfaceTub,
          extSurfaceTub_B=Ec_IP_extSurfaceTub,
          use_T=use_T_Ec,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Ec,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Ec,
          fluidNomPressure_A=fluidHPNomPressure_Ec,
          fluidNomPressure_B=fluidIPNomPressure_Ec,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.ParallelHE(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec1_HP_N_G,
          N_F_A=Ec1_HP_N_F,
          N_F_B=Ec_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G_A=Ec1_HP_exchSurface_G,
          exchSurface_G_B=Ec_IP_exchSurface_G,
          exchSurface_F_A=Ec1_HP_exchSurface_F,
          exchSurface_F_B=Ec_IP_exchSurface_F,
          gasVol=Ec1_HP_gasVol,
          fluidVol_A=Ec1_HP_fluidVol,
          fluidVol_B=Ec_IP_fluidVol,
          metalVol_A=Ec1_HP_metalVol,
          metalVol_B=Ec_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec1_HP_Tstart_G_In,
          Tstart_G_Out=Ec1_HP_Tstart_G_Out,
          Tstart_M_A_In=Ec1_HP_Tstart_M_In,
          Tstart_M_A_Out=Ec1_HP_Tstart_M_Out,
          Tstart_F_A_In=Ec1_HP_Tstart_F_In,
          Tstart_F_A_Out=Ec1_HP_Tstart_F_Out,
          hstart_F_A_In=Ec1_HP_hstart_F_In,
          hstart_F_A_Out=Ec1_HP_hstart_F_Out,
          Tstart_M_B_In=Ec_IP_Tstart_M_In,
          Tstart_M_B_Out=Ec_IP_Tstart_M_Out,
          Tstart_F_B_In=Ec_IP_Tstart_F_In,
          Tstart_F_B_Out=Ec_IP_Tstart_F_Out,
          hstart_F_B_In=Ec_IP_hstart_F_In,
          hstart_F_B_Out=Ec_IP_hstart_F_Out,
          lambda=lambda,
          extSurfaceTub_A=Ec1_HP_extSurfaceTub,
          extSurfaceTub_B=Ec_IP_extSurfaceTub,
          use_T=use_T_Ec,
          fluidNomFlowRate_A=fluidHPNomFlowRate_Ec,
          fluidNomFlowRate_B=fluidIPNomFlowRate_Ec,
          fluidNomPressure_A=fluidHPNomPressure_Ec,
          fluidNomPressure_B=fluidIPNomPressure_Ec,
          rhomcm_A=rhomcm,
          rhomcm_B=rhomcm) 
                         annotation (extent=[178,-12; 202,12]);
        replaceable Components.HE Ec_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_LP_N_G,
          N_F=Ec_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ec_LP_exchSurface_G,
          exchSurface_F=Ec_LP_exchSurface_F,
          gasVol=Ec_LP_gasVol,
          fluidVol=Ec_LP_fluidVol,
          metalVol=Ec_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_LP_Tstart_G_In,
          Tstart_G_Out=Ec_LP_Tstart_G_Out,
          Tstart_M_In=Ec_LP_Tstart_M_In,
          Tstart_M_Out=Ec_LP_Tstart_M_Out,
          Tstart_F_In=Ec_LP_Tstart_F_In,
          Tstart_F_Out=Ec_LP_Tstart_F_Out,
          hstart_F_In=Ec_LP_hstart_F_In,
          hstart_F_Out=Ec_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ec_LP_extSurfaceTub,
          use_T=use_T_Ec,
          fluidNomFlowRate=fluidLPNomFlowRate_Ec,
          fluidNomPressure=fluidLPNomPressure_Ec) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ec_LP_N_G,
          N_F=Ec_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ec_LP_exchSurface_G,
          exchSurface_F=Ec_LP_exchSurface_F,
          gasVol=Ec_LP_gasVol,
          fluidVol=Ec_LP_fluidVol,
          metalVol=Ec_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ec_LP_Tstart_G_In,
          Tstart_G_Out=Ec_LP_Tstart_G_Out,
          Tstart_M_In=Ec_LP_Tstart_M_In,
          Tstart_M_Out=Ec_LP_Tstart_M_Out,
          Tstart_F_In=Ec_LP_Tstart_F_In,
          Tstart_F_Out=Ec_LP_Tstart_F_Out,
          hstart_F_In=Ec_LP_hstart_F_In,
          hstart_F_Out=Ec_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ec_LP_extSurfaceTub,
          use_T=use_T_Ec,
          fluidNomFlowRate=fluidLPNomFlowRate_Ec,
          fluidNomPressure=fluidLPNomPressure_Ec) 
                                         annotation (extent=[368,-12; 392,12]);
        replaceable Components.HE2ph Ev_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_LP_N_G,
          N_F=Ev_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ev_LP_exchSurface_G,
          exchSurface_F=Ev_LP_exchSurface_F,
          gasVol=Ev_LP_gasVol,
          fluidVol=Ev_LP_fluidVol,
          metalVol=Ev_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_LP_Tstart_G_In,
          Tstart_G_Out=Ev_LP_Tstart_G_Out,
          Tstart_M_In=Ev_LP_Tstart_M_In,
          Tstart_M_Out=Ev_LP_Tstart_M_Out,
          Tstart_F_In=Ev_LP_Tstart_F_In,
          Tstart_F_Out=Ev_LP_Tstart_F_Out,
          hstart_F_In=Ev_LP_hstart_F_In,
          hstart_F_Out=Ev_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ev_LP_extSurfaceTub,
          use_T=use_T_Ev,
          fluidNomFlowRate=fluidLPNomFlowRate_Ev,
          fluidNomPressure=fluidLPNomPressure_Ev) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_LP_N_G,
          N_F=Ev_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ev_LP_exchSurface_G,
          exchSurface_F=Ev_LP_exchSurface_F,
          gasVol=Ev_LP_gasVol,
          fluidVol=Ev_LP_fluidVol,
          metalVol=Ev_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_LP_Tstart_G_In,
          Tstart_G_Out=Ev_LP_Tstart_G_Out,
          Tstart_M_In=Ev_LP_Tstart_M_In,
          Tstart_M_Out=Ev_LP_Tstart_M_Out,
          Tstart_F_In=Ev_LP_Tstart_F_In,
          Tstart_F_Out=Ev_LP_Tstart_F_Out,
          hstart_F_In=Ev_LP_hstart_F_In,
          hstart_F_Out=Ev_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ev_LP_extSurfaceTub,
          use_T=use_T_Ev,
          fluidNomFlowRate=fluidLPNomFlowRate_Ev,
          fluidNomPressure=fluidLPNomPressure_Ev) 
                         annotation (extent=[288,-12; 312,12]);
        replaceable Components.HE2ph Ev_IP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_IP_N_G,
          N_F=Ev_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ev_IP_exchSurface_G,
          exchSurface_F=Ev_IP_exchSurface_F,
          gasVol=Ev_IP_gasVol,
          fluidVol=Ev_IP_fluidVol,
          metalVol=Ev_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_IP_Tstart_G_In,
          Tstart_G_Out=Ev_IP_Tstart_G_Out,
          Tstart_M_In=Ev_IP_Tstart_M_In,
          Tstart_M_Out=Ev_IP_Tstart_M_Out,
          Tstart_F_In=Ev_IP_Tstart_F_In,
          Tstart_F_Out=Ev_IP_Tstart_F_Out,
          hstart_F_In=Ev_IP_hstart_F_In,
          hstart_F_Out=Ev_IP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ev_IP_extSurfaceTub,
          use_T=use_T_Ev,
          fluidNomFlowRate=fluidIPNomFlowRate_Ev,
          fluidNomPressure=fluidIPNomPressure_Ev) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_IP_N_G,
          N_F=Ev_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Ev_IP_exchSurface_G,
          exchSurface_F=Ev_IP_exchSurface_F,
          gasVol=Ev_IP_gasVol,
          fluidVol=Ev_IP_fluidVol,
          metalVol=Ev_IP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_IP_Tstart_G_In,
          Tstart_G_Out=Ev_IP_Tstart_G_Out,
          Tstart_M_In=Ev_IP_Tstart_M_In,
          Tstart_M_Out=Ev_IP_Tstart_M_Out,
          Tstart_F_In=Ev_IP_Tstart_F_In,
          Tstart_F_Out=Ev_IP_Tstart_F_Out,
          hstart_F_In=Ev_IP_hstart_F_In,
          hstart_F_Out=Ev_IP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Ev_IP_extSurfaceTub,
          use_T=use_T_Ev,
          fluidNomFlowRate=fluidIPNomFlowRate_Ev,
          fluidNomPressure=fluidIPNomPressure_Ev) 
          annotation (extent=[-12,-12; 12,12]);
        replaceable Components.HE Sh_LP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_LP_N_G,
          N_F=Sh_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Sh_LP_exchSurface_G,
          exchSurface_F=Sh_LP_exchSurface_F,
          gasVol=Sh_LP_gasVol,
          fluidVol=Sh_LP_fluidVol,
          metalVol=Sh_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh_LP_Tstart_G_In,
          Tstart_G_Out=Sh_LP_Tstart_G_Out,
          Tstart_M_In=Sh_LP_Tstart_M_In,
          Tstart_M_Out=Sh_LP_Tstart_M_Out,
          Tstart_F_In=Sh_LP_Tstart_F_In,
          Tstart_F_Out=Sh_LP_Tstart_F_Out,
          hstart_F_In=Sh_LP_hstart_F_In,
          hstart_F_Out=Sh_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Sh_LP_extSurfaceTub,
          use_T=use_T_Sh,
          fluidNomFlowRate=fluidLPNomFlowRate_Sh,
          fluidNomPressure=fluidLPNomPressure_Sh) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_LP_N_G,
          N_F=Sh_LP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Sh_LP_exchSurface_G,
          exchSurface_F=Sh_LP_exchSurface_F,
          gasVol=Sh_LP_gasVol,
          fluidVol=Sh_LP_fluidVol,
          metalVol=Sh_LP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Sh_LP_Tstart_G_In,
          Tstart_G_Out=Sh_LP_Tstart_G_Out,
          Tstart_M_In=Sh_LP_Tstart_M_In,
          Tstart_M_Out=Sh_LP_Tstart_M_Out,
          Tstart_F_In=Sh_LP_Tstart_F_In,
          Tstart_F_Out=Sh_LP_Tstart_F_Out,
          hstart_F_In=Sh_LP_hstart_F_In,
          hstart_F_Out=Sh_LP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          extSurfaceTub=Sh_LP_extSurfaceTub,
          use_T=use_T_Sh,
          fluidNomFlowRate=fluidLPNomFlowRate_Sh,
          fluidNomPressure=fluidLPNomPressure_Sh) 
                                   annotation (extent=[108,-12; 132,12]);
        replaceable Components.HE2ph Ev_HP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_HP_N_G,
          N_F=Ev_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ev,
          fluidNomPressure=fluidHPNomPressure_Ev,
          exchSurface_G=Ev_HP_exchSurface_G,
          exchSurface_F=Ev_HP_exchSurface_F,
          extSurfaceTub=Ev_HP_extSurfaceTub,
          gasVol=Ev_HP_gasVol,
          fluidVol=Ev_HP_fluidVol,
          metalVol=Ev_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_HP_Tstart_G_In,
          Tstart_G_Out=Ev_HP_Tstart_G_Out,
          Tstart_M_In=Ev_HP_Tstart_M_In,
          Tstart_M_Out=Ev_HP_Tstart_M_Out,
          Tstart_F_In=Ev_HP_Tstart_F_In,
          Tstart_F_Out=Ev_HP_Tstart_F_Out,
          hstart_F_In=Ev_HP_hstart_F_In,
          hstart_F_Out=Ev_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Ev_HP_N_G,
          N_F=Ev_HP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          gasNomPressure=gasNomPressure,
          fluidNomFlowRate=fluidHPNomFlowRate_Ev,
          fluidNomPressure=fluidHPNomPressure_Ev,
          exchSurface_G=Ev_HP_exchSurface_G,
          exchSurface_F=Ev_HP_exchSurface_F,
          extSurfaceTub=Ev_HP_extSurfaceTub,
          gasVol=Ev_HP_gasVol,
          fluidVol=Ev_HP_fluidVol,
          metalVol=Ev_HP_metalVol,
          SSInit=SSInit,
          Tstart_G_In=Ev_HP_Tstart_G_In,
          Tstart_G_Out=Ev_HP_Tstart_G_Out,
          Tstart_M_In=Ev_HP_Tstart_M_In,
          Tstart_M_Out=Ev_HP_Tstart_M_Out,
          Tstart_F_In=Ev_HP_Tstart_F_In,
          Tstart_F_Out=Ev_HP_Tstart_F_Out,
          hstart_F_In=Ev_HP_hstart_F_In,
          hstart_F_Out=Ev_HP_hstart_F_Out,
          rhomcm=rhomcm,
          lambda=lambda,
          use_T=use_T_Ev) 
          annotation (extent=[-312,-12; -288,12]);
        replaceable Components.HE Sh_IP(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_IP_N_G,
          N_F=Sh_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidIPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          exchSurface_G=Sh_IP_exchSurface_G,
          exchSurface_F=Sh_IP_exchSurface_F,
          extSurfaceTub=Sh_IP_extSurfaceTub,
          gasVol=Sh_IP_gasVol,
          fluidVol=Sh_IP_fluidVol,
          metalVol=Sh_IP_metalVol,
          rhomcm=rhomcm,
          use_T=use_T_Sh,
          SSInit=SSInit,
          Tstart_G_In=Sh_IP_Tstart_G_In,
          Tstart_G_Out=Sh_IP_Tstart_G_Out,
          Tstart_M_In=Sh_IP_Tstart_M_In,
          Tstart_M_Out=Sh_IP_Tstart_M_Out,
          Tstart_F_In=Sh_IP_Tstart_F_In,
          Tstart_F_Out=Sh_IP_Tstart_F_Out,
          hstart_F_In=Sh_IP_hstart_F_In,
          hstart_F_Out=Sh_IP_hstart_F_Out,
          fluidNomPressure=fluidIPNomPressure_Sh,
          lambda=lambda) extends 
          ThermoPower.PowerPlants.HRSG.Interfaces.HeatExchanger(
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          N_G=Sh_IP_N_G,
          N_F=Sh_IP_N_F,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate=fluidIPNomFlowRate_Sh,
          gasNomPressure=gasNomPressure,
          fluidNomPressure=fluidNomPressure_Sh,
          exchSurface_G=Sh_IP_exchSurface_G,
          exchSurface_F=Sh_IP_exchSurface_F,
          extSurfaceTub=Sh_IP_extSurfaceTub,
          gasVol=Sh_IP_gasVol,
          fluidVol=Sh_IP_fluidVol,
          metalVol=Sh_IP_metalVol,
          rhomcm=rhomcm,
          lambda=lamba,
          use_T=use_T_Sh,
          SSInit=SSInit,
          Tstart_G_In=Sh_IP_Tstart_G_In,
          Tstart_G_Out=Sh_IP_Tstart_G_Out,
          Tstart_M_In=Sh_IP_Tstart_M_In,
          Tstart_M_Out=Sh_IP_Tstart_M_Out,
          Tstart_F_In=Sh_IP_Tstart_F_In,
          Tstart_F_Out=Sh_IP_Tstart_F_Out,
          hstart_F_In=Sh_IP_hstart_F_In,
          hstart_F_Out=Sh_IP_hstart_F_Out) annotation (extent=[-218,-12; -194,
              12]);
        replaceable Components.BaseReader_water stateEvHP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-310,60; -290,40], rotation=90);
        replaceable Components.BaseReader_water stateShIP_in(
                                                  redeclare package Medium = 
              FluidMedium) extends Components.BaseReader_water( redeclare 
            package Medium = 
              FluidMedium) annotation (extent=[-216,60; -196,40], rotation=90);
        replaceable Components.BaseReader_water stateEc2HP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-90,-60; -70,-40], rotation=90);
        replaceable Components.BaseReader_water stateEvIP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-10,60; 10,40],    rotation=90);
        replaceable Components.BaseReader_water stateShLP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[110,40; 130,60],
                                                              rotation=270);
        replaceable Components.BaseReader_water stateEcIP_in(
                                           redeclare package Medium = 
              FluidMedium) extends Components.BaseReader_water( redeclare 
            package Medium = 
              FluidMedium) annotation (extent=[186,60; 206,40],   rotation=90);
        replaceable Components.BaseReader_water stateEvLP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[290,40; 310,60],
                                                              rotation=270);
        replaceable Components.BaseReader_water stateEcLP_in(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[370,40; 390,60],
                                                              rotation=270);
        replaceable Components.BaseReader_water stateEvHP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = 
                     FlueGasMedium) 
                           annotation (extent=[-310,-60; -290,-40],
                                                                  rotation=270);
        replaceable Components.BaseReader_water stateShIP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-216,-40; -196,-60],
                                                                  rotation=90);
        replaceable Components.BaseReader_water stateEc2HP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-90,38; -70,18],   rotation=270);
        replaceable Components.BaseReader_water stateEvIP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[-10,-60; 10,-40],  rotation=270);
        replaceable Components.BaseReader_water stateShLP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[110,-40; 130,-60], rotation=90);
        replaceable Components.BaseReader_water stateEcIP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[186,-60; 206,-40], rotation=270);
        replaceable Components.BaseReader_water stateEvLP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[310,-40; 290,-60],
                                                              rotation=90);
        replaceable Components.BaseReader_water stateEcLP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water(  redeclare package Medium = FluidMedium)
                           annotation (extent=[390,-40; 370,-60],
                                                              rotation=90);
        replaceable Components.BaseReader_water stateEc1HP_in(
                                           redeclare package Medium = 
              FluidMedium) extends Components.BaseReader_water( redeclare 
            package Medium = 
              FluidMedium) annotation (extent=[174,60; 194,40],   rotation=90);
        replaceable Components.BaseReader_water stateEc1HP_out(
            redeclare package Medium = FluidMedium) extends 
          Components.BaseReader_water( redeclare package Medium = FluidMedium) 
                           annotation (extent=[174,-60; 194,-40], rotation=270);
        replaceable Components.BaseReader_water stateSh1HP_in(redeclare package
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-450,80; -430,60], rotation=90);
        replaceable Components.BaseReader_water stateRh1IP_in(redeclare package
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-430,80; -410,60], rotation=90);
        replaceable Components.BaseReader_water stateRh2IP_out(redeclare 
            package Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-430,-60; -410,-80],
                                                                  rotation=90);
        replaceable Components.BaseReader_water stateSh2HP_out(redeclare 
            package Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-450,-60; -430,-80],
                                                                  rotation=90);
        Water.Mixer mixIP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          V=mixIP_V,
          hstart=Rh1_IP_hstart_F_In,
          pstart=fluidIPNomPressure_Rh) 
                          annotation (extent=[-280,-120; -260,-100],
                                                             rotation=180);
        Water.FlowSplit flowSplit2(           redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false) 
          annotation (extent=[-230,140; -210,160],
                                               rotation=270);
        Water.FlowSplit flowSplit1(           redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false) 
          annotation (extent=[70,150; 90,170], rotation=270);
      equation 
        connect(SensorsBus.T_Sh2HP_In, ShHP_RhIP.T_intermedA) annotation (
            points=[500,140; -360,140; -360,14; -382,14; -405.76,14.16],
                                                                style(color=84,
              rgbcolor={255,170,213}));
        connect(SensorsBus.T_Rh2IP_In, ShHP_RhIP.T_intermedB) annotation (
            points=[500,140; -360,140; -360,10; -382,10; -405.76,9.36],
                                                                style(
            color=84,
            rgbcolor={255,170,213},
            thickness=1));
        connect(SensorsBus.T_Sh2HP_Out, ShHPOut_T.T) annotation (points=[500,140;
              -250,140; -250,-150; -362,-150],         style(color=84, rgbcolor=
               {255,170,213}));
        connect(SensorsBus.T_Rh2IP_Out, RhIPOut_T.T) annotation (points=[500,140;
              -250,140; -250,-124; -362,-124],style(color=84, rgbcolor={255,170,
                213}));
        connect(ActuatorsBus.theta_attShHP, ShHP_RhIP.theta_valveA) annotation (
           points=[500,80; 440,80; 440,-92; -360,-92; -360,-8; -362,-10;
              -405.76,-10.08],                                 style(color=52,
              rgbcolor={213,255,170}));
        connect(ActuatorsBus.theta_attRhIP, ShHP_RhIP.theta_valveB) annotation (
           points=[500,80; 440,80; 440,-92; -360,-92; -360,-16; -382,-16;
              -405.76,-15.12],                                 style(color=52,
              rgbcolor={213,255,170}));
        connect(stateEcLP_in.outlet,Ec_LP. waterIn) annotation (points=[380,44;
              380,12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvLP_in.outlet,Ev_LP. waterIn) annotation (points=[300,44;
              300,12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_LP.waterIn,stateShLP_in. outlet) annotation (points=[120,12;
              120,44],    style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_IP.waterIn,stateEvIP_in. outlet) annotation (points=[0,12; 0,
              44; 3.67382e-016,44],
            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_IP.waterIn,stateShIP_in. outlet) annotation (points=[-206,12;
              -206,44],                   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_HP.waterIn,stateEvHP_in. outlet) annotation (points=[-300,12;
              -300,44], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvIP_out.inlet,Ev_IP. waterOut) annotation (points=[
              1.10215e-015,-44; 1.10215e-015,-36.5; 0,-36.5; 0,-12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShLP_out.inlet,Sh_LP. waterOut) annotation (points=[120,-44;
              120,-12],style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShIP_out.inlet,Sh_IP. waterOut) annotation (points=[-206,-44;
              -206,-12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcLP_out.inlet,Ec_LP. waterOut) annotation (points=[380,-44;
              380,-12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvLP_out.inlet,Ev_LP. waterOut) annotation (points=[300,-44;
              300,-12],                   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvHP_out.inlet,Ev_HP. waterOut) annotation (points=[-300,-44;
              -300,-12], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec1HP_EcIP.waterInB,stateEcIP_in. outlet) annotation (points=[194.8,12;
              196,12; 196,44],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc2HP_out.inlet,Ec2_HP. waterOut) annotation (points=[-80,22;
              -80,12],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ec2_HP.waterIn,stateEc2HP_in. outlet) annotation (points=[-80,-12;
              -80,-44],    style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc2HP_in.inlet,stateEc1HP_out. outlet) annotation (points=[-80,-56;
              -80,-90; 184,-90; 184,-56],                            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcIP_out.inlet,Ec1HP_EcIP. waterOutB) annotation (points=[196,-44;
              196,-12; 194.8,-12],          style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc1HP_out.inlet,Ec1HP_EcIP. waterOutA) annotation (points=[184,-44;
              184,-12; 185.2,-12],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc1HP_in.outlet,Ec1HP_EcIP. waterInA) annotation (points=[184,44;
              185.2,44; 185.2,12],         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcLP_in.inlet, Ec_LP_In) annotation (points=[380,56; 380,
              200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvLP_in.inlet, Ev_LP_In) annotation (points=[300,56; 300,
              200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShLP_in.inlet, Sh_LP_In) annotation (points=[120,56; 120,
              130; 220,130; 220,200],
                                 style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcIP_out.outlet, Ec_IP_Out) annotation (points=[196,-56;
              196,-70; 40,-70; 40,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShIP_in.inlet, Sh_IP_In) annotation (points=[-206,56; -206,
              90; -80,90; -80,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvIP_in.inlet, Ev_IP_In) annotation (points=[-3.67382e-016,
              56; 1.77636e-015,56; 1.77636e-015,200],
                                style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvIP_out.outlet, Ev_IP_Out) annotation (points=[
              -1.10215e-015,-56; 0,-64; 0,-70; -40,-70; -40,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateShLP_out.outlet, Sh_LP_Out) annotation (points=[120,-56;
              120,-110; 300,-110; 300,-198],
                                         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateSh1HP_in.inlet, Sh_HP_In) annotation (points=[-440,76;
              -440,140; -380,140; -380,200],
                         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvHP_out.outlet, Ev_HP_Out) annotation (points=[-300,-56;
              -300,-70; -340,-70; -340,200],                     style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEvLP_out.outlet, Ev_LP_Out) annotation (points=[300,-56;
              300,-70; 260,-70; 260,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEcLP_out.outlet, Ec_LP_Out) annotation (points=[380,-56;
              380,-70; 340,-70; 340,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateEc2HP_out.outlet, Ec_HP_Out) annotation (points=[-80,34;
              -80,60; -260,60; -260,200],                       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(ShHPOut_T.inlet, stateSh2HP_out.outlet) annotation (points=[-376,
              -160; -440,-160; -440,-76],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(RhIPOut_T.inlet, stateRh2IP_out.outlet) annotation (points=[-376,
              -134; -420,-134; -420,-76],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(RhIPOut_T.outlet, Rh_IP_Out) annotation (points=[-364,-134;
              -190,-134; -190,-200],
                               style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(ShHPOut_T.outlet, Sh_HP_Out) annotation (points=[-364,-160;
              -300,-160; -300,-200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(ShHP_RhIP.gasIn, GasIn) annotation (points=[-454,0; -462,0;
              -462,1.77636e-015; -500,1.77636e-015],                    style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(GasOut, Ec_LP.gasOut) annotation (points=[500,1.77636e-015; 426,
              0; 392,0],                   style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Ev_HP.gasIn, ShHP_RhIP.gasOut) annotation (points=[-312,0; -406,
              0],                                       style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Sh_IP.gasIn, Ev_HP.gasOut) annotation (points=[-218,0; -288,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ec2_HP.gasIn, Sh_IP.gasOut) annotation (points=[-92,0; -194,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ev_IP.gasIn, Ec2_HP.gasOut) annotation (points=[-12,0; -68,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Sh_LP.gasIn, Ev_IP.gasOut) annotation (points=[108,0; 12,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ec1HP_EcIP.gasIn, Sh_LP.gasOut) annotation (points=[178,0; 132,
              0],              style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ev_LP.gasIn, Ec1HP_EcIP.gasOut) annotation (points=[288,0; 202,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(Ec_LP.gasIn, Ev_LP.gasOut) annotation (points=[368,0; 312,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(stateRh2IP_out.inlet, ShHP_RhIP.waterOutB) annotation (points=[
              -420,-64; -420,-24; -420.4,-24], style(thickness=2, fillPattern=1));
        connect(stateSh2HP_out.inlet, ShHP_RhIP.waterOutA) annotation (points=[
              -440,-64; -440,-24; -439.6,-24], style(thickness=2, fillPattern=1));
        connect(stateSh1HP_in.outlet, ShHP_RhIP.waterInA) annotation (points=[
              -440,64; -440,24; -439.6,24], style(thickness=2, fillPattern=1));
        connect(ShHP_RhIP.waterInB, stateRh1IP_in.outlet) annotation (points=[
              -420.4,24; -420.4,44; -420,44; -420,64], style(thickness=2,
              fillPattern=1));
        connect(stateEvHP_in.inlet, Ev_HP_In) annotation (points=[-300,56; -300,
              200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(mixIP.out, stateRh1IP_in.inlet) annotation (points=[-280,-110;
              -380,-110; -380,100; -420,100; -420,76], style(thickness=2));
        connect(Rh_IP_In, mixIP.in1) annotation (points=[-130,-200; -130,-116;
              -262,-116], style(thickness=2));
        connect(mixIP.in2, stateShIP_out.outlet) annotation (points=[-262.1,
              -104; -206,-104; -206,-56], style(thickness=2));
        connect(flowSplit2.in1, Ec_HP_In) 
          annotation (points=[-220,156; -220,200], style(thickness=2));
        connect(flowSplit2.out1, stateEc1HP_in.inlet) annotation (points=[
              -215.8,144; -214,144; -214,110; 184,110; 184,56], style(thickness=
               2));
        connect(flowSplit2.out2, ShHP_RhIP.LiquidWaterIn_A) annotation (points=
              [-224,144; -224,110; -472,110; -472,-10.08; -454,-10.08], style(
              thickness=2));
        connect(flowSplit1.in1, Ec_IP_In) 
          annotation (points=[80,166; 80,200], style(thickness=2));
        connect(flowSplit1.out1, stateEcIP_in.inlet) annotation (points=[84.2,
              154; 86,154; 86,120; 196,120; 196,56], style(thickness=2));
        connect(flowSplit1.out2, ShHP_RhIP.LiquidWaterIn_B) annotation (points=
              [76,154; 76,120; -480,120; -480,-16.8; -454,-16.8], style(
              thickness=2));
      end HEG_3LRh_des;
      
      model DG_2L_NC 
        "Drums Group with two pressure level, natural circulation, all ideal feed pumps and desuperheater" 
        extends ThermoPower.PowerPlants.HRSG.Interfaces.DG_2L;
        annotation (Diagram, Icon);
        Water.SinkW blowDownHP(            redeclare package Medium = 
              FluidMedium, w0=0) 
          annotation (extent=[-246,-100; -206,-60],
                                            rotation=270);
        Components.DrumSensors HPd(
          redeclare package Medium = FluidMedium,
          rint=HPd_rint,
          rext=HPd_rext,
          L=HPd_L,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          pstart=HPd_pstart,
          hlstart=HPd_hlstart,
          hvstart=HPd_hvstart,
          Tmstart=HPd_Tmstart,
          Cm=HPd_Cm) "HP drum" 
                        annotation (extent=[-186,-20; -266,60]);
        Water.SensW steamHP_w(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-308,-200; -268,-240],
                                                  rotation=90);
        Water.SinkW blowDownLP(            redeclare package Medium = 
              FluidMedium,
            w0=0) 
          annotation (extent=[72,-66; 112,-26],
                                            rotation=270);
        Components.DrumSensors LPd(
          redeclare package Medium = FluidMedium,
          rint=LPd_rint,
          rext=LPd_rext,
          L=LPd_L,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          pstart=LPd_pstart,
          hlstart=LPd_hlstart,
          hvstart=LPd_hvstart,
          Tmstart=LPd_Tmstart,
          Cm=LPd_Cm) "LP drum" 
                        annotation (extent=[144,0; 64,80]);
        Water.SensW steamLP_w(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[32,-200; 72,-240],
                                              rotation=90);
        Water.FlowSplit flowSplit1(           redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false) 
          annotation (extent=[112,-128; 152,-88],
                                               rotation=270);
        Water.ThroughW IPfeedPump(redeclare package Medium = FluidMedium) 
                                annotation (extent=[-30,-188; -70,-148],
            rotation=0);
      equation 
        connect(SensorsBus.p_drumHP,HPd. p_out) annotation (points=[400,80; 200,
              80; 200,120; -60,120; -60,50; -211.6,50; -211.6,49.6],
                                                           style(color=84,
              rgbcolor={255,170,213}));
        connect(SensorsBus.y_drumHP,HPd. y_out) annotation (points=[400,80; 200,
              80; 200,120; -60,120; -60,36; -198.8,36],  style(color=84,
              rgbcolor={255,170,213}));
        connect(Steam_HP_Out,steamHP_w. outlet) annotation (points=[-280,-300;
              -280,-232], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(steamHP_w.inlet,HPd. steam) annotation (points=[-280,-208; -280,
              80; -244,80; -244,48; -249.2,48], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Feed_HP,HPd. feedwater) annotation (points=[-160,-300; -160,14;
              -187.2,14], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(blowDownHP.flange,HPd. blowdown) annotation (points=[-226,-60;
              -226,-19.2],                   style(thickness=2));
        connect(SensorsBus.p_drumLP,LPd. p_out) annotation (points=[400,80; 140,
              80; 140,68; 118.4,68; 118.4,69.6],      style(color=84, rgbcolor=
                {255,170,213}));
        connect(SensorsBus.y_drumLP,LPd. y_out) annotation (points=[400,80; 140,
              80; 140,56; 131.2,56],              style(color=84, rgbcolor={255,
                170,213}));
        connect(steamLP_w.inlet,LPd. steam) annotation (points=[60,-208; 60,78;
              80.8,78; 80.8,68],   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Feed_LP,LPd. feedwater) annotation (points=[180,-300; 180,34;
              142.8,34], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(blowDownLP.flange,LPd. blowdown) 
          annotation (points=[92,-26; 92,0.8; 104,0.8],
                                                 style(thickness=2));
        connect(Steam_LP_Out, steamLP_w.outlet) annotation (points=[60,-300; 60,
              -232], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(flowSplit1.in1, LPd.downcomer) annotation (points=[132,-96; 132,
              12],          style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(IPfeedPump.inlet, flowSplit1.out2) annotation (points=[-30,-168;
              124,-168; 124,-120], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(ActuatorsBus.flowRate_feedHP, IPfeedPump.in_w0) annotation (
            points=[400,20; 302,20; 302,-140; -42,-140; -42,-156], style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(WaterForHP, IPfeedPump.outlet) annotation (points=[-120,-300;
              -120,-168; -70,-168], style(thickness=2));
        connect(Downcomer_HP, HPd.downcomer) annotation (points=[-200,-300;
              -200,-8; -198,-8], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Riser_HP, HPd.riser) annotation (points=[-240,-300; -240,-120;
              -257.2,-120; -257.2,-2.8], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Downcomer_LP, flowSplit1.out1) annotation (points=[140,-300;
              140,-120; 140.4,-120], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Riser_LP, LPd.riser) annotation (points=[100,-300; 100,-82;
              72.8,-82; 72.8,17.2], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end DG_2L_NC;
      
      model DG_3L_CC 
        "Drums Group with three pressure level, controlled circulation" 
        extends ThermoPower.PowerPlants.HRSG.Interfaces.DG_3L_p;
        parameter SI.MassFlowRate RiserHPFlowRate=0 
          "Nominal mass flowrate through the riser of HP";
        parameter SI.MassFlowRate RiserIPFlowRate=0 
          "Nominal mass flowrate through the riser of IP";
        parameter SI.MassFlowRate RiserLPFlowRate=0 
          "Nominal mass flowrate through the riser of LP";
        
        annotation (Diagram);
        Water.FlowSplit flowSplit3(
          redeclare package Medium = FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false) 
          annotation (extent=[132,-150; 172,-110],
                                               rotation=180);
        Water.ThroughW feedPumpIP(            redeclare package Medium = 
              FluidMedium)      annotation (extent=[60,-202; 100,-162],
                                                                     rotation=
              270);
        Water.ThroughW feedPumpHP(            redeclare package Medium = 
              FluidMedium)      annotation (extent=[-240,-200; -200,-160],
                                                                       rotation=
             270);
        Water.ThroughW circulationPumpLP(            redeclare package Medium 
            = FluidMedium, w0=RiserLPFlowRate) 
                                annotation (extent=[280,-188; 320,-228],
                                                                   rotation=90);
        Water.SinkW blowDownHP(            redeclare package Medium = 
              FluidMedium, w0=0) 
          annotation (extent=[-348,-78; -308,-38],
                                            rotation=270);
        Water.SinkW blowDownLP(            redeclare package Medium = 
              FluidMedium,
            w0=0) 
          annotation (extent=[244,-80; 284,-40],
                                            rotation=270);
        Water.SinkW blowDownIP(            redeclare package Medium = 
              FluidMedium,
            w0=0) 
          annotation (extent=[-48,-84; -8,-44],
                                            rotation=270);
        Components.DrumSensors HPd(
          redeclare package Medium = FluidMedium,
          rint=HPd_rint,
          rext=HPd_rext,
          L=HPd_L,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          pstart=HPd_pstart,
          hlstart=HPd_hlstart,
          hvstart=HPd_hvstart,
          Tmstart=HPd_Tmstart,
          Cm=HPd_Cm) "HP drum" 
                        annotation (extent=[-288,0; -368,80]);
        Components.DrumSensors IPd(
          redeclare package Medium = FluidMedium,
          rint=IPd_rint,
          rext=IPd_rext,
          L=IPd_L,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          pstart=IPd_pstart,
          hlstart=IPd_hlstart,
          hvstart=IPd_hvstart,
          Tmstart=IPd_Tmstart,
          Cm=IPd_Cm) "IP drum" 
                        annotation (extent=[10,0; -70,80]);
        Components.DrumSensors LPd(
          redeclare package Medium = FluidMedium,
          rint=LPd_rint,
          rext=LPd_rext,
          L=LPd_L,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          pstart=LPd_pstart,
          hlstart=LPd_hlstart,
          hvstart=LPd_hvstart,
          Tmstart=LPd_Tmstart,
          Cm=LPd_Cm) "LP drum" 
                        annotation (extent=[304,-4; 224,76]);
        Water.ThroughW circulationPumpIP(            redeclare package Medium 
            = FluidMedium, w0=RiserIPFlowRate) 
                                annotation (extent=[20,-200; -20,-160],
                                                                     rotation=
              270);
        Water.ThroughW circulationPumpHP(            redeclare package Medium 
            = FluidMedium, w0=RiserHPFlowRate) 
                                annotation (extent=[-280,-200; -320,-160],
                                                                       rotation=
             270);
        Water.SensW steamHP_w(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-408,-200; -368,-240],
                                                  rotation=90);
        Water.SensW steamIP_w(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-108,-200; -68,-240],
                                               rotation=90);
        Water.SensW steamLP_w(            redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[192,-200; 232,-240],
                                              rotation=90);
        Water.FlowSplit flowSplit1(           redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false) 
          annotation (extent=[272,-120; 312,-80],
                                               rotation=270);
      equation 
        connect(ActuatorsBus.flowRate_feedHP,feedPumpHP. in_w0) annotation (
            points=[500,20; 440,20; 440,-20; -140,-20; -140,-172; -208,-172],
            style(color=52, rgbcolor={213,255,170}));
        connect(ActuatorsBus.flowRate_feedIP,feedPumpIP. in_w0) annotation (
            points=[500,20; 440,20; 440,-20; 120,-20; 120,-174; 92,-174],
                                                                 style(color=52,
              rgbcolor={213,255,170}));
        connect(SensorsBus.w_steamHP,steamHP_w. w) annotation (points=[500,80;
              400,80; 400,160; -420,160; -420,-260; -400,-260; -400,-236],
                    style(color=84, rgbcolor={255,170,213}));
        connect(SensorsBus.w_steamIP,steamIP_w. w) annotation (points=[500,80;
              400,80; 400,160; -120,160; -120,-260; -100,-260; -100,-236],
                                                                 style(color=84,
              rgbcolor={255,170,213}));
        connect(SensorsBus.w_steamLP,steamLP_w. w) annotation (points=[500,80;
              400,80; 400,160; 180,160; 180,-260; 200,-260; 200,-236],
                                                style(color=84, rgbcolor={255,
                170,213}));
        connect(SensorsBus.p_drumHP,HPd. p_out) annotation (points=[500,80; 400,
              80; 400,160; -160,160; -160,68; -313.6,68; -313.6,69.6],
                                                           style(color=84,
              rgbcolor={255,170,213}));
        connect(SensorsBus.p_drumIP,IPd. p_out) annotation (points=[500,80; 400,
              80; 400,160; 40,160; 40,69.6; -15.6,69.6],
                                          style(color=84, rgbcolor={255,170,213}));
        connect(SensorsBus.p_drumLP,LPd. p_out) annotation (points=[500,80; 400,
              80; 400,68; 278.4,68; 278.4,65.6],      style(color=84, rgbcolor=
                {255,170,213}));
        connect(SensorsBus.y_drumHP,HPd. y_out) annotation (points=[500,80; 380,
              80; 380,160; -160,160; -160,56; -300.8,56],style(color=84,
              rgbcolor={255,170,213}));
        connect(SensorsBus.y_drumIP,IPd. y_out) annotation (points=[500,80; 400,
              80; 400,160; 40,160; 40,56; -2.8,56],
                                      style(color=84, rgbcolor={255,170,213}));
        connect(SensorsBus.y_drumLP,LPd. y_out) annotation (points=[500,80; 400,
              80; 400,52; 291.2,52],              style(color=84, rgbcolor={255,
                170,213}));
        connect(Steam_HP_Out,steamHP_w. outlet) annotation (points=[-380,-300;
              -380,-232], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(steamHP_w.inlet,HPd. steam) annotation (points=[-380,-208; -380,
              100; -352,100; -352,68; -351.2,68],
                                                style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(circulationPumpHP.inlet,HPd. downcomer) annotation (points=[-300,
              -160; -300,12],                       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Feed_HP,HPd. feedwater) annotation (points=[-260,-300; -260,34;
              -289.2,34], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Steam_IP_Out,steamIP_w. outlet) annotation (points=[-80,-300;
              -80,-232],
                     style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(circulationPumpIP.inlet,IPd. downcomer) annotation (points=[
              -3.67382e-015,-160; 0,-160; 0,12; -2,12],            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(steamIP_w.inlet,IPd. steam) annotation (points=[-80,-208; -80,
              80; -38,80; -38,68; -53.2,68],
                                         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Feed_IP,IPd. feedwater) annotation (points=[40,-300; 40,34; 8.8,
              34], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(steamLP_w.inlet,LPd. steam) annotation (points=[220,-208; 220,
              80; 240.8,80; 240.8,64],
                                   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Steam_LP_Out,steamLP_w. outlet) annotation (points=[220,-300;
              220,-232],
                     style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Feed_LP,LPd. feedwater) annotation (points=[340,-300; 340,30;
              302.8,30], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(circulationPumpLP.inlet,flowSplit1. out1) annotation (points=[300,-188;
              300,-112; 300.4,-112],           style(thickness=2));
        connect(flowSplit1.in1,LPd. downcomer) annotation (points=[292,-88; 292,
              8],                          style(thickness=2));
        connect(flowSplit3.out1,feedPumpIP. inlet) 
                                                  annotation (points=[140,
              -138.4; 140,-140; 80,-140; 80,-162],
                                           style(thickness=2));
        connect(flowSplit3.out2,feedPumpHP. inlet) 
                                                  annotation (points=[140,-122;
              -220,-122; -220,-160], style(thickness=2));
        connect(blowDownHP.flange,HPd. blowdown) annotation (points=[-328,-38;
              -328,0.8],                     style(thickness=2));
        connect(blowDownIP.flange,IPd. blowdown) 
          annotation (points=[-28,-44; -28,0.8; -30,0.8], style(thickness=2));
        connect(blowDownLP.flange,LPd. blowdown) 
          annotation (points=[264,-40; 264,-3.2],style(thickness=2));
        connect(flowSplit1.out2, flowSplit3.in1) annotation (points=[284,-112;
              284,-130; 164,-130], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(WaterForHP, feedPumpHP.outlet) annotation (points=[-220,-300;
              -220,-200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(WaterForIP, feedPumpIP.outlet) annotation (points=[80,-300; 80,
              -202], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Riser_LP, LPd.riser) annotation (points=[260,-300; 260,-100;
              232.8,-100; 232.8,13.2], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Downcomer_LP, circulationPumpLP.outlet) annotation (points=[300,
              -300; 300,-300; 300,-228], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Downcomer_IP, circulationPumpIP.outlet) annotation (points=[
              1.77636e-015,-300; 1.77636e-015,-252; 3.67382e-015,-252;
              3.67382e-015,-200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Riser_IP, IPd.riser) annotation (points=[-40,-300; -40,-100;
              -61.2,-100; -61.2,17.2], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Downcomer_HP, circulationPumpHP.outlet) annotation (points=[
              -300,-300; -300,-200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Riser_HP, HPd.riser) annotation (points=[-340,-300; -340,-100;
              -359.2,-100; -359.2,17.2], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end DG_3L_CC;
      
      model HRSG_3LRh 
        "Heat Recovery Steam Generator with three pressure level and reheat" 
        extends Interfaces.HRSG_3LRh;
        
        annotation (Diagram);
        ThermoPower.PowerPlants.HRSG.Examples.DG_3L_CC drums(
          HPd_rint=1.067,
          HPd_rext=1.167,
          HPd_L=11.920,
          HPd_Cm=0,
          IPd_rint=0.915,
          IPd_rext=1.015,
          IPd_L=7,
          IPd_Cm=0,
          LPd_rint=1.143,
          LPd_rext=1.243,
          LPd_L=11.503,
          LPd_Cm=0,
          RiserHPFlowRate=175.5,
          RiserIPFlowRate=67.5,
          RiserLPFlowRate=41.5,
          redeclare package FluidMedium = FluidMedium,
          SSInit=false,
          fluidLPNomPressure=7.19048e5,
          LPd_hvstart=2.76051e6,
          LPd_hlstart=6.68356e5,
          fluidHPNomPressure=1.626e7,
          fluidIPNomPressure=3.3816e6,
          HPd_hvstart=2.53239e6,
          HPd_hlstart=1.5879e6,
          IPd_hvstart=2.794e6,
          IPd_hlstart=1.0265e6) annotation (extent=[-102,40; 98,160]);
        ThermoPower.PowerPlants.HRSG.Examples.HEG_3LRh HeatExchangersGroup(
          gasNomFlowRate=585.5,
          fluidHPNomFlowRate_Sh=70.59,
          fluidHPNomPressure_Sh=134.3e5,
          fluidHPNomFlowRate_Ev=175.5,
          fluidHPNomPressure_Ev=137.1e5,
          fluidHPNomFlowRate_Ec=70.10,
          fluidHPNomPressure_Ec=138.9e5,
          fluidIPNomFlowRate_Rh=81.10,
          fluidIPNomPressure_Rh=28.4e5,
          fluidIPNomFlowRate_Sh=13.5,
          fluidIPNomPressure_Sh=29.5e5,
          fluidIPNomFlowRate_Ev=67.5,
          fluidIPNomPressure_Ev=37.16e5,
          fluidIPNomFlowRate_Ec=21.8,
          fluidIPNomPressure_Ec=48.6e5,
          fluidLPNomFlowRate_Sh=6.91,
          fluidLPNomPressure_Sh=6.6e5,
          fluidLPNomFlowRate_Ev=41.49,
          fluidLPNomPressure_Ev=15.34e5,
          fluidLPNomFlowRate_Ec=122.4,
          fluidLPNomPressure_Ec=19.8e5,
          Sh2_HP_N_G=3,
          Sh2_HP_N_F=5,
          Sh2_HP_exchSurface_G=3636,
          Sh2_HP_exchSurface_F=421.844,
          Sh2_HP_extSurfaceTub=540.913,
          Sh2_HP_gasVol=10,
          Sh2_HP_fluidVol=2.615,
          Sh2_HP_metalVol=1.685,
          Sh1_HP_N_G=3,
          Sh1_HP_N_F=7,
          Sh1_HP_exchSurface_G=8137.2,
          Sh1_HP_exchSurface_F=612.387,
          Sh1_HP_extSurfaceTub=721.256,
          Sh1_HP_gasVol=10,
          Sh1_HP_fluidVol=4.134,
          Sh1_HP_metalVol=1.600,
          Ev_HP_N_G=4,
          Ev_HP_N_F=4,
          Ev_HP_exchSurface_G=30501.9,
          Ev_HP_exchSurface_F=2296.328,
          Ev_HP_extSurfaceTub=2704.564,
          Ev_HP_gasVol=10,
          Ev_HP_fluidVol=15.500,
          Ev_HP_metalVol=6.001,
          Ec2_HP_N_G=3,
          Ec2_HP_N_F=6,
          Ec2_HP_exchSurface_G=20335,
          Ec2_HP_exchSurface_F=1451.506,
          Ec2_HP_extSurfaceTub=1803.043,
          Ec2_HP_gasVol=10,
          Ec2_HP_fluidVol=9.290,
          Ec2_HP_metalVol=5.045,
          Ec1_HP_N_G=3,
          Ec1_HP_N_F=5,
          Ec1_HP_exchSurface_G=12201.2,
          Ec1_HP_exchSurface_F=870.904,
          Ec1_HP_extSurfaceTub=1081.826,
          Ec1_HP_gasVol=10,
          Ec1_HP_fluidVol=5.574,
          Ec1_HP_metalVol=3.027,
          Rh2_IP_N_F=7,
          Rh2_IP_exchSurface_G=4630.2,
          Rh2_IP_exchSurface_F=873.079,
          Rh2_IP_extSurfaceTub=1009.143,
          Rh2_IP_fluidVol=8.403,
          Rh2_IP_metalVol=2.823,
          Rh1_IP_N_F=5,
          Rh1_IP_exchSurface_G=4630,
          Rh1_IP_exchSurface_F=900.387,
          Rh1_IP_extSurfaceTub=1009.250,
          Rh1_IP_fluidVol=8.936,
          Rh1_IP_metalVol=2.292,
          Sh_IP_N_G=3,
          Sh_IP_N_F=7,
          Sh_IP_exchSurface_G=2314.8,
          Sh_IP_exchSurface_F=450.218,
          Sh_IP_extSurfaceTub=504.652,
          Sh_IP_gasVol=10,
          Sh_IP_fluidVol=4.468,
          Sh_IP_metalVol=1.146,
          Ev_IP_N_G=4,
          Ev_IP_N_F=4,
          Ev_IP_exchSurface_G=24402,
          Ev_IP_exchSurface_F=1837.063,
          Ev_IP_extSurfaceTub=2163.652,
          Ev_IP_gasVol=10,
          Ev_IP_fluidVol=12.400,
          Ev_IP_metalVol=4.801,
          Ec_IP_N_G=3,
          Ec_IP_N_F=5,
          Ec_IP_exchSurface_G=4067.2,
          Ec_IP_exchSurface_F=306.177,
          Ec_IP_extSurfaceTub=360.609,
          Ec_IP_gasVol=10,
          Ec_IP_fluidVol=2.067,
          Ec_IP_metalVol=0.800,
          Sh_LP_N_G=3,
          Sh_LP_N_F=7,
          Sh_LP_exchSurface_G=1708.2,
          Sh_LP_exchSurface_F=225.073,
          Sh_LP_extSurfaceTub=252.286,
          Sh_LP_gasVol=10,
          Sh_LP_fluidVol=2.234,
          Sh_LP_metalVol=0.573,
          Ev_LP_N_G=4,
          Ev_LP_N_F=4,
          Ev_LP_exchSurface_G=24402,
          Ev_LP_exchSurface_F=2292.926,
          Ev_LP_extSurfaceTub=2592.300,
          Ev_LP_gasVol=10,
          Ev_LP_fluidVol=19.318,
          Ev_LP_metalVol=5.374,
          Ec_LP_N_G=3,
          Ec_LP_N_F=6,
          Ec_LP_exchSurface_G=40095.9,
          Ec_LP_exchSurface_F=3439.389,
          Ec_LP_extSurfaceTub=3888.449,
          Ec_LP_gasVol=10,
          Ec_LP_fluidVol=28.977,
          Ec_LP_metalVol=8.061,
          use_T_Ev=false,
          Sh2_HP_Tstart_G_In=884.65,
          Sh2_HP_Tstart_G_Out=837.31,
          Sh2_HP_Tstart_M_In=742.65,
          Sh2_HP_Tstart_M_Out=823.28,
          Sh2_HP_Tstart_F_In=742.65,
          Sh2_HP_Tstart_F_Out=823.28,
          Sh1_HP_Tstart_G_In=837.15,
          Sh1_HP_Tstart_G_Out=748.60,
          Sh1_HP_Tstart_M_In=606.59,
          Sh1_HP_Tstart_M_Out=743.18,
          Sh1_HP_Tstart_F_In=606.59,
          Sh1_HP_Tstart_F_Out=743.18,
          Ev_HP_Tstart_G_In=747.15,
          Ev_HP_Tstart_G_Out=620.30,
          Ev_HP_hstart_F_In=1.514e6,
          Ev_HP_hstart_F_Out=2.002e6,
          Ec2_HP_Tstart_G_In=614.15,
          Ec2_HP_Tstart_G_Out=558.98,
          Ec2_HP_Tstart_M_In=494.67,
          Ec2_HP_Tstart_M_Out=594.14,
          Ec2_HP_Tstart_F_In=494.67,
          Ec2_HP_Tstart_F_Out=594.14,
          Ec1_HP_Tstart_M_In=431.77,
          Ec1_HP_Tstart_M_Out=494.67,
          Ec1_HP_Tstart_F_In=431.77,
          Ec1_HP_Tstart_F_Out=494.67,
          Rh2_IP_Tstart_M_In=717.31,
          Rh2_IP_Tstart_M_Out=813.09,
          Rh2_IP_Tstart_F_In=717.31,
          Rh2_IP_Tstart_F_Out=813.09,
          Rh1_IP_Tstart_M_In=618.31,
          Rh1_IP_Tstart_M_Out=717.34,
          Rh1_IP_Tstart_F_In=618.31,
          Rh1_IP_Tstart_F_Out=717.34,
          Sh_IP_Tstart_G_In=620.15,
          Sh_IP_Tstart_G_Out=614.50,
          Sh_IP_Tstart_M_In=506.08,
          Sh_IP_Tstart_M_Out=605.11,
          Sh_IP_Tstart_F_In=506.08,
          Sh_IP_Tstart_F_Out=605.11,
          Ev_IP_Tstart_G_In=558.15,
          Ev_IP_Tstart_G_Out=518.51,
          Ev_IP_hstart_F_In=9.92e5,
          Ev_IP_hstart_F_Out=1.37e6,
          Ec_IP_Tstart_G_In=517.15,
          Ec_IP_Tstart_G_Out=480.11,
          Ec_IP_Tstart_M_In=430.24,
          Ec_IP_Tstart_M_Out=493.16,
          Ec_IP_Tstart_F_In=430.24,
          Ec_IP_Tstart_F_Out=493.16,
          Sh_LP_Tstart_G_In=519.15,
          Sh_LP_Tstart_G_Out=517.44,
          Sh_LP_Tstart_M_In=435.75,
          Sh_LP_Tstart_M_Out=505.04,
          Sh_LP_Tstart_F_In=435.75,
          Sh_LP_Tstart_F_Out=505.75,
          Ev_LP_Tstart_G_In=480.00,
          Ev_LP_Tstart_G_Out=453.47,
          Ev_LP_hstart_F_In=6.625e5,
          Ev_LP_hstart_F_Out=1.065e6,
          Ec_LP_Tstart_G_In=453.47,
          Ec_LP_Tstart_G_Out=372.35,
          Ec_LP_Tstart_M_In=331.05,
          Ec_LP_Tstart_M_Out=428.09,
          Ec_LP_Tstart_F_In=331.05,
          Ec_LP_Tstart_F_Out=428.09,
          rhomcm=7900*578.05,
          lambda=20,
          Ec_LP(gamma_G=46.8, gamma_F=4000),
          Ev_LP(gamma_G=127, gamma_F=20000),
          Ec1HP_EcIP(
            gamma_G_A=42,
            gamma_G_B=45,
            gamma_F_A=4000,
            gamma_F_B=4000),
          Sh_LP(gamma_G=16.6, gamma_F=4000),
          Ev_IP(gamma_G=58.5, gamma_F=20000),
          Ec2_HP(gamma_G=56, gamma_F=4000),
          Sh_IP(gamma_G=33, gamma_F=4000),
          Ev_HP(gamma_G=46.5, gamma_F=20000),
          gasNomPressure=1e5,
          Sh1HP_Rh1IP(
            gamma_G_A=70,
            gamma_G_B=80,
            gamma_F_A=4000,
            gamma_F_B=4000),
          Sh2HP_Rh2IP(
            gamma_G_A=83.97,
            gamma_G_B=80,
            gamma_F_A=4000,
            gamma_F_B=4000),
          redeclare package FlueGasMedium = FlueGasMedium,
          redeclare package FluidMedium = FluidMedium,
          Ev_HP_Tstart_M_In=FluidMedium.temperature_ph(137.1e5, 1.514e6),
          Ev_HP_Tstart_M_Out=FluidMedium.temperature_ph(137.1e5, 2.002e6),
          Ev_IP_Tstart_M_In=FluidMedium.temperature_ph(37.16e5, 9.92e5),
          Ev_IP_Tstart_M_Out=FluidMedium.temperature_ph(37.16e5, 1.37e6),
          Ev_LP_Tstart_M_In=FluidMedium.temperature_ph(15.34e5, 6.625e5),
          Ev_LP_Tstart_M_Out=FluidMedium.temperature_ph(15.34e5, 1.065e6)) 
                             annotation (extent=[-102,-80; 98,0]);
        replaceable Components.BaseReader_water stateInletWater(redeclare 
            package Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[150,-120; 170,-100],
                                                                  rotation=90);
      equation 
        connect(ActuatorsBus, drums.ActuatorsBus)           annotation (points=[200,100;
              158,100; 158,104; 98,104],            style(color=52, rgbcolor={
                213,255,170}));
        connect(drums.SensorsBus, SensorsBus)           annotation (points=[98,116;
              140,116; 140,160; 200,160],              style(color=84, rgbcolor=
               {255,170,213}));
        connect(HeatExchangersGroup.GasIn, GasIn) annotation (points=[-102,-40;
              -200,-40],                     style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.GasOut, GasOut) annotation (points=[98,-40;
              200,-40],                   style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_HP_Out, HeatExchangersGroup.Sh_HP_Out) annotation (points=[-160,
              -200; -160,-106; -62,-106; -62,-80],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Sh_LP_Out, HeatExchangersGroup.Sh_LP_Out) annotation (points=[80,-200;
              80,-160; 58,-160; 58,-79.6],         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Rh_IP_Out, HeatExchangersGroup.Rh_IP_Out) annotation (points=[-40,-200;
              -40,-80],                                style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(Rh_IP_In, HeatExchangersGroup.Rh_IP_In) annotation (points=[-100,
              -200; -100,-120; -28,-120; -28,-80],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(WaterIn, stateInletWater.inlet) 
          annotation (points=[160,-200; 160,-116], style(thickness=2));
        connect(stateInletWater.outlet, HeatExchangersGroup.Ec_LP_In) 
          annotation (points=[160,-104; 160,20; 74,20; 74,0], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.SensorsBus, drums.SensorsBus) annotation (
            points=[98,-12; 140,-12; 140,116; 98,116],   style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.ActuatorsBus, drums.ActuatorsBus) 
          annotation (points=[98,-24; 128,-24; 128,104; 98,104],   style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Sh_HP_In, drums.Steam_HP_Out) annotation (
            points=[-78,0; -78,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ev_HP_Out, drums.Riser_HP) annotation (
            points=[-70,0; -70,0; -70,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ev_HP_In, drums.Downcomer_HP) annotation (
            points=[-62,0; -62,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ec_HP_Out, drums.Feed_HP) annotation (
            points=[-54,0; -54,40; -54,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ec_HP_In, drums.WaterForHP) annotation (
            points=[-46,0; -46,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Sh_IP_In, drums.Steam_IP_Out) annotation (
            points=[-18,0; -18,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ev_IP_Out, drums.Riser_IP) annotation (
            points=[-10,0; -10,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ev_IP_In, drums.Downcomer_IP) annotation (
            points=[-2,0; -2,10.5; -2,10.5; -2,21; -2,40; -2,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ec_IP_Out, drums.Feed_IP) annotation (
            points=[6,0; 6,40; 6,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ec_IP_In, drums.WaterForIP) annotation (
            points=[14,0; 14,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Sh_LP_In, drums.Steam_LP_Out) annotation (
            points=[42,0; 42,0; 42,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ev_LP_Out, drums.Riser_LP) annotation (
            points=[50,0; 50,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ev_LP_In, drums.Downcomer_LP) annotation (
            points=[58,0; 58,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HeatExchangersGroup.Ec_LP_Out, drums.Feed_LP) annotation (
            points=[66,0; 66,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end HRSG_3LRh;
      
    end Examples;
    
    package Tests "Test cases" 
      model computation_states 
        package FluidMedium = ThermoPower.Water.StandardWater;
        parameter SI.SpecificEnthalpy h=2.39102e6 "value of specific enthalpy";
        parameter SI.Pressure p=5389.2 "value of pressure";
        parameter SI.Temperature T=288 "value of temperature";
        
        SI.Temperature Tf "Temperature (p,h)";
        SI.SpecificEnthalpy hf "Specific enthalpy (p,T)";
        SI.Temperature Ts "Saturation temperature (p)";
        SI.SpecificEnthalpy hv "Dew specific enthalpy (p)";
        SI.SpecificEnthalpy hl "Bubble specific enthalpy (p)";
        SI.Density rhov "Dew density (p)";
        SI.Density rhol "Bubble density (p)";
        FluidMedium.SaturationProperties sat "Saturation properties";
        
      equation 
        sat.psat = p;
        sat.Tsat = FluidMedium.saturationTemperature(p);
        
        Ts = sat.Tsat;
        hl=FluidMedium.bubbleEnthalpy(sat);
        hv=FluidMedium.dewEnthalpy(sat);
        Tf=FluidMedium.temperature_ph(p,h);
        hf=FluidMedium.specificEnthalpy_pT(p,T);
        rhov=FluidMedium.density_ph(p,h);
        rhol=FluidMedium.bubbleDensity(sat);
      end computation_states;
      
      model TestHE_simp 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        Water.SourceW sourseW_water(redeclare package Medium = FluidMedium,
              w0=70.59,
          p0=129.6e5,
          h=3.251e6)          annotation (extent=[-20,60; 0,80],  rotation=270);
        Water.SinkP sinkP_water(redeclare package Medium = FluidMedium,
          p0=129.6e5,
          h=3.328e6)      annotation (extent=[-20,-98; 0,-78],  rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=862) 
                         annotation (extent=[50,-12; 76,12]);
        Gas.SourceW sourceW_gas(
          w0=585.5,
          redeclare package Medium = FlueGasMedium,
          T=884.65)          annotation (extent=[-90,-14; -68,14]);
        annotation (Diagram);
        Components.HE_simp hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=3,
          N_F=5,
          exchSurface_G=540.913,
          exchSurface_F=421.844,
          gasVol=10,
          fluidVol=2.615,
          metalVol=1.685,
          rhomcm=7780*688,
          lambda=20,
          gasNomFlowRate=585.5,
          gamma_G=83.97,
          gamma_F=4000,
          Tstart_G_In=884.65,
          Tstart_F_In=742.65,
          fluidNomFlowRate=70.59,
          fluidNomPressure=129.6e5,
          gasNomPressure=0,
          Tstart_M_In=743,
          Tstart_M_Out=824.28,
          Tstart_F_Out=769.84,
          SSInit=false,
          Tstart_G_Out=876.35) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SensT T_waterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,32; 4,52], rotation=270);
        Water.SensT T_waterOut(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,-62; 4,-42], rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 50,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -61,0; -61,1.77636e-015; -68,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(sinkP_water.flange, T_waterOut.outlet) annotation (points=[-10,-78;
              -10,-68; -10,-68; -10,-58],      style(thickness=2));
        connect(T_waterOut.inlet, hE.waterOut) 
          annotation (points=[-10,-46; -10,-20], style(thickness=2));
        connect(T_waterIn.inlet, sourseW_water.flange) annotation (points=[-10,48;
              -10,54; -10,60; -10,60],     style(thickness=2));
        connect(T_waterIn.outlet, hE.waterIn) 
          annotation (points=[-10,36; -10,20], style(thickness=2));
      end TestHE_simp;
      
      model TestEcLP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 "Nominal mass flowrate";
        parameter SI.Temperature Tstart_G_In=453.47 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=372.35 
          "Outlet gas temperature start value";
        
        //fluid
        parameter SI.MassFlowRate fluidNomFlowRate=89.82 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure=6e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G=35 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_In=307.05 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_Out=428.09 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_In=307.05 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_Out=428.09 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_Out) 
          "Nominal specific enthalpy";
        parameter Boolean SSInit = false "Steady-state initialization";
        
        Water.SourceW sourseW_water(redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate,
          p0=fluidNomPressure,
          h=hstart_F_In)      annotation (extent=[-10,50; 10,70], rotation=270);
        Water.SinkP sinkP_water(redeclare package Medium = FluidMedium,
          p0=fluidNomPressure,
          h=hstart_F_Out) annotation (extent=[-10,-70; 10,-50], rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[50,-10; 70,10]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          w0=gasNomFlowRate,
          T=Tstart_G_In)     annotation (extent=[-70,-10; -50,10]);
        annotation (Diagram);
        Components.HE hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=3,
          N_F=6,
          exchSurface_G=40095.9,
          exchSurface_F=3439.389,
          extSurfaceTub=3888.449,
          gasVol=10,
          fluidVol=28.977,
          metalVol=8.061,
          rhomcm=7900*578.05,
          lambda=20,
          gasNomFlowRate=585.5,
          fluidNomFlowRate=fluidNomFlowRate,
          fluidNomPressure=fluidNomPressure,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_In=Tstart_M_In,
          Tstart_M_Out=Tstart_M_Out,
          Tstart_F_In=Tstart_F_In,
          Tstart_F_Out=Tstart_F_Out,
          gamma_G=gamma_G,
          gamma_F=gamma_F,
          SSInit=SSInit,
          gasNomPressure=5e5) 
          annotation (extent=[-30,-30; 30,30]);
      equation 
        connect(hE.waterIn, sourseW_water.flange) annotation (points=[0,30; 0,
              50; -1.83691e-015,50], style(thickness=2));
        connect(hE.waterOut, sinkP_water.flange) annotation (points=[0,-30; 0,
              -50; 1.83691e-015,-50], style(thickness=2));
        connect(hE.gasIn, sourceW_gas.flange) annotation (points=[-30,0; -50,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(hE.gasOut, sinkP_gas.flange) annotation (points=[30,0; 50,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
      end TestEcLP;
      
      model TestEvLP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 "Nominal mass flowrate";
        parameter SI.Temperature Tstart_G_In=480 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=453.47 
          "Outlet gas temperature start value";
        
        //fluid
        parameter SI.MassFlowRate fluidNomFlowRate=41.49 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure=15.34e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G=40 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_In=Tstart_F_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_Out=Tstart_F_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_In=FluidMedium.temperature_ph(fluidNomPressure,hstart_F_In) 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_Out=FluidMedium.temperature_ph(fluidNomPressure,hstart_F_Out) 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_In=6.626e5 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_Out=1.065e6 
          "Nominal specific enthalpy";
        
        Water.SourceW sourseW_water(redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate,
          p0=fluidNomPressure,
          h=hstart_F_In)      annotation (extent=[-20,60; 0,80],  rotation=270);
        Water.SinkP sinkP_water(redeclare package Medium = FluidMedium,
          p0=fluidNomPressure,
          h=hstart_F_Out) annotation (extent=[-20,-98; 0,-78],  rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[50,-12; 76,12]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          w0=gasNomFlowRate,
          T=Tstart_G_In)     annotation (extent=[-90,-14; -68,14]);
        annotation (Diagram);
        Components.HE2ph hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=4,
          N_F=4,
          exchSurface_G=24402,
          exchSurface_F=2292.926,
          extSurfaceTub=2592.300,
          gasVol=10,
          fluidVol=19.318,
          metalVol=5.374,
          rhomcm=7900*578.05,
          lambda=20,
          gasNomFlowRate=585.5,
          gasNomPressure=0,
          fluidNomFlowRate=fluidNomFlowRate,
          fluidNomPressure=fluidNomPressure,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_In=Tstart_M_In,
          Tstart_M_Out=Tstart_M_Out,
          Tstart_F_In=Tstart_F_In,
          Tstart_F_Out=Tstart_F_Out,
          gamma_G=gamma_G,
          gamma_F=gamma_F,
          SSInit=true) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SensT T_waterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,32; 4,52], rotation=270);
        Water.SensT T_waterOut(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,-62; 4,-42], rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 50,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -61,0; -61,1.77636e-015; -68,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(sinkP_water.flange, T_waterOut.outlet) annotation (points=[-10,-78;
              -10,-68; -10,-68; -10,-58],      style(thickness=2));
        connect(T_waterOut.inlet, hE.waterOut) 
          annotation (points=[-10,-46; -10,-20], style(thickness=2));
        connect(T_waterIn.inlet, sourseW_water.flange) annotation (points=[-10,48;
              -10,54; -10,60; -10,60],     style(thickness=2));
        connect(T_waterIn.outlet, hE.waterIn) 
          annotation (points=[-10,36; -10,20], style(thickness=2));
      end TestEvLP;
      
      model TestShLP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 "Nominal mass flowrate";
        parameter SI.Temperature Tstart_G_In=519.15 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=517.44 
          "Outlet gas temperature start value";
        
        //fluid
        parameter SI.MassFlowRate fluidNomFlowRate=6.91 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure=6.6e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G=50 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_In=Tstart_F_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_Out=Tstart_F_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_In=435.75 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_Out=505.04 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_Out) 
          "Nominal specific enthalpy";
        
        Water.SourceW sourseW_water(redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate,
          p0=fluidNomPressure,
          h=hstart_F_In)      annotation (extent=[-20,60; 0,80],  rotation=270);
        Water.SinkP sinkP_water(redeclare package Medium = FluidMedium,
          p0=fluidNomPressure,
          h=hstart_F_Out) annotation (extent=[-20,-98; 0,-78],  rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[50,-12; 76,12]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          w0=gasNomFlowRate,
          T=Tstart_G_In)     annotation (extent=[-90,-14; -68,14]);
        annotation (Diagram);
        Components.HE hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=3,
          N_F=7,
          exchSurface_G=1708.2,
          exchSurface_F=225.073,
          extSurfaceTub=252.286,
          gasVol=10,
          fluidVol=2.234,
          metalVol=0.573,
          rhomcm=7900*578.05,
          lambda=20,
          gasNomFlowRate=585.5,
          gasNomPressure=0,
          fluidNomFlowRate=fluidNomFlowRate,
          fluidNomPressure=fluidNomPressure,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_In=Tstart_M_In,
          Tstart_M_Out=Tstart_M_Out,
          Tstart_F_In=Tstart_F_In,
          Tstart_F_Out=Tstart_F_Out,
          gamma_G=gamma_G,
          gamma_F=gamma_F,
          SSInit=true) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SensT T_waterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,32; 4,52], rotation=270);
        Water.SensT T_waterOut(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,-62; 4,-42], rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 50,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -61,0; -61,1.77636e-015; -68,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(sinkP_water.flange, T_waterOut.outlet) annotation (points=[-10,-78;
              -10,-68; -10,-68; -10,-58],      style(thickness=2));
        connect(T_waterOut.inlet, hE.waterOut) 
          annotation (points=[-10,-46; -10,-20], style(thickness=2));
        connect(T_waterIn.inlet, sourseW_water.flange) annotation (points=[-10,48;
              -10,54; -10,60; -10,60],     style(thickness=2));
        connect(T_waterIn.outlet, hE.waterIn) 
          annotation (points=[-10,36; -10,20], style(thickness=2));
      end TestShLP;
      
      model TestEc1HP_EcIP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 
          "Nominal flow rate through the gas side";
        parameter SI.Temperature Tstart_G_In=517.15 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=480.11 
          "Outlet gas temperature start value";
        
        //fluid A
        parameter SI.MassFlowRate fluidNomFlowRate_A=70.1 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure_A=138.9e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G_A=40 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F_A=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_A_In=Tstart_F_A_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_A_Out=Tstart_F_A_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_A_In=431.77 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_A_Out=494.67 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_A_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,Tstart_F_A_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_A_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,Tstart_F_A_Out) 
          "Nominal specific enthalpy";
        
        //fluid B
        parameter SI.MassFlowRate fluidNomFlowRate_B=21.8 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure_B=48.6e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G_B=40 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F_B=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_B_In=Tstart_F_B_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_B_Out=Tstart_F_B_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_B_In=430.24 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_B_Out=493.16 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_B_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,Tstart_F_B_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_B_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,Tstart_F_B_Out) 
          "Nominal specific enthalpy";
        Water.SourceW sourseW_water_A(
          redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate_A,
          p0=fluidNomPressure_A,
          h=hstart_F_A_In)    annotation (extent=[-38,64; -18,84],rotation=270);
        Water.SinkP sinkP_water_A(
          redeclare package Medium = FluidMedium,
          p0=fluidNomPressure_A,
          h=hstart_F_A_Out) 
                          annotation (extent=[-38,-96; -18,-76],rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[60,-10; 80,10]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          T=Tstart_G_In,
          w0=gasNomFlowRate) annotation (extent=[-94,-10; -74,10]);
        annotation (Diagram);
        Components.ParHE hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=3,
          gasVol=10,
          lambda=20,
          N_F_A=5,
          exchSurface_G_A=12201.200,
          exchSurface_F_A=870.904,
          extSurfaceTub_A=1081.826,
          fluidVol_A=5.574,
          metalVol_A=3.027,
          rhomcm_A=7900*578.05,
          N_F_B=5,
          exchSurface_G_B=4067.200,
          exchSurface_F_B=306.177,
          extSurfaceTub_B=360.609,
          fluidVol_B=2.067,
          metalVol_B=0.800,
          rhomcm_B=7900*578.05,
          SSInit=true,
          counterCurrent_A=true,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate_A=fluidNomFlowRate_A,
          fluidNomFlowRate_B=fluidNomFlowRate_B,
          fluidNomPressure_A=fluidNomPressure_A,
          fluidNomPressure_B=fluidNomPressure_B,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_A_In=Tstart_M_A_In,
          Tstart_M_A_Out=Tstart_M_A_Out,
          Tstart_F_A_In=Tstart_F_A_In,
          Tstart_F_A_Out=Tstart_F_A_Out,
          Tstart_M_B_In=Tstart_M_B_In,
          Tstart_M_B_Out=Tstart_M_B_Out,
          Tstart_F_B_In=Tstart_F_B_In,
          Tstart_F_B_Out=Tstart_F_B_Out,
          gamma_G_A=gamma_G_A,
          gamma_G_B=gamma_G_B,
          gamma_F_A=gamma_F_A,
          gamma_F_B=gamma_F_B,
          counterCurrent_B=true,
          gasNomPressure=1e5) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SinkP sinkP_water_B(
          redeclare package Medium = FluidMedium,
          p0=fluidNomPressure_B,
          h=hstart_F_B_Out) 
                          annotation (extent=[-2,-96; 18,-76],  rotation=270);
        Water.SourceW sourseW_water_B(
          redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate_B,
          p0=fluidNomPressure_B,
          h=hstart_F_B_In)    annotation (extent=[-4,64; 16,84],  rotation=270);
        Water.SensT T_waterIn_A(redeclare package Medium = FluidMedium) 
          annotation (extent=[-34,28; -14,48],
                                             rotation=270);
        Water.SensT T_waterOut_A(redeclare package Medium = FluidMedium) 
          annotation (extent=[-34,-52; -14,-32],
                                               rotation=270);
        Water.SensT T_waterOut_B(redeclare package Medium = FluidMedium) 
          annotation (extent=[2,-52; 22,-32],  rotation=270);
        Water.SensT T_waterIn_B(redeclare package Medium = FluidMedium) 
          annotation (extent=[0,28; 20,48],  rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
        
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 60,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -74,0],                                     style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_waterOut_A.outlet, sinkP_water_A.flange) annotation (points=[-28,-48;
              -28,-62; -28,-76; -28,-76],          style(thickness=2));
        connect(sinkP_water_B.flange, T_waterOut_B.outlet) annotation (points=[8,-76; 8,
              -69; 8,-69; 8,-62; 8,-48; 8,-48],              style(thickness=2));
        connect(T_waterOut_B.inlet, hE.waterOutB) annotation (points=[8,-36; 8,
              -26; -2,-26; -2,-20], style(thickness=2));
        connect(T_waterOut_A.inlet, hE.waterOutA) annotation (points=[-28,-36;
              -28,-28; -18,-28; -18,-20], style(thickness=2));
        connect(hE.waterInA, T_waterIn_A.outlet) annotation (points=[-18,20;
              -18,24; -28,24; -28,32], style(thickness=2));
        connect(hE.waterInB, T_waterIn_B.outlet) 
          annotation (points=[-2,20; -2,24; 6,24; 6,32], style(thickness=2));
        connect(T_waterIn_B.inlet, sourseW_water_B.flange) 
          annotation (points=[6,44; 6,49; 6,49; 6,54; 6,64; 6,64],
                                                       style(thickness=2));
        connect(T_waterIn_A.inlet, sourseW_water_A.flange) annotation (points=[-28,44;
              -28,54; -28,64; -28,64],         style(thickness=2));
      end TestEc1HP_EcIP;
      
      model TestEvIP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 "Nominal mass flowrate";
        parameter SI.Temperature Tstart_G_In=558.15 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=518.51 
          "Outlet gas temperature start value";
        
        //fluid
        parameter SI.MassFlowRate fluidNomFlowRate=67.5 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure=37.16e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G=40 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_In=Tstart_F_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_Out=Tstart_F_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_In=FluidMedium.temperature_ph(fluidNomPressure,hstart_F_In) 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_Out=FluidMedium.temperature_ph(fluidNomPressure,hstart_F_Out) 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_In=9.920e5 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_Out=1.370e6 
          "Nominal specific enthalpy";
        
        Water.SourceW sourseW_water(redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate,
          p0=fluidNomPressure,
          h=hstart_F_In)      annotation (extent=[-20,60; 0,80],  rotation=270);
        Water.SinkP sinkP_water(redeclare package Medium = FluidMedium,
          p0=fluidNomPressure,
          h=hstart_F_Out) annotation (extent=[-20,-98; 0,-78],  rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[50,-12; 76,12]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          w0=gasNomFlowRate,
          T=Tstart_G_In)     annotation (extent=[-90,-14; -68,14]);
        annotation (Diagram);
        Components.HE2ph hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=4,
          N_F=4,
          exchSurface_G=24402,
          exchSurface_F=1837.063,
          extSurfaceTub=2163.652,
          gasVol=10,
          fluidVol=12.400,
          metalVol=4.801,
          rhomcm=7900*578.05,
          lambda=20,
          gasNomFlowRate=585.5,
          gasNomPressure=0,
          fluidNomFlowRate=fluidNomFlowRate,
          fluidNomPressure=fluidNomPressure,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_In=Tstart_M_In,
          Tstart_M_Out=Tstart_M_Out,
          Tstart_F_In=Tstart_F_In,
          Tstart_F_Out=Tstart_F_Out,
          gamma_G=gamma_G,
          gamma_F=gamma_F,
          SSInit=true) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SensT T_waterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,32; 4,52], rotation=270);
        Water.SensT T_waterOut(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,-62; 4,-42], rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 50,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -61,0; -61,1.77636e-015; -68,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(sinkP_water.flange, T_waterOut.outlet) annotation (points=[-10,-78;
              -10,-68; -10,-68; -10,-58],      style(thickness=2));
        connect(T_waterOut.inlet, hE.waterOut) 
          annotation (points=[-10,-46; -10,-20], style(thickness=2));
        connect(T_waterIn.inlet, sourseW_water.flange) annotation (points=[-10,48;
              -10,54; -10,60; -10,60],     style(thickness=2));
        connect(T_waterIn.outlet, hE.waterIn) 
          annotation (points=[-10,36; -10,20], style(thickness=2));
      end TestEvIP;
      
      model TestShIP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 "Nominal mass flowrate";
        parameter SI.Temperature Tstart_G_In=620.14 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=614.5 
          "Outlet gas temperature start value";
        
        //fluid
        parameter SI.MassFlowRate fluidNomFlowRate=13.5 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure=29.5e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G=33 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_In=Tstart_F_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_Out=Tstart_F_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_In=506.08 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_Out=605.11 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_Out) 
          "Nominal specific enthalpy";
        
        Water.SourceW sourseW_water(redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate,
          p0=fluidNomPressure,
          h=hstart_F_In)      annotation (extent=[-20,60; 0,80],  rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[50,-12; 76,12]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          w0=gasNomFlowRate,
          T=Tstart_G_In)     annotation (extent=[-90,-14; -68,14]);
        annotation (Diagram);
        Components.HE hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=3,
          N_F=7,
          exchSurface_G=2314.8,
          exchSurface_F=450.218,
          extSurfaceTub=504.652,
          gasVol=10,
          fluidVol=4.468,
          metalVol=1.146,
          rhomcm=7900*578.05,
          lambda=20,
          gasNomFlowRate=585.5,
          gasNomPressure=0,
          fluidNomFlowRate=fluidNomFlowRate,
          fluidNomPressure=fluidNomPressure,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_In=Tstart_M_In,
          Tstart_M_Out=Tstart_M_Out,
          Tstart_F_In=Tstart_F_In,
          Tstart_F_Out=Tstart_F_Out,
          gamma_G=gamma_G,
          gamma_F=gamma_F,
          SSInit=true) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SensT T_waterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,32; 4,52], rotation=270);
        Water.SensT T_waterOut(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,-62; 4,-42], rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
        Water.SinkP OutIP(redeclare package Medium = FluidMedium, p0=0) 
                           annotation (extent=[-42,-88; -54,-76],
                                                                rotation=0);
        Water.ValveLin valveLinIP(redeclare package Medium = FluidMedium, Kv=
              fluidNomFlowRate/fluidNomPressure) 
                                 annotation (extent=[-20,-88; -32,-76]);
        Modelica.Blocks.Sources.Constant const 
          annotation (extent=[-86,-50; -76,-40]);
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 50,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -61,0; -61,1.77636e-015; -68,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_waterOut.inlet, hE.waterOut) 
          annotation (points=[-10,-46; -10,-20], style(thickness=2));
        connect(T_waterIn.inlet, sourseW_water.flange) annotation (points=[-10,48;
              -10,54; -10,60; -10,60],     style(thickness=2));
        connect(T_waterIn.outlet, hE.waterIn) 
          annotation (points=[-10,36; -10,20], style(thickness=2));
        connect(valveLinIP.outlet,OutIP. flange) 
          annotation (points=[-32,-82; -42,-82],
                                               style(thickness=2));
        connect(valveLinIP.inlet, T_waterOut.outlet) 
          annotation (points=[-20,-82; -10,-82; -10,-58], style(thickness=2));
        connect(const.y, valveLinIP.cmd) annotation (points=[-75.5,-45; -26,-45;
              -26,-77.2], style(color=74, rgbcolor={0,0,127}));
      end TestShIP;
      
      model TestEc2HP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 "Nominal mass flowrate";
        parameter SI.Temperature Tstart_G_In=614.15 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=558.98 
          "Outlet gas temperature start value";
        
        //fluid
        parameter SI.MassFlowRate fluidNomFlowRate=70.1 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure=138.6e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G=60 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_In=Tstart_F_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_Out=Tstart_F_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_In=484.77 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_Out=594.14 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure,Tstart_F_Out) 
          "Nominal specific enthalpy";
        
        Water.SourceW sourseW_water(redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate,
          p0=fluidNomPressure,
          h=hstart_F_In)      annotation (extent=[-20,60; 0,80],  rotation=270);
        Water.SinkP sinkP_water(redeclare package Medium = FluidMedium,
          p0=fluidNomPressure,
          h=hstart_F_Out) annotation (extent=[-20,-98; 0,-78],  rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[50,-12; 76,12]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          w0=gasNomFlowRate,
          T=Tstart_G_In)     annotation (extent=[-90,-14; -68,14]);
        annotation (Diagram);
        Components.HE hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=3,
          N_F=6,
          exchSurface_G=20335,
          exchSurface_F=1451.506,
          extSurfaceTub=1803.043,
          gasVol=10,
          fluidVol=9.290,
          metalVol=5.045,
          rhomcm=7900*578.05,
          lambda=20,
          gasNomFlowRate=585.5,
          gasNomPressure=0,
          fluidNomFlowRate=fluidNomFlowRate,
          fluidNomPressure=fluidNomPressure,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_In=Tstart_M_In,
          Tstart_M_Out=Tstart_M_Out,
          Tstart_F_In=Tstart_F_In,
          Tstart_F_Out=Tstart_F_Out,
          gamma_G=gamma_G,
          gamma_F=gamma_F,
          SSInit=true) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SensT T_waterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,32; 4,52], rotation=270);
        Water.SensT T_waterOut(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,-62; 4,-42], rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 50,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -61,0; -61,1.77636e-015; -68,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(sinkP_water.flange, T_waterOut.outlet) annotation (points=[-10,-78;
              -10,-68; -10,-68; -10,-58],      style(thickness=2));
        connect(T_waterOut.inlet, hE.waterOut) 
          annotation (points=[-10,-46; -10,-20], style(thickness=2));
        connect(T_waterIn.inlet, sourseW_water.flange) annotation (points=[-10,48;
              -10,54; -10,60; -10,60],     style(thickness=2));
        connect(T_waterIn.outlet, hE.waterIn) 
          annotation (points=[-10,36; -10,20], style(thickness=2));
      end TestEc2HP;
      
      model TestEvHP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 "Nominal mass flowrate";
        parameter SI.Temperature Tstart_G_In=747.15 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=620.3 
          "Outlet gas temperature start value";
        
        //fluid
        parameter SI.MassFlowRate fluidNomFlowRate=175.5 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure=137.1e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G=40 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_In=Tstart_F_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_Out=Tstart_F_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_In=FluidMedium.temperature_ph(fluidNomPressure,hstart_F_In) 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_Out=FluidMedium.temperature_ph(fluidNomPressure,hstart_F_Out) 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_In=1.514e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_Out=2.002e6 
          "Nominal specific enthalpy";
        
        Water.SourceW sourseW_water(redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate,
          p0=fluidNomPressure,
          h=hstart_F_In)      annotation (extent=[-20,60; 0,80],  rotation=270);
        Water.SinkP sinkP_water(redeclare package Medium = FluidMedium,
          p0=fluidNomPressure,
          h=hstart_F_Out) annotation (extent=[-20,-98; 0,-78],  rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[50,-12; 76,12]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          w0=gasNomFlowRate,
          T=Tstart_G_In)     annotation (extent=[-90,-14; -68,14]);
        annotation (Diagram);
        Components.HE2ph hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=4,
          N_F=4,
          exchSurface_G=30501.9,
          exchSurface_F=2296.328,
          extSurfaceTub=2704.564,
          gasVol=10,
          fluidVol=15.500,
          metalVol=6.001,
          rhomcm=7900*612.58,
          lambda=20,
          gasNomFlowRate=585.5,
          gasNomPressure=0,
          fluidNomFlowRate=fluidNomFlowRate,
          fluidNomPressure=fluidNomPressure,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_In=Tstart_M_In,
          Tstart_M_Out=Tstart_M_Out,
          Tstart_F_In=Tstart_F_In,
          Tstart_F_Out=Tstart_F_Out,
          gamma_G=gamma_G,
          gamma_F=gamma_F,
          SSInit=true) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SensT T_waterIn(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,32; 4,52], rotation=270);
        Water.SensT T_waterOut(redeclare package Medium = FluidMedium) 
          annotation (extent=[-16,-62; 4,-42], rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 50,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -61,0; -61,1.77636e-015; -68,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(sinkP_water.flange, T_waterOut.outlet) annotation (points=[-10,-78;
              -10,-68; -10,-68; -10,-58],      style(thickness=2));
        connect(T_waterOut.inlet, hE.waterOut) 
          annotation (points=[-10,-46; -10,-20], style(thickness=2));
        connect(T_waterIn.inlet, sourseW_water.flange) annotation (points=[-10,48;
              -10,54; -10,60; -10,60],     style(thickness=2));
        connect(T_waterIn.outlet, hE.waterIn) 
          annotation (points=[-10,36; -10,20], style(thickness=2));
      end TestEvHP;
      
      model TestSh1HP_Rh1IP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 
          "Nominal flow rate through the gas side";
        parameter SI.Temperature Tstart_G_In=837.15 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=748.6 
          "Outlet gas temperature start value";
        
        //fluid A
        parameter SI.MassFlowRate fluidNomFlowRate_A=70.59 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure_A=134.3e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G_A=40 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F_A=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_A_In=Tstart_F_A_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_A_Out=Tstart_F_A_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_A_In=606.59 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_A_Out=743.18 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_A_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,Tstart_F_A_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_A_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,Tstart_F_A_Out) 
          "Nominal specific enthalpy";
        
        //fluid B
        parameter SI.MassFlowRate fluidNomFlowRate_B=81.1 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure_B=28.4e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G_B=40 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F_B=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_B_In=Tstart_F_B_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_B_Out=Tstart_F_B_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_B_In=618.31 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_B_Out=717.34 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_B_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,Tstart_F_B_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_B_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,Tstart_F_B_Out) 
          "Nominal specific enthalpy";
        
        Water.SourceW sourseW_water_A(
          redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate_A,
          p0=fluidNomPressure_A,
          h=hstart_F_A_In)    annotation (extent=[-38,64; -18,84],rotation=270);
        Water.SinkP sinkP_water_A(
          redeclare package Medium = FluidMedium,
          p0=fluidNomPressure_A,
          h=hstart_F_A_Out) 
                          annotation (extent=[-38,-96; -18,-76],rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[50,-12; 76,12]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          T=Tstart_G_In,
          w0=gasNomFlowRate) annotation (extent=[-90,-14; -68,14]);
        annotation (Diagram);
        Components.ParHE hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=3,
          gasVol=10,
          lambda=20,
          gasNomPressure=0,
          N_F_A=7,
          exchSurface_G_A=8137.2,
          exchSurface_F_A=612.387,
          extSurfaceTub_A=721.256,
          fluidVol_A=4.134,
          metalVol_A=1.600,
          rhomcm_A=7800*575.22,
          N_F_B=5,
          exchSurface_F_B=4630,
          exchSurface_G_B=900.387,
          extSurfaceTub_B=1009.250,
          fluidVol_B=8.936,
          metalVol_B=2.292,
          rhomcm_B=7850*575.64,
          SSInit=true,
          counterCurrent_A=true,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate_A=fluidNomFlowRate_A,
          fluidNomFlowRate_B=fluidNomFlowRate_B,
          fluidNomPressure_A=fluidNomPressure_A,
          fluidNomPressure_B=fluidNomPressure_B,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_A_In=Tstart_M_A_In,
          Tstart_M_A_Out=Tstart_M_A_Out,
          Tstart_F_A_In=Tstart_F_A_In,
          Tstart_F_A_Out=Tstart_F_A_Out,
          Tstart_M_B_In=Tstart_M_B_In,
          Tstart_M_B_Out=Tstart_M_B_Out,
          Tstart_F_B_In=Tstart_F_B_In,
          Tstart_F_B_Out=Tstart_F_B_Out,
          gamma_G_A=gamma_G_A,
          gamma_G_B=gamma_G_B,
          gamma_F_A=gamma_F_A,
          gamma_F_B=gamma_F_B,
          counterCurrent_B=true) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SinkP sinkP_water_B(
          redeclare package Medium = FluidMedium,
          p0=fluidNomPressure_B,
          h=hstart_F_B_Out) 
                          annotation (extent=[-2,-96; 18,-76],  rotation=270);
        Water.SourceW sourseW_water_B(
          redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate_B,
          p0=fluidNomPressure_B,
          h=hstart_F_B_In)    annotation (extent=[-4,64; 16,84],  rotation=270);
        Water.SensT T_waterIn_A(redeclare package Medium = FluidMedium) 
          annotation (extent=[-34,28; -14,48],
                                             rotation=270);
        Water.SensT T_waterOut_A(redeclare package Medium = FluidMedium) 
          annotation (extent=[-34,-52; -14,-32],
                                               rotation=270);
        Water.SensT T_waterOut_B(redeclare package Medium = FluidMedium) 
          annotation (extent=[2,-52; 22,-32],  rotation=270);
        Water.SensT T_waterIn_B(redeclare package Medium = FluidMedium) 
          annotation (extent=[0,28; 20,48],  rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 50,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -61,0; -61,1.77636e-015; -68,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_waterOut_A.outlet, sinkP_water_A.flange) annotation (points=[-28,-48;
              -28,-62; -28,-76; -28,-76],          style(thickness=2));
        connect(sinkP_water_B.flange, T_waterOut_B.outlet) annotation (points=[8,-76; 8,
              -69; 8,-69; 8,-62; 8,-48; 8,-48],              style(thickness=2));
        connect(T_waterOut_B.inlet, hE.waterOutB) annotation (points=[8,-36; 8,
              -26; -2,-26; -2,-20], style(thickness=2));
        connect(T_waterOut_A.inlet, hE.waterOutA) annotation (points=[-28,-36;
              -28,-28; -18,-28; -18,-20], style(thickness=2));
        connect(hE.waterInA, T_waterIn_A.outlet) annotation (points=[-18,20;
              -18,24; -28,24; -28,32], style(thickness=2));
        connect(hE.waterInB, T_waterIn_B.outlet) 
          annotation (points=[-2,20; -2,24; 6,24; 6,32], style(thickness=2));
        connect(T_waterIn_B.inlet, sourseW_water_B.flange) 
          annotation (points=[6,44; 6,49; 6,49; 6,54; 6,64; 6,64],
                                                       style(thickness=2));
        connect(T_waterIn_A.inlet, sourseW_water_A.flange) annotation (points=[-28,44;
              -28,54; -28,64; -28,64],         style(thickness=2));
      end TestSh1HP_Rh1IP;
      
      model TestSh2HP_Rh2IP 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        //gas
        parameter SI.MassFlowRate gasNomFlowRate=585.5 
          "Nominal flow rate through the gas side";
        parameter SI.Temperature Tstart_G_In=884.65 
          "Inlet gas temperature start value";
        parameter SI.Temperature Tstart_G_Out=823.15 
          "Outlet gas temperature start value";
        
        //fluid A
        parameter SI.MassFlowRate fluidNomFlowRate_A=70.59 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure_A=130.5e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G_A=40 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F_A=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_A_In=Tstart_F_A_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_A_Out=Tstart_F_A_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_A_In=769.486 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_A_Out=823.28 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_A_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,Tstart_F_A_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_A_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure_A,Tstart_F_A_Out) 
          "Nominal specific enthalpy";
        
        //fluid B
        parameter SI.MassFlowRate fluidNomFlowRate_B=81.1 
          "Nominal flow rate through the fluid side";
        parameter SI.Pressure fluidNomPressure_B=28.2e5 
          "Nominal pressure in the fluid side inlet";
        parameter SI.CoefficientOfHeatTransfer gamma_G_B=40 
          "Constant heat transfer coefficient in the gas side";
        parameter SI.CoefficientOfHeatTransfer gamma_F_B=4000 
          "Constant heat transfer coefficient in the fluid side";
        parameter SI.Temperature Tstart_M_B_In=Tstart_F_B_In 
          "Inlet metal wall temperature start value";
        parameter SI.Temperature Tstart_M_B_Out=Tstart_F_B_Out 
          "Outlet metal wall temperature start value";
        parameter SI.Temperature Tstart_F_B_In=757.896 
          "Inlet fluid temperature start value";
        parameter SI.Temperature Tstart_F_B_Out=813.09 
          "Outlet fluid temperature start value";
        parameter SI.SpecificEnthalpy hstart_F_B_In=FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,Tstart_F_B_In) 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy hstart_F_B_Out=FluidMedium.specificEnthalpy_pT(fluidNomPressure_B,Tstart_F_B_Out) 
          "Nominal specific enthalpy";
        
        Water.SourceW sourseW_water_A(
          redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate_A,
          p0=fluidNomPressure_A,
          h=hstart_F_A_In)    annotation (extent=[-38,64; -18,84],rotation=270);
        Water.SinkP sinkP_water_A(
          redeclare package Medium = FluidMedium,
          p0=fluidNomPressure_A,
          h=hstart_F_A_Out) 
                          annotation (extent=[-38,-96; -18,-76],rotation=270);
        Gas.SinkP sinkP_gas(redeclare package Medium = FlueGasMedium, T=Tstart_G_Out) 
                         annotation (extent=[50,-12; 76,12]);
        Gas.SourceW sourceW_gas(
          redeclare package Medium = FlueGasMedium,
          T=Tstart_G_In,
          w0=gasNomFlowRate) annotation (extent=[-90,-14; -68,14]);
        annotation (Diagram);
        Components.ParHE hE(redeclare package FluidMedium = FluidMedium,
          redeclare package FlueGasMedium = FlueGasMedium,
          N_G=3,
          gasVol=10,
          lambda=20,
          gasNomPressure=0,
          N_F_A=5,
          exchSurface_G_A=3636,
          exchSurface_F_A=421.844,
          extSurfaceTub_A=540.913,
          fluidVol_A=2.615,
          metalVol_A=1.685,
          rhomcm_A=7780*688.31,
          N_F_B=7,
          exchSurface_F_B=4630.2,
          exchSurface_G_B=873.079,
          extSurfaceTub_B=1009.143,
          fluidVol_B=8.403,
          metalVol_B=2.823,
          rhomcm_B=7800*574.93,
          SSInit=true,
          counterCurrent_A=true,
          gasNomFlowRate=gasNomFlowRate,
          fluidNomFlowRate_A=fluidNomFlowRate_A,
          fluidNomFlowRate_B=fluidNomFlowRate_B,
          fluidNomPressure_A=fluidNomPressure_A,
          fluidNomPressure_B=fluidNomPressure_B,
          Tstart_G_In=Tstart_G_In,
          Tstart_G_Out=Tstart_G_Out,
          Tstart_M_A_In=Tstart_M_A_In,
          Tstart_M_A_Out=Tstart_M_A_Out,
          Tstart_F_A_In=Tstart_F_A_In,
          Tstart_F_A_Out=Tstart_F_A_Out,
          Tstart_M_B_In=Tstart_M_B_In,
          Tstart_M_B_Out=Tstart_M_B_Out,
          Tstart_F_B_In=Tstart_F_B_In,
          Tstart_F_B_Out=Tstart_F_B_Out,
          gamma_G_A=gamma_G_A,
          gamma_G_B=gamma_G_B,
          gamma_F_A=gamma_F_A,
          gamma_F_B=gamma_F_B,
          counterCurrent_B=true) 
          annotation (extent=[-30,-20; 10,20]);
        Water.SinkP sinkP_water_B(
          redeclare package Medium = FluidMedium,
          p0=fluidNomPressure_B,
          h=hstart_F_B_Out) 
                          annotation (extent=[-2,-96; 18,-76],  rotation=270);
        Water.SourceW sourseW_water_B(
          redeclare package Medium = FluidMedium,
          w0=fluidNomFlowRate_B,
          p0=fluidNomPressure_B,
          h=hstart_F_B_In)    annotation (extent=[-4,64; 16,84],  rotation=270);
        Water.SensT T_waterIn_A(redeclare package Medium = FluidMedium) 
          annotation (extent=[-34,28; -14,48],
                                             rotation=270);
        Water.SensT T_waterOut_A(redeclare package Medium = FluidMedium) 
          annotation (extent=[-34,-52; -14,-32],
                                               rotation=270);
        Water.SensT T_waterOut_B(redeclare package Medium = FluidMedium) 
          annotation (extent=[2,-52; 22,-32],  rotation=270);
        Water.SensT T_waterIn_B(redeclare package Medium = FluidMedium) 
          annotation (extent=[0,28; 20,48],  rotation=270);
        Gas.SensT T_gasIn(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[-58,-6; -38,14]);
        Gas.SensT T_gasOut(redeclare package Medium = FlueGasMedium) 
          annotation (extent=[20,-6; 40,14]);
        
      equation 
        connect(T_gasOut.inlet, hE.gasOut) annotation (points=[24,0; 10,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasOut.outlet, sinkP_gas.flange) annotation (points=[36,0; 50,
              0], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.outlet, hE.gasIn) annotation (points=[-42,0; -30,0],
            style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_gasIn.inlet, sourceW_gas.flange) annotation (points=[-54,0;
              -61,0; -61,1.77636e-015; -68,1.77636e-015], style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(T_waterOut_A.outlet, sinkP_water_A.flange) annotation (points=[-28,-48;
              -28,-62; -28,-76; -28,-76],          style(thickness=2));
        connect(sinkP_water_B.flange, T_waterOut_B.outlet) annotation (points=[8,-76; 8,
              -69; 8,-69; 8,-62; 8,-48; 8,-48],              style(thickness=2));
        connect(T_waterOut_B.inlet, hE.waterOutB) annotation (points=[8,-36; 8,
              -26; -2,-26; -2,-20], style(thickness=2));
        connect(T_waterOut_A.inlet, hE.waterOutA) annotation (points=[-28,-36;
              -28,-28; -18,-28; -18,-20], style(thickness=2));
        connect(hE.waterInA, T_waterIn_A.outlet) annotation (points=[-18,20;
              -18,24; -28,24; -28,32], style(thickness=2));
        connect(hE.waterInB, T_waterIn_B.outlet) 
          annotation (points=[-2,20; -2,24; 6,24; 6,32], style(thickness=2));
        connect(T_waterIn_B.inlet, sourseW_water_B.flange) 
          annotation (points=[6,44; 6,49; 6,49; 6,54; 6,64; 6,64],
                                                       style(thickness=2));
        connect(T_waterIn_A.inlet, sourseW_water_A.flange) annotation (points=[-28,44;
              -28,54; -28,64; -28,64],         style(thickness=2));
      end TestSh2HP_Rh2IP;
      
      model TestDG_3LCC 
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        ThermoPower.PowerPlants.HRSG.Examples.DG_3L_CC Drums(
          HPd_rint=1.067,
          HPd_rext=1.167,
          HPd_L=11.920,
          HPd_Cm=0,
          IPd_rint=0.915,
          IPd_rext=1.015,
          IPd_L=7,
          IPd_Cm=0,
          LPd_rint=1.143,
          LPd_rext=1.243,
          LPd_L=11.503,
          LPd_Cm=0,
          RiserHPFlowRate=175.5,
          RiserIPFlowRate=67.5,
          RiserLPFlowRate=41.5,
          redeclare package FluidMedium = FluidMedium,
          fluidLPNomPressure=7.19048e5,
          LPd_hvstart=2.76051e6,
          LPd_hlstart=6.68356e5,
          fluidHPNomPressure=1.626e7,
          fluidIPNomPressure=3.3816e6,
          HPd_hvstart=2.53239e6,
          HPd_hlstart=1.5879e6,
          IPd_hvstart=2.794e6,
          IPd_hlstart=1.0265e5,
          SSInit=false)         annotation (extent=[-180,20; 20,140]);
        annotation (Diagram, Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        Water.SourceW sourceFromEcLP_w(
          h=6.544e5,
          w0=89.8,
          p0=7.19048e5) 
                     annotation (extent=[-6,-30; -18,-18],  rotation=90);
        Water.SinkP sinkToRiserLP_p(p0=1.534e6, h=1.6325e6) 
          annotation (extent=[-26,-48; -14,-36], rotation=270);
        Water.SourceW sourceFromRiserLP_w(
          w0=41.49,
          h=1.065e6,
          p0=7.19046e5) 
                     annotation (extent=[-22,-30; -34,-18], rotation=90);
        Water.SinkP sinkToShLP_p(          h=2.760e6,
          redeclare package Medium = FluidMedium,
          p0=5.398e3) 
          annotation (extent=[-2,-66; 10,-54]);
        Water.SinkP sinkToEcHP_p(p0=1.389e7, h=6.775e5) 
          annotation (extent=[-130,-162; -118,-150],
                                                 rotation=270);
        Water.SourceW sourceFromEcHP_w(
          w0=69.5,
          h=1.463e6,
          p0=1.626e7) 
                     annotation (extent=[-126,-148; -138,-136],
                                                            rotation=90);
        Water.SinkP sinkToRiserHP_p(p0=1.371e7, h=1.6325e6) 
          annotation (extent=[-146,-162; -134,-150],
                                                 rotation=270);
        Water.SourceW sourceFromRiserHP_w(
          w0=175.5,
          h=2.002e6,
          p0=1.626e7) 
                     annotation (extent=[-142,-148; -154,-136],
                                                            rotation=90);
        Water.SinkP sinkToShHP_p(            h=2.657e6,
          redeclare package Medium = FluidMedium,
          p0=2.98e6) 
          annotation (extent=[-126,-178; -114,-166],
                                                 rotation=0);
        Water.SinkP sinkToEcIP_p(p0=4.86e6, h=6.654e5) 
          annotation (extent=[-70,-110; -58,-98],
                                                rotation=270);
        Water.SourceW sourceFromEcIP_w(
          h=9.444e5,
          w0=12.7,
          p0=3.3816e6) 
                     annotation (extent=[-66,-96; -78,-84], rotation=90);
        Water.SinkP sinkToRiserIP_p(p0=3.716e6, h=1.0223e6) 
          annotation (extent=[-86,-110; -74,-98],rotation=270);
        Water.SourceW sourceFromRiserIP_w(
          w0=67.5,
          h=1.370e6,
          p0=3.3816e6) 
                     annotation (extent=[-82,-96; -94,-84], rotation=90);
        Water.SinkP sinkToShIP_p(           h=2.802e6,
          p0=6e5,
          redeclare package Medium = FluidMedium) 
          annotation (extent=[-62,-126; -50,-114],
                                                 rotation=0);
        Water.ValveLin valveLP(redeclare package Medium = FluidMedium, Kv=6.9/6.738e5) 
          annotation (extent=[-28,-52; -12,-68]);
        Water.ValveLin valveIP(redeclare package Medium = FluidMedium, Kv=13.5/
              2.959e6) annotation (extent=[-88,-112; -72,-128]);
        Water.ValveLin valveHP(redeclare package Medium = FluidMedium, Kv=70/1.344e7) 
          annotation (extent=[-152,-164; -136,-180]);
      public 
        Modelica.Blocks.Sources.Constant com_valve(k=1) 
          annotation (extent=[60,-196; 40,-176]);
      public 
        Components.StateReader_water stateToEcHP(
                                           redeclare package Medium = 
              FluidMedium) annotation (extent=[-130,-106; -118,-118],
                                                                  rotation=90);
        Components.StateReader_water stateReturnEcHP(
                                               redeclare package Medium = 
              FluidMedium) annotation (extent=[-138,-118; -126,-106],
                                                                  rotation=90);
        Components.StateReader_water stateToRiserHP(
                                              redeclare package Medium = 
              FluidMedium) annotation (extent=[-146,-106; -134,-118],
                                                                  rotation=90);
        Components.StateReader_water stateReturnRiserHP(
                                                  redeclare package Medium = 
              FluidMedium) annotation (extent=[-154,-118; -142,-106],
                                                                  rotation=90);
        Components.StateReader_water stateToShHP(
                                           redeclare package Medium = 
              FluidMedium) annotation (extent=[-162,-106; -150,-118],
                                                                  rotation=90);
        Components.StateReader_water stateToEcIP(
                                           redeclare package Medium = 
              FluidMedium) annotation (extent=[-70,-50; -58,-62],
                                                                rotation=90);
        Components.StateReader_water stateReturnEcIP(
                                               redeclare package Medium = 
              FluidMedium) annotation (extent=[-78,-62; -66,-50],rotation=90);
        Components.StateReader_water stateToRiserIP(
                                              redeclare package Medium = 
              FluidMedium) annotation (extent=[-86,-50; -74,-62], rotation=90);
        Components.StateReader_water stateReturnRiserIP(
                                                  redeclare package Medium = 
              FluidMedium) annotation (extent=[-94,-62; -82,-50], rotation=90);
        Components.StateReader_water stateToShIP(
                                           redeclare package Medium = 
              FluidMedium) annotation (extent=[-102,-50; -90,-62],rotation=90);
        Components.StateReader_water stateReturnEcLP(
                                               redeclare package Medium = 
              FluidMedium) annotation (extent=[-18,-10; -6,2],rotation=90);
        Components.StateReader_water stateToRiserLP(
                                              redeclare package Medium = 
              FluidMedium) annotation (extent=[-14,2; -26,-10],
                                                              rotation=90);
        Components.StateReader_water stateReturnRiserLP(
                                                  redeclare package Medium = 
              FluidMedium) annotation (extent=[-34,-10; -22,2],
                                                              rotation=90);
        Components.StateReader_water stateToShLP(
                                           redeclare package Medium = 
              FluidMedium) annotation (extent=[-30,2; -42,-10],
                                                              rotation=90);
      public 
        Modelica.Blocks.Sources.Ramp IPpump_com(
          duration=10,
          offset=12.7,
          height=0) 
          annotation (extent=[160,-114; 140,-94]);
        Modelica.Blocks.Sources.Ramp HPpump_com(
          duration=10,
          startTime=0,
          offset=69.5,
          height=0) 
          annotation (extent=[160,-154; 140,-134]);
      public 
        Modelica.Blocks.Sources.Constant LPpump_com(k=89.8) 
          annotation (extent=[160,-80; 140,-60]);
      protected 
        Buses.Actuators actuators annotation (extent=[92,78; 80,90]);
      public 
        Modelica.Blocks.Sources.Constant LPpump_com1(k=1500) 
          annotation (extent=[160,-20; 140,0]);
      equation 
        connect(IPpump_com.y,actuators. flowRate_feedIP) 
                                                     annotation (points=[139,-104;
              110,-104; 110,84; 86,84],
                              style(color=74, rgbcolor={0,0,127}));
        connect(HPpump_com.y,actuators. flowRate_feedHP) 
                                                     annotation (points=[139,-144;
              100,-144; 100,84; 86,84],
                                     style(color=74, rgbcolor={0,0,127}));
        connect(stateReturnEcHP.outlet, Drums.Feed_HP) annotation (points=[-132,
              -108.4; -132,20],              style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateToEcHP.inlet, Drums.WaterForHP) annotation (points=[-124,
              -108.4; -124,20],              style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateToShHP.inlet, Drums.Steam_HP_Out) annotation (points=[-156,
              -108.4; -156,20],              style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateReturnEcIP.outlet, Drums.Feed_IP) annotation (points=[-72,
              -52.4; -72,20], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateToEcIP.inlet, Drums.WaterForIP) annotation (points=[-64,
              -52.4; -64,20], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateToShIP.inlet, Drums.Steam_IP_Out) annotation (points=[-96,
              -52.4; -96,20],  style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sinkToShLP_p.flange, valveLP.outlet) annotation (points=[-2,-60;
              -12,-60],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(valveLP.inlet, stateToShLP.outlet) annotation (points=[-28,-60;
              -36,-60; -36,-7.6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sinkToRiserLP_p.flange, stateToRiserLP.outlet) annotation (
            points=[-20,-36; -20,-7.6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sourceFromRiserLP_w.flange, stateReturnRiserLP.inlet) 
          annotation (points=[-28,-18; -28,-7.6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sourceFromEcLP_w.flange, stateReturnEcLP.inlet) annotation (
            points=[-12,-18; -12,-7.6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateReturnEcLP.outlet, Drums.Feed_LP) annotation (points=[-12,-0.4;
              -12,20],                    style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateToShLP.inlet, Drums.Steam_LP_Out) annotation (points=[-36,-0.4;
              -36,20],                    style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(actuators, Drums.ActuatorsBus) annotation (points=[86,84; 20,84],
                                            style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sinkToEcIP_p.flange, stateToEcIP.outlet) annotation (points=[-64,-98;
              -64,-59.6],          style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sourceFromEcIP_w.flange, stateReturnEcIP.inlet) annotation (
            points=[-72,-84; -72,-59.6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sinkToRiserIP_p.flange, stateToRiserIP.outlet) annotation (
            points=[-80,-98; -80,-59.6],   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sourceFromRiserIP_w.flange, stateReturnRiserIP.inlet) 
          annotation (points=[-88,-84; -88,-59.6],   style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(valveIP.inlet, stateToShIP.outlet) annotation (points=[-88,-120;
              -96,-120; -96,-59.6],         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sinkToShIP_p.flange, valveIP.outlet) annotation (points=[-62,-120;
              -72,-120],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sinkToEcHP_p.flange, stateToEcHP.outlet) annotation (points=[-124,
              -150; -124,-115.6],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sourceFromEcHP_w.flange, stateReturnEcHP.inlet) annotation (
            points=[-132,-136; -132,-115.6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sinkToRiserHP_p.flange, stateToRiserHP.outlet) annotation (
            points=[-140,-150; -140,-115.6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sourceFromRiserHP_w.flange, stateReturnRiserHP.inlet) 
          annotation (points=[-148,-136; -148,-115.6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(valveHP.inlet, stateToShHP.outlet) annotation (points=[-152,
              -172; -156,-172; -156,-115.6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(valveHP.outlet, sinkToShHP_p.flange) annotation (points=[-136,
              -172; -126,-172], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(com_valve.y, valveHP.cmd) annotation (points=[39,-186; -144,
              -186; -144,-178.4], style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(com_valve.y, valveIP.cmd) annotation (points=[39,-186; -80,-186;
              -80,-126.4],        style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(com_valve.y, valveLP.cmd) annotation (points=[39,-186; -20,-186;
              -20,-66.4], style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(HPpump_com.y, sourceFromEcHP_w.in_w0) annotation (points=[139,-144;
              -128.4,-144; -128.4,-144.4],       style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(IPpump_com.y, sourceFromEcIP_w.in_w0) annotation (points=[139,-104;
              39.5,-104; 39.5,-92.4; -68.4,-92.4],       style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(LPpump_com.y, sourceFromEcLP_w.in_w0) annotation (points=[139,-70;
              72,-70; 72,-26.4; -8.4,-26.4],       style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(LPpump_com1.y, actuators.nPump_feedLP) annotation (points=[139,-10;
              110,-10; 110,84; 86,84],        style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateToRiserLP.inlet, Drums.Downcomer_LP) annotation (points=[
              -20,-0.4; -20,20], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateReturnRiserLP.outlet, Drums.Riser_LP) annotation (points=[
              -28,-0.4; -28,20; -28,20], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateReturnRiserIP.outlet, Drums.Riser_IP) annotation (points=[
              -88,-52.4; -88,20; -88,20], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateToRiserIP.inlet, Drums.Downcomer_IP) annotation (points=[
              -80,-52.4; -80,20; -80,20], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateToRiserHP.inlet, Drums.Downcomer_HP) annotation (points=[
              -140,-108.4; -140,20], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateReturnRiserHP.outlet, Drums.Riser_HP) annotation (points=[
              -148,-108.4; -148,20], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end TestDG_3LCC;
      
      model TestHEG_3LRh 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        //HP
        parameter SI.MassFlowRate fluidHPNomFlowRate_Sh=67.6 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidHPNomPressure_Sh=138.9e5 "Nominal pressure";
        parameter SI.MassFlowRate fluidHPNomFlowRate_Ev=175.5 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidHPNomPressure_Ev=138.9e5 "Nominal pressure";
        parameter SI.MassFlowRate fluidHPNomFlowRate_Ec=67.6 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidHPNomPressure_Ec=138.9e5 "Nominal pressure";
        parameter SI.SpecificEnthalpy Sh1_HP_hstart_F_In= 2.657e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ev_HP_hstart_F_In= 1.514e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ev_HP_hstart_F_Out= 1.9423e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ec1_HP_hstart_F_In= 6.775e5 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ec2_HP_hstart_F_Out= 1.45077e6 
          "Nominal specific enthalpy";
        
        //IP
        parameter SI.MassFlowRate fluidIPNomFlowRate_Rh=81.10 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidIPNomPressure_Rh=33.8e5 "Nominal pressure";
        parameter SI.MassFlowRate fluidIPNomFlowRate_Sh=13.5 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidIPNomPressure_Sh=33.8e5 "Nominal pressure";
        parameter SI.MassFlowRate fluidIPNomFlowRate_Ev=67.5 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidIPNomPressure_Ev=33.8e5 "Nominal pressure";
        parameter SI.MassFlowRate fluidIPNomFlowRate_Ec=13.5 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidIPNomPressure_Ec=33.8e5 "Nominal pressure";
        parameter SI.SpecificEnthalpy Rh1_IP_hstart_F_In= 3.22e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Rh2_IP_hstart_F_Out=3.64853e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Sh_IP_hstart_F_In=2.802e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ev_IP_hstart_F_In= 9.92e5 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ev_IP_hstart_F_Out= 1.36395e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ec_IP_hstart_F_In=6.654e5 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ec_IP_hstart_F_Out=9.70768e5 
          "Nominal specific enthalpy";
        
        //LP
        parameter SI.MassFlowRate fluidLPNomFlowRate_Sh=8.72 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidLPNomPressure_Sh=7.19e5 "Nominal pressure";
        parameter SI.MassFlowRate fluidLPNomFlowRate_Ev=41.49 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidLPNomPressure_Ev=7.19e5 "Nominal pressure";
        parameter SI.MassFlowRate fluidLPNomFlowRate_Ec=89.82 
          "Nominal mass flowrate";
        parameter SI.Pressure fluidLPNomPressure_Ec=7.19e5 "Nominal pressure";
        parameter SI.SpecificEnthalpy Sh_LP_hstart_F_In= 2.760e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ev_LP_hstart_F_In= 6.626e5 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ev_LP_hstart_F_Out=1.0227e6 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ec_LP_hstart_F_In=2.44e5 
          "Nominal specific enthalpy";
        parameter SI.SpecificEnthalpy Ec_LP_hstart_F_Out=6.39929e5 
          "Nominal specific enthalpy";
        
        Water.SinkP OutHP(redeclare package Medium = FluidMedium, p0=0) 
                          annotation (extent=[-88,-90; -100,-78],
                                                              rotation=0);
        Gas.SinkP sinkGas(redeclare package Medium = FlueGasMedium, T=375.386) 
                         annotation (extent=[88,-10; 100,2]);
        Gas.SourceW sourceGas(
          redeclare package Medium = FlueGasMedium,
          w0=585.5,
          T=884.65)          annotation (extent=[-100,-10; -88,2]);
        Water.SinkP sinkEvHP(
          redeclare package Medium = FluidMedium,
          p0=fluidHPNomPressure_Ev,
          h=Ev_HP_hstart_F_Out) 
                           annotation (extent=[-66,70; -54,82], rotation=90);
        Water.ValveLin valveLinHP(redeclare package Medium = FluidMedium, Kv=
              fluidHPNomFlowRate_Sh/fluidHPNomPressure_Sh) 
                                annotation (extent=[-68,-78; -80,-90]);
        Water.SinkP OutLP(redeclare package Medium = FluidMedium, p0=0) 
                           annotation (extent=[70,-92; 82,-80], rotation=0);
        Water.ValveLin valveLinLP(redeclare package Medium = FluidMedium, Kv=
              fluidLPNomFlowRate_Sh/fluidLPNomPressure_Sh) 
                                 annotation (extent=[50,-80; 62,-92]);
        annotation (Diagram, Icon);
        Water.SourceW sourceEvHP(
          redeclare package Medium = FluidMedium,
          w0=fluidHPNomFlowRate_Ev,
          p0=fluidHPNomPressure_Ev,
          h=Ev_HP_hstart_F_In) 
                              annotation (extent=[-58,58; -46,70], rotation=270);
        Water.SourceW sourceRhIP(
          redeclare package Medium = FluidMedium,
          w0=fluidIPNomFlowRate_Rh,
          p0=fluidIPNomPressure_Rh,
          h=Rh1_IP_hstart_F_In) 
                              annotation (extent=[-24,-58; -14,-48],
                                                                   rotation=90);
        Water.SinkP sinkRhIP(
          redeclare package Medium = FluidMedium,
          p0=fluidIPNomPressure_Rh,
          h=Rh2_IP_hstart_F_Out) 
                           annotation (extent=[-32,-64; -20,-76],
                                                                rotation=90);
        Modelica.Blocks.Sources.Constant commandValve 
          annotation (extent=[-50,-90; -40,-80], rotation=270);
        Water.SinkP sinkEcHP(
          redeclare package Medium = FluidMedium,
          p0=fluidHPNomPressure_Ec,
          h=Ec2_HP_hstart_F_Out) 
                           annotation (extent=[-50,70; -38,82], rotation=90);
        Water.SourceW sourceEcHP(
          redeclare package Medium = FluidMedium,
          w0=fluidHPNomFlowRate_Ec,
          p0=fluidHPNomPressure_Ec,
          h=Ec1_HP_hstart_F_In) 
                              annotation (extent=[-44,58; -32,70], rotation=270);
        Water.SinkP sinkEvIP(
          redeclare package Medium = FluidMedium,
          p0=fluidIPNomPressure_Ev,
          h=Ev_IP_hstart_F_Out) 
                           annotation (extent=[-16,70; -4,82],  rotation=90);
        Water.SourceW sourceEvIP(
          redeclare package Medium = FluidMedium,
          w0=fluidIPNomFlowRate_Ev,
          p0=fluidIPNomPressure_Ev,
          h=Ev_IP_hstart_F_In) 
                              annotation (extent=[-12,58; 0,70],   rotation=270);
        Water.SinkP sinkEcIP(
          redeclare package Medium = FluidMedium,
          p0=fluidIPNomPressure_Ec,
          h=Ec_IP_hstart_F_Out) 
                           annotation (extent=[-2,70; 10,82],   rotation=90);
        Water.SourceW sourceEcIP(
          redeclare package Medium = FluidMedium,
          w0=fluidIPNomFlowRate_Ec,
          p0=fluidIPNomPressure_Ec,
          h=Ec_IP_hstart_F_In) 
                              annotation (extent=[4,58; 16,70],    rotation=270);
        Water.SinkP sinkEvLP(
          redeclare package Medium = FluidMedium,
          p0=fluidLPNomPressure_Ev,
          h=Ev_LP_hstart_F_Out) 
                           annotation (extent=[38,70; 50,82],   rotation=90);
        Water.SourceW sourceEvLP(
          redeclare package Medium = FluidMedium,
          w0=fluidLPNomFlowRate_Ev,
          p0=fluidLPNomPressure_Ev,
          h=Ev_LP_hstart_F_In) 
                              annotation (extent=[46,70; 58,58],   rotation=90);
        Water.SinkP sinkEcLP(
          redeclare package Medium = FluidMedium,
          p0=fluidLPNomPressure_Ec,
          h=Ec_LP_hstart_F_Out) 
                           annotation (extent=[54,70; 66,82],   rotation=90);
        Water.SourceW sourceEcLP(
          redeclare package Medium = FluidMedium,
          w0=fluidLPNomFlowRate_Ec,
          p0=fluidLPNomPressure_Ec,
          h=Ec_LP_hstart_F_In) 
                              annotation (extent=[62,70; 74,58],   rotation=90);
        Water.SourceP sourceShHP(p0=fluidHPNomPressure_Sh, h=Sh1_HP_hstart_F_In) 
          annotation (extent=[-62,58; -74,70], rotation=270);
        Water.SourceW sourceShIP(p0=fluidIPNomPressure_Sh, h=Sh_IP_hstart_F_In,
          w0=fluidIPNomFlowRate_Sh) 
          annotation (extent=[-14,58; -26,70],
                                             rotation=270);
        Water.SourceP sourceShLP(p0=fluidLPNomPressure_Sh, h=Sh_LP_hstart_F_In) 
          annotation (extent=[28,70; 40,58], rotation=90);
        ThermoPower.PowerPlants.HRSG.Components.StateReader_gas stateGas_in(
                                               redeclare package Medium = 
              FlueGasMedium) annotation (extent=[-84,-12; -68,4], rotation=0);
        ThermoPower.PowerPlants.HRSG.Components.StateReader_gas stateGas_out(
                                                redeclare package Medium = 
              FlueGasMedium) annotation (extent=[64,-12; 80,4], rotation=0);
        Examples.HEG_3LRh hEG_3LRh_simp(
          gasNomFlowRate=585.5,
          fluidHPNomFlowRate_Sh=70.59,
          fluidHPNomPressure_Sh=134.3e5,
          fluidHPNomFlowRate_Ev=175.5,
          fluidHPNomPressure_Ev=137.1e5,
          fluidHPNomFlowRate_Ec=70.10,
          fluidHPNomPressure_Ec=138.9e5,
          fluidIPNomFlowRate_Rh=81.10,
          fluidIPNomPressure_Rh=28.4e5,
          fluidIPNomFlowRate_Sh=13.5,
          fluidIPNomPressure_Sh=29.5e5,
          fluidIPNomFlowRate_Ev=67.5,
          fluidIPNomPressure_Ev=37.16e5,
          fluidIPNomFlowRate_Ec=21.8,
          fluidIPNomPressure_Ec=48.6e5,
          fluidLPNomFlowRate_Sh=6.91,
          fluidLPNomPressure_Sh=6.6e5,
          fluidLPNomFlowRate_Ev=41.49,
          fluidLPNomPressure_Ev=15.34e5,
          fluidLPNomFlowRate_Ec=122.4,
          fluidLPNomPressure_Ec=19.8e5,
          Sh2_HP_N_G=3,
          Sh2_HP_N_F=5,
          Sh2_HP_exchSurface_G=3636,
          Sh2_HP_exchSurface_F=421.844,
          Sh2_HP_extSurfaceTub=540.913,
          Sh2_HP_gasVol=10,
          Sh2_HP_fluidVol=2.615,
          Sh2_HP_metalVol=1.685,
          Sh1_HP_N_G=3,
          Sh1_HP_N_F=7,
          Sh1_HP_exchSurface_G=8137.2,
          Sh1_HP_exchSurface_F=612.387,
          Sh1_HP_extSurfaceTub=721.256,
          Sh1_HP_gasVol=10,
          Sh1_HP_fluidVol=4.134,
          Sh1_HP_metalVol=1.600,
          Ev_HP_N_G=4,
          Ev_HP_N_F=4,
          Ev_HP_exchSurface_G=30501.9,
          Ev_HP_exchSurface_F=2296.328,
          Ev_HP_extSurfaceTub=2704.564,
          Ev_HP_gasVol=10,
          Ev_HP_fluidVol=15.500,
          Ev_HP_metalVol=6.001,
          Ec2_HP_N_G=3,
          Ec2_HP_N_F=6,
          Ec2_HP_exchSurface_G=20335,
          Ec2_HP_exchSurface_F=1451.506,
          Ec2_HP_extSurfaceTub=1803.043,
          Ec2_HP_gasVol=10,
          Ec2_HP_fluidVol=9.290,
          Ec2_HP_metalVol=5.045,
          Ec1_HP_N_G=3,
          Ec1_HP_N_F=5,
          Ec1_HP_exchSurface_G=12201.2,
          Ec1_HP_exchSurface_F=870.904,
          Ec1_HP_extSurfaceTub=1081.826,
          Ec1_HP_gasVol=10,
          Ec1_HP_fluidVol=5.574,
          Ec1_HP_metalVol=3.027,
          Rh2_IP_N_F=7,
          Rh2_IP_exchSurface_G=4630.2,
          Rh2_IP_exchSurface_F=873.079,
          Rh2_IP_extSurfaceTub=1009.143,
          Rh2_IP_fluidVol=8.403,
          Rh2_IP_metalVol=2.823,
          Rh1_IP_N_F=5,
          Rh1_IP_exchSurface_G=4630,
          Rh1_IP_exchSurface_F=900.387,
          Rh1_IP_extSurfaceTub=1009.250,
          Rh1_IP_fluidVol=8.936,
          Rh1_IP_metalVol=2.292,
          Sh_IP_N_G=3,
          Sh_IP_N_F=7,
          Sh_IP_exchSurface_G=2314.8,
          Sh_IP_exchSurface_F=450.218,
          Sh_IP_extSurfaceTub=504.652,
          Sh_IP_gasVol=10,
          Sh_IP_fluidVol=4.468,
          Sh_IP_metalVol=1.146,
          Ev_IP_N_G=4,
          Ev_IP_N_F=4,
          Ev_IP_exchSurface_G=24402,
          Ev_IP_exchSurface_F=1837.063,
          Ev_IP_extSurfaceTub=2163.652,
          Ev_IP_gasVol=10,
          Ev_IP_fluidVol=12.400,
          Ev_IP_metalVol=4.801,
          Ec_IP_N_G=3,
          Ec_IP_N_F=5,
          Ec_IP_exchSurface_G=4067.2,
          Ec_IP_exchSurface_F=306.177,
          Ec_IP_extSurfaceTub=360.609,
          Ec_IP_gasVol=10,
          Ec_IP_fluidVol=2.067,
          Ec_IP_metalVol=0.800,
          Sh_LP_N_G=3,
          Sh_LP_N_F=7,
          Sh_LP_exchSurface_G=1708.2,
          Sh_LP_exchSurface_F=225.073,
          Sh_LP_extSurfaceTub=252.286,
          Sh_LP_gasVol=10,
          Sh_LP_fluidVol=2.234,
          Sh_LP_metalVol=0.573,
          Ev_LP_N_G=4,
          Ev_LP_N_F=4,
          Ev_LP_exchSurface_G=24402,
          Ev_LP_exchSurface_F=2292.926,
          Ev_LP_extSurfaceTub=2592.300,
          Ev_LP_gasVol=10,
          Ev_LP_fluidVol=19.318,
          Ev_LP_metalVol=5.374,
          Ec_LP_N_G=3,
          Ec_LP_N_F=6,
          Ec_LP_exchSurface_G=40095.9,
          Ec_LP_exchSurface_F=3439.389,
          Ec_LP_extSurfaceTub=3888.449,
          Ec_LP_gasVol=10,
          Ec_LP_fluidVol=28.977,
          Ec_LP_metalVol=8.061,
          use_T_Ev=false,
          Sh2_HP_Tstart_G_In=884.65,
          Sh2_HP_Tstart_G_Out=837.31,
          Sh2_HP_Tstart_M_In=742.65,
          Sh2_HP_Tstart_M_Out=823.28,
          Sh2_HP_Tstart_F_In=742.65,
          Sh2_HP_Tstart_F_Out=823.28,
          Sh1_HP_Tstart_G_In=837.15,
          Sh1_HP_Tstart_G_Out=748.60,
          Sh1_HP_Tstart_M_In=606.59,
          Sh1_HP_Tstart_M_Out=743.18,
          Sh1_HP_Tstart_F_In=606.59,
          Sh1_HP_Tstart_F_Out=743.18,
          Ev_HP_Tstart_G_In=747.15,
          Ev_HP_Tstart_G_Out=620.30,
          Ev_HP_Tstart_M_In=FluidMedium.temperature_ph(fluidHPNomPressure_Ev, 1.514e6),
          Ev_HP_Tstart_M_Out=FluidMedium.temperature_ph(fluidHPNomPressure_Ev,
              2.002e6),
          Ev_HP_hstart_F_In=1.514e6,
          Ev_HP_hstart_F_Out=2.002e6,
          Ec2_HP_Tstart_G_In=614.15,
          Ec2_HP_Tstart_G_Out=558.98,
          Ec2_HP_Tstart_M_In=494.67,
          Ec2_HP_Tstart_M_Out=594.14,
          Ec2_HP_Tstart_F_In=494.67,
          Ec2_HP_Tstart_F_Out=594.14,
          Ec1_HP_Tstart_M_In=431.77,
          Ec1_HP_Tstart_M_Out=494.67,
          Ec1_HP_Tstart_F_In=431.77,
          Ec1_HP_Tstart_F_Out=494.67,
          Rh2_IP_Tstart_M_In=717.31,
          Rh2_IP_Tstart_M_Out=813.09,
          Rh2_IP_Tstart_F_In=717.31,
          Rh2_IP_Tstart_F_Out=813.09,
          Rh1_IP_Tstart_M_In=618.31,
          Rh1_IP_Tstart_M_Out=717.34,
          Rh1_IP_Tstart_F_In=618.31,
          Rh1_IP_Tstart_F_Out=717.34,
          Sh_IP_Tstart_G_In=620.15,
          Sh_IP_Tstart_G_Out=614.50,
          Sh_IP_Tstart_M_In=506.08,
          Sh_IP_Tstart_M_Out=605.11,
          Sh_IP_Tstart_F_In=506.08,
          Sh_IP_Tstart_F_Out=605.11,
          Ev_IP_Tstart_G_In=558.15,
          Ev_IP_Tstart_G_Out=518.51,
          Ev_IP_Tstart_M_In=FluidMedium.temperature_ph(fluidIPNomPressure_Ev, 9.92e5),
          Ev_IP_Tstart_M_Out=FluidMedium.temperature_ph(fluidIPNomPressure_Ev, 1.37e6),
          Ev_IP_hstart_F_In=9.92e5,
          Ev_IP_hstart_F_Out=1.37e6,
          Ec_IP_Tstart_G_In=517.15,
          Ec_IP_Tstart_G_Out=480.11,
          Ec_IP_Tstart_M_In=430.24,
          Ec_IP_Tstart_M_Out=493.16,
          Ec_IP_Tstart_F_In=430.24,
          Ec_IP_Tstart_F_Out=493.16,
          Sh_LP_Tstart_G_In=519.15,
          Sh_LP_Tstart_G_Out=517.44,
          Sh_LP_Tstart_M_In=435.75,
          Sh_LP_Tstart_M_Out=505.04,
          Sh_LP_Tstart_F_In=435.75,
          Sh_LP_Tstart_F_Out=505.75,
          Ev_LP_Tstart_G_In=480.00,
          Ev_LP_Tstart_G_Out=453.47,
          Ev_LP_Tstart_M_In=FluidMedium.temperature_ph(fluidLPNomPressure_Ev, 6.625e5),
          Ev_LP_Tstart_M_Out=FluidMedium.temperature_ph(fluidLPNomPressure_Ev,
              1.065e6),
          Ev_LP_hstart_F_In=6.625e5,
          Ev_LP_hstart_F_Out=1.065e6,
          Ec_LP_Tstart_G_In=453.47,
          Ec_LP_Tstart_G_Out=372.35,
          Ec_LP_Tstart_M_In=331.05,
          Ec_LP_Tstart_M_Out=428.09,
          Ec_LP_Tstart_F_In=331.05,
          Ec_LP_Tstart_F_Out=428.09,
          rhomcm=7900*578.05,
          lambda=20,
          Ec_LP(gamma_G=46.8, gamma_F=4000),
          Ev_LP(gamma_G=127, gamma_F=20000),
          Ec1HP_EcIP(
            gamma_G_A=42,
            gamma_G_B=45,
            gamma_F_A=4000,
            gamma_F_B=4000),
          Sh_LP(gamma_G=16.6, gamma_F=4000),
          Ev_IP(gamma_G=58.5, gamma_F=20000),
          Ec2_HP(gamma_G=56, gamma_F=4000),
          Sh_IP(gamma_G=33, gamma_F=4000),
          Ev_HP(gamma_G=46.5, gamma_F=20000),
          gasNomPressure=1e5,
          Sh1HP_Rh1IP(
            gamma_G_A=70,
            gamma_G_B=80,
            gamma_F_A=4000,
            gamma_F_B=4000),
          Sh2HP_Rh2IP(
            gamma_G_A=83.97,
            gamma_G_B=80,
            gamma_F_A=4000,
            gamma_F_B=4000),
          SSInit=true)       annotation (extent=[-60,-28; 50,20]);
      equation 
        connect(OutHP.flange, valveLinHP.outlet) 
          annotation (points=[-88,-84; -80,-84],style(thickness=2));
        connect(valveLinLP.outlet, OutLP.flange) 
          annotation (points=[62,-86; 70,-86], style(thickness=2));
        connect(commandValve.y, valveLinHP.cmd) annotation (points=[-45,-90.5;
              -45,-98; -74,-98; -74,-88.8], style(color=74, rgbcolor={0,0,127}));
        connect(commandValve.y, valveLinLP.cmd) annotation (points=[-45,-90.5;
              -45,-98; 56,-98; 56,-90.8], style(color=74, rgbcolor={0,0,127}));
        connect(stateGas_in.inlet, sourceGas.flange) annotation (points=[-80.8,-4;
              -88,-4],                style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=76,
            rgbfillColor={170,170,255},
            fillPattern=1));
        connect(sinkGas.flange, stateGas_out.outlet) annotation (points=[88,-4;
              76.8,-4],        style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=76,
            rgbfillColor={170,170,255},
            fillPattern=1));
        connect(stateGas_out.inlet, hEG_3LRh_simp.GasOut) annotation (points=[67.2,-4;
              50,-4],                   style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(stateGas_in.outlet, hEG_3LRh_simp.GasIn) annotation (points=[-71.2,-4;
              -60,-4],                     style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(sinkRhIP.flange, hEG_3LRh_simp.Rh_IP_Out) annotation (points=[-26,-64;
              -26,-48; -25.9,-48; -25.9,-28],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(sourceRhIP.flange, hEG_3LRh_simp.Rh_IP_In) annotation (points=[-19,-48;
              -19,-28; -19.3,-28],
                                 style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(valveLinHP.inlet, hEG_3LRh_simp.Sh_HP_Out) annotation (points=[-68,-84;
              -58,-84; -58,-54; -38,-54; -38,-28],          style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(valveLinLP.inlet, hEG_3LRh_simp.Sh_LP_Out) annotation (points=[50,-86;
              28,-86; 28,-27.76],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ev_IP_In, sourceEvIP.flange) annotation (points=[-5,20;
              -5,58; -6,58],                       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ec_IP_Out, sinkEcIP.flange) annotation (points=[-0.6,20;
              -0.6,52; 4,52; 4,70],  style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ev_IP_Out, sinkEvIP.flange) annotation (points=[-9.4,20;
              -9.4,52; -10,52; -10,70],    style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ec_IP_In, sourceEcIP.flange) annotation (points=[3.8,20;
              3.8,40; 10,40; 10,58],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Sh_IP_In, sourceShIP.flange) annotation (points=[-13.8,20;
              -13.8,40; -20,40; -20,58],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Sh_HP_In, sourceShHP.flange) annotation (points=[-46.8,20;
              -46.8,26; -68,26; -68,58],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ev_HP_Out, sinkEvHP.flange) annotation (points=[-42.4,20;
              -42.4,30; -60,30; -60,70],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ec_HP_Out, sinkEcHP.flange) annotation (points=[-33.6,20;
              -33.6,38; -44,38; -44,70],       style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ec_LP_In, sourceEcLP.flange) annotation (points=[36.8,20;
              36.8,26; 68,26; 68,58],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ec_LP_Out, sinkEcLP.flange) annotation (points=[32.4,20;
              32.4,30; 60,30; 60,70],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ev_LP_In, sourceEvLP.flange) annotation (points=[28,20;
              28,34; 52,34; 52,58],        style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ev_LP_Out, sinkEvLP.flange) annotation (points=[23.6,20;
              23.6,38; 44,38; 44,70],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Sh_LP_In, sourceShLP.flange) annotation (points=[19.2,20;
              19.2,42; 34,42; 34,58],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ev_HP_In, sourceEvHP.flange) annotation (points=[-38,20;
              -38,36; -52,36; -52,58],         style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(hEG_3LRh_simp.Ec_HP_In, sourceEcHP.flange) annotation (points=[-29.2,20;
              -29.2,44; -38,44; -38,58],           style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end TestHEG_3LRh;
      
    end Tests;
    
    package TestsControl "Closed-loop test cases" 
      model TestPumpControl 
        import ThermoPower;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        Water.SinkP sinkToEcLP_p(           h=2.440e5, p0=7.19e5) 
          annotation (extent=[60,-20; 80,0]);
        Water.SourceP sourceShLP(
          redeclare package Medium = FluidMedium,
          p0=5398.2,
          h=1.43495e5) 
          annotation (extent=[-60,0; -80,-20],
                                             rotation=180);
      public 
        Modelica.Blocks.Sources.Ramp n_pump(
          duration=100,
          offset=89.9,
          startTime=500,
          height=0) 
          annotation (extent=[64,34; 52,46]);
        ThermoPower.PowerPlants.Control.PID pIDController(
          CSmin=500,
          PVmin=-1,
          PVmax=1,
          CSmax=2500,
          Ti=200,
          Kp=4) annotation (extent=[12,46; 32,26],  rotation=180);
      public 
        ThermoPower.PowerPlants.HRSG.Components.PrescribedSpeedPump pump(
          rho_nom=1000,
          redeclare package WaterMedium = FluidMedium,
          q_nom={0.0898,0,0.1},
          head_nom={72.74,130,0},
          nominalSteamPressure=7.19048e5,
          nominalCondensationPressure=5398.2,
          nominalSteamFlow=89.8,
          hstart=1.43495e5,
          n0=1500,
          feedWaterPump(redeclare function efficiencyCharacteristic = 
                Functions.PumpCharacteristics.constantEfficiency (eta_nom=0.6))) 
                                annotation (extent=[0,0; -20,-20],   rotation=180);
        ThermoPower.Water.SensW feed_w(redeclare package Medium = FluidMedium) 
          annotation (extent=[36,4; 16,-16],      rotation=180);
        annotation (Diagram);
        Modelica.Blocks.Continuous.FirstOrder firstOrder(y_start=1512, T=1) 
          annotation (extent=[-4,26; -24,46], rotation=0);
      equation 
        connect(n_pump.y, pIDController.SP) annotation (points=[51.4,40; 32,
              40], style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(feed_w.inlet, pump.outlet)          annotation (points=[20,-10;
              10,-10; 10,-10; 0,-10],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(sourceShLP.flange, pump.inlet)          annotation (points=[-60,-10;
              -50,-10; -50,-10; -40,-10; -40,-10; -20,-10],          style(
            thickness=2,
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(feed_w.outlet, sinkToEcLP_p.flange) annotation (points=[32,
              -10; 60,-10], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(pIDController.PV, feed_w.w) annotation (points=[32,32; 40,32;
              40,-8.88178e-016; 34,-8.88178e-016],       style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={240,240,240},
            fillPattern=1));
        connect(pIDController.CS, firstOrder.u) annotation (points=[11.4,36; -2,
              36],    style(color=74, rgbcolor={0,0,127}));
        connect(firstOrder.y, pump.pumpSpeed_rpm)          annotation (points=[-25,36;
              -40,36; -40,-4; -20.8,-4],         style(color=74, rgbcolor={0,
                0,127}));
      end TestPumpControl;
      
      model TestHRSG_wac 
        package FlueGasMedium = ThermoPower.Media.FlueGas;
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        annotation (Diagram);
        Gas.SourceW sourceGas(
          redeclare package Medium = FlueGasMedium,
          w0=585.5,
          T=884.65)          annotation (extent=[-100,18; -84,34]);
        Gas.SinkP sinkGas(redeclare package Medium = FlueGasMedium, T=379.448) 
                         annotation (extent=[74,18; 90,34]);
        ThermoPower.PowerPlants.HRSG.Components.StateReader_gas stateGas_in(
                                               redeclare package Medium = 
              FlueGasMedium) annotation (extent=[-76,16; -56,36],   rotation=0);
        ThermoPower.PowerPlants.HRSG.Components.StateReader_gas stateGas_out(
                                                redeclare package Medium = 
              FlueGasMedium) annotation (extent=[42,16; 62,36],   rotation=0);
        ThermoPower.PowerPlants.HRSG.Components.StateReader_water stateShHP_out(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-48,-16; -32,-32], rotation=90);
        ThermoPower.PowerPlants.HRSG.Components.StateReader_water stateShLP_out(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[0,-16; 16,-32],    rotation=90);
        ThermoPower.PowerPlants.HRSG.Components.StateReader_water stateRhIP_out(
                                                   redeclare package Medium = 
              FluidMedium) annotation (extent=[-24,-16; -8,-32],  rotation=90);
        ThermoPower.PowerPlants.HRSG.Components.StateReader_water stateRhIP_in(
                                                  redeclare package Medium = 
              FluidMedium) annotation (extent=[-36,-16; -20,-32], rotation=270);
        Examples.HRSG_3LRh hRSG(redeclare package FlueGasMedium = 
              FlueGasMedium,                                                     redeclare 
            package FluidMedium = FluidMedium,
          drums(SSInit=true),
          HeatExchangersGroup(SSInit=true)) 
          annotation (extent=[-48,-6; 32,74]);
        Water.SinkP OutLP(redeclare package Medium = FluidMedium,
          p0=5389.2,
          h=2.3854e6)      annotation (extent=[64,-80; 76,-68], rotation=0);
        Water.SteamTurbineStodola turbineHP(
          pstart_in=1.28e7,
          pstart_out=2.98e6,
          wstart=67.6,
          hstartin=3.47e6,
          hstartout=3.1076e6,
          wnom=67.6,
          eta_iso_nom=0.833,
          Kt=0.0032078) 
                      annotation (extent=[-50,-92; -30,-72]);
        Water.SteamTurbineStodola turbineIP(
          pstart_in=2.68e6,
          pstart_out=6e5,
          wstart=81.10,
          hstartin=3.554e6,
          hstartout=3.1281e6,
          wnom=81.10,
          eta_iso_nom=0.903,
          Kt=0.018883)     annotation (extent=[-10,-92; 10,-72]);
        Water.SteamTurbineStodola turbineLP(
          pstart_in=6e5,
          pstart_out=5.3982e3,
          wstart=89.82,
          hstartin=3.109e6,
          hstartout=2.3854e6,
          wnom=89.82,
          eta_iso_nom=0.903,
          Kt=0.078004)     annotation (extent=[34,-92; 54,-72]);
        Water.Mixer mixLP(
          redeclare package Medium = FluidMedium,
          V=10,
          pstart=7.19e5,
          hstart=2.89295e6,
          initOpt=ThermoPower.Choices.Init.Options.steadyState) 
                          annotation (extent=[18,-76; 28,-66],
                                                            rotation=0);
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
                      annotation (extent=[98,-88; 88,-78]);
        Water.SourceP sourceShLP(
          redeclare package Medium = FluidMedium,
          p0=5398.2,
          h=1.43495e5) 
          annotation (extent=[78,-34; 90,-46],
                                             rotation=180);
        ThermoPower.PowerPlants.HRSG.Control.levelsControl 
          levelsControlSimplified(
          Level_HP(steadyStateInit=true),
          Level_IP(steadyStateInit=true),
          Level_LP(steadyStateInit=true)) 
          annotation (extent=[60,60; 80,80]);
        Components.PrescribedSpeedPump totalFeedPump(
          redeclare package WaterMedium = FluidMedium,
          rho_nom=1000,
          q_nom={0.0898,0,0.1},
          head_nom={72.74,130,0},
          nominalCondensationPressure=5398.2,
          nominalSteamFlow=89.8,
          n0=1500,
          hstart=1.43495e5,
          nominalSteamPressure=6e5) 
                                  annotation (extent=[38,-34; 50,-46],
            rotation=180);
      protected 
        Buses.Actuators actuators annotation (extent=[60,-20; 68,-12]);
      equation 
        connect(stateGas_in.inlet, sourceGas.flange) annotation (points=[-72,26;
              -84,26],       style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=76,
            rgbfillColor={170,170,255},
            fillPattern=1));
        connect(sinkGas.flange, stateGas_out.outlet) annotation (points=[74,26;
              58,26],  style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2,
            fillColor=76,
            rgbfillColor={170,170,255},
            fillPattern=1));
        connect(stateGas_in.outlet, hRSG.GasIn) annotation (points=[-60,26; -48,
              26],         style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(stateGas_out.inlet, hRSG.GasOut) annotation (points=[46,26; 32,
              26],        style(
            color=76,
            rgbcolor={159,159,223},
            thickness=2));
        connect(stateShLP_out.inlet, hRSG.Sh_LP_Out) 
          annotation (points=[8,-19.2; 8,-6], style(thickness=2));
        connect(stateRhIP_out.inlet, hRSG.Rh_IP_Out) 
          annotation (points=[-16,-19.2; -16,-6], style(thickness=2));
        connect(stateRhIP_in.outlet, hRSG.Rh_IP_In) 
          annotation (points=[-28,-19.2; -28,-6], style(thickness=2));
        connect(stateShHP_out.inlet, hRSG.Sh_HP_Out) 
          annotation (points=[-40,-19.2; -40,-6], style(thickness=2));
        connect(OutLP.flange, turbineLP.outlet) annotation (points=[64,-74; 52,
              -74],                      style(thickness=2));
        connect(turbineHP.shaft_b, turbineIP.shaft_a) annotation (points=[-33.6,
              -82; -6.6,-82],       style(color=0, rgbcolor={0,0,0}));
        connect(turbineIP.shaft_b, turbineLP.shaft_a) annotation (points=[6.4,-82;
              37.4,-82],           style(color=0, rgbcolor={0,0,0}));
        connect(mixLP.in2, turbineIP.outlet) 
          annotation (points=[19.05,-74; 8,-74],    style(thickness=2));
        connect(constantSpeed.flange, turbineLP.shaft_b) annotation (points=[88,-83;
              88,-82; 50.4,-82],         style(color=0, rgbcolor={0,0,0}));
        connect(turbineLP.inlet, mixLP.out) annotation (points=[36,-74; 36,-71;
              28,-71],           style(thickness=2));
        connect(mixLP.in1, stateShLP_out.outlet) 
          annotation (points=[19,-68; 8,-68; 8,-28.8], style(thickness=2));
        connect(turbineHP.inlet, stateShHP_out.outlet) annotation (points=[-48,-74;
              -48,-52.05; -40,-52.05; -40,-28.8],                style(
              thickness=2));
        connect(turbineIP.inlet, stateRhIP_out.outlet) annotation (points=[-8,-74;
              -16,-74; -16,-28.8],               style(thickness=2));
        connect(levelsControlSimplified.SensorsBus, hRSG.SensorsBus) 
          annotation (points=[60,70; 46,70; 46,66; 32,66],         style(
              color=84, rgbcolor={255,170,213}));
        connect(levelsControlSimplified.ActuatorsBus, hRSG.ActuatorsBus) 
          annotation (points=[80,70; 92,70; 92,54; 32,54],       style(color=
                52, rgbcolor={213,255,170}));
        connect(sourceShLP.flange, totalFeedPump.inlet) annotation (points=[78,
              -40; 50,-40], style(
            color=3,
            rgbcolor={0,0,255},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(actuators, levelsControlSimplified.ActuatorsBus) annotation (
            points=[64,-16; 92,-16; 92,70; 80,70], style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.pumpSpeed_rpm, actuators.nPump_feedLP) 
          annotation (points=[50.48,-36.4; 64,-36.4; 64,-16], style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.outlet, hRSG.WaterIn) annotation (points=[38,-40;
              24,-40; 24,-6], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(turbineHP.outlet, stateRhIP_in.inlet) annotation (points=[-32,
              -74; -28,-74; -28,-28.8], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end TestHRSG_wac;
      
    end TestsControl;
    annotation (Documentation(revisions="<html>
<ul>
<li><i>15 Apr 2008</i>
    by <a>Luca Savoldelli</a>:<br>
       First release.</li>
</ul>
</html>"));
  end HRSG;
  
  package SteamTurbineGroup 
    "Models and tests of the steam turbine group and its main components" 
    package Interfaces "Interface definitions" 
      partial model ST_2L 
        "Base class for Steam Turbine with two pressure levels" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Turbines parameter  
        parameter SI.MassFlowRate steamHPNomFlowRate 
          "Nominal HP steam flow rate";
        parameter SI.MassFlowRate steamLPNomFlowRate 
          "Nominal LP steam flowr rate";
        parameter SI.Pressure steamHPNomPressure "Nominal HP steam pressure";
        parameter SI.Pressure steamLPNomPressure "Nominal LP steam pressure";
        parameter SI.Area HPT_Kt "Flow coefficient"           annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "HP Turbine"));
        parameter SI.Area LPT_Kt "Flow coefficient"           annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "LP Turbine"));
        
        //Start value
        parameter Boolean SSInit=false "Steady-state initialization" 
                                       annotation (Dialog(tab = "Initialization"));
        //HPT
        parameter SI.Pressure HPT_pstart_in = steamHPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.Pressure HPT_pstart_out = steamLPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.MassFlowRate HPT_wstart = steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_out 
          "Outlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        //LPT
        parameter SI.Pressure LPT_pstart_in = steamLPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.Pressure LPT_pstart_out = pcond 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.MassFlowRate LPT_wstart = steamLPNomFlowRate + steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_out 
          "Outlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        
        //other parameter
        parameter SI.Pressure pcond "Condenser pressure";
        
        ThermoPower.Water.FlangeA HPT_In(redeclare package Medium = FluidMedium)
                                         annotation (extent=[-140,180; -100,220]);
        ThermoPower.Water.FlangeA LPT_In(redeclare package Medium = FluidMedium)
                                         annotation (extent=[-20,180; 20,220]);
        annotation (Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-200,14; 200,-14],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[-120,30; -120,200], style(color=3, rgbcolor={0,0,255})),
            Line(points=[0,30; 0,200], style(color=3, rgbcolor={0,0,255})),
            Line(points=[60,-90; 60,-192], style(color=3, rgbcolor={0,0,255})),
            Polygon(points=[-126,148; -114,148; -120,132; -126,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-6,148; 6,148; 0,132; -6,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[54,-132; 66,-132; 60,-148; 54,-132], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-120,30; -120,-30; -60,-90; -60,90; -120,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[0,30; 0,-30; 60,-90; 60,90; 0,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1))),
                                Diagram,
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        
        Water.FlangeB LPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[40,-220; 80,-180]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end ST_2L;
      
      partial model ST_2LRh 
        "Base class for Steam Turbine with two pressure levels and reheat" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Turbines parameter
        parameter SI.MassFlowRate steamHPNomFlowRate 
          "Nominal HP steam flow rate";
        parameter SI.MassFlowRate steamIPNomFlowRate 
          "Nominal IP steam flow rate";
        parameter SI.MassFlowRate steamLPNomFlowRate 
          "Nominal LP steam flowr rate";
        parameter SI.Pressure steamHPNomPressure "Nominal HP steam pressure";
        parameter SI.Pressure steamIPNomPressure 
          "Nominal IP steam pressure (pressure in the reheat)";
        parameter SI.Pressure steamLPNomPressure "Nominal LP steam pressure";
        parameter SI.Area HPT_Kt "Flow coefficient"           annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "HP Turbine"));
        parameter SI.Area IPT_Kt "Flow coefficient"           annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "IP Turbine"));
        parameter SI.Area LPT_Kt "Flow coefficient"           annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "LP Turbine"));
        
        //Start value
        parameter Boolean SSInit=false "Steady-state initialization" 
                                       annotation (Dialog(tab = "Initialization"));
        //HPT
        parameter SI.Pressure HPT_pstart_in = steamHPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.Pressure HPT_pstart_out = steamIPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.MassFlowRate HPT_wstart = steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_out 
          "Outlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        //IPT
        parameter SI.Pressure IPT_pstart_in = steamIPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.Pressure IPT_pstart_out = steamLPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.MassFlowRate IPT_wstart = steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.SpecificEnthalpy IPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.SpecificEnthalpy IPT_hstart_out 
          "Outlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        //LPT
        parameter SI.Pressure LPT_pstart_in = steamLPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.Pressure LPT_pstart_out = pcond 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.MassFlowRate LPT_wstart = steamLPNomFlowRate + steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_out 
          "Outlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        
        //other parameter
        parameter SI.Pressure pcond "Condenser pressure";
        
        annotation (Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[-160,198; -160,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-100,200; -100,88], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,200; -40,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Rectangle(extent=[-200,14; 200,-14],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[20,58; 20,140; 80,140; 80,198], style(color=3,
                  rgbcolor={0,0,255})),
            Line(points=[80,-90; 80,-198], style(color=3, rgbcolor={0,0,255})),
            Polygon(points=[-166,148; -154,148; -160,132; -166,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-106,134; -94,134; -100,150; -106,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,148; -34,148; -40,132; -46,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[14,116; 26,116; 20,100; 14,116], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,-132; 86,-132; 80,-148; 74,-132], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-160,30; -160,-30; -100,-90; -100,90; -160,30],
                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-40,30; -40,-30; 80,-90; 80,90; -40,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1))),
                                Diagram,
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        
        Water.FlangeA HPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-180,180; -140,220]);
        Water.FlangeB HPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[-120,180; -80,220]);
        Water.FlangeA IPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-60,180; -20,220]);
        Water.FlangeA LPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[60,180; 100,220]);
        Water.FlangeB LPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[60,-220; 100,-180]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end ST_2LRh;
      
      partial model ST_3LRh 
        "Base class for Steam Turbine with three pressure levels and reheat" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Turbines parameter
        parameter SI.MassFlowRate steamHPNomFlowRate 
          "Nominal HP steam flow rate";
        parameter SI.MassFlowRate steamIPNomFlowRate 
          "Nominal IP steam flow rate";
        parameter SI.MassFlowRate steamLPNomFlowRate 
          "Nominal LP steam flowr rate";
        parameter SI.Pressure steamHPNomPressure "Nominal HP steam pressure";
        parameter SI.Pressure steamIPNomPressure "Nominal IP steam pressure";
        parameter SI.Pressure steamLPNomPressure "Nominal LP steam pressure";
        parameter SI.Area HPT_Kt "Flow coefficient"           annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "HP Turbine"));
        parameter SI.Area IPT_Kt "Flow coefficient"           annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "IP Turbine"));
        parameter SI.Area LPT_Kt "Flow coefficient"           annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "LP Turbine"));
        
        //Start value
        parameter Boolean SSInit=false "Steady-state initialization" 
                                       annotation (Dialog(tab = "Initialization"));
        //HPT
        parameter SI.Pressure HPT_pstart_in = steamHPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.Pressure HPT_pstart_out = steamIPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.MassFlowRate HPT_wstart = steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_out 
          "Outlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        //IPT
        parameter SI.Pressure IPT_pstart_in = steamIPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.Pressure IPT_pstart_out = steamLPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.MassFlowRate IPT_wstart = steamIPNomFlowRate + steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.SpecificEnthalpy IPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.SpecificEnthalpy IPT_hstart_out 
          "Outlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        //LPT
        parameter SI.Pressure LPT_pstart_in = steamLPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.Pressure LPT_pstart_out = pcond 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.MassFlowRate LPT_wstart = steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_out 
          "Outlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        
        //Other parameter
        parameter SI.Pressure pcond "Condenser pressure";
        
        annotation (Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-200,14; 200,-14], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[-100,200; -100,90], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,30; -40,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[80,30; 80,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[20,90; 20,130; 80,130], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[76,134; 84,126],
                                           style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[140,-90; 140,-200], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[-160,30; -160,120; -160,146; -160,204], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Polygon(points=[-166,148; -154,148; -160,132; -166,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,148; -34,148; -40,132; -46,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-106,134; -94,134; -100,150; -106,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[40,136; 40,124; 56,130; 40,136], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,166; 86,166; 80,150; 74,166], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,106; 86,106; 80,90; 74,106], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[134,-132; 146,-132; 140,-148; 134,-132], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-160,30; -160,-30; -100,-90; -100,90; -160,30],
                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-40,30; -40,-30; 20,-90; 20,90; -40,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[80,30; 80,-30; 140,-90; 140,90; 80,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1))),
                                Diagram,
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        
        Water.FlangeA HPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-180,180; -140,220]);
        Water.FlangeB HPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[-120,180; -80,220]);
        Water.FlangeA IPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-60,180; -20,220]);
        Water.FlangeA LPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[60,180; 100,220]);
        Water.FlangeB LPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[120,-220; 160,
              -180]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end ST_3LRh;
      
      partial model ST_2LHU 
        "Base class for Steam Turbine with two pressure levels and with coupling Heat Usage" 
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Turbines parameter  
        parameter SI.MassFlowRate steamHPNomFlowRate 
          "Nominal HP steam flow rate";
        parameter SI.MassFlowRate steamLPNomFlowRate 
          "Nominal LP steam flowr rate";
        parameter SI.Pressure steamHPNomPressure "Nominal HP steam pressure";
        parameter SI.Pressure steamLPNomPressure "Nominal LP steam pressure";
        parameter SI.Area HPT_Kt "Flow coefficient"           annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "HP Turbine"));
        parameter SI.Area LPT_Kt "Flow coefficient"           annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "LP Turbine"));
        
        //Start value
        parameter Boolean SSInit=false "Steady-state initialization" 
                                       annotation (Dialog(tab = "Initialization"));
        //HPT
        parameter SI.Pressure HPT_pstart_in = steamHPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.Pressure HPT_pstart_out = steamLPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.MassFlowRate HPT_wstart = steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        //LPT
        parameter SI.Pressure LPT_pstart_in = steamLPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.Pressure LPT_pstart_out = pcond 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.MassFlowRate LPT_wstart = steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        
        //other parameter
        parameter SI.Pressure pcond "Condenser pressure";
        
        ThermoPower.Water.FlangeB SteamForHU(redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-80,-220; -40,
              -180]);
        ThermoPower.Water.FlangeB LPT_Out(redeclare package Medium = 
              FluidMedium)                annotation (extent=[28,-220; 68,-180]);
        annotation (Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[-20,134; -12,126], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-60,-198; -60,-140; 140,-140; 140,130; 0,130], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-60,90; -60,130; 0,130], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[0,130; 0,198], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Rectangle(extent=[-200,14; 200,-14],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[-120,28; -120,200], style(color=3, rgbcolor={0,0,255})),
            Line(points=[48,-90; 48,-192], style(color=3, rgbcolor={0,0,255})),
            Line(points=[-12,130; -12,120; -12,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-50,136; -50,124; -34,130; -50,136], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-126,148; -114,148; -120,132; -126,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-6,168; 6,168; 0,152; -6,168], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[42,-152; 54,-152; 48,-168; 42,-152], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-18,102; -6,102; -12,86; -18,102], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[26,136; 26,124; 42,130; 26,136], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[0,134; 8,126], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Rectangle(extent=[-16,134; 4,126], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-120,30; -120,-30; -60,-90; -60,90; -120,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-12,30; -12,-30; 48,-90; 48,90; -12,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1))),
                                Diagram,
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        
        Water.FlangeA HPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-140,180; -100,220]);
        Water.FlangeA LPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-20,180; 20,220]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end ST_2LHU;
      
      partial model ST_2LRhHU 
        "Base class for Steam Turbine with two pressure levels, reheat and with coupling Heat Usage" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Turbines parameter
        parameter SI.MassFlowRate steamHPNomFlowRate 
          "Nominal HP steam flow rate";
        parameter SI.MassFlowRate steamLPNomFlowRate 
          "Nominal LP steam flowr rate";
        parameter SI.Pressure steamHPNomPressure "Nominal HP steam pressure";
        parameter SI.Pressure steamLPNomPressure "Nominal LP steam pressure";
        parameter SI.Area HPT_Kt "Flow coefficient"           annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "HP Turbine"));
        parameter SI.Area IPT_Kt "Flow coefficient"           annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "IP Turbine"));
        parameter SI.Area LPT_Kt "Flow coefficient"           annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "LP Turbine"));
        
        //Start value
        parameter Boolean SSInit=false "Steady-state initialization" 
                                       annotation (Dialog(tab = "Initialization"));
        //HPT
        parameter SI.Pressure HPT_pstart_in = steamHPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.Pressure HPT_pstart_out = steamLPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.MassFlowRate HPT_wstart = steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        //LPT
        parameter SI.Pressure LPT_pstart_in = steamLPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.Pressure LPT_pstart_out = pcond 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.MassFlowRate LPT_wstart = steamLPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        
        //other parameter
        parameter SI.Pressure pcond "Condenser pressure";
        
        annotation (Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[76,136; 84,128], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-98,132; 140,132; 140,-140; -40,-140; -40,-198], style(
                  color=3, rgbcolor={0,0,255})),
            Line(points=[80,132; 80,192], style(color=3, rgbcolor={0,0,255})),
            Line(points=[-160,198; -160,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-100,200; -100,88], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,200; -40,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Rectangle(extent=[-200,14; 200,-14],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[80,-90; 80,-198], style(color=3, rgbcolor={0,0,255})),
            Ellipse(extent=[-104,136; -96,128], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-74,138; -74,126; -58,132; -74,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-166,128; -154,128; -160,112; -166,128], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[104,138; 104,126; 120,132; 104,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,170; 86,170; 80,154; 74,170], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,106; -34,106; -40,90; -46,106], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,-152; 86,-152; 80,-168; 74,-152], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-106,152; -94,152; -100,168; -106,152], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-160,30; -160,-30; -100,-90; -100,90; -160,30],
                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-40,30; -40,-30; 80,-90; 80,90; -40,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1))),
                                Diagram,
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        
        Water.FlangeB SteamForHU(            redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-60,-220; -20,
              -180]);
        Water.FlangeB LPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[60,-220; 100,-180]);
        Water.FlangeA HPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-180,180; -140,220]);
        Water.FlangeB HPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[-120,180; -80,220]);
        Water.FlangeA IPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-60,180; -20,220]);
        Water.FlangeA LPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[60,180; 100,220]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end ST_2LRhHU;
      
      partial model ST_2LRh_3t 
        "Base class for Steam Turbine with two pressure levels, reheat and three tappings" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Turbines parameter
        parameter SI.MassFlowRate steamHPNomFlowRate 
          "Nominal HP steam flow rate";
        parameter SI.MassFlowRate steamIPNomFlowRate 
          "Nominal IP steam flow rate";
        parameter SI.MassFlowRate steamLPNomFlowRate 
          "Nominal LP steam flowr rate";
        parameter SI.Pressure steamHPNomPressure "Nominal HP steam pressure";
        parameter SI.Pressure steamIPNomPressure "Nominal IP steam pressure";
        parameter SI.Pressure steamLPNomPressure "Nominal LP steam pressure";
        parameter SI.Area HPT_Kt "Flow coefficient"           annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "HP Turbine"));
        parameter SI.Area IPT_Kt "Flow coefficient"           annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "IP Turbine"));
        parameter SI.Area LPT_Kt "Flow coefficient"           annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "LP Turbine"));
        
        //Start value
        //HPT
        parameter SI.Pressure HPT_pstart_in = steamHPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.Pressure HPT_pstart_out = steamIPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.MassFlowRate HPT_wstart = steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        //IPT
        parameter SI.Pressure IPT_pstart_in = steamIPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.Pressure IPT_pstart_out = steamLPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.MassFlowRate IPT_wstart = steamIPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.SpecificEnthalpy IPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.SpecificEnthalpy IPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        //LPT
        parameter SI.Pressure LPT_pstart_in = steamLPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.Pressure LPT_pstart_out = pcond 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.MassFlowRate LPT_wstart = steamLPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        
        //other parameter
        parameter SI.Pressure pcond "Condenser pressure";
        
        annotation (Diagram, Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[-160,198; -160,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-100,200; -100,86], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,200; -40,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Rectangle(extent=[-200,14; 202,-14],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[20,52; 20,140; 80,140; 80,198], style(color=3,
                  rgbcolor={0,0,255})),
            Line(points=[80,-76; 80,-198], style(color=3, rgbcolor={0,0,255})),
            Line(points=[-130,-50; -130,-200], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Line(points=[-40,-200; -40,-100; -10,-100; -10,-40], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Line(points=[0,-200; 0,-130; 50,-130; 50,-66], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Polygon(points=[-40,30; -40,-30; 80,-90; 80,92; -40,30],   style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-160,30; -160,-30; -100,-90; -100,90; -160,30],
                                                                       style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-166,148; -154,148; -160,132; -166,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-106,134; -94,134; -100,150; -106,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,148; -34,148; -40,132; -46,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[14,108; 26,108; 20,92; 14,108], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,-134; 86,-134; 80,-150; 74,-134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        
        ThermoPower.Water.FlangeB Tap1(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[-140,-210; -120,
              -190]);
        ThermoPower.Water.FlangeB Tap2(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[-50,-210; -30,
              -190]);
        ThermoPower.Water.FlangeB Tap3(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[-10,-210; 10,-190]);
        
        Water.FlangeA HPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-180,180; -140,220]);
        Water.FlangeB HPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[-120,180; -80,220]);
        Water.FlangeA IPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-60,180; -20,220]);
        Water.FlangeA LPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[60,180; 100,220]);
        Water.FlangeB LPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[60,-220; 100,-180]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end ST_2LRh_3t;
      
      partial model ST_3LRh_7t 
        "Base class for Steam Turbine with two pressure levels, reheat and seven tappings" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Turbines parameter
        parameter SI.MassFlowRate steamHPNomFlowRate 
          "Nominal HP steam flow rate";
        parameter SI.MassFlowRate steamIPNomFlowRate 
          "Nominal IP steam flow rate";
        parameter SI.MassFlowRate steamLPNomFlowRate 
          "Nominal LP steam flowr rate";
        parameter SI.Pressure steamHPNomPressure "Nominal HP steam pressure";
        parameter SI.Pressure steamIPNomPressure "Nominal IP steam pressure";
        parameter SI.Pressure steamLPNomPressure "Nominal LP steam pressure";
        parameter SI.Area HPT_Kt "Flow coefficient"           annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "HP Turbine"));
        parameter Real HPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "HP Turbine"));
        parameter SI.Area IPT_Kt "Flow coefficient"           annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "IP Turbine"));
        parameter Real IPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "IP Turbine"));
        parameter SI.Area LPT_Kt "Flow coefficient"           annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_mech=0.98 "Mechanical efficiency"            annotation (Dialog(group = "LP Turbine"));
        parameter Real LPT_eta_iso_nom=0.92 "Nominal isentropic efficiency" 
                                                     annotation (Dialog(group = "LP Turbine"));
        
        //Start value
        //HPT
        parameter SI.Pressure HPT_pstart_in = steamHPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.Pressure HPT_pstart_out = steamIPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.MassFlowRate HPT_wstart = steamHPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        parameter SI.SpecificEnthalpy HPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "HP Turbine"));
        //IPT
        parameter SI.Pressure IPT_pstart_in = steamIPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.Pressure IPT_pstart_out = steamLPNomPressure 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.MassFlowRate IPT_wstart = steamIPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.SpecificEnthalpy IPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        parameter SI.SpecificEnthalpy IPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "IP Turbine"));
        //LPT
        parameter SI.Pressure LPT_pstart_in = steamLPNomPressure 
          "Inlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.Pressure LPT_pstart_out = pcond 
          "Outlet pressure start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.MassFlowRate LPT_wstart = steamLPNomFlowRate 
          "Mass flow rate start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_in 
          "Inlet specific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        parameter SI.SpecificEnthalpy LPT_hstart_out 
          "Outletspecific enthalpy start value" 
           annotation (Dialog(tab = "Initialization",
                              group = "LP Turbine"));
        
        //other parameter
        parameter SI.Pressure pcond "Condenser pressure";
        
        annotation (Diagram, Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-202,14; 200,-14],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[-100,200; -100,88], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,30; -40,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[80,30; 80,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[20,84; 20,130; 80,130], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[76,134; 84,126], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[140,-84; 140,-200], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[-160,30; -160,120; -160,146; -160,204], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[-170,-200; -170,-108; -140,-108; -140,-44], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Line(points=[-130,-200; -130,-120; -120,-120; -120,-58], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Line(points=[-50,-196; -50,-120; 0,-120; 0,-60], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Line(points=[-90,-198; -90,-108; -20,-108; -20,-40], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Line(points=[70,-198; 70,-150; 124,-150; 124,-64], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Line(points=[30,-194; 30,-140; 110,-140; 110,-52], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Line(points=[-10,-198; -10,-130; 92,-130; 92,-36], style(
                color=3,
                rgbcolor={0,0,255},
                pattern=2)),
            Polygon(points=[-160,30; -160,-30; -100,-90; -100,90; -160,30],
                                                                       style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-40,30; -40,-30; 20,-90; 20,90; -40,30],   style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[80,30; 80,-30; 140,-90; 140,90; 80,30],    style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-166,148; -154,148; -160,132; -166,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-106,134; -94,134; -100,150; -106,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,148; -34,148; -40,132; -46,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[134,-134; 146,-134; 140,-150; 134,-134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[40,136; 40,124; 56,130; 40,136], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,166; 86,166; 80,150; 74,166], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,106; 86,106; 80,90; 74,106], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        ThermoPower.Water.FlangeB Tap1(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[-180,-210; -160,
              -190]);
        ThermoPower.Water.FlangeB Tap2(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[-140,-210; -120,
              -190]);
        ThermoPower.Water.FlangeB Tap3(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[-100,-210; -80,
              -190]);
        ThermoPower.Water.FlangeB Tap4(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[-60,-210; -40,
              -190]);
        ThermoPower.Water.FlangeB Tap5(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[-20,-210; 0,-190]);
        ThermoPower.Water.FlangeB Tap6(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[20,-210; 40,-190]);
        ThermoPower.Water.FlangeB Tap7(redeclare package Medium = FluidMedium) 
                                          annotation (extent=[60,-210; 80,-190]);
        
        Water.FlangeA HPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-180,180; -140,220]);
        Water.FlangeB HPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[-120,180; -80,220]);
        Water.FlangeA IPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-60,180; -20,220]);
        Water.FlangeA LPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[60,180; 100,220]);
        Water.FlangeB LPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[120,-220; 160,
              -180]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end ST_3LRh_7t;
      
      partial model Condenser "Base class for condenser" 
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Nominal parameter
        parameter SI.MassFlowRate condNomFlowRate 
          "Nominal flow rate through the condensing fluid side";
        parameter SI.MassFlowRate coolNomFlowRate 
          "Nominal flow rate through the cooling fluid side";
        parameter SI.Pressure condNomPressure 
          "Nominal pressure in the condensing fluid side inlet";
        parameter SI.Pressure coolNomPressure 
          "Nominal pressure in the cooling fluid side inlet";
        
        //Physical Parameter
        parameter Integer N_cool=2 "Number of node of the cooling fluid side";
        parameter SI.Area condExchSurface 
          "Exchange surface between condensing fluid - metal";
        parameter SI.Area coolExchSurface 
          "Exchange surface between metal - cooling fluid";
        parameter SI.Volume condVol "Condensing fluid volume";
        parameter SI.Volume coolVol "Cooling fluid volume";
        parameter SI.Volume metalVol "Volume of the metal part in the tubes";
        parameter SI.SpecificHeatCapacity cm "Specific heat capacity of metal";
        parameter SI.Density rhoMetal "Density of metal";
        
        //Initialization conditions
        parameter Boolean use_T = true 
          "Select: -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the cooling fluid side"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        //Start value
        parameter SI.Pressure pstart_cond = condNomPressure 
          "Condensing fluid pressure start value"                annotation(Dialog(tab = "Initialization"));
        parameter SI.Volume Vlstart_cond = condVol*0.15 
          "Start value of the liquid water volume, condensation side"             annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M_in 
          "Inlet metal wall temperature start value"       annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M_out 
          "Outlet metal wall temperature start value"      annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M[N_cool]=linspace(Tstart_M_in,Tstart_M_out,N_cool) 
          "Start value of metal wall temperature vector (initialized by default)"
                                                                                  annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_cool_in=FluidMedium.temperature_ph(coolNomPressure,hstart_cool_in) 
          "Inlet cooling fluid temperature start value"     annotation(Dialog(enable = use_T,
                                                                              tab = "Initialization"));
        parameter SI.Temperature Tstart_cool_out=FluidMedium.temperature_ph(coolNomPressure,hstart_cool_out) 
          "Outlet cooling fluid temperature start value"    annotation(Dialog(enable = use_T,
                                                                              tab = "Initialization"));
        parameter SI.Temperature Tstart_cool[N_cool]=linspace(Tstart_cool_in,Tstart_cool_out,N_cool) 
          "Start value of cooling fluid temperature vector (initialized by default)"
                                                                          annotation(Dialog(tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_cool_in=FluidMedium.specificEnthalpy_pT(coolNomPressure,Tstart_cool_in) 
          "Inlet cooling fluid specific enthalpy start value"           annotation(Dialog(enable = not use_T,
                                                                                          tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_cool_out=FluidMedium.specificEnthalpy_pT(coolNomPressure,Tstart_cool_out) 
          "Outlet cooling fluid specific enthalpy start value"          annotation(Dialog(enable = not use_T,
                                                                                          tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_cool[N_cool]=linspace(hstart_cool_in,hstart_cool_out,N_cool) 
          "Start value of cooling fluid enthalpy vector (initialized by default)"
                                                                          annotation(Dialog(tab = "Initialization"));
        parameter SI.Pressure pstartin_cool = coolNomPressure 
          "Inlet cooling fluid pressure start value"        annotation(Dialog(tab = "Initialization"));
        parameter SI.Pressure pstartout_cool = coolNomPressure 
          "Outlet cooling fluid pressure start value"        annotation(Dialog(tab = "Initialization"));
        
        ThermoPower.Water.FlangeB waterOut(redeclare package Medium = 
              FluidMedium)                annotation (extent=[-30,-120; 10,-80]);
        ThermoPower.Water.FlangeA coolingIn(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[80,-60; 120,-20]);
        ThermoPower.Water.FlangeB coolingOut(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[80,40; 120,80]);
        annotation (Diagram, Icon(
            Rectangle(extent=[-100,100; 80,-60], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-90,-60; 70,-100],style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[100,-40; -60,-40; 10,10; -60,60; 100,60], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2)),
            Text(
              extent=[-100,-113; 100,-143],
              string="%name",
              style(color=71, rgbcolor={85,170,255}))));
        ThermoPower.Water.FlangeA steamIn(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[-30,80; 10,120]);
        
      end Condenser;
      
      partial model CondPlant_base 
        "Base class for Condensation Plant (simplified version)" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        parameter SI.Pressure p "Pressure of condesation";
        parameter SI.Volume Vtot "Total volume condensing fluid side";
        parameter SI.Volume Vlstart "Start value of the liquid volume";
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        ThermoPower.Water.FlangeA SteamIn(redeclare package Medium = 
              FluidMedium)                annotation (extent=[-20,80; 20,120]);
        ThermoPower.Water.FlangeB WaterOut(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[-20,-120; 20,-80]);
        Buses.Sensors SensorsBus annotation (extent=[88,-50; 108,-30]);
        Buses.Actuators ActuatorsBus annotation (extent=[108,-82; 88,-62]);
        annotation (Diagram, Icon(Rectangle(extent=[-100,100; 100,-100], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[-60,60; 60,-60], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[40,30; -40,30; 38,0; -40,-30; 40,-30], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[0,60; 0,100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[0,-60; 0,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1))));
        
      end CondPlant_base;
      
      partial model CondPlantHU_base 
        "Base class for Condensation Plant with coupling Heat Usage (simplified version)" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        parameter SI.Pressure p "Pressure of condesation";
        parameter SI.Volume Vtot "Total volume condensing fluid side";
        parameter SI.Volume Vlstart "Start value of the liquid volume";
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        ThermoPower.Water.FlangeA CondensatedFromHU(redeclare package Medium = 
              FluidMedium)                  annotation (extent=[-70,80; -30,120]);
        ThermoPower.Water.FlangeA SteamIn(redeclare package Medium = 
              FluidMedium)                annotation (extent=[30,80; 70,120]);
        ThermoPower.Water.FlangeB WaterOut(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[-20,-120; 20,-80]);
        Buses.Sensors SensorsBus annotation (extent=[88,-50; 108,-30]);
        Buses.Actuators ActuatorsBus annotation (extent=[108,-82; 88,-62]);
        annotation (Diagram, Icon(Rectangle(extent=[-100,100; 100,-100], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[-60,60; 60,-60], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[40,30; -40,30; 38,0; -40,-30; 40,-30], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[0,-60; 0,-98], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[-60,0; -80,0; -80,70; -50,70; -50,102], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[0,60; 0,70; 50,70; 50,88], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1))));
        
      end CondPlantHU_base;
      
      partial model CondPlant "Base class for Condensation Plant" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Nominal parameter
        parameter SI.MassFlowRate condNomFlowRate 
          "Nominal flow rate through the condensing fluid side";
        parameter SI.MassFlowRate coolNomFlowRate 
          "Nominal flow rate through the cooling fluid side";
        parameter SI.Pressure condNomPressure 
          "Nominal pressure in the condensing fluid side inlet";
        parameter SI.Pressure coolNomPressure 
          "Nominal pressure in the cooling fluid side inlet";
        
        //Physical Parameter
        parameter Integer N_cool=2 "Number of node of the cooling fluid side";
        parameter SI.Area condExchSurface 
          "Exchange surface between condensing fluid - metal";
        parameter SI.Area coolExchSurface 
          "Exchange surface between metal - cooling fluid";
        parameter SI.Volume condVol "Condensing fluid volume";
        parameter SI.Volume coolVol "Cooling fluid volume";
        parameter SI.Volume metalVol "Volume of the metal part in the tubes";
        parameter SI.SpecificHeatCapacity cm "Specific heat capacity of metal";
        parameter SI.Density rhoMetal "Density of metal";
        
        //Initialization conditions
        parameter Boolean use_T = true 
          "Select: -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the cooling fluid side"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        //Start value
        parameter SI.Pressure pstart_cond = condNomPressure 
          "Condensing fluid pressure start value"                annotation(Dialog(tab = "Initialization"));
        parameter SI.Volume Vlstart_cond = condVol*0.15 
          "Start value of the liquid water volume, condensation side"             annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M_in 
          "Inlet metal wall temperature start value"       annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M_out 
          "Outlet metal wall temperature start value"      annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_cool_in=FluidMedium.temperature_ph(coolNomPressure,hstart_cool_in) 
          "Inlet cooling fluid temperature start value"     annotation(Dialog(enable = use_T,
                                                                              tab = "Initialization"));
        parameter SI.Temperature Tstart_cool_out=FluidMedium.temperature_ph(coolNomPressure,hstart_cool_out) 
          "Outlet cooling fluid temperature start value"    annotation(Dialog(enable = use_T,
                                                                              tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_cool_in=FluidMedium.specificEnthalpy_pT(coolNomPressure,Tstart_cool_in) 
          "Inlet cooling fluid specific enthalpy start value"           annotation(Dialog(enable = not use_T,
                                                                                          tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_cool_out=FluidMedium.specificEnthalpy_pT(coolNomPressure,Tstart_cool_out) 
          "Outlet cooling fluid specific enthalpy start value"          annotation(Dialog(enable = not use_T,
                                                                                          tab = "Initialization"));
        
        ThermoPower.Water.FlangeA SteamIn(redeclare package Medium = 
              FluidMedium)                annotation (extent=[-20,80; 20,120]);
        ThermoPower.Water.FlangeB WaterOut(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[-20,-120; 20,-80]);
        Buses.Sensors SensorsBus annotation (extent=[88,-50; 108,-30]);
        Buses.Actuators ActuatorsBus annotation (extent=[108,-82; 88,-62]);
        annotation (Diagram, Icon(Rectangle(extent=[-100,100; 100,-100], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[-60,60; 60,-60], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[40,30; -40,30; 38,0; -40,-30; 40,-30], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[0,60; 0,100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[0,-60; 0,-100], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1))));
        
      end CondPlant;
      
      partial model CondPlantHU 
        "Base class for Condensation Plant with coupling Heat Usage" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Nominal parameter
        parameter SI.MassFlowRate condNomFlowRate 
          "Nominal flow rate through the condensing fluid side";
        parameter SI.MassFlowRate coolNomFlowRate 
          "Nominal flow rate through the cooling fluid side";
        parameter SI.Pressure condNomPressure 
          "Nominal pressure in the condensing fluid side inlet";
        parameter SI.Pressure coolNomPressure 
          "Nominal pressure in the cooling fluid side inlet";
        
        //Physical Parameter
        parameter Integer N_cool=2 "Number of node of the cooling fluid side";
        parameter SI.Area condExchSurface 
          "Exchange surface between condensing fluid - metal";
        parameter SI.Area coolExchSurface 
          "Exchange surface between metal - cooling fluid";
        parameter SI.Volume condVol "Condensing fluid volume";
        parameter SI.Volume coolVol "Cooling fluid volume";
        parameter SI.Volume metalVol "Volume of the metal part in the tubes";
        parameter SI.SpecificHeatCapacity cm "Specific heat capacity of metal";
        parameter SI.Density rhoMetal "Density of metal";
        
        //Initialization conditions
        parameter Boolean use_T = true 
          "Select: -true- to insert the initial temperature or -false- to insert the initial specifc enthalpy of the cooling fluid side"
           annotation(Dialog(tab = "Initialization Conditions"));
        parameter Boolean SSInit = false "Steady-state initialization" annotation(Dialog(tab = "Initialization Conditions"));
        //Start value
        parameter SI.Pressure pstart_cond = condNomPressure 
          "Condensing fluid pressure start value"                annotation(Dialog(tab = "Initialization"));
        parameter SI.Volume Vlstart_cond = condVol*0.15 
          "Start value of the liquid water volume, condensation side"             annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M_in 
          "Inlet metal wall temperature start value"       annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_M_out 
          "Outlet metal wall temperature start value"      annotation(Dialog(tab = "Initialization"));
        parameter SI.Temperature Tstart_cool_in=FluidMedium.temperature_ph(coolNomPressure,hstart_cool_in) 
          "Inlet cooling fluid temperature start value"     annotation(Dialog(enable = use_T,
                                                                              tab = "Initialization"));
        parameter SI.Temperature Tstart_cool_out=FluidMedium.temperature_ph(coolNomPressure,hstart_cool_out) 
          "Outlet cooling fluid temperature start value"    annotation(Dialog(enable = use_T,
                                                                              tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_cool_in=FluidMedium.specificEnthalpy_pT(coolNomPressure,Tstart_cool_in) 
          "Inlet cooling fluid specific enthalpy start value"           annotation(Dialog(enable = not use_T,
                                                                                          tab = "Initialization"));
        parameter SI.SpecificEnthalpy hstart_cool_out=FluidMedium.specificEnthalpy_pT(coolNomPressure,Tstart_cool_out) 
          "Outlet cooling fluid specific enthalpy start value"          annotation(Dialog(enable = not use_T,
                                                                                          tab = "Initialization"));
        
        ThermoPower.Water.FlangeA CondensatedFromHU(redeclare package Medium = 
              FluidMedium)                  annotation (extent=[-70,80; -30,120]);
        ThermoPower.Water.FlangeA SteamIn(redeclare package Medium = 
              FluidMedium)                annotation (extent=[30,80; 70,120]);
        ThermoPower.Water.FlangeB WaterOut(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[-20,-120; 20,-80]);
        Buses.Sensors SensorsBus annotation (extent=[88,-50; 108,-30]);
        Buses.Actuators ActuatorsBus annotation (extent=[108,-82; 88,-62]);
        annotation (Diagram, Icon(Rectangle(extent=[-100,100; 100,-100], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Ellipse(extent=[-60,60; 60,-60], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[40,30; -40,30; 38,0; -40,-30; 40,-30], style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[0,-60; 0,-98], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[-60,0; -80,0; -80,80; -50,80; -50,102], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Line(points=[0,60; 0,80; 50,80; 50,98], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1))));
        
      end CondPlantHU;
      
      partial model STGroup2L 
        "Base class for Steam Turbine Group with two pressure levels" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        annotation (Diagram, Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[46,-162; 74,-170], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[60,-126; 60,-78], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[40,-126; 80,-166], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[70,-136; 50,-136; 70,-146; 50,-156; 70,-156; 70,-156], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[60,-170; 60,-180; 140,-180; 140,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[126,-132; 154,-160], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[140,-160; 140,-132; 152,-154; 140,-132; 128,-154],
                style(color=3, rgbcolor={0,0,255})),
            Rectangle(extent=[-200,14; 200,-14],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[-120,30; -120,200], style(color=3, rgbcolor={0,0,255})),
            Line(points=[0,30; 0,200], style(color=3, rgbcolor={0,0,255})),
            Polygon(points=[-126,148; -114,148; -120,132; -126,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-6,148; 6,148; 0,132; -6,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-120,30; -120,-30; -60,-90; -60,90; -120,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[0,30; 0,-30; 60,-90; 60,90; 0,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[134,134; 146,134; 140,150; 134,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[94,-174; 94,-186; 110,-180; 94,-174], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        ThermoPower.Water.FlangeA From_SH_HP(redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-140,180; -100,
              220]);
        ThermoPower.Water.FlangeA From_SH_LP(redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-20,180; 20,
              220]);
        ThermoPower.Water.FlangeB WaterOut(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[120,180; 160,220]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end STGroup2L;
      
      partial model STGroup2LRh 
        "Base class for Steam Turbine Group with two pressure levels and reheat" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        annotation (Diagram, Icon(Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[66,-162; 94,-170], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[80,-128; 80,-80], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[60,-126; 100,-166], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[90,-136; 70,-136; 90,-146; 70,-156; 90,-156; 90,-156], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[80,-170; 80,-180; 160,-180; 160,196], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[146,-132; 174,-160], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[160,-160; 160,-132; 172,-154; 160,-132; 148,-154],
                style(color=3, rgbcolor={0,0,255})),
            Line(points=[-160,198; -160,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-100,200; -100,88], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,200; -40,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Rectangle(extent=[-200,14; 200,-14],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[20,58; 20,140; 80,140; 80,198], style(color=3,
                  rgbcolor={0,0,255})),
            Polygon(points=[-166,148; -154,148; -160,132; -166,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-106,134; -94,134; -100,150; -106,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,148; -34,148; -40,132; -46,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[14,116; 26,116; 20,100; 14,116], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-160,30; -160,-30; -100,-90; -100,90; -160,30],
                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-40,30; -40,-30; 80,-90; 80,90; -40,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[154,134; 166,134; 160,150; 154,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[114,-174; 114,-186; 130,-180; 114,-174], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        ThermoPower.Water.FlangeA From_SH_HP(redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-180,180; -140,
              220]);
        ThermoPower.Water.FlangeB To_RH_IP(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[-120,180; -80,
              220]);
        ThermoPower.Water.FlangeA From_RH_IP(redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-60,180; -20,
              220]);
        ThermoPower.Water.FlangeA From_SH_LP(redeclare package Medium = 
              FluidMedium)                   annotation (extent=[60,180; 100,
              220]);
        ThermoPower.Water.FlangeB WaterOut(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[140,180; 180,220]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end STGroup2LRh;
      
      partial model STGroup3LRh 
        "Base class for Steam Turbine Group with three pressure levels and reheat" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        annotation (Diagram, Icon(Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[86,-162; 114,-170], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[80,-126; 120,-166], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[110,-136; 90,-136; 110,-146; 90,-156; 110,-156; 110,
                  -156], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[100,-168; 100,-186; 160,-186; 160,202], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Line(points=[140,-88; 140,-110; 100,-110; 100,-126], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[146,-132; 174,-160], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[160,-160; 160,-132; 172,-154; 160,-132; 148,-154],
                style(color=3, rgbcolor={0,0,255})),
            Rectangle(extent=[-200,14; 200,-14], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[-100,200; -100,90], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,30; -40,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[80,30; 80,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[20,90; 20,130; 80,130], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[76,134; 84,126],
                                           style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-160,30; -160,120; -160,146; -160,204], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Polygon(points=[-166,148; -154,148; -160,132; -166,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,148; -34,148; -40,132; -46,148], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-106,134; -94,134; -100,150; -106,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[40,136; 40,124; 56,130; 40,136], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,166; 86,166; 80,150; 74,166], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,106; 86,106; 80,90; 74,106], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-160,30; -160,-30; -100,-90; -100,90; -160,30],
                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-40,30; -40,-30; 20,-90; 20,90; -40,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[80,30; 80,-30; 140,-90; 140,90; 80,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[154,134; 166,134; 160,150; 154,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[126,-180; 126,-192; 142,-186; 126,-180], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        Water.FlangeA From_SH_HP(            redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-180,180; -140,
              220]);
        Water.FlangeB To_RH_IP(            redeclare package Medium = 
              FluidMedium)                 annotation (extent=[-120,180; -80,
              220]);
        Water.FlangeA From_RH_IP(            redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-60,180; -20,
              220]);
        Water.FlangeA From_SH_LP(            redeclare package Medium = 
              FluidMedium)                   annotation (extent=[60,180; 100,
              220]);
        Water.FlangeB WaterOut(            redeclare package Medium = 
              FluidMedium)                 annotation (extent=[140,180; 180,220]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end STGroup3LRh;
      
      partial model STGroup2LHU 
        "Base class for Steam Turbine Group with two pressure levels and coupling Heat Usage" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        annotation (Diagram, Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[34,-168; 62,-176], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[48,-134; 48,-70], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[28,-132; 68,-172], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[58,-142; 38,-142; 58,-152; 38,-162; 58,-162; 58,-162], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[136,-136; 144,-144], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[140,-140; 140,-198], style(color=3, rgbcolor={0,0,255})),
            Line(points=[140,-140; 140,190], style(color=3, rgbcolor={0,0,255})),
            Line(points=[48,-174; 48,-186; 110,-186; 110,-140; 140,-140], style(
                  color=3, rgbcolor={0,0,255})),
            Ellipse(extent=[126,-92; 154,-120],  style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[140,-120; 140,-92; 152,-114; 140,-92; 128,-114],
                style(color=3, rgbcolor={0,0,255})),
            Ellipse(extent=[-20,136; -12,128], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-60,-196; -60,-110; 100,-110; 100,132; 0,132], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-60,92; -60,132; 0,132], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[0,132; 0,200], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Rectangle(extent=[-200,16; 200,-12],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[-120,30; -120,202], style(color=3, rgbcolor={0,0,255})),
            Line(points=[-12,132; -12,122; -12,32], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-50,138; -50,126; -34,132; -50,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-126,150; -114,150; -120,134; -126,150], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-6,170; 6,170; 0,154; -6,170], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-18,104; -6,104; -12,88; -18,104], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[26,138; 26,126; 42,132; 26,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[0,136; 8,128], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Rectangle(extent=[-16,136; 4,128], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-120,32; -120,-28; -60,-88; -60,92; -120,32], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-12,32; -12,-28; 48,-88; 48,92; -12,32], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-66,-134; -54,-134; -60,-150; -66,-134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[134,134; 146,134; 140,150; 134,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[134,-170; 146,-170; 140,-154; 134,-170], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[80,-180; 80,-192; 96,-186; 80,-180], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        ThermoPower.Water.FlangeB WaterOut(redeclare package Medium = 
              FluidMedium)                 annotation (extent=[120,180; 160,220]);
        ThermoPower.Water.FlangeA WaterFromHU(redeclare package Medium = 
              FluidMedium)                    annotation (extent=[120,-220; 160,
              -180]);
        ThermoPower.Water.FlangeB SteamForHU(redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-80,-220; -40,
              -180]);
        Water.FlangeA HPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-140,180; -100,220]);
        Water.FlangeA LPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-20,180; 20,220]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end STGroup2LHU;
      
      partial model STGroup2LRhHU 
        "Base class for Steam Turbine Group with two pressure levels and reheat, and with coupling Heat Usage" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        annotation (Diagram, Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[66,-168; 94,-176], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[80,-134; 80,-74], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[60,-132; 100,-172], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[90,-142; 70,-142; 90,-152; 70,-162; 90,-162; 90,-162], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Ellipse(extent=[156,-136; 164,-144], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[160,-140; 160,-198], style(color=3, rgbcolor={0,0,255})),
            Line(points=[160,-140; 160,194], style(color=3, rgbcolor={0,0,255})),
            Line(points=[80,-176; 80,-186; 130,-186; 130,-140; 160,-140], style(
                  color=3, rgbcolor={0,0,255})),
            Ellipse(extent=[146,-92; 174,-120],  style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
            Line(points=[160,-120; 160,-92; 172,-114; 160,-92; 148,-114],
                style(color=3, rgbcolor={0,0,255})),
            Ellipse(extent=[76,136; 84,128], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-98,132; 134,132; 134,-110; -40,-110; -40,-198], style(
                  color=3, rgbcolor={0,0,255})),
            Line(points=[80,132; 80,192], style(color=3, rgbcolor={0,0,255})),
            Line(points=[-160,198; -160,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-100,200; -100,88], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Line(points=[-40,200; -40,30], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Rectangle(extent=[-200,14; 200,-14],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Ellipse(extent=[-104,136; -96,128], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-74,138; -74,126; -58,132; -74,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-166,128; -154,128; -160,112; -166,128], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[104,138; 104,126; 120,132; 104,138], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[74,170; 86,170; 80,154; 74,170], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,106; -34,106; -40,90; -46,106], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-106,152; -94,152; -100,168; -106,152], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-160,30; -160,-30; -100,-90; -100,90; -160,30],
                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-40,30; -40,-30; 80,-90; 80,90; -40,30], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)),
            Polygon(points=[154,-170; 166,-170; 160,-154; 154,-170], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[100,-180; 100,-192; 116,-186; 100,-180], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[-46,-134; -34,-134; -40,-150; -46,-134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1)),
            Polygon(points=[154,134; 166,134; 160,150; 154,134], style(
                color=3,
                rgbcolor={0,0,255},
                fillPattern=1))),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        ThermoPower.Water.FlangeB SteamForHU(redeclare package Medium = 
              FluidMedium)                   annotation (extent=[-58,-220; -18,
              -180]);
        ThermoPower.Water.FlangeA WaterFromHU(redeclare package Medium = 
              FluidMedium)                    annotation (extent=[140,-220; 180,
              -180]);
        Water.FlangeB WaterOut(            redeclare package Medium = 
              FluidMedium)                 annotation (extent=[140,180; 180,220]);
        Water.FlangeA HPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-180,180; -140,220]);
        Water.FlangeB HPT_Out(            redeclare package Medium = 
              FluidMedium)                annotation (extent=[-120,180; -80,220]);
        Water.FlangeA IPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[-60,180; -20,220]);
        Water.FlangeA LPT_In(            redeclare package Medium = FluidMedium)
                                         annotation (extent=[60,180; 100,220]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b Shaft_b 
          annotation (extent=[180,-20; 220,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a Shaft_a 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end STGroup2LRhHU;
    end Interfaces;
    
    package Components "Component definitions" 
      
      model Condenser "Condenser" 
        extends Interfaces.Condenser(pstartout_cool=coolNomPressure-dpnom_cool);
        parameter SI.CoefficientOfHeatTransfer gamma_cond 
          "Coefficient of heat transfer on condensation surfaces";
        parameter SI.CoefficientOfHeatTransfer gamma_cool 
          "Coefficient of heat transfer of cooling fluid side";
        parameter Choices.Flow1D.FFtypes.Temp FFtype_cool = Choices.Flow1D.FFtypes.NoFriction 
          "Friction Factor Type";
        parameter Choices.Flow1D.HCtypes.Temp HCtype_cool = Choices.Flow1D.HCtypes.Downstream 
          "Location of the hydraulic capacitance";
        parameter SI.Pressure dpnom_cool=0 
          "Nominal pressure drop (friction term only!)";
        parameter Density rhonom_cool=0 "Nominal inlet density";
        
        //other data
        constant Real pi=Modelica.Constants.pi;
        
        Water.Flow1D flowCooling(
          Nt=1,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          L=coolExchSurface^2/(coolVol*pi*4),
          A=(coolVol*4/coolExchSurface)^2/4*pi,
          omega=coolVol*4/coolExchSurface*pi,
          Dhyd=coolVol*4/coolExchSurface,
          wnom=coolNomFlowRate,
          pstartin=pstartin_cool,
          pstartout=pstartout_cool,
          hstart=hstart_cool,
          N=N_cool,
          FFtype=FFtype_cool,
          dpnom=dpnom_cool,
          rhonom=rhonom_cool,
          HydraulicCapacitance=HCtype_cool,
          hstartin=hstart_cool_in,
          hstartout=hstart_cool_out) 
                            annotation (extent=[10,-2; 30,22],
          rotation=90);
        Thermal.ConvHT convHT(
          Tstart1=Tstart_M,
          gamma=gamma_cool,
          N=N_cool,
          Tstart2=Tstart_cool,
          Tstart11=Tstart_M_in,
          Tstart1N=Tstart_M_out,
          Tstart21=Tstart_cool_in,
          Tstart2N=Tstart_cool_out) 
          annotation (extent=[-32,0; -12,20],rotation=90);
        CondenserShell condenserShell(
          redeclare package Medium = FluidMedium,
          V=condVol,
          Mm=metalVol*rhoMetal,
          Ac=condExchSurface,
          Af=coolExchSurface,
          cm=cm,
          hc=gamma_cond,
          Nc=N_cool,
          pstart=pstart_cond,
          Vlstart=Vlstart_cond,
          initOpt=if SSInit then Options.steadyState else Options.noInit) 
          annotation (extent=[-66,-6; -34,26]);
        annotation (Diagram, Icon);
        Modelica.Blocks.Interfaces.RealOutput ratio_VvonVtot 
          annotation (extent=[-110,0; -90,20], rotation=180);
      equation 
        connect(flowCooling.infl, coolingIn) 
          annotation (points=[20,-2; 20,-40; 100,-40],
                                                  style(thickness=2));
        connect(flowCooling.outfl, coolingOut) 
          annotation (points=[20,22; 20,60; 100,60],style(thickness=2));
        connect(convHT.side2,flowCooling. wall) annotation (points=[-18.9,10;
              15,10],
                   style(color=45, rgbcolor={255,127,0}));
        connect(condenserShell.steam, steamIn) 
          annotation (points=[-50,26; -50,100; -10,100],
                                                     style(thickness=2));
        connect(condenserShell.condensate, waterOut) 
          annotation (points=[-50,-6; -50,-100; -10,-100],
                                                      style(thickness=2));
        connect(condenserShell.coolingFluid,convHT. side1) annotation (points=[-50,10;
              -25,10],       style(color=45, rgbcolor={255,127,0}));
        connect(condenserShell.ratio_VvVtot, ratio_VvonVtot) annotation (points=[-61.2,10;
              -100,10],          style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end Condenser;
      
      model CondenserPreP "Ideal condenser with prescribed pressure" 
        replaceable package Medium = Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialMedium "Medium model";
        //Parameters
        parameter SI.Pressure p "Nominal inlet pressure";
        parameter SI.Volume Vtot = 10 "Total volume of the fluid side";
        parameter SI.Volume Vlstart = 0.15*Vtot 
          "Start value of the liquid water volume";
        
        //Variable
        SI.Density rhol "Density of saturated liquid";
        SI.Density rhov "Density of saturated steam";
        Medium.SaturationProperties sat "Saturation properties";
        Medium.SpecificEnthalpy hl "Specific enthalpy of saturated liquid";
        SI.Mass M "Total mass, steam+liquid";
        SI.Mass Ml "Liquid mass";
        SI.Mass Mv "Steam mass";
        SI.Volume Vl(start=Vlstart) "Liquid volume";
        SI.Volume Vv "Steam volume";
        SI.Energy E "Internal energy";
        SI.Power Q "Thermal power";
        
        //Connectors
        Water.FlangeA steamIn( redeclare package Medium = Medium) 
                       annotation (extent=[-20,80; 20,120]);
        Water.FlangeB waterOut( redeclare package Medium = Medium) annotation (extent=[-20,-120;
              20,-80]);
        Modelica.Blocks.Interfaces.RealOutput Qcond annotation (extent=[-110,30;
              -90,50], rotation=180);
        Modelica.Blocks.Interfaces.RealOutput ratio_Vv_Vtot 
          annotation (extent=[-110,-30; -90,-10],rotation=180);
        annotation (
          Icon(
            Rectangle(extent=[-90,100; 90,-60], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-80,-60; 80,-100],style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[60,-40; -60,-40; 30,10; -60,60; 60,60],   style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2))),
          Diagram);
        
      equation 
        steamIn.p = p;
        steamIn.hAB = hl;
        sat.psat = p;
        sat.Tsat = Medium.saturationTemperature(p);
        hl = Medium.bubbleEnthalpy(sat);
        waterOut.p = p;
        waterOut.hBA = hl;
        rhol = Medium.bubbleDensity(sat);
        rhov = Medium.density_ph(steamIn.p,steamIn.hBA);
        
        Ml = Vl*rhol;
        Mv = Vv*rhov;
        Vtot= Vv+Vl;
        M = Ml + Mv;
        E = Ml*hl + Mv*steamIn.hBA - p*Vtot;
        
        //Energy and Mass Bilances 
        der(M) = steamIn.w + waterOut.w;
        der(E) = steamIn.w*steamIn.hBA + waterOut.w*hl - Q;
        
        //Output signal
        ratio_Vv_Vtot=Vv/Vtot;
        Qcond = Q;
        
      end CondenserPreP;
      
      model CondenserPreP_tap 
        "Ideal condenser with prescribed pressure and external tapping for ratio control" 
        replaceable package Medium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialTwoPhaseMedium;
        parameter SI.Pressure p "Nominal inlet pressure";
        parameter SI.Volume Vtot = 10 "Total volume of the fluid side";
        parameter SI.Volume Vlstart = 0.15*Vtot;
        
        SI.Density rhol "Density of saturated liquid";
        SI.Density rhov "Density of saturated steam";
        Medium.SaturationProperties sat "Saturation properties";
        Medium.SpecificEnthalpy hl "Specific enthalpy of saturated liquid";
        
        SI.Mass M "Total mass, steam+liquid";
        SI.Mass Ml "Liquid mass";
        SI.Mass Mv "Steam mass";
        SI.Volume Vl(start=Vlstart) "Liquid volume";
        SI.Volume Vv "Steam volume";
        SI.Energy E "Internal energy";
        SI.Power Q "Thermal power";
        
        Water.FlangeA steamIn( redeclare package Medium = Medium) 
                       annotation (extent=[-20,80; 20,120]);
        Water.FlangeB waterOut( redeclare package Medium = Medium) annotation (extent=[-20,-120;
              20,-80]);
        Modelica.Blocks.Interfaces.RealOutput Qcond annotation (extent=[-110,30;
              -90,50], rotation=180);
        Modelica.Blocks.Interfaces.RealOutput ratio_Vv_Vtot 
          annotation (extent=[-110,-30; -90,-10],rotation=180);
        Water.FlangeA tapWater(redeclare package Medium = Medium) 
                       annotation (extent=[-90,-90; -70,-70]);
        annotation (
          Icon(
            Rectangle(extent=[-90,100; 90,-60], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-80,-60; 80,-100],style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Line(points=[60,-40; -60,-40; 30,10; -60,60; 60,60],   style(
                color=3,
                rgbcolor={0,0,255},
                thickness=2))),
          Diagram);
        
      equation 
        steamIn.p = p;
        steamIn.hAB = hl;
        sat.psat = p;
        sat.Tsat = Medium.saturationTemperature(p);
        hl = Medium.bubbleEnthalpy(sat);
        waterOut.p = p;
        waterOut.hBA = hl;
        rhol = Medium.bubbleDensity(sat);
        rhov = Medium.density_ph(steamIn.p,steamIn.hBA);
        
        hl=tapWater.hAB;
        tapWater.p=p;
        
        Ml = Vl*rhol;
        Mv = Vv*rhov;
        Vtot= Vv + Vl;
        M = Ml + Mv;
        E = Ml*hl + Mv*steamIn.hBA - p*Vtot;
        
        //Energy and Mass Bilances 
        der(M) = steamIn.w + (waterOut.w + tapWater.w);
        der(E) = steamIn.w*steamIn.hBA + (waterOut.w + tapWater.w)*hl - Q;
        
        //Output signal
        ratio_Vv_Vtot = Vv/Vtot;
        Qcond = Q;
        
      end CondenserPreP_tap;
      
      model CondenserShell 
        replaceable package Medium = 
            Modelica.Media.Interfaces.PartialTwoPhaseMedium "Medium model";
        parameter SI.Volume V "Total volume of condensation cavity";
        parameter SI.Mass Mm "Total mass of shell and tubes";
        parameter SI.Area Ac "Area of condensation surfaces";
        parameter SI.Area Af "Area of contact with the cooling fluid";
        parameter SI.CoefficientOfHeatTransfer hc 
          "Coefficient of heat transfer on condensation surfaces";
        parameter SI.SpecificHeatCapacity cm 
          "Specific heat capacity of the metal";
        parameter Integer Nc = 2 "Number of nodes for coolingFluid connector";
        parameter SI.Pressure pstart "Pressure start value" 
          annotation(Dialog(tab = "Initialisation"));
        parameter SI.Volume Vlstart "Start value of the liquid water volume" 
          annotation(Dialog(tab = "Initialisation"));
        parameter ThermoPower.Choices.Init.Options.Temp initOpt=ThermoPower.Choices.Init.Options.noInit 
          "Initialisation option" annotation(Dialog(tab = "Initialisation"));
        
        SI.Mass Ml "Liquid water mass";
        SI.Mass Mv "Steam mass";
        SI.Mass M "Total liquid+steam mass";
        SI.Energy E "Total liquid+steam energy";
        SI.Energy Em "Total energy of metal masses";
        SI.Volume Vl(start=Vlstart, stateSelect = StateSelect.prefer) 
          "Liquid water total volume";
        SI.Volume Vv "Steam volume";
        Medium.SaturationProperties sat "Saturation properties";
        Medium.AbsolutePressure p(start=pstart,stateSelect=StateSelect.prefer) 
          "Drum pressure";
        Medium.MassFlowRate ws "Steam mass flowrate";
        Medium.MassFlowRate wc "Condensate mass flowrate";
        SI.HeatFlowRate Qcool "Heat flow from the metal to the cooling fluid";
        SI.HeatFlowRate Qcond 
          "Heat flow from the condensing fluid to the metal";
        Medium.SpecificEnthalpy hs "Specific enthalpy of entering steam";
        Medium.SpecificEnthalpy hl "Specific enthalpy of saturated liquid";
        Medium.SpecificEnthalpy hv "Specific enthalpy of saturated steam";
        Medium.Temperature Ts "Saturation temperature";
        Medium.Temperature Tm(start = Medium.saturationTemperature(pstart), stateSelect = StateSelect.prefer) 
          "Average temperature of metal walls";
        Medium.Density rhol "Density of saturated liquid";
        Medium.Density rhov "Density of saturated steam";
        
        ThermoPower.Water.FlangeA steam(redeclare package Medium = Medium) 
                                        annotation (extent=[-20,80; 20,120]);
        ThermoPower.Water.FlangeB condensate(redeclare package Medium = Medium) 
                                             annotation (extent=[-20,-120; 20,-80]);
        ThermoPower.Thermal.DHT coolingFluid(N=Nc) annotation (extent=[-6,-40; 6,40]);
        annotation (Diagram, Icon(
            Rectangle(extent=[-54,88; -14,-86],style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Rectangle(extent=[-14,88; -6,-86],  style(
                color=10,
                rgbcolor={135,135,135},
                fillColor=10,
                rgbfillColor={135,135,135})),
            Rectangle(extent=[6,88; 14,-86],  style(
                color=10,
                rgbcolor={135,135,135},
                fillColor=10,
                rgbfillColor={135,135,135})),
            Rectangle(extent=[14,88; 54,-86],  style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255})),
            Rectangle(extent=[-54,88; 54,-86], style(
                color=10,
                rgbcolor={135,135,135},
                thickness=4)),
            Text(
              extent=[-102,-130; 108,-148],
              style(color=3, rgbcolor={0,0,255}),
              string="%name")));
        Modelica.Blocks.Interfaces.RealOutput ratio_VvVtot 
          annotation (extent=[-90,-20; -50,20], rotation=180);
      equation 
        Ml=Vl*rhol "Mass of liquid";
        Mv=Vv*rhov "Mass of vapour";
        M=Ml+Mv "Total mass";
        V = Vl + Vv "Total volume";
        E=Ml*hl+Mv*hv-p*V "Total liquid+steam energy";
        Em = Mm*cm*Tm "Metal tubes energy";
        der(M) = ws - wc "Mass balance";
        der(E) = ws*hs - wc*hl - Qcond "Energy balance (liquid+steam)";
        der(Em) = Qcond - Qcool "Energy balance (metal tubes)";
        Qcond = hc*Ac*(Ts - Tm);
        
        // Boundary conditions
        p = steam.p;
        p = condensate.p;
        ws = steam.w;
        hs = steam.hBA "Flow reversal not supported";
        steam.hAB = hv;
        wc = -condensate.w;
        condensate.hBA = hl;
        Qcool = -Af/Nc*sum(coolingFluid.phi);
        coolingFluid.T = ones(Nc)*Tm;
        
        // Fluid properties
        // sat = Medium.setSat_p(p);
        
        sat.psat = p;
        sat.Tsat = Medium.saturationTemperature(p);
        
        Ts = sat.Tsat;
        rhol=Medium.bubbleDensity(sat);
        rhov=Medium.dewDensity(sat);
        hl=Medium.bubbleEnthalpy(sat);
        hv=Medium.dewEnthalpy(sat);
        ratio_VvVtot=Vv/V;
      initial equation 
        if initOpt == ThermoPower.Choices.Init.Options.noInit then
          // do nothing
        elseif initOpt == ThermoPower.Choices.Init.Options.steadyState then
          der(M) = 0;
          der(E) = 0;
        else
          assert(false, "Unsupported initialisation option");
        end if;
      end CondenserShell;
      
      model SteamTurbineVarEta "Steam turbine with variable efficiency" 
        extends ThermoPower.Water.SteamTurbineBase;
        
        //Parameter
        parameter SI.Area Kt "Kt coefficient of Stodola's law";
        replaceable function curveEta_iso = Functions.curveEfficiency extends 
          ThermoPower.PowerPlants.SteamTurbineGroup.Functions 
          "Characteristich curve of efficiency";
        parameter Real eta_iso_nom=0.92 "Nominal isentropic efficiency";
        parameter Real x=0.5 "Degree of reaction";
        parameter Length Rm "Mean ray";
        parameter Integer n "Number of stages";
        
        //Variable
        Real y "Ratio uf/wiso";
        Real uf "Mean peripheral velocity of the rotor";
        Real viso "Mean velocity of the fluid in isentropic conditions";
        
      equation 
        w = Kt*partialArc*sqrt(steam_in.p*steam_in.d)*ThermoPower.Functions.sqrtReg(1-(1/PR)^2) 
          "Stodola's law";
        
        uf=omega*Rm;
        viso=sqrt(2*(hin-hiso)/n);
        y=uf/viso;
        eta_iso = curveEta_iso(eta_iso_nom, y, x) "Variable efficiency";
      end SteamTurbineVarEta;
      
      model SteamTurbine_1tapping "Turbine with one tapping" 
        
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance;
        
        //Turbines parameters
        parameter SI.MassFlowRate wn "Inlet nominal flowrate";
        parameter SI.Area Kt "Coefficient of Stodola's law";
        
        //Start Value
        parameter SI.Pressure pstart_in "Inlet pressure start value" 
            annotation (Dialog(tab= "Initialization"));
        parameter SI.Pressure pstart_tap "Tapping pressure start value" 
            annotation (Dialog(tab= "Initialization"));
        parameter SI.Pressure pstart_out "Outlet pressure start value" 
            annotation (Dialog(tab= "Initialization"));
        parameter SI.MassFlowRate wstart "Mass flow rate start value" 
            annotation (Dialog(tab= "Initialization"));
        parameter SI.SpecificEnthalpy hstartin "Inlet enthalpy start value," 
            annotation (Dialog(tab= "Initialization"));
        parameter SI.SpecificEnthalpy hstarttap "Tapping enthalpy start value,"
            annotation (Dialog(tab= "Initialization"));
        parameter SI.SpecificEnthalpy hstartout "Outlet enthalpy start value" 
            annotation (Dialog(tab= "Initialization"));
        
        annotation (Diagram, Icon(
            Polygon(points=[-26,76; -26,28; -20,28; -20,82; -58,82; -58,76; -26,
                  76],          style(
                color=0,
                fillColor=3,
                thickness=2)),
            Polygon(points=[28,56; 34,56; 34,76; 62,76; 62,82; 28,82; 28,56],
                        style(
                color=0,
                fillColor=3,
                thickness=2)),
            Rectangle(extent=[-58,8; 62,-8],   style(
                color=76,
                gradient=3,
                fillColor=9)),
            Polygon(points=[-26,28; -26,-26; 34,-60; 34,60; -26,28],      style(
                color=0,
                fillColor=3,
                thickness=2,
                fillPattern=1)),
               Text(extent=[-126,136; 132,96],    string="%name"),
            Polygon(points=[2,-78; 4,-78; 4,-42; 2,-42; 2,-78], style(
                color=0,
                rgbcolor={0,0,0},
                fillPattern=1))));
        Modelica.Blocks.Interfaces.RealInput partialArc 
          annotation (extent=[-68,-48; -48,-28]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a shaft_a 
          annotation (extent=[-76,-10; -56,10]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_b shaft_b 
          annotation (extent=[56,-10; 76,10]);
        Water.FlangeB tap1(redeclare package Medium = FluidMedium) 
                                         annotation (extent=[-6,-88; 12,-70]);
        Water.SteamTurbineStodola ST_firstStages(
          redeclare package Medium = FluidMedium,
          pstart_in=pstart_in,
          pstart_out=pstart_tap,
          wstart=wstart,
          hstartin=hstartin,
          hstartout=hstarttap,
          wnom=wn,
          Kt=Kt)    annotation (extent=[-44,-18; -8,18]);
        Water.SteamTurbineStodola ST_secondStages(
          redeclare package Medium = FluidMedium,
          pstart_in=pstart_tap,
          pstart_out=pstart_out,
          wstart=wstart,
          hstartin=hstarttap,
          hstartout=hstartout,
          wnom=wn,
          Kt=Kt)    annotation (extent=[18,-18; 56,18]);
        Water.FlowSplit flowSplit annotation (extent=[-8,36; 6,50]);
        Water.FlangeA flangeA(redeclare package Medium = FluidMedium) 
          annotation (extent=[-90,62; -56,96]);
        Water.FlangeB flangeB(redeclare package Medium = FluidMedium) 
          annotation (extent=[62,62; 96,96]);
        
      equation 
        connect(shaft_b, ST_secondStages.shaft_b) annotation (points=[66,0;
              49.16,0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_secondStages.shaft_a, ST_firstStages.shaft_b) annotation (
            points=[24.46,0; -14.48,0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(flowSplit.out1, ST_secondStages.inlet) annotation (points=[3.2,
              45.94; 12,45.94; 12,46; 21.8,46; 21.8,14.4],
                                                       style(thickness=2));
        connect(flowSplit.out2, tap1) 
          annotation (points=[3.2,40.2; 3.2,-79; 3,-79], style(thickness=2));
        connect(flangeA, ST_firstStages.inlet) 
          annotation (points=[-73,79; -40.4,79; -40.4,14.4],
                                                         style(thickness=2));
        connect(flangeB, ST_secondStages.outlet) 
          annotation (points=[79,79; 52.2,79; 52.2,14.4],
                                                        style(thickness=2));
        connect(flowSplit.in1, ST_firstStages.outlet) annotation (points=[-5.2,43;
              -11.6,43; -11.6,14.4],   style(thickness=2));
        connect(partialArc, ST_firstStages.partialArc) annotation (points=[-58,-38;
              -36,-38; -36,-26; -52,-26; -52,-7.2; -35,-7.2],      style(color=
                74, rgbcolor={0,0,127}));
        connect(ST_firstStages.shaft_a, shaft_a) annotation (points=[-37.88,0;
              -66,0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
      end SteamTurbine_1tapping;
      
      model EffectHE "Effect of ideal HE" 
        extends Icons.Water.PressDrop;
        parameter SI.SpecificEnthalpy dh "Increase of specific enthalpy";
        parameter SI.Pressure dp "Drop pressure";
        
        Water.FlangeA in1 annotation (extent=[-140,-40; -62,40]);
        Water.FlangeB out1 annotation (extent=[60,-40; 140,40]);
        annotation (Icon(
            Text(extent=[-100,-56; 100,-84],   string="%name")));
      equation 
        out1.p = in1.p - dp;
        in1.w + out1.w = 0;
        in1.hAB + dh= out1.hAB;
        in1.hBA = out1.hBA - dh;
      end EffectHE;
      
      model Comp_bubble_h 
        "Computation Specific enthalpy of saturated liquid, imposted pressure" 
        replaceable package FluidMedium = ThermoPower.Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialTwoPhaseMedium;
        parameter SI.Pressure p "Pressure value";
        FluidMedium.SaturationProperties sat "Saturation properties";
        SI.SpecificEnthalpy hl "Specific enthalpy of saturated liquid";
        
        Modelica.Blocks.Interfaces.RealOutput h 
          annotation (extent=[80,-10; 100,10]);
        annotation (Icon(Rectangle(extent=[-80,40; 80,-40], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=7,
                rgbfillColor={255,255,255})),
                                        Text(
              extent=[-60,20; 60,-20],
              style(color=3, rgbcolor={0,0,255}),
              string="Computation")));
      equation 
        sat.psat = p;
        sat.Tsat = FluidMedium.saturationTemperature(p);
        hl = FluidMedium.bubbleEnthalpy(sat);
        
        //Output signal
        h = hl;
      end Comp_bubble_h;
      
      model PrescribedSpeed "External imposed Speed" 
        extends 
          Modelica.Mechanics.Rotational.Interfaces.PartialSpeedDependentTorque;
        
        annotation (
          Diagram,
          Icon(Line(points=[0,-100; 0,100], style(color=3))),
          Documentation(info="<HTML>
<p>
Model of <b>fixed</b> angular verlocity of flange, not dependent on torque.
</p>
</HTML>"));
        Modelica.Blocks.Interfaces.RealInput w_fixed 
                                         annotation (extent=[-116,44; -84,76]);
      equation 
        w = w_fixed;
      end PrescribedSpeed;
      
      model BaseReader_water 
        "Base reader for the visualization of the state in the simulation (water)" 
        replaceable package Medium = Water.StandardWater extends 
          Modelica.Media.Interfaces.PartialPureSubstance "Medium model";
        ThermoPower.Water.FlangeA inlet(redeclare package Medium = Medium) 
                                        annotation (extent=[-80,-20; -40,20]);
        ThermoPower.Water.FlangeB outlet(redeclare package Medium = Medium) 
                                         annotation (extent=[40,-20; 80,20]);
        annotation (
          Diagram,
          Icon(Polygon(points=[-80,0; 0,32; 80,0; 0,-32; -80,0], style(
                color=3,
                rgbcolor={0,0,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1)), Text(
              extent=[-80,20; 80,-20],
              string="S",
              style(
                color=7,
                rgbcolor={255,255,255},
                fillColor=3,
                rgbfillColor={0,0,255},
                fillPattern=1))),
          Documentation(info="<HTML>
<p>This component can be inserted in a hydraulic circuit to measure the temperature of the fluid flowing through it.
<p>Flow reversal is supported.
</HTML>",   revisions="<html>
<ul>
<li><i>16 Dec 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Standard medium definition added.</li>
<li><i>1 Jul 2004</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       Adapted to Modelica.Media.</li>
<li><i>1 Oct 2003</i>
    by <a href=\"mailto:francesco.casella@polimi.it\">Francesco Casella</a>:<br>
       First release.</li>
</ul>
</html>"));
      equation 
        inlet.w + outlet.w = 0 "Mass balance";
        inlet.p = outlet.p "No pressure drop";
        
        // Boundary conditions
        inlet.hAB = outlet.hAB;
        inlet.hBA = outlet.hBA;
      end BaseReader_water;
      
      model StateReader_water 
        "State reader for the visualization of the state in the simulation (water)" 
        extends BaseReader_water;
        SI.Temperature T "Temperature";
        SI.Pressure p "Pressure";
        SI.SpecificEnthalpy h "Specific enthalpy";
        SI.MassFlowRate w "Mass flow rate";
        Medium.BaseProperties fluid;
      equation 
        // Set fluid properties
        fluid.p=inlet.p;
        fluid.h = if inlet.w >= 0 then inlet.hBA else inlet.hAB;
        
        T=fluid.T;
        p=fluid.p;
        h=fluid.h;
        w=inlet.w;
      end StateReader_water;
    end Components;
    
    package Examples "Example implementations" 
      
      model ST2LRh_base "Steam turbine with two pressure levels and reheat" 
        extends Interfaces.ST_2LRh;
        
        //Mixers Parameters
        parameter SI.Volume mixLP_V "Internal volume of the LP mixer";
        parameter SI.Enthalpy mixLP_hstart = LPT_hstart_in 
          "Enthalpy start value of the LP mixer" annotation (Dialog(tab = "Initialization",
                                                                    group = "LP mixer"));
        parameter SI.Pressure mixLP_pstart = steamLPNomPressure 
          "Pressure start value of the LP mixer" 
              annotation (Dialog(tab = "Initialization",
                                 group = "LP mixer"));
        
        annotation (Diagram);
        Water.SteamTurbineStodola ST_HP(
          redeclare package Medium = FluidMedium,
          wnom=steamHPNomFlowRate,
          eta_mech=HPT_eta_mech,
          eta_iso_nom=HPT_eta_iso_nom,
          Kt=HPT_Kt,
          hstartin=HPT_hstart_in,
          hstartout=HPT_hstart_out,
          pstart_in=HPT_pstart_in,
          pstart_out=HPT_pstart_out,
          wstart=HPT_wstart)            annotation (extent=[-150,-20; -110,20]);
        Water.SteamTurbineStodola ST_IP(
          redeclare package Medium = FluidMedium,
          eta_mech=IPT_eta_mech,
          eta_iso_nom=IPT_eta_iso_nom,
          Kt=IPT_Kt,
          hstartin=IPT_hstart_in,
          hstartout=IPT_hstart_out,
          pstart_in=IPT_pstart_in,
          pstart_out=IPT_pstart_out,
          wstart=IPT_wstart,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[-20,-20; 20,20]);
        Water.SteamTurbineStodola ST_LP(
          redeclare package Medium = FluidMedium,
          eta_mech=LPT_eta_mech,
          eta_iso_nom=LPT_eta_iso_nom,
          Kt=LPT_Kt,
          hstartin=LPT_hstart_in,
          hstartout=LPT_hstart_out,
          pstart_in=LPT_pstart_in,
          pstart_out=LPT_pstart_out,
          wstart=LPT_wstart,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[100,-20; 140,20]);
        Water.Mixer mixLP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          hstart=mixLP_hstart,
          V=mixLP_V,
          pstart=mixLP_pstart) 
                          annotation (extent=[64,80; 84,100],rotation=270);
      protected 
        replaceable Components.BaseReader_water stateHPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-170,80; -150,60], rotation=90);
        replaceable Components.BaseReader_water stateHPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-110,80; -90,60],  rotation=270);
        replaceable Components.BaseReader_water stateIPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-50,80; -30,60],   rotation=90);
        replaceable Components.BaseReader_water stateIPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[30,80; 50,60],     rotation=270);
        replaceable Components.BaseReader_water stateLPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[64,60; 84,40],     rotation=90);
        replaceable Components.BaseReader_water stateLPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[150,-20; 170,-40], rotation=90);
      equation 
        connect(ST_HP.shaft_a, Shaft_a) annotation (points=[-143.2,0; -200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(Shaft_b,ST_LP. shaft_b) annotation (points=[200,0; 132.8,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_a,ST_IP. shaft_b) annotation (points=[106.8,0; 12.8,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_IP.shaft_a,ST_HP. shaft_b) annotation (points=[-13.2,0;
              -117.2,0],
                  style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(mixLP.in1, LPT_In) 
          annotation (points=[80,98; 80,200; 80,200], style(thickness=2));
        connect(ST_HP.inlet, stateHPT_in.outlet) annotation (points=[-146,16;
              -160,16; -160,64],   style(thickness=2));
        connect(stateHPT_in.inlet, HPT_In) 
          annotation (points=[-160,76; -160,200], style(thickness=2));
        connect(ST_HP.outlet, stateHPT_out.inlet) annotation (points=[-114,16;
              -100,16; -100,64], style(thickness=2));
        connect(stateHPT_out.outlet, HPT_Out) 
          annotation (points=[-100,76; -100,200], style(thickness=2));
        connect(ST_IP.inlet, stateIPT_in.outlet) 
          annotation (points=[-16,16; -40,16; -40,64],     style(thickness=2));
        connect(stateIPT_in.inlet, IPT_In) 
          annotation (points=[-40,76; -40,200; -40,200], style(thickness=2));
        connect(ST_IP.outlet, stateIPT_out.inlet) 
          annotation (points=[16,16; 40,16; 40,64],   style(thickness=2));
        connect(stateIPT_out.outlet, mixLP.in2) annotation (points=[40,76; 40,
              120; 68,120; 68,97.9], style(thickness=2));
        connect(stateLPT_in.inlet, mixLP.out) 
          annotation (points=[74,56; 74,80; 74,80], style(thickness=2));
        connect(stateLPT_in.outlet, ST_LP.inlet) 
          annotation (points=[74,44; 74,16; 104,16],     style(thickness=2));
        connect(ST_LP.outlet, stateLPT_out.inlet) 
          annotation (points=[136,16; 160,16; 160,-24],   style(thickness=2));
        connect(LPT_Out, stateLPT_out.outlet) annotation (points=[80,-200; 80,
              -50; 160,-50; 160,-36], style(thickness=2));
      end ST2LRh_base;
      
      model ST3LRh_base "Steam turbine with three pressure levels and reheat" 
        extends Interfaces.ST_3LRh;
        
        //Mixers Parameters
        parameter SI.Volume mixLP_V "Internal volume of the LP mixer";
        parameter SI.Enthalpy mixLP_hstart = LPT_hstart_in 
          "Enthalpy start value of the LP mixer" annotation (Dialog(tab = "Initialization",
                                                                    group = "LP mixer"));
        parameter SI.Pressure mixLP_pstart = steamLPNomPressure 
          "Pressure start value of the LP mixer" 
              annotation (Dialog(tab = "Initialization",
                                 group = "LP mixer"));
        
        annotation (Diagram);
        Water.SteamTurbineStodola ST_HP(
          redeclare package Medium = FluidMedium,
          wnom=steamHPNomFlowRate,
          eta_mech=HPT_eta_mech,
          eta_iso_nom=HPT_eta_iso_nom,
          Kt=HPT_Kt,
          hstartin=HPT_hstart_in,
          hstartout=HPT_hstart_out,
          pstart_in=HPT_pstart_in,
          pstart_out=HPT_pstart_out,
          wstart=HPT_wstart)            annotation (extent=[-154,-20; -114,20]);
        Water.SteamTurbineStodola ST_IP(
          redeclare package Medium = FluidMedium,
          eta_mech=IPT_eta_mech,
          eta_iso_nom=IPT_eta_iso_nom,
          Kt=IPT_Kt,
          hstartin=IPT_hstart_in,
          hstartout=IPT_hstart_out,
          pstart_in=IPT_pstart_in,
          pstart_out=IPT_pstart_out,
          wstart=IPT_wstart,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[-20,-20; 20,20]);
        Water.SteamTurbineStodola ST_LP(
          redeclare package Medium = FluidMedium,
          eta_mech=LPT_eta_mech,
          eta_iso_nom=LPT_eta_iso_nom,
          Kt=LPT_Kt,
          hstartin=LPT_hstart_in,
          hstartout=LPT_hstart_out,
          pstart_in=LPT_pstart_in,
          pstart_out=LPT_pstart_out,
          wstart=LPT_wstart,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[100,-20; 140,20]);
        Water.Mixer mixLP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          hstart=mixLP_hstart,
          V=mixLP_V,
          pstart=mixLP_pstart) 
                          annotation (extent=[64,80; 84,100],rotation=270);
      protected 
        replaceable Components.BaseReader_water stateHPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-170,80; -150,60], rotation=90);
        replaceable Components.BaseReader_water stateHPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-110,80; -90,60],  rotation=270);
        replaceable Components.BaseReader_water stateIPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-50,80; -30,60],   rotation=90);
        replaceable Components.BaseReader_water stateIPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[30,80; 50,60],     rotation=270);
        replaceable Components.BaseReader_water stateLPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[64,60; 84,40],     rotation=90);
        replaceable Components.BaseReader_water stateLPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[150,-20; 170,-40], rotation=90);
      equation 
        connect(ST_HP.shaft_a, Shaft_a) annotation (points=[-147.2,0; -200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(Shaft_b,ST_LP. shaft_b) annotation (points=[200,0; 132.8,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_a,ST_IP. shaft_b) annotation (points=[106.8,0; 12.8,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_IP.shaft_a,ST_HP. shaft_b) annotation (points=[-13.2,0;
              -121.2,0],
                  style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(mixLP.in1, LPT_In) 
          annotation (points=[80,98; 80,200], style(thickness=2));
        connect(ST_HP.inlet, stateHPT_in.outlet) annotation (points=[-150,16;
              -160,16; -160,64],   style(thickness=2));
        connect(stateHPT_in.inlet, HPT_In) 
          annotation (points=[-160,76; -160,200], style(thickness=2));
        connect(ST_HP.outlet, stateHPT_out.inlet) annotation (points=[-118,16;
              -100,16; -100,64], style(thickness=2));
        connect(ST_IP.inlet, stateIPT_in.outlet) 
          annotation (points=[-16,16; -40,16; -40,64],     style(thickness=2));
        connect(stateIPT_in.inlet, IPT_In) 
          annotation (points=[-40,76; -40,200], style(thickness=2));
        connect(ST_IP.outlet, stateIPT_out.inlet) 
          annotation (points=[16,16; 40,16; 40,64],   style(thickness=2));
        connect(stateIPT_out.outlet, mixLP.in2) annotation (points=[40,76; 40,
              120; 68,120; 68,97.9], style(thickness=2));
        connect(stateLPT_in.inlet, mixLP.out) 
          annotation (points=[74,56; 74,80], style(thickness=2));
        connect(stateLPT_in.outlet, ST_LP.inlet) 
          annotation (points=[74,44; 74,16; 104,16],     style(thickness=2));
        connect(ST_LP.outlet, stateLPT_out.inlet) 
          annotation (points=[136,16; 160,16; 160,-24],   style(thickness=2));
        connect(LPT_Out, stateLPT_out.outlet) annotation (points=[140,-200; 140,
              -50; 160,-50; 160,-36], style(thickness=2));
        connect(stateHPT_out.outlet, HPT_Out) 
          annotation (points=[-100,76; -100,200], style(thickness=2));
      end ST3LRh_base;
      
      model ST2LRh_valve 
        "Steam turbine with two pressure levels and reheat, inlet valves" 
        extends Interfaces.ST_2LRh;
        
        //Mixers Parameters
        parameter SI.Volume mixLP_V "Internal volume of the LP mixer";
        parameter SI.Enthalpy mixLP_hstart = LPT_hstart_in 
          "Enthalpy start value of the LP mixer" annotation (Dialog(tab = "Initialization",
                                                                    group = "LP mixer"));
        parameter SI.Pressure mixLP_pstart = steamLPNomPressure 
          "Pressure start value of the LP mixer" 
              annotation (Dialog(tab = "Initialization",
                                 group = "LP mixer"));
        
        //Valves Parameters
        parameter Real valveHP_Cv=0 "Cv (US) flow coefficient of the HP valve" annotation(Dialog(group= "HP valve"));
        parameter SI.Pressure valveHP_dpnom 
          "Nominal pressure drop of the HP valve"                                   annotation(Dialog(group= "HP valve"));
        parameter Real valveIP_Cv=0 "Cv (US) flow coefficient of the IP valve" annotation(Dialog(group= "IP valve"));
        parameter SI.Pressure valveIP_dpnom 
          "Nominal pressure drop of the IP valve"                                   annotation(Dialog(group= "IP valve"));
        parameter Real valveLP_Cv=0 "Cv (US) flow coefficient of the LP valve" annotation(Dialog(group= "LP valve"));
        parameter SI.Pressure valveLP_dpnom 
          "Nominal pressure drop of the LP valve"                                   annotation(Dialog(group= "LP valve"));
        
        annotation (Diagram, Icon);
        Water.SteamTurbineStodola ST_HP(
          redeclare package Medium = FluidMedium,
          wnom=steamHPNomFlowRate,
          eta_mech=HPT_eta_mech,
          eta_iso_nom=HPT_eta_iso_nom,
          Kt=HPT_Kt,
          hstartin=HPT_hstart_in,
          hstartout=HPT_hstart_out,
          pstart_in=HPT_pstart_in,
          pstart_out=HPT_pstart_out,
          wstart=HPT_wstart)            annotation (extent=[-120,-20; -80,20]);
        Water.SteamTurbineStodola ST_IP(
          redeclare package Medium = FluidMedium,
          eta_mech=IPT_eta_mech,
          eta_iso_nom=IPT_eta_iso_nom,
          Kt=IPT_Kt,
          hstartin=IPT_hstart_in,
          hstartout=IPT_hstart_out,
          pstart_in=IPT_pstart_in,
          pstart_out=IPT_pstart_out,
          wstart=IPT_wstart,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[0,-20; 40,20]);
        Water.SteamTurbineStodola ST_LP(
          redeclare package Medium = FluidMedium,
          eta_mech=LPT_eta_mech,
          eta_iso_nom=LPT_eta_iso_nom,
          Kt=LPT_Kt,
          hstartin=LPT_hstart_in,
          hstartout=LPT_hstart_out,
          pstart_in=LPT_pstart_in,
          pstart_out=LPT_pstart_out,
          wstart=LPT_wstart,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[120,-20; 160,20]);
        Water.Mixer mixLP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          hstart=mixLP_hstart,
          V=mixLP_V,
          pstart=mixLP_pstart) 
                          annotation (extent=[64,80; 84,100],rotation=270);
        Water.ValveVap valveHP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveHP_Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          dpnom=valveHP_dpnom,
          redeclare package Medium = FluidMedium,
          hstart=HPT_hstart_in) 
                               annotation (extent=[-130,26; -150,6],
                                                                   rotation=180);
        Water.ValveVap valveIP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveIP_Cv,
          pnom=steamIPNomPressure,
          dpnom=valveIP_dpnom,
          redeclare package Medium = FluidMedium,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate,
          hstart=IPT_hstart_in) 
                               annotation (extent=[-10,26; -30,6],rotation=180);
        Water.ValveVap valveLP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          dpnom=valveLP_dpnom,
          redeclare package Medium = FluidMedium,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate,
          Cv=valveLP_Cv,
          hstart=LPT_hstart_in) 
                               annotation (extent=[110,26; 90,6],rotation=180);
      protected 
        replaceable Components.BaseReader_water stateHPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-170,80; -150,60], rotation=90);
        replaceable Components.BaseReader_water stateHPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-110,80; -90,60],  rotation=270);
        replaceable Components.BaseReader_water stateIPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-50,80; -30,60],   rotation=90);
        replaceable Components.BaseReader_water stateIPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[30,80; 50,60],     rotation=270);
        replaceable Components.BaseReader_water stateLPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[64,60; 84,40],     rotation=90);
        replaceable Components.BaseReader_water stateLPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[170,-20; 190,-40], rotation=90);
      equation 
        connect(Shaft_b, Shaft_b) annotation (points=[200,0; 200,0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(valveIP.outlet,ST_IP. inlet) annotation (points=[-10,16; -10,16;
              4,16],             style(thickness=2));
        connect(ActuatorsBus.Opening_valveHP,valveHP. theta) annotation (points=[200,-140;
              -180,-140; -180,30; -140,30; -140,24],       style(color=52, rgbcolor={
                213,255,170}));
        connect(ActuatorsBus.Opening_valveIP,valveIP. theta) annotation (points=[200,-140;
              -60,-140; -60,34; -20,34; -20,24],      style(color=52, rgbcolor={213,
                255,170}));
        connect(ActuatorsBus.Opening_valveLP,valveLP. theta) annotation (points=[200,-140;
              60,-140; 60,32; 100,32; 100,24],
                                        style(color=52, rgbcolor={213,255,170}));
        connect(ST_HP.inlet,valveHP. outlet) annotation (points=[-116,16;
              -126.5,16; -126.5,16; -130,16],   style(thickness=2));
        connect(ST_HP.shaft_a, Shaft_a) annotation (points=[-113.2,0; -200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.inlet,valveLP. outlet) annotation (points=[124,16; 118.5,
              16; 118.5,16; 110,16],         style(thickness=2));
        connect(Shaft_b,ST_LP. shaft_b) annotation (points=[200,0; 152.8,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_a,ST_IP. shaft_b) annotation (points=[126.8,0; 32.8,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_IP.shaft_a,ST_HP. shaft_b) annotation (points=[6.8,0; -87.2,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(mixLP.in1, LPT_In) 
          annotation (points=[80,98; 80,200], style(thickness=2));
        connect(stateHPT_in.inlet, HPT_In) 
          annotation (points=[-160,76; -160,200], style(thickness=2));
        connect(ST_HP.outlet, stateHPT_out.inlet) annotation (points=[-84,16;
              -70,16; -70,40; -100,40; -100,64], style(thickness=2));
        connect(stateHPT_out.outlet, HPT_Out) 
          annotation (points=[-100,76; -100,200], style(thickness=2));
        connect(stateIPT_in.inlet, IPT_In) 
          annotation (points=[-40,76; -40,200], style(thickness=2));
        connect(ST_IP.outlet, stateIPT_out.inlet) 
          annotation (points=[36,16; 40,16; 40,64],   style(thickness=2));
        connect(stateIPT_out.outlet, mixLP.in2) annotation (points=[40,76; 40,
              120; 68,120; 68,97.9], style(thickness=2));
        connect(stateLPT_in.inlet, mixLP.out) 
          annotation (points=[74,56; 74,80], style(thickness=2));
        connect(ST_LP.outlet, stateLPT_out.inlet) 
          annotation (points=[156,16; 180,16; 180,-24],   style(thickness=2));
        connect(LPT_Out, stateLPT_out.outlet) annotation (points=[80,-200; 80,
              -50; 180,-50; 180,-36], style(thickness=2));
        connect(valveHP.inlet, stateHPT_in.outlet) 
          annotation (points=[-150,16; -160,16; -160,64], style(thickness=2));
        connect(valveIP.inlet, stateIPT_in.outlet) 
          annotation (points=[-30,16; -40,16; -40,64], style(thickness=2));
        connect(valveLP.inlet, stateLPT_in.outlet) 
          annotation (points=[90,16; 74,16; 74,44], style(thickness=2));
      end ST2LRh_valve;
      
      model ST3LRh_valve 
        "Steam turbine with three pressure levels and reheat, inlet valves" 
        extends Interfaces.ST_3LRh;
        
        //Mixers Parameters
        parameter SI.Volume mixLP_V "Internal volume of the LP mixer";
        parameter SI.Enthalpy mixLP_hstart = LPT_hstart_in 
          "Enthalpy start value of the LP mixer" annotation (Dialog(tab = "Initialization",
                                                                    group = "LP mixer"));
        parameter SI.Pressure mixLP_pstart = steamLPNomPressure 
          "Pressure start value of the LP mixer" 
              annotation (Dialog(tab = "Initialization",
                                 group = "LP mixer"));
        
        //Valves Parameters
        parameter Real valveHP_Cv=0 "Cv (US) flow coefficient of the HP valve" annotation(Dialog(group= "HP valve"));
        parameter SI.Pressure valveHP_dpnom 
          "Nominal pressure drop of the HP valve"                                   annotation(Dialog(group= "HP valve"));
        parameter Real valveIP_Cv=0 "Cv (US) flow coefficient of the IP valve" annotation(Dialog(group= "IP valve"));
        parameter SI.Pressure valveIP_dpnom 
          "Nominal pressure drop of the IP valve"                                   annotation(Dialog(group= "IP valve"));
        parameter Real valveLP_Cv=0 "Cv (US) flow coefficient of the LP valve" annotation(Dialog(group= "LP valve"));
        parameter SI.Pressure valveLP_dpnom 
          "Nominal pressure drop of the LP valve"                                   annotation(Dialog(group= "LP valve"));
        
        annotation (Diagram, Icon);
        Water.SteamTurbineStodola ST_HP(
          redeclare package Medium = FluidMedium,
          wnom=steamHPNomFlowRate,
          eta_mech=HPT_eta_mech,
          eta_iso_nom=HPT_eta_iso_nom,
          Kt=HPT_Kt,
          hstartin=HPT_hstart_in,
          hstartout=HPT_hstart_out,
          pstart_in=HPT_pstart_in,
          pstart_out=HPT_pstart_out,
          wstart=HPT_wstart)            annotation (extent=[-122,-20; -82,20]);
        Water.SteamTurbineStodola ST_IP(
          redeclare package Medium = FluidMedium,
          eta_mech=IPT_eta_mech,
          eta_iso_nom=IPT_eta_iso_nom,
          Kt=IPT_Kt,
          hstartin=IPT_hstart_in,
          hstartout=IPT_hstart_out,
          pstart_in=IPT_pstart_in,
          pstart_out=IPT_pstart_out,
          wstart=IPT_wstart,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[0,-20; 40,20]);
        Water.SteamTurbineStodola ST_LP(
          redeclare package Medium = FluidMedium,
          eta_mech=LPT_eta_mech,
          eta_iso_nom=LPT_eta_iso_nom,
          Kt=LPT_Kt,
          hstartin=LPT_hstart_in,
          hstartout=LPT_hstart_out,
          pstart_in=LPT_pstart_in,
          pstart_out=LPT_pstart_out,
          wstart=LPT_wstart,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[120,-20; 160,20]);
        Water.Mixer mixLP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          hstart=mixLP_hstart,
          V=mixLP_V,
          pstart=mixLP_pstart) 
                          annotation (extent=[64,80; 84,100],rotation=270);
        Water.ValveVap valveHP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveHP_Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          dpnom=valveHP_dpnom,
          redeclare package Medium = FluidMedium,
          hstart=HPT_hstart_in) 
                               annotation (extent=[-130,26; -150,6],
                                                                   rotation=180);
        Water.ValveVap valveIP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveIP_Cv,
          pnom=steamIPNomPressure,
          dpnom=valveIP_dpnom,
          redeclare package Medium = FluidMedium,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate,
          hstart=IPT_hstart_in) 
                               annotation (extent=[-10,26; -30,6],rotation=180);
        Water.ValveVap valveLP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          dpnom=valveLP_dpnom,
          redeclare package Medium = FluidMedium,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate,
          Cv=valveLP_Cv,
          hstart=LPT_hstart_in) 
                               annotation (extent=[110,26; 90,6],rotation=180);
      protected 
        replaceable Components.BaseReader_water stateHPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-170,80; -150,60], rotation=90);
        replaceable Components.BaseReader_water stateHPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-84,80; -64,60],   rotation=270);
        replaceable Components.BaseReader_water stateIPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-50,80; -30,60],   rotation=90);
        replaceable Components.BaseReader_water stateIPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[36,80; 56,60],     rotation=270);
        replaceable Components.BaseReader_water stateLPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[64,60; 84,40],     rotation=90);
        replaceable Components.BaseReader_water stateLPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[160,-20; 180,-40], rotation=90);
      equation 
        connect(valveIP.outlet,ST_IP. inlet) annotation (points=[-10,16; -10,16;
              4,16],             style(thickness=2));
        connect(ActuatorsBus.Opening_valveHP,valveHP. theta) annotation (points=[200,-140;
              -180,-140; -180,30; -140,30; -140,24],       style(color=52, rgbcolor={
                213,255,170}));
        connect(ActuatorsBus.Opening_valveIP,valveIP. theta) annotation (points=[200,-140;
              -60,-140; -60,34; -20,34; -20,24],      style(color=52, rgbcolor={213,
                255,170}));
        connect(ActuatorsBus.Opening_valveLP,valveLP. theta) annotation (points=[200,-140;
              60,-140; 60,32; 100,32; 100,24],
                                        style(color=52, rgbcolor={213,255,170}));
        connect(ST_HP.inlet,valveHP. outlet) annotation (points=[-118,16; -124,
              16; -124,16; -130,16],            style(thickness=2));
        connect(ST_HP.shaft_a, Shaft_a) annotation (points=[-115.2,0; -200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.inlet,valveLP. outlet) annotation (points=[124,16; 118.5,
              16; 118.5,16; 110,16],         style(thickness=2));
        connect(Shaft_b,ST_LP. shaft_b) annotation (points=[200,0; 152.8,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_a,ST_IP. shaft_b) annotation (points=[126.8,0; 32.8,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_IP.shaft_a,ST_HP. shaft_b) annotation (points=[6.8,0; -89.2,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(mixLP.in1, LPT_In) 
          annotation (points=[80,98; 80,200; 80,200], style(thickness=2));
        connect(stateHPT_in.inlet, HPT_In) 
          annotation (points=[-160,76; -160,200], style(thickness=2));
        connect(stateIPT_in.inlet, IPT_In) 
          annotation (points=[-40,76; -40,200], style(thickness=2));
        connect(ST_IP.outlet, stateIPT_out.inlet) 
          annotation (points=[36,16; 46,16; 46,64],   style(thickness=2));
        connect(stateIPT_out.outlet, mixLP.in2) annotation (points=[46,76; 46,120;
              68,120; 68,97.9], style(thickness=2));
        connect(stateLPT_in.inlet, mixLP.out) 
          annotation (points=[74,56; 74,80], style(thickness=2));
        connect(ST_LP.outlet, stateLPT_out.inlet) 
          annotation (points=[156,16; 170,16; 170,-24],   style(thickness=2));
        connect(LPT_Out, stateLPT_out.outlet) annotation (points=[140,-200; 140,
              -50; 170,-50; 170,-36], style(thickness=2));
        connect(valveIP.inlet, stateIPT_in.outlet) 
          annotation (points=[-30,16; -40,16; -40,64], style(thickness=2));
        connect(ST_HP.outlet, stateHPT_out.inlet) annotation (points=[-86,16;
              -74,16; -74,64],                   style(thickness=2));
        connect(valveHP.inlet, stateHPT_in.outlet) 
          annotation (points=[-150,16; -160,16; -160,64], style(thickness=2));
        connect(valveLP.inlet, stateLPT_in.outlet) 
          annotation (points=[90,16; 74,16; 74,44], style(thickness=2));
        connect(stateHPT_out.outlet, HPT_Out) annotation (points=[-74,76; -74,
              120; -100,120; -100,200], style(thickness=2));
      end ST3LRh_valve;
      
      model ST2LRh_bypass 
        "Steam turbine with two pressure levels and reheat, inlet and by-pass valves " 
        extends Interfaces.ST_2LRh;
        
        //Mixers Parameters
        parameter SI.Volume mixLP_V "Internal volume of the LP mixer";
        parameter SI.Enthalpy mixLP_hstart = LPT_hstart_in 
          "Enthalpy start value of the LP mixer" annotation (Dialog(tab = "Initialization",
                                                                    group = "LP mixer"));
        parameter SI.Pressure mixLP_pstart = steamLPNomPressure 
          "Pressure start value of the LP mixer" 
              annotation (Dialog(tab = "Initialization",
                                 group = "LP mixer"));
        
        //Valves Parameters
        parameter Real valveHP_Cv=0 "Cv (US) flow coefficient of the HP valve" annotation(Dialog(group= "HP valves"));
        parameter Modelica.SIunits.Pressure valveHP_dpnom 
          "Nominal pressure drop of the HP valve"                                   annotation(Dialog(group= "HP valves"));
        parameter Real bypassHP_Cv=0 
          "Cv (US) flow coefficient of the HP valve of bypass"                                 annotation(Dialog(group= "HP valves"));
        parameter Real valveIP_Cv=0 "Cv (US) flow coefficient of the IP valve" annotation(Dialog(group= "IP valves"));
        parameter Modelica.SIunits.Pressure valveIP_dpnom 
          "Nominal pressure drop of the IP valve"                                   annotation(Dialog(group= "IP valves"));
        parameter Real bypassIP_Cv=0 
          "Cv (US) flow coefficient of the IP valve of bypass" annotation(Dialog(group= "IP valves"));
        parameter Real valveLP_Cv=0 "Cv (US) flow coefficient of the LP valve" annotation(Dialog(group= "LP valves"));
        parameter Modelica.SIunits.Pressure valveLP_dpnom 
          "Nominal pressure drop of the LP valve"                                   annotation(Dialog(group= "LP valves"));
        parameter Real bypassLP_Cv=0 
          "Cv (US) flow coefficient of the HP valve of bypass" annotation(Dialog(group= "LP valves"));
        parameter Real valveDrumLP_Cv=0 
          "Cv (US) flow coefficient of the valve of pressurization LP drum" 
                                                                          annotation(Dialog(group= "LP valves"));
        
        annotation (Diagram, Icon);
        Water.SteamTurbineStodola ST_HP(
          redeclare package Medium = FluidMedium,
          wnom=steamHPNomFlowRate,
          eta_mech=HPT_eta_mech,
          eta_iso_nom=HPT_eta_iso_nom,
          Kt=HPT_Kt,
          hstartin=HPT_hstart_in,
          hstartout=HPT_hstart_out,
          pstart_in=HPT_pstart_in,
          pstart_out=HPT_pstart_out,
          wstart=HPT_wstart)            annotation (extent=[-120,-20; -80,20]);
        Water.SteamTurbineStodola ST_IP(
          redeclare package Medium = FluidMedium,
          eta_mech=IPT_eta_mech,
          eta_iso_nom=IPT_eta_iso_nom,
          Kt=IPT_Kt,
          hstartin=IPT_hstart_in,
          hstartout=IPT_hstart_out,
          pstart_in=IPT_pstart_in,
          pstart_out=IPT_pstart_out,
          wstart=IPT_wstart,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[0,-20; 40,20]);
        Water.SteamTurbineStodola ST_LP(
          redeclare package Medium = FluidMedium,
          eta_mech=LPT_eta_mech,
          eta_iso_nom=LPT_eta_iso_nom,
          Kt=LPT_Kt,
          hstartin=LPT_hstart_in,
          hstartout=LPT_hstart_out,
          pstart_in=LPT_pstart_in,
          pstart_out=LPT_pstart_out,
          wstart=LPT_wstart,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[120,-20; 160,20]);
        Water.Mixer mixLP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          hstart=mixLP_hstart,
          V=mixLP_V,
          pstart=mixLP_pstart) 
                          annotation (extent=[64,80; 84,100],rotation=270);
        Water.ValveVap valveHP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveHP_Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          dpnom=valveHP_dpnom,
          redeclare package Medium = FluidMedium,
          hstart=HPT_hstart_in) 
                               annotation (extent=[-130,26; -150,6],
                                                                   rotation=180);
        Water.ValveVap valveIP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveIP_Cv,
          pnom=steamIPNomPressure,
          dpnom=valveIP_dpnom,
          redeclare package Medium = FluidMedium,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate,
          hstart=IPT_hstart_in) 
                               annotation (extent=[-10,26; -30,6],rotation=180);
        Water.ValveVap valveLP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          dpnom=valveLP_dpnom,
          redeclare package Medium = FluidMedium,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate,
          Cv=valveLP_Cv,
          hstart=LPT_hstart_in) 
                               annotation (extent=[110,26; 90,6],rotation=180);
        Water.ValveVap byPassHP(
          redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          dpnom=steamHPNomPressure - steamIPNomPressure,
          Cv=bypassHP_Cv,
          hstart=HPT_hstart_in) annotation (extent=[-110,50; -130,30],
                                                                     rotation=180);
        Water.ValveVap byPassIP(
          redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamIPNomPressure,
          dpnom=steamIPNomPressure - steamLPNomPressure,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate,
          Cv=bypassIP_Cv,
          hstart=IPT_hstart_in) annotation (extent=[10,50; -10,30],rotation=180);
        Water.ValveVap byPassLP(
          redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate,
          Cv=bypassLP_Cv,
          dpnom=steamLPNomPressure - pcond,
          hstart=LPT_hstart_in)  annotation (extent=[130,50; 110,30],
                                                                    rotation=180);
        Water.FlowJoin flowHPJoin(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_in2=false,
          rev_out=false)        annotation (extent=[-86,36; -66,56], rotation=
              90);
        Water.FlowSplit flowHPSplit(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false)         annotation (extent=[-170,36; -150,56],
                                                                       rotation=270);
        Water.FlowJoin flowIPJoin(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_in2=false,
          rev_out=false)         annotation (extent=[34,36; 54,56], rotation=90);
        Water.FlowSplit flowIPSplit(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false)          annotation (extent=[-50,36; -30,56],rotation=270);
        Water.FlowJoin flowLPJoin(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_in2=false,
          rev_out=false)         annotation (extent=[166,0; 186,20], rotation=
              270);
        Water.FlowSplit flowLPSplit(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false)          annotation (extent=[64,36; 84,56], rotation=270);
        Water.SensP p_ST_LP(            redeclare package Medium = FluidMedium)
          annotation (extent=[90,74; 110,94]);
      protected 
        replaceable Components.BaseReader_water stateHPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-170,100; -150,80],rotation=90);
        replaceable Components.BaseReader_water stateHPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-110,100; -90,80], rotation=270);
        replaceable Components.BaseReader_water stateIPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-50,100; -30,80],  rotation=90);
        replaceable Components.BaseReader_water stateIPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[34,100; 54,80],    rotation=270);
        replaceable Components.BaseReader_water stateLPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[64,76; 84,56],     rotation=90);
        replaceable Components.BaseReader_water stateLPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[166,-10; 186,-30], rotation=90);
      equation 
        connect(valveIP.outlet,ST_IP. inlet) annotation (points=[-10,16; -10,16;
              4,16],             style(thickness=2));
        connect(ActuatorsBus.Opening_valveHP,valveHP. theta) annotation (points=[200,-140;
              -180,-140; -180,30; -140,30; -140,24],       style(color=52, rgbcolor={
                213,255,170}));
        connect(ActuatorsBus.Opening_valveIP,valveIP. theta) annotation (points=[200,-140;
              -60,-140; -60,34; -20,34; -20,24],      style(color=52, rgbcolor={213,
                255,170}));
        connect(ActuatorsBus.Opening_valveLP,valveLP. theta) annotation (points=[200,-140;
              60,-140; 60,32; 100,32; 100,24],
                                        style(color=52, rgbcolor={213,255,170}));
        connect(ActuatorsBus.Opening_byPassHP,byPassHP. theta) annotation (points=[200,-140;
              -180,-140; -180,30; -140,30; -140,60; -120,60; -120,48],
                                             style(color=52, rgbcolor={213,255,170}));
        connect(ActuatorsBus.Opening_byPassIP,byPassIP. theta) annotation (points=[200,-140;
              -60,-140; -60,34; -20,34; -20,60; 9.79685e-016,60; 9.79685e-016,
              48],                       style(color=52, rgbcolor={213,255,170}));
        connect(ActuatorsBus.Opening_byPassLP,byPassLP. theta) annotation (points=[200,-140;
              60,-140; 60,32; 100,32; 100,60; 120,60; 120,48],
                                             style(color=52, rgbcolor={213,255,170}));
        connect(SensorsBus.p_STLP_in,p_ST_LP. p) annotation (points=[200,-80;
              166,-80; 166,90; 108,90],
                                  style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ST_HP.inlet,valveHP. outlet) annotation (points=[-116,16;
              -126.5,16; -126.5,16; -130,16],   style(thickness=2));
        connect(valveHP.inlet,flowHPSplit. out2) 
          annotation (points=[-150,16; -164,16; -164,40], style(thickness=2));
        connect(byPassHP.inlet,flowHPSplit. out1) 
          annotation (points=[-130,40; -155.8,40], style(thickness=2));
        connect(flowHPJoin.in2,ST_HP. outlet) 
          annotation (points=[-72,40; -72,16; -84,16],   style(thickness=2));
        connect(flowHPJoin.in1,byPassHP. outlet) 
          annotation (points=[-80,40; -110,40], style(thickness=2));
        connect(ST_HP.shaft_a, Shaft_a) annotation (points=[-113.2,0; -200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(valveIP.inlet,flowIPSplit. out2) 
          annotation (points=[-30,16; -44,16; -44,40], style(thickness=2));
        connect(byPassIP.inlet,flowIPSplit. out1) 
          annotation (points=[-10,40; -35.8,40], style(thickness=2));
        connect(ST_IP.outlet,flowIPJoin. in2) 
          annotation (points=[36,16; 48,16; 48,40],   style(thickness=2));
        connect(flowIPJoin.in1,byPassIP. outlet) 
          annotation (points=[40,40; 10,40], style(thickness=2));
        connect(ST_LP.inlet,valveLP. outlet) annotation (points=[124,16; 118.5,
              16; 118.5,16; 110,16],         style(thickness=2));
        connect(valveLP.inlet,flowLPSplit. out2) 
          annotation (points=[90,16; 70,16; 70,40], style(thickness=2));
        connect(byPassLP.inlet,flowLPSplit. out1) 
          annotation (points=[110,40; 78.2,40], style(thickness=2));
        connect(flowLPJoin.in2,ST_LP. outlet) 
          annotation (points=[172,16; 156,16],   style(thickness=2));
        connect(flowLPJoin.in1,byPassLP. outlet) 
          annotation (points=[180,16; 180,40; 130,40], style(thickness=2));
        connect(p_ST_LP.flange,mixLP. out) annotation (points=[100,80; 74,80],
            style(
            color=3,
            rgbcolor={0,0,255},
            thickness=2));
        connect(Shaft_b,ST_LP. shaft_b) annotation (points=[200,0; 152.8,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_a,ST_IP. shaft_b) annotation (points=[126.8,0; 32.8,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_IP.shaft_a,ST_HP. shaft_b) annotation (points=[6.8,0; -87.2,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(stateHPT_in.inlet, HPT_In) 
          annotation (points=[-160,96; -160,200], style(thickness=2));
        connect(stateHPT_out.outlet, HPT_Out) 
          annotation (points=[-100,96; -100,200], style(thickness=2));
        connect(stateIPT_in.inlet, IPT_In) 
          annotation (points=[-40,96; -40,200], style(thickness=2));
        connect(stateIPT_out.outlet, mixLP.in2) annotation (points=[44,96; 44,140;
              68,140; 68,97.9], style(thickness=2));
        connect(stateLPT_in.inlet, mixLP.out) 
          annotation (points=[74,72; 74,80], style(thickness=2));
        connect(LPT_Out, stateLPT_out.outlet) annotation (points=[80,-200; 80,
              -40; 176,-40; 176,-26], style(thickness=2));
        connect(flowHPSplit.in1, stateHPT_in.outlet) 
          annotation (points=[-160,52; -160,84; -160,84], style(thickness=2));
        connect(flowHPJoin.out, stateHPT_out.inlet) annotation (points=[-76,52;
              -76,66; -100,66; -100,84], style(thickness=2));
        connect(flowIPSplit.in1, stateIPT_in.outlet) 
          annotation (points=[-40,52; -40,84], style(thickness=2));
        connect(flowIPJoin.out, stateIPT_out.inlet) 
          annotation (points=[44,52; 44,84], style(thickness=2));
        connect(flowLPSplit.in1, stateLPT_in.outlet) 
          annotation (points=[74,52; 74,60], style(thickness=2));
        connect(stateLPT_out.inlet, flowLPJoin.out) 
          annotation (points=[176,-14; 176,4], style(thickness=2));
        connect(mixLP.in1, LPT_In) annotation (points=[80,98; 80,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end ST2LRh_bypass;
      
      model ST3LRh_bypass 
        "Steam turbine with three pressure levels and reheat, inlet and by-pass valves " 
        extends Interfaces.ST_3LRh;
        
        //Mixers Parameters
        parameter SI.Volume mixLP_V "Internal volume of the LP mixer";
        parameter SI.Enthalpy mixLP_hstart = LPT_hstart_in 
          "Enthalpy start value of the LP mixer" annotation (Dialog(tab = "Initialization",
                                                                    group = "LP mixer"));
        parameter SI.Pressure mixLP_pstart = steamLPNomPressure 
          "Pressure start value of the LP mixer" 
              annotation (Dialog(tab = "Initialization",
                                 group = "LP mixer"));
        
        //Valves Parameters
        parameter Real valveHP_Cv=0 "Cv (US) flow coefficient of the HP valve" annotation(Dialog(group= "HP valves"));
        parameter Modelica.SIunits.Pressure valveHP_dpnom 
          "Nominal pressure drop of the HP valve"                                   annotation(Dialog(group= "HP valves"));
        parameter Real bypassHP_Cv=0 
          "Cv (US) flow coefficient of the HP valve of bypass"                                 annotation(Dialog(group= "HP valves"));
        parameter Real valveIP_Cv=0 "Cv (US) flow coefficient of the IP valve" annotation(Dialog(group= "IP valves"));
        parameter Modelica.SIunits.Pressure valveIP_dpnom 
          "Nominal pressure drop of the IP valve"                                   annotation(Dialog(group= "IP valves"));
        parameter Real bypassIP_Cv=0 
          "Cv (US) flow coefficient of the IP valve of bypass" annotation(Dialog(group= "IP valves"));
        parameter Real valveDrumIP_Cv=0 
          "Cv (US) flow coefficient of the valve of pressurization IP drum" 
                                                                          annotation(Dialog(group= "IP valves"));
        parameter Real valveLP_Cv=0 "Cv (US) flow coefficient of the LP valve" annotation(Dialog(group= "LP valves"));
        parameter Modelica.SIunits.Pressure valveLP_dpnom 
          "Nominal pressure drop of the LP valve"                                   annotation(Dialog(group= "LP valves"));
        parameter Real bypassLP_Cv=0 
          "Cv (US) flow coefficient of the HP valve of bypass" annotation(Dialog(group= "LP valves"));
        parameter Real valveDrumLP_Cv=0 
          "Cv (US) flow coefficient of the valve of pressurization LP drum" 
                                                                          annotation(Dialog(group= "LP valves"));
        
        annotation (Diagram, Icon);
        Water.SteamTurbineStodola ST_HP(
          redeclare package Medium = FluidMedium,
          wnom=steamHPNomFlowRate,
          eta_mech=HPT_eta_mech,
          eta_iso_nom=HPT_eta_iso_nom,
          Kt=HPT_Kt,
          hstartin=HPT_hstart_in,
          hstartout=HPT_hstart_out,
          pstart_in=HPT_pstart_in,
          pstart_out=HPT_pstart_out,
          wstart=HPT_wstart)            annotation (extent=[-120,-20; -80,20]);
        Water.SteamTurbineStodola ST_IP(
          redeclare package Medium = FluidMedium,
          eta_mech=IPT_eta_mech,
          eta_iso_nom=IPT_eta_iso_nom,
          Kt=IPT_Kt,
          hstartin=IPT_hstart_in,
          hstartout=IPT_hstart_out,
          pstart_in=IPT_pstart_in,
          pstart_out=IPT_pstart_out,
          wstart=IPT_wstart,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[0,-20; 40,20]);
        Water.SteamTurbineStodola ST_LP(
          redeclare package Medium = FluidMedium,
          eta_mech=LPT_eta_mech,
          eta_iso_nom=LPT_eta_iso_nom,
          Kt=LPT_Kt,
          hstartin=LPT_hstart_in,
          hstartout=LPT_hstart_out,
          pstart_in=LPT_pstart_in,
          pstart_out=LPT_pstart_out,
          wstart=LPT_wstart,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate) 
                                        annotation (extent=[120,-20; 160,20]);
        Water.Mixer mixLP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then ThermoPower.Choices.Init.Options.steadyState else 
                    ThermoPower.Choices.Init.Options.noInit,
          hstart=mixLP_hstart,
          V=mixLP_V,
          pstart=mixLP_pstart) 
                          annotation (extent=[64,80; 84,100],rotation=270);
        Water.ValveVap valveHP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveHP_Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          dpnom=valveHP_dpnom,
          redeclare package Medium = FluidMedium,
          hstart=HPT_hstart_in) 
                               annotation (extent=[-130,26; -150,6],
                                                                   rotation=180);
        Water.ValveVap valveIP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveIP_Cv,
          pnom=steamIPNomPressure,
          dpnom=valveIP_dpnom,
          redeclare package Medium = FluidMedium,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate,
          hstart=IPT_hstart_in) 
                               annotation (extent=[-10,26; -30,6],rotation=180);
        Water.ValveVap valveLP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          dpnom=valveLP_dpnom,
          redeclare package Medium = FluidMedium,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate,
          Cv=valveLP_Cv,
          hstart=LPT_hstart_in) 
                               annotation (extent=[110,26; 90,6],rotation=180);
        Water.ValveVap byPassHP(
          redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          dpnom=steamHPNomPressure - steamIPNomPressure,
          Cv=bypassHP_Cv,
          hstart=HPT_hstart_in) annotation (extent=[-110,50; -130,30],
                                                                     rotation=180);
        Water.ValveVap byPassIP(
          redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamIPNomPressure,
          dpnom=steamIPNomPressure - steamLPNomPressure,
          wnom=steamIPNomFlowRate + steamHPNomFlowRate,
          Cv=bypassIP_Cv,
          hstart=IPT_hstart_in) annotation (extent=[10,50; -10,30],rotation=180);
        Water.ValveVap byPassLP(
          redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          wnom=steamLPNomFlowRate + steamIPNomFlowRate + steamHPNomFlowRate,
          Cv=bypassLP_Cv,
          dpnom=steamLPNomPressure - pcond,
          hstart=LPT_hstart_in)  annotation (extent=[130,50; 110,30],
                                                                    rotation=180);
        Water.FlowJoin flowHPJoin(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_in2=false,
          rev_out=false)        annotation (extent=[-86,36; -66,56], rotation=
              90);
        Water.FlowSplit flowHPSplit(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false)         annotation (extent=[-170,36; -150,56],
                                                                       rotation=270);
        Water.FlowJoin flowIPJoin(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_in2=false,
          rev_out=false)         annotation (extent=[34,36; 54,56], rotation=90);
        Water.FlowSplit flowIPSplit(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false)          annotation (extent=[-50,36; -30,56],rotation=270);
        Water.FlowJoin flowLPJoin(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_in2=false,
          rev_out=false)         annotation (extent=[166,0; 186,20], rotation=
              270);
        Water.FlowSplit flowLPSplit(            redeclare package Medium = 
              FluidMedium,
          rev_in1=false,
          rev_out1=false,
          rev_out2=false)          annotation (extent=[64,36; 84,56], rotation=270);
        Water.SensP p_ST_LP(            redeclare package Medium = FluidMedium)
          annotation (extent=[90,74; 110,94]);
      protected 
        replaceable Components.BaseReader_water stateHPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-170,100; -150,80],rotation=90);
        replaceable Components.BaseReader_water stateHPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-110,100; -90,80], rotation=270);
        replaceable Components.BaseReader_water stateIPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-50,100; -30,80],  rotation=90);
        replaceable Components.BaseReader_water stateIPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[34,100; 54,80],    rotation=270);
        replaceable Components.BaseReader_water stateLPT_in(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[64,76; 84,56],     rotation=90);
        replaceable Components.BaseReader_water stateLPT_out(redeclare package 
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[166,-10; 186,-30], rotation=90);
      equation 
        connect(valveIP.outlet,ST_IP. inlet) annotation (points=[-10,16; -10,16;
              4,16],             style(thickness=2));
        connect(ActuatorsBus.Opening_valveHP,valveHP. theta) annotation (points=[200,-140;
              -180,-140; -180,30; -140,30; -140,24],       style(color=52, rgbcolor={
                213,255,170}));
        connect(ActuatorsBus.Opening_valveIP,valveIP. theta) annotation (points=[200,-140;
              -60,-140; -60,34; -20,34; -20,24],      style(color=52, rgbcolor={213,
                255,170}));
        connect(ActuatorsBus.Opening_valveLP,valveLP. theta) annotation (points=[200,-140;
              60,-140; 60,32; 100,32; 100,24],
                                        style(color=52, rgbcolor={213,255,170}));
        connect(ActuatorsBus.Opening_byPassHP,byPassHP. theta) annotation (points=[200,-140;
              -180,-140; -180,30; -140,30; -140,60; -120,60; -120,48],
                                             style(color=52, rgbcolor={213,255,170}));
        connect(ActuatorsBus.Opening_byPassIP,byPassIP. theta) annotation (points=[200,-140;
              -60,-140; -60,34; -20,34; -20,60; 9.79685e-016,60; 9.79685e-016,
              48],                       style(color=52, rgbcolor={213,255,170}));
        connect(ActuatorsBus.Opening_byPassLP,byPassLP. theta) annotation (points=[200,-140;
              60,-140; 60,32; 100,32; 100,60; 120,60; 120,48],
                                             style(color=52, rgbcolor={213,255,170}));
        connect(SensorsBus.p_STLP_in,p_ST_LP. p) annotation (points=[200,-80;
              160,-80; 160,90; 108,90],
                                  style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ST_HP.inlet,valveHP. outlet) annotation (points=[-116,16;
              -126.5,16; -126.5,16; -130,16],   style(thickness=2));
        connect(valveHP.inlet,flowHPSplit. out2) 
          annotation (points=[-150,16; -164,16; -164,40], style(thickness=2));
        connect(byPassHP.inlet,flowHPSplit. out1) 
          annotation (points=[-130,40; -155.8,40], style(thickness=2));
        connect(flowHPJoin.in2,ST_HP. outlet) 
          annotation (points=[-72,40; -72,16; -84,16],   style(thickness=2));
        connect(flowHPJoin.in1,byPassHP. outlet) 
          annotation (points=[-80,40; -110,40], style(thickness=2));
        connect(ST_HP.shaft_a, Shaft_a) annotation (points=[-113.2,0; -200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(valveIP.inlet,flowIPSplit. out2) 
          annotation (points=[-30,16; -44,16; -44,40], style(thickness=2));
        connect(byPassIP.inlet,flowIPSplit. out1) 
          annotation (points=[-10,40; -35.8,40], style(thickness=2));
        connect(ST_IP.outlet,flowIPJoin. in2) 
          annotation (points=[36,16; 48,16; 48,40],   style(thickness=2));
        connect(flowIPJoin.in1,byPassIP. outlet) 
          annotation (points=[40,40; 10,40], style(thickness=2));
        connect(ST_LP.inlet,valveLP. outlet) annotation (points=[124,16; 118.5,
              16; 118.5,16; 110,16],         style(thickness=2));
        connect(valveLP.inlet,flowLPSplit. out2) 
          annotation (points=[90,16; 70,16; 70,40], style(thickness=2));
        connect(byPassLP.inlet,flowLPSplit. out1) 
          annotation (points=[110,40; 78.2,40], style(thickness=2));
        connect(flowLPJoin.in2,ST_LP. outlet) 
          annotation (points=[172,16; 156,16],   style(thickness=2));
        connect(flowLPJoin.in1,byPassLP. outlet) 
          annotation (points=[180,16; 180,40; 130,40], style(thickness=2));
        connect(p_ST_LP.flange,mixLP. out) annotation (points=[100,80; 74,80],
            style(
            color=3,
            rgbcolor={0,0,255},
            thickness=2));
        connect(Shaft_b,ST_LP. shaft_b) annotation (points=[200,0; 152.8,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_a,ST_IP. shaft_b) annotation (points=[126.8,0; 32.8,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_IP.shaft_a,ST_HP. shaft_b) annotation (points=[6.8,0; -87.2,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(stateHPT_in.inlet, HPT_In) 
          annotation (points=[-160,96; -160,200], style(thickness=2));
        connect(stateIPT_in.inlet, IPT_In) 
          annotation (points=[-40,96; -40,200], style(thickness=2));
        connect(stateIPT_out.outlet, mixLP.in2) annotation (points=[44,96; 44,140;
              68,140; 68,97.9], style(thickness=2));
        connect(stateLPT_in.inlet, mixLP.out) 
          annotation (points=[74,72; 74,80], style(thickness=2));
        connect(LPT_Out, stateLPT_out.outlet) annotation (points=[140,-200; 140,
              -40; 176,-40; 176,-26], style(thickness=2));
        connect(flowLPSplit.in1, stateLPT_in.outlet) 
          annotation (points=[74,52; 74,60], style(thickness=2));
        connect(flowIPJoin.out, stateIPT_out.inlet) 
          annotation (points=[44,52; 44,84], style(thickness=2));
        connect(flowIPSplit.in1, stateIPT_in.outlet) 
          annotation (points=[-40,52; -40,84], style(thickness=2));
        connect(flowHPJoin.out, stateHPT_out.inlet) annotation (points=[-76,52;
              -76,66; -100,66; -100,84], style(thickness=2));
        connect(flowHPSplit.in1, stateHPT_in.outlet) 
          annotation (points=[-160,52; -160,84], style(thickness=2));
        connect(stateLPT_out.inlet, flowLPJoin.out) 
          annotation (points=[176,-14; 176,4; 176,4], style(thickness=2));
        connect(stateHPT_out.outlet, HPT_Out) 
          annotation (points=[-100,96; -100,200], style(thickness=2));
        connect(mixLP.in1, LPT_In) 
          annotation (points=[80,98; 80,200; 80,200], style(thickness=2));
      end ST3LRh_bypass;
      
      model ST_2LHU 
        "Steam Turbine Group (two pressure levels) with coupling Heat Usage" 
        extends Interfaces.ST_2LHU;
        
        //Mixers Parameters
        parameter SI.Volume mixLP_V "Internal volume of the LP mixer";
        parameter SI.Enthalpy mixLP_hstart 
          "Enthalpy start value of the LP mixer" annotation (Dialog(tab = "Initialization",
                                                                    group = "LP mixer"));
        parameter SI.Pressure mixLP_pstart 
          "Pressure start value of the LP mixer" 
              annotation (Dialog(tab = "Initialization",
                                 group = "LP mixer"));
        
        //Valves Parameters
        parameter Real valveHP_Cv=0 "Cv (US) flow coefficient of the HP valve" annotation(Dialog(group= "HP valves"));
        parameter SI.Pressure valveHP_dpnom 
          "Nominal pressure drop of the HP valve"                                   annotation(Dialog(group= "HP valves"));
        parameter Real bypassHP_Cv=0 
          "Cv (US) flow coefficient of the HP valve of bypass"                                 annotation(Dialog(group= "HP valves"));
        parameter Real valveLP_Cv=0 "Cv (US) flow coefficient of the LP valve" annotation(Dialog(group= "LP valves"));
        parameter SI.Pressure valveLP_dpnom 
          "Nominal pressure drop of the LP valve"                                   annotation(Dialog(group= "LP valves"));
        parameter Real bypassLP_Cv=0 
          "Cv (US) flow coefficient of the HP valve of bypass" annotation(Dialog(group= "LP valves"));
        parameter Real valveHU_Cv=0 
          "Cv (US) flow coefficient of the valve for HU"                           annotation(Dialog(group= "Heat Usage valve"));
        parameter SI.Pressure valveHU_dpnom 
          "Nominal pressure drop of the valve for HU"                                   annotation(Dialog(group= "Heat Usage valve"));
        
        Water.SteamTurbineStodola ST_HP(
          redeclare package Medium = FluidMedium,
          wnom=steamHPNomFlowRate,
          eta_mech=HPT_eta_mech,
          eta_iso_nom=HPT_eta_iso_nom,
          Kt=HPT_Kt,
          hstartin=HPT_hstart_in,
          hstartout=HPT_hstart_out,
          pstart_in=HPT_pstart_in,
          pstart_out=HPT_pstart_out,
          wstart=HPT_wstart)            annotation (extent=[-86,-20; -46,20]);
        Water.SteamTurbineStodola ST_LP(
          redeclare package Medium = FluidMedium,
          eta_mech=LPT_eta_mech,
          eta_iso_nom=LPT_eta_iso_nom,
          Kt=LPT_Kt,
          hstartin=LPT_hstart_in,
          hstartout=LPT_hstart_out,
          pstart_in=LPT_pstart_in,
          pstart_out=LPT_pstart_out,
          wstart=LPT_wstart,
          wnom=steamHPNomFlowRate)      annotation (extent=[94,-20; 134,20]);
        Water.Mixer mixLP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          hstart=mixLP_hstart,
          V=mixLP_V,
          pstart=mixLP_pstart) 
                          annotation (extent=[-50,-120; -30,-100],
                                                             rotation=180);
        Water.ValveVap valveHU(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          CheckValve=true,
          redeclare package Medium = FluidMedium,
          Cv=valveHU_Cv,
          dpnom=valveHU_dpnom,
          pnom=steamLPNomPressure,
          wnom=steamHPNomFlowRate,
          hstart=HPT_hstart_out) 
                               annotation (extent=[-10,-16; -28,-34],
                                                                  rotation=90);
        Water.ValveVap valveLP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          CheckValve=true,
          dpnom=valveLP_dpnom,
          redeclare package Medium = FluidMedium,
          Cv=valveLP_Cv,
          wnom=steamHPNomFlowRate,
          hstart=LPT_hstart_in) 
                               annotation (extent=[60,6; 80,26], rotation=0);
        Water.FlowSplit flowHUSplit(redeclare package Medium = FluidMedium) 
                                  annotation (extent=[-26,20; -6,40],  rotation=270);
        Water.ValveVap valveHP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveHP_Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          CheckValve=true,
          dpnom=valveHP_dpnom,
          redeclare package Medium = FluidMedium,
          hstart=HPT_hstart_in) 
                               annotation (extent=[-118,6; -98,26],rotation=0);
        Water.ValveVap byPassHP(redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          CheckValve=true,
          Cv=bypassHP_Cv,
          dpnom=steamHPNomPressure - steamLPNomPressure,
          hstart=HPT_hstart_in) annotation (extent=[-100,40; -80,60],rotation=0);
        Water.FlowJoin flowHPJoin(redeclare package Medium = FluidMedium) 
                                annotation (extent=[-44,36; -24,56],rotation=0);
        Water.FlowSplit flowHPSplit1(
                                    redeclare package Medium = FluidMedium) 
                                  annotation (extent=[-130,90; -110,110],
                                                                       rotation=270);
        Water.ValveVap byPassLP(redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          CheckValve=true,
          Cv=bypassLP_Cv,
          dpnom=steamLPNomPressure - pcond,
          wnom=steamHPNomFlowRate,
          hstart=LPT_hstart_in)  annotation (extent=[80,40; 100,60],rotation=0);
        Water.FlowSplit flowLPSplit(redeclare package Medium = FluidMedium) 
                                   annotation (extent=[20,10; 40,30], rotation=0);
        Water.FlowJoin flowLPJoin(redeclare package Medium = FluidMedium) 
                                 annotation (extent=[140,36; 160,56],
                                                                    rotation=0);
        annotation (Diagram);
      equation 
        connect(ST_HP.shaft_a, Shaft_a) annotation (points=[-79.2,0; -200,0],
                       style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_b, Shaft_b) annotation (points=[126.8,0; 200,0],
                            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_a, ST_HP.shaft_b) annotation (points=[100.8,0;
              -53.2,0],   style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(flowHPSplit1.in1, HPT_In) 
          annotation (points=[-120,106; -120,200],         style(thickness=2));
        connect(flowHPSplit1.out2, valveHP.inlet) annotation (points=[-124,94;
              -124,16; -118,16],               style(thickness=2));
        connect(valveHP.outlet, ST_HP.inlet) 
          annotation (points=[-98,16; -82,16],           style(thickness=2));
        connect(flowHPJoin.in1, byPassHP.outlet) annotation (points=[-40,50;
              -80,50],                         style(thickness=2));
        connect(mixLP.out, SteamForHU) 
          annotation (points=[-50,-110; -60,-110; -60,-200],
                                                           style(thickness=2));
        connect(valveHU.inlet, flowHUSplit.out2) annotation (points=[-19,-16;
              -20,-16; -20,24],       style(thickness=2));
        connect(LPT_In, mixLP.in1) annotation (points=[0,200; 0,-116; -32,-116],
                                            style(thickness=2));
        connect(valveLP.outlet, ST_LP.inlet) 
          annotation (points=[80,16; 98,16],          style(thickness=2));
        connect(flowHPJoin.in2, ST_HP.outlet) annotation (points=[-40,42; -50,
              42; -50,16],                  style(thickness=2));
        connect(mixLP.in2, valveHU.outlet) annotation (points=[-32.1,-104; -19,
              -104; -19,-34],      style(thickness=2));
        connect(byPassLP.inlet,flowLPSplit. out1) annotation (points=[80,50; 36,
              50; 36,24.2],              style(thickness=2));
        connect(flowLPSplit.out2, valveLP.inlet) annotation (points=[36,16; 60,
              16],                  style(thickness=2));
        connect(flowLPSplit.in1, flowHUSplit.out1) annotation (points=[24,20;
              -12,20; -12,22; -11.8,24],                      style(thickness=2));
        connect(flowHUSplit.in1, flowHPJoin.out) 
          annotation (points=[-16,36; -16,46; -28,46],
                                                  style(thickness=2));
        connect(byPassHP.inlet, flowHPSplit1.out1) annotation (points=[-100,50;
              -115.8,50; -115.8,94],           style(thickness=2));
        connect(flowLPJoin.out, LPT_Out) annotation (points=[156,46; 160,46;
              160,-120; 60,-120; 60,-200],
                                     style(thickness=2));
        connect(flowLPJoin.in1, byPassLP.outlet) 
          annotation (points=[144,50; 100,50],            style(thickness=2));
        connect(flowLPJoin.in2, ST_LP.outlet) annotation (points=[144,42; 130,
              42; 130,16],        style(thickness=2));
        connect(ActuatorsBus.Opening_valveHP, valveHP.theta) annotation (points=[200,-140;
              -140,-140; -140,32; -108,32; -108,24],
                                                style(color=52, rgbcolor={213,
                255,170}));
        connect(ActuatorsBus.Opening_valveLP, valveLP.theta) annotation (points=[200,-140;
              50,-140; 50,32; 70,32; 70,24],
                                          style(color=52, rgbcolor={213,255,170}));
        connect(ActuatorsBus.Opening_byPassHP, byPassHP.theta) annotation (
            points=[200,-140; 20,-140; 20,70; -90,70; -90,58],
                                                      style(color=52, rgbcolor=
                {213,255,170}));
        connect(ActuatorsBus.Opening_byPassLP, byPassLP.theta) annotation (
            points=[200,-140; 50,-140; 50,70; 90,70; 90,58],
                                                 style(color=52, rgbcolor={213,
                255,170}));
        connect(ActuatorsBus.Opening_valveHU, valveHU.theta) annotation (points=[200,-140;
              50,-140; 50,-26; 20,-26; -11.8,-25],
                                               style(color=52, rgbcolor={213,
                255,170}));
      end ST_2LHU;
      
      model ST_2LRhHU 
        "Steam Turbine Group (two pressure levels and reheat) with coupling Heat Usage" 
        extends Interfaces.ST_2LRhHU;
        
        //Mixers Parameters
        parameter SI.Volume mixLP_V "Internal volume of the LP mixer";
        parameter SI.Enthalpy mixLP_hstart 
          "Enthalpy start value of the LP mixer" annotation (Dialog(tab = "Initialization",
                                                                    group = "LP mixer"));
        parameter SI.Pressure mixLP_pstart 
          "Pressure start value of the LP mixer" 
              annotation (Dialog(tab = "Initialization",
                                 group = "LP mixer"));
        
        //Valves Parameters
        parameter Real valveHP_Cv=0 "Cv (US) flow coefficient of the HP valve" annotation(Dialog(group= "HP valves"));
        parameter SI.Pressure valveHP_dpnom 
          "Nominal pressure drop of the HP valve"                                   annotation(Dialog(group= "HP valves"));
        parameter Real bypassHP_Cv=0 
          "Cv (US) flow coefficient of the HP valve of bypass"                                 annotation(Dialog(group= "HP valves"));
        parameter Real valveLP_Cv=0 "Cv (US) flow coefficient of the LP valve" annotation(Dialog(group= "LP valves"));
        parameter SI.Pressure valveLP_dpnom 
          "Nominal pressure drop of the LP valve"                                   annotation(Dialog(group= "LP valves"));
        parameter Real bypassLP_Cv=0 
          "Cv (US) flow coefficient of the HP valve of bypass" annotation(Dialog(group= "LP valves"));
        
        annotation (Diagram);
        Water.SteamTurbineStodola ST_HP(
          redeclare package Medium = FluidMedium,
          wnom=steamHPNomFlowRate,
          eta_mech=HPT_eta_mech,
          eta_iso_nom=HPT_eta_iso_nom,
          Kt=HPT_Kt,
          hstartin=HPT_hstart_in,
          hstartout=HPT_hstart_out,
          pstart_in=HPT_pstart_in,
          pstart_out=HPT_pstart_out,
          wstart=HPT_wstart)            annotation (extent=[-126,-20; -86,20]);
        Water.SteamTurbineStodola ST_LP(
          redeclare package Medium = FluidMedium,
          eta_mech=LPT_eta_mech,
          eta_iso_nom=LPT_eta_iso_nom,
          Kt=LPT_Kt,
          hstartin=LPT_hstart_in,
          hstartout=LPT_hstart_out,
          pstart_in=LPT_pstart_in,
          pstart_out=LPT_pstart_out,
          wstart=LPT_wstart,
          wnom=steamHPNomFlowRate)      annotation (extent=[54,-20; 94,20]);
        Water.Mixer mixLP(
          redeclare package Medium = FluidMedium,
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          hstart=mixLP_hstart,
          V=mixLP_V,
          pstart=mixLP_pstart) 
                          annotation (extent=[-54,-94; -34,-74],
                                                             rotation=270);
        Water.ValveVap valveLP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          CheckValve=true,
          dpnom=valveLP_dpnom,
          redeclare package Medium = FluidMedium,
          Cv=valveLP_Cv,
          wnom=steamHPNomFlowRate,
          hstart=LPT_hstart_in) 
                               annotation (extent=[20,6; 40,26], rotation=0);
        Water.FlowSplit flowHUSplit(redeclare package Medium = FluidMedium) 
                                  annotation (extent=[-54,-138; -34,-118],
                                                                       rotation=270);
        Water.ValveVap valveHP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          Cv=valveHP_Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          CheckValve=true,
          dpnom=valveHP_dpnom,
          redeclare package Medium = FluidMedium,
          hstart=HPT_hstart_in) 
                               annotation (extent=[-158,6; -138,26],
                                                                   rotation=0);
        Water.ValveVap byPassHP(redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamHPNomPressure,
          wnom=steamHPNomFlowRate,
          CheckValve=true,
          Cv=bypassHP_Cv,
          dpnom=steamHPNomPressure - steamLPNomPressure,
          hstart=HPT_hstart_in) annotation (extent=[-140,40; -120,60],
                                                                     rotation=0);
        Water.FlowJoin flowHPJoin(redeclare package Medium = FluidMedium) 
                                annotation (extent=[-84,36; -64,56],rotation=0);
        Water.FlowSplit flowHPSplit1(
                                    redeclare package Medium = FluidMedium) 
                                  annotation (extent=[-170,90; -150,110],
                                                                       rotation=270);
        Water.ValveVap byPassLP(redeclare package Medium = FluidMedium,
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          pnom=steamLPNomPressure,
          CheckValve=true,
          Cv=bypassLP_Cv,
          dpnom=steamLPNomPressure - pcond,
          wnom=steamHPNomFlowRate,
          hstart=LPT_hstart_in)  annotation (extent=[40,40; 60,60], rotation=0);
        Water.FlowSplit flowLPSplit(redeclare package Medium = FluidMedium) 
                                   annotation (extent=[-20,10; 0,30], rotation=0);
        Water.FlowJoin flowLPJoin(redeclare package Medium = FluidMedium) 
                                 annotation (extent=[100,10; 120,30],
                                                                    rotation=0);
      equation 
        connect(ST_HP.shaft_a, Shaft_a) annotation (points=[-119.2,0; -200,0],
                       style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_b, Shaft_b) annotation (points=[86.8,0; 200,0],
                            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ST_LP.shaft_a,ST_HP. shaft_b) annotation (points=[60.8,0; -93.2,
              0],         style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(flowHPSplit1.in1, HPT_In) 
          annotation (points=[-160,106; -160,200],         style(thickness=2));
        connect(flowHPSplit1.out2,valveHP. inlet) annotation (points=[-164,94;
              -164,16; -158,16],               style(thickness=2));
        connect(valveHP.outlet,ST_HP. inlet) 
          annotation (points=[-138,16; -122,16],         style(thickness=2));
        connect(flowHPJoin.in1,byPassHP. outlet) annotation (points=[-80,50;
              -120,50],                        style(thickness=2));
        connect(LPT_In,mixLP. in1) annotation (points=[80,200; 80,-60; -38,-60;
              -38,-76],                     style(thickness=2));
        connect(valveLP.outlet,ST_LP. inlet) 
          annotation (points=[40,16; 58,16],          style(thickness=2));
        connect(flowHPJoin.in2,ST_HP. outlet) annotation (points=[-80,42; -90,
              42; -90,16],                  style(thickness=2));
        connect(byPassLP.inlet,flowLPSplit. out1) annotation (points=[40,50; 6,
              50; 6,24; -4,24; -4,24.2], style(thickness=2));
        connect(flowLPSplit.out2,valveLP. inlet) annotation (points=[-4,16; 20,
              16],                  style(thickness=2));
        connect(byPassHP.inlet,flowHPSplit1. out1) annotation (points=[-140,50;
              -155.8,50; -155.8,94],           style(thickness=2));
        connect(flowLPJoin.out, LPT_Out) annotation (points=[116,20; 140,20;
              140,-120; 80,-120; 80,-200],
                                     style(thickness=2));
        connect(flowLPJoin.in1,byPassLP. outlet) 
          annotation (points=[104,24; 98,24; 98,50; 60,50],
                                                          style(thickness=2));
        connect(flowLPJoin.in2,ST_LP. outlet) annotation (points=[104,16; 90,16],
                                  style(thickness=2));
        connect(ActuatorsBus.Opening_valveHP,valveHP. theta) annotation (points=[200,-140;
              -180,-140; -180,32; -148,32; -148,24],
                                                style(color=52, rgbcolor={213,
                255,170}));
        connect(ActuatorsBus.Opening_valveLP,valveLP. theta) annotation (points=[200,-140;
              10,-140; 10,32; 30,32; 30,24],
                                          style(color=52, rgbcolor={213,255,170}));
        connect(ActuatorsBus.Opening_byPassHP,byPassHP. theta) annotation (
            points=[200,-140; -20,-140; -20,70; -130,70; -130,58],
                                                      style(color=52, rgbcolor=
                {213,255,170}));
        connect(ActuatorsBus.Opening_byPassLP,byPassLP. theta) annotation (
            points=[200,-140; 10,-140; 10,70; 50,70; 50,58],
                                                 style(color=52, rgbcolor={213,
                255,170}));
        connect(SteamForHU, flowHUSplit.out1) annotation (points=[-40,-200; -40,
              -134; -39.8,-134], style(thickness=2));
        connect(flowHUSplit.in1, mixLP.out) 
          annotation (points=[-44,-122; -44,-94], style(thickness=2));
        connect(mixLP.in2, flowHPJoin.out) 
          annotation (points=[-50,-76.1; -50,46; -68,46], style(thickness=2));
        connect(flowLPSplit.in1, IPT_In) 
          annotation (points=[-16,20; -40,20; -40,200], style(thickness=2));
        connect(flowHUSplit.out2, HPT_Out) annotation (points=[-48,-134; -48,
              -160; -60,-160; -60,160; -100,160; -100,200], style(thickness=2));
      end ST_2LRhHU;
      
      model CondPlant "Condensation Plant" 
        extends Interfaces.CondPlant;
        replaceable Components.Condenser condenser(
          redeclare package FluidMedium = FluidMedium,
          N_cool=N_cool,
          condNomFlowRate=condNomFlowRate,
          coolNomFlowRate=coolNomFlowRate,
          condNomPressure=condNomPressure,
          coolNomPressure=coolNomPressure,
          condExchSurface=condExchSurface,
          coolExchSurface=coolExchSurface,
          condVol=condVol,
          coolVol=coolVol,
          metalVol=metalVol,
          cm=cm,
          rhoMetal=rhoMetal,
          use_T=use_T,
          SSInit=SSInit,
          pstart_cond=pstart_cond,
          Tstart_M_in=Tstart_M_in,
          Tstart_M_out=Tstart_M_out,
          Tstart_cool_in=Tstart_cool_in,
          Tstart_cool_out=Tstart_cool_out,
          hstart_cool_in=hstart_cool_in,
          hstart_cool_out=hstart_cool_out)       extends Interfaces.Condenser(
          redeclare package FluidMedium = FluidMedium,
          N_cool=N_cool,
          condNomFlowRate=condNomFlowRate,
          coolNomFlowRate=coolNomFlowRate,
          condNomPressure=condNomPressure,
          coolNomPressure=coolNomPressure,
          condExchSurface=condExchSurface,
          coolExchSurface=coolExchSurface,
          condVol=condVol,
          coolVol=coolVol,
          metalVol=metalVol,
          cm=cm,
          rhoMetal=rhoMetal,
          use_T=use_T,
          SSInit=SSInit,
          pstart_cond=pstart_cond,
          Tstart_M_in=Tstart_M_in,
          Tstart_M_out=Tstart_M_out,
          Tstart_cool_in=Tstart_cool_in,
          Tstart_cool_out=Tstart_cool_out,
          hstart_cool_in=hstart_cool_in,
          hstart_cool_out=hstart_cool_out) 
                                       annotation (extent=[-18,-22; 22,18]);
        annotation (Diagram);
        Water.SourceW coolingIn(redeclare package Medium = FluidMedium,
          p0=coolNomPressure,
          w0=coolNomFlowRate,
          h=hstart_cool_in)   annotation (extent=[90,-24; 70,-4]);
        Water.SinkP coolingOut(redeclare package Medium = FluidMedium,
          p0=coolNomPressure,
          h=hstart_cool_out) 
                          annotation (extent=[70,30; 90,50]);
        replaceable Components.BaseReader_water stateCoolingOut(redeclare 
            package Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[40,30; 60,50]);
        replaceable Components.BaseReader_water stateCoolingIn(redeclare 
            package Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[40,-24; 60,-4], rotation=180);
        Modelica.Blocks.Interfaces.RealOutput ratio_VvonVtot 
          annotation (extent=[-116,-10; -96,10], rotation=180);
      equation 
        connect(condenser.steamIn, SteamIn) 
          annotation (points=[0,18; 0,100], style(thickness=2));
        connect(condenser.waterOut, WaterOut) 
          annotation (points=[0,-22; 0,-100], style(thickness=2));
        connect(stateCoolingOut.outlet, coolingOut.flange) annotation (points=[
              56,40; 70,40], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateCoolingOut.inlet, condenser.coolingOut) annotation (points=[44,40;
              34,40; 34,10; 22,10],      style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateCoolingIn.inlet, coolingIn.flange) annotation (points=[56,
              -14; 70,-14], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(stateCoolingIn.outlet, condenser.coolingIn) annotation (points=[44,-14;
              34,-14; 34,-10; 22,-10],            style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(condenser.ratio_VvonVtot, ratio_VvonVtot) annotation (points=[-18,0;
              -106,0],        style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end CondPlant;
      
      model CondPlant_cc 
        "Condensation plant with condenser ratio control (type ImpPressureCondenser_tap)" 
        extends Interfaces.CondPlant_base;
        parameter Real setPoint_ratio 
          "SetPoint ratio Steam Volume / Total Volume";
        
        Control.PID pID(
          PVmax=1,
          PVmin=0.1,
          CSmin=-0.6,
          CSmax=0.6,
          Ti=2000,
          Kp=-3)      annotation (extent=[-40,34; -60,54], rotation=0);
        Modelica.Blocks.Sources.Constant setPoint(k=setPoint_ratio) 
                    annotation (extent=[-6,42; -18,54]);
        ThermoPower.PowerPlants.SteamTurbineGroup.Components.CondenserPreP_tap 
          condenserIdeal_tap(                            redeclare package 
            Medium = FluidMedium,
          p=p,
          Vtot=Vtot,
          Vlstart=Vlstart)                  annotation (extent=[16,-26; 76,34]);
        Water.SourceW sourceTap(
          redeclare package Medium = FluidMedium,
          p0=p)       annotation (extent=[-80,-30; -60,-10]);
        annotation (Diagram);
        ThermoPower.PowerPlants.SteamTurbineGroup.Components.Comp_bubble_h 
          BubbleEnthalpy(redeclare package FluidMedium = FluidMedium, p=p) 
                                annotation (extent=[-30,-10; -60,10]);
      protected 
        replaceable Components.BaseReader_water stateTappings(redeclare package
            Medium = FluidMedium) extends Components.BaseReader_water(
            redeclare package Medium = FluidMedium) 
                           annotation (extent=[-40,-10; -20,-30], rotation=0);
      equation 
        
        connect(setPoint.y, pID.SP) 
                                 annotation (points=[-18.6,48; -40,48],
                             style(color=74, rgbcolor={0,0,127}));
        connect(pID.CS,sourceTap. in_w0) annotation (points=[-60.6,44; -74,44;
              -74,-14],                                 style(color=74,
              rgbcolor={0,0,127}));
        connect(pID.PV,condenserIdeal_tap. ratio_Vv_Vtot) annotation (points=[-40,40;
              -28,40; -28,-2; 16,-2],                          style(color=74,
              rgbcolor={0,0,127}));
        connect(condenserIdeal_tap.steamIn, SteamIn) 
          annotation (points=[46,34; 46,64; 0,64; 0,100], style(thickness=2));
        connect(condenserIdeal_tap.waterOut, WaterOut) annotation (points=[46,-26;
              46,-52; 0,-52; 0,-100],     style(thickness=2));
        connect(BubbleEnthalpy.h, sourceTap.in_h)                  annotation (
            points=[-58.5,0; -66,0; -66,-14],                 style(color=74,
              rgbcolor={0,0,127}));
        connect(SensorsBus.Cond_ratio, condenserIdeal_tap.ratio_Vv_Vtot) 
          annotation (points=[98,-40; -8,-40; -8,-2; 16,-2], style(color=84,
              rgbcolor={255,170,213}));
        connect(SensorsBus.Cond_Q, condenserIdeal_tap.Qcond) annotation (points=[98,-40;
              -8,-40; -8,16; 16,16],         style(color=84, rgbcolor={255,170,
                213}));
        connect(stateTappings.inlet, sourceTap.flange) 
          annotation (points=[-36,-20; -60,-20], style(thickness=2));
        connect(condenserIdeal_tap.tapWater, stateTappings.outlet) 
          annotation (points=[22,-20; -24,-20], style(thickness=2));
      end CondPlant_cc;
      
      model CondPlantHU_cc 
        "Condensation plant with condenser ratio control (type ImpPressureCondenser_tap) and coupling Heat Usage" 
        extends Interfaces.CondPlantHU_base;
        parameter SI.Volume mixCondenser_V "Internal volume of the mixer";
        parameter SI.Enthalpy mixCondenser_hstart=1e5 "Enthalpy start value";
        parameter SI.Pressure mixCondenser_pstart=1e5 "Pressure start value";
        
        CondPlant_cc condPlant_cc annotation (extent=[20,0; 80,60]);
        annotation (Diagram);
        Water.Mixer mixCondenser(
          initOpt=if SSInit then Options.steadyState else Options.noInit,
          redeclare package Medium = FluidMedium,
          V=mixCondenser_V,
          pstart=mixCondenser_pstart,
          hstart=mixCondenser_hstart) 
                          annotation (extent=[-10,-60; 10,-40],  rotation=270);
      equation 
        connect(condPlant_cc.SteamIn, SteamIn) 
          annotation (points=[50,60; 50,100], style(thickness=2));
        connect(WaterOut, mixCondenser.out) annotation (points=[0,-100; 0,-60;
              -1.83691e-015,-60], style(thickness=2));
        connect(mixCondenser.in1, condPlant_cc.WaterOut) 
          annotation (points=[6,-42; 6,-20; 50,-20; 50,0], style(thickness=2));
        connect(SensorsBus, condPlant_cc.SensorsBus) annotation (points=[98,-40;
              80,-40; 80,-20; 96,-20; 96,18; 79.4,18], style(color=84, rgbcolor=
               {255,170,213}));
        connect(ActuatorsBus, condPlant_cc.ActuatorsBus) annotation (points=[98,-72;
              72,-72; 72,-12; 88,-12; 88,8.4; 79.4,8.4],      style(color=52,
              rgbcolor={213,255,170}));
        connect(mixCondenser.in2, CondensatedFromHU) annotation (points=[-6,
              -42.1; -6,-20; -50,-20; -50,100], style(thickness=2));
      end CondPlantHU_cc;
      
      model STG_3LRh "Steam turbine group (three pressure levels, reheat)" 
        extends Interfaces.STGroup3LRh;
        annotation (Diagram);
        ST3LRh_base steamTurbines(
          redeclare package FluidMedium = FluidMedium,
          steamHPNomFlowRate=67.6,
          steamHPNomPressure=1.28e7,
          steamIPNomPressure=2.68e6,
          steamLPNomPressure=6e5,
          pcond=5.3982e3,
          HPT_hstart_in=3.47e6,
          HPT_eta_iso_nom=0.833,
          IPT_eta_iso_nom=0.903,
          LPT_eta_iso_nom=0.876,
          HPT_Kt=0.0032078,
          IPT_Kt=0.018883,
          LPT_Kt=0.078004,
          HPT_hstart_out=3.10745e6,
          IPT_hstart_in=3.54348e6,
          IPT_hstart_out=3.12011e6,
          LPT_hstart_in=3.11911e6,
          LPT_hstart_out=2.39102e6,
          mixLP_hstart=3.11911e6,
          mixLP_V=5,
          steamIPNomFlowRate=81.3524 - 67.6,
          steamLPNomFlowRate=89.438 - 81.3524) 
                       annotation (extent=[-80,-50; 20,50]);
        ThermoPower.PowerPlants.SteamTurbineGroup.Components.CondenserPreP 
          condenserIdeal(redeclare package Medium = FluidMedium, p=5389.2) 
          annotation (extent=[26,-140; -14,-100],
                                               rotation=0);
        HRSG.Components.PrescribedSpeedPump totalFeedPump(
          redeclare package WaterMedium = FluidMedium,
          rho_nom=1000,
          q_nom={0.0898,0,0.1},
          head_nom={72.74,130,0},
          nominalCondensationPressure=5398.2,
          nominalSteamFlow=89.8,
          n0=1500,
          hstart=1.43495e5,
          nominalSteamPressure=6e5) 
                                  annotation (extent=[92,-148; 60,-180],
            rotation=180);
      equation 
        connect(steamTurbines.Shaft_b, Shaft_b) annotation (points=[20,0; 200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(steamTurbines.Shaft_a, Shaft_a) annotation (points=[-80,0; -200,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(SensorsBus.ratioVvVtot, condenserIdeal.ratio_Vv_Vtot) 
          annotation (points=[200,-80; 76,-80; 76,-124; 26,-124],      style(
              color=84, rgbcolor={255,170,213}));
        connect(SensorsBus.Qout_condenser, condenserIdeal.Qcond) annotation (
            points=[200,-80; 76,-80; 76,-112; 26,-112],      style(color=84,
              rgbcolor={255,170,213}));
        connect(condenserIdeal.steamIn, steamTurbines.LPT_Out) annotation (
            points=[6,-100; 6,-50; 5,-50], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.IPT_In, From_RH_IP) annotation (points=[-40,50;
              -40,200],         style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_Out, To_RH_IP) annotation (points=[-55,50;
              -55,140; -100,140; -100,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.LPT_In, From_SH_LP) annotation (points=[-10,50;
              -10,100; 80,100; 80,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_In, From_SH_HP) annotation (points=[-70,50;
              -70,100; -160,100; -160,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ActuatorsBus, steamTurbines.ActuatorsBus) annotation (points=[
              200,-140; 60,-140; 60,-35; 20,-35], style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(SensorsBus, steamTurbines.SensorsBus) annotation (points=[200,
              -80; 76,-80; 76,-20; 20,-20], style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(totalFeedPump.inlet, condenserIdeal.waterOut) annotation (
            points=[60,-164; 6,-164; 6,-140], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.outlet, WaterOut) annotation (points=[92,-164;
              160,-164; 160,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(ActuatorsBus.nPump_feedLP, totalFeedPump.pumpSpeed_rpm) 
          annotation (points=[200,-140; 40,-140; 40,-154.4; 58.72,-154.4],
            style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end STG_3LRh;
      
      model STG_3LRh_valve 
        "Steam turbine group (three pressure levels, reheat) with inlet valves" 
        extends Interfaces.STGroup3LRh;
        annotation (Diagram);
        ST3LRh_valve steamTurbines(
          redeclare package FluidMedium = FluidMedium,
          steamHPNomFlowRate=67.6,
          steamHPNomPressure=1.28e7,
          steamIPNomFlowRate=81.10 - 67.5,
          steamIPNomPressure=2.68e6,
          steamLPNomPressure=6e5,
          pcond=5.3982e3,
          steamLPNomFlowRate=89.82 - 81.10,
          HPT_hstart_in=3.47e6,
          HPT_hstart_out=3.1076e6,
          IPT_hstart_in=3.554e6,
          IPT_hstart_out=3.12810e6,
          LPT_hstart_in=3.109e6,
          LPT_hstart_out=2.3854e6,
          valveHP_dpnom=1.6e5,
          valveIP_dpnom=5e4,
          mixLP_hstart=3.109e6,
          HPT_eta_iso_nom=0.833,
          IPT_eta_iso_nom=0.903,
          LPT_eta_iso_nom=0.876,
          mixLP_V=20,
          valveHP_Cv=1165,
          valveIP_Cv=5625,
          valveLP_Cv=14733,
          valveLP_dpnom=2.964e4,
          HPT_Kt=0.0032078,
          IPT_Kt=0.018883,
          LPT_Kt=0.078004) 
                       annotation (extent=[-80,-50; 20,50]);
        ThermoPower.PowerPlants.SteamTurbineGroup.Components.CondenserPreP 
          condenserIdeal(                        p=5398.2,
          redeclare package Medium = FluidMedium) 
          annotation (extent=[26,-140; -14,-100],
                                               rotation=0);
        HRSG.Components.PrescribedSpeedPump totalFeedPump(
          redeclare package WaterMedium = FluidMedium,
          rho_nom=1000,
          q_nom={0.0898,0,0.1},
          head_nom={72.74,130,0},
          nominalCondensationPressure=5398.2,
          nominalSteamFlow=89.8,
          n0=1500,
          hstart=1.43495e5,
          nominalSteamPressure=6e5) 
                                  annotation (extent=[92,-148; 60,-180],
            rotation=180);
      equation 
        connect(steamTurbines.Shaft_a, Shaft_a) annotation (points=[-80,0; -200,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(steamTurbines.Shaft_b, Shaft_b) annotation (points=[20,0; 200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ActuatorsBus, steamTurbines.ActuatorsBus) annotation (points=[200,-140;
              60,-140; 60,-35; 20,-35],        style(color=52, rgbcolor={213,
                255,170}));
        connect(SensorsBus.ratioVvVtot, condenserIdeal.ratio_Vv_Vtot) annotation (
            points=[200,-80; 66,-80; 66,-124; 26,-124],              style(
              color=84, rgbcolor={255,170,213}));
        connect(SensorsBus.Qout_condenser, condenserIdeal.Qcond) annotation (
            points=[200,-80; 66,-80; 66,-112; 26,-112],              style(
              color=84, rgbcolor={255,170,213}));
        connect(condenserIdeal.steamIn, steamTurbines.LPT_Out) annotation (
            points=[6,-100; 6,-50; 5,-50], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.IPT_In, From_RH_IP) annotation (points=[-40,50;
              -40,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.LPT_In, From_SH_LP) annotation (points=[-10,50;
              -10,100; 80,100; 80,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_Out, To_RH_IP) annotation (points=[-55,50;
              -55,140; -100,140; -100,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_In, From_SH_HP) annotation (points=[-70,50;
              -70,100; -160,100; -160,200],         style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(SensorsBus, steamTurbines.SensorsBus) annotation (points=[200,
              -80; 66,-80; 66,-20; 20,-20], style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ActuatorsBus.nPump_feedLP, totalFeedPump.pumpSpeed_rpm) 
          annotation (points=[200,-140; 40,-140; 40,-154.4; 58.72,-154.4],
            style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.inlet, condenserIdeal.waterOut) annotation (
            points=[60,-164; 6,-164; 6,-140], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.outlet, WaterOut) annotation (points=[92,-164;
              160,-164; 160,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end STG_3LRh_valve;
      
      model STG_3LRh_cc 
        "Steam turbine group (three pressure levels, reheat) and controlled condenser" 
        extends Interfaces.STGroup3LRh;
        annotation (Diagram);
        ST3LRh_base steamTurbines(
          redeclare package FluidMedium = FluidMedium,
          steamHPNomFlowRate=67.6,
          steamHPNomPressure=1.28e7,
          steamIPNomPressure=2.68e6,
          steamLPNomPressure=6e5,
          pcond=5.3982e3,
          HPT_hstart_in=3.47e6,
          HPT_eta_iso_nom=0.833,
          IPT_eta_iso_nom=0.903,
          LPT_eta_iso_nom=0.876,
          HPT_Kt=0.0032078,
          IPT_Kt=0.018883,
          LPT_Kt=0.078004,
          HPT_hstart_out=3.10745e6,
          IPT_hstart_in=3.54348e6,
          IPT_hstart_out=3.12011e6,
          LPT_hstart_in=3.11911e6,
          LPT_hstart_out=2.39102e6,
          mixLP_hstart=3.11911e6,
          mixLP_V=5,
          steamIPNomFlowRate=81.3524 - 67.6,
          steamLPNomFlowRate=89.438 - 81.3524) 
                       annotation (extent=[-80,-50; 20,50]);
        ThermoPower.PowerPlants.SteamTurbineGroup.Examples.CondPlant_cc 
          controlledCondeser(
          redeclare package FluidMedium = FluidMedium,
          p=5398.2,
          Vtot=10,
          Vlstart=1.5,
          setPoint_ratio=0.85) annotation (extent=[-26,-140; 34,-80],
                                                                    rotation=0);
        HRSG.Components.PrescribedSpeedPump totalFeedPump(
          redeclare package WaterMedium = FluidMedium,
          rho_nom=1000,
          q_nom={0.0898,0,0.1},
          head_nom={72.74,130,0},
          nominalCondensationPressure=5398.2,
          nominalSteamFlow=89.8,
          n0=1500,
          hstart=1.43495e5,
          nominalSteamPressure=6e5) 
                                  annotation (extent=[112,-154; 80,-186],
            rotation=180);
      equation 
        connect(steamTurbines.Shaft_b, Shaft_b) annotation (points=[20,0; 200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(steamTurbines.Shaft_a, Shaft_a) annotation (points=[-80,0; -200,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(controlledCondeser.SteamIn, steamTurbines.LPT_Out) annotation (
            points=[4,-80; 4,-50; 5,-50],  style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.IPT_In, From_RH_IP) annotation (points=[-40,50;
              -40,200],          style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_Out, To_RH_IP) annotation (points=[-55,50;
              -55,140; -100,140; -100,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_In, From_SH_HP) annotation (points=[-70,50;
              -70,100; -160,100; -160,200],           style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.LPT_In, From_SH_LP) annotation (points=[-10,50;
              -10,100; 80,100; 80,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(SensorsBus, steamTurbines.SensorsBus) annotation (points=[200,
              -80; 80,-80; 80,-20; 20,-20], style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(SensorsBus, controlledCondeser.SensorsBus) annotation (points=[200,-80;
              80,-80; 80,-122; 33.4,-122],          style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ActuatorsBus, steamTurbines.ActuatorsBus) annotation (points=[
              200,-140; 100,-140; 100,-35; 20,-35], style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ActuatorsBus, controlledCondeser.ActuatorsBus) annotation (
            points=[200,-140; 100,-140; 100,-131.6; 33.4,-131.6], style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ActuatorsBus.nPump_feedLP, totalFeedPump.pumpSpeed_rpm) 
          annotation (points=[200,-140; 60,-140; 60,-160.4; 78.72,-160.4],
            style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.inlet, controlledCondeser.WaterOut) annotation (
            points=[80,-170; 4,-170; 4,-140], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.outlet, WaterOut) annotation (points=[112,-170;
              160,-170; 160,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end STG_3LRh_cc;
      
      model STG_3LRh_valve_cc 
        "Steam turbine group (three pressure levels, reheat) with inlet valves and controlled condenser" 
        extends Interfaces.STGroup3LRh;
        annotation (Diagram);
        ST3LRh_valve steamTurbines(
          redeclare package FluidMedium = FluidMedium,
          steamHPNomFlowRate=67.6,
          steamHPNomPressure=1.28e7,
          steamIPNomPressure=2.68e6,
          steamLPNomPressure=6e5,
          pcond=5.3982e3,
          HPT_hstart_in=3.47e6,
          HPT_eta_iso_nom=0.833,
          IPT_eta_iso_nom=0.903,
          LPT_eta_iso_nom=0.876,
          HPT_Kt=0.0032078,
          IPT_Kt=0.018883,
          LPT_Kt=0.078004,
          HPT_hstart_out=3.10745e6,
          IPT_hstart_in=3.54348e6,
          IPT_hstart_out=3.12011e6,
          LPT_hstart_in=3.11911e6,
          LPT_hstart_out=2.39102e6,
          mixLP_hstart=3.11911e6,
          mixLP_V=5,
          steamIPNomFlowRate=81.3524 - 67.6,
          steamLPNomFlowRate=89.438 - 81.3524,
          valveHP_Cv=1165,
          valveHP_dpnom=1.6e5,
          valveIP_Cv=5625,
          valveIP_dpnom=5e4,
          valveLP_Cv=14733,
          valveLP_dpnom=2.964e4) 
                       annotation (extent=[-80,-50; 20,50]);
        ThermoPower.PowerPlants.SteamTurbineGroup.Examples.CondPlant_cc 
          controlledCondeser(
          redeclare package FluidMedium = FluidMedium,
          p=5398.2,
          Vtot=10,
          Vlstart=1.5,
          setPoint_ratio=0.85) annotation (extent=[-24,-140; 36,-80],
                                                                    rotation=0);
        HRSG.Components.PrescribedSpeedPump totalFeedPump(
          redeclare package WaterMedium = FluidMedium,
          rho_nom=1000,
          q_nom={0.0898,0,0.1},
          head_nom={72.74,130,0},
          nominalCondensationPressure=5398.2,
          nominalSteamFlow=89.8,
          n0=1500,
          hstart=1.43495e5,
          nominalSteamPressure=6e5) 
                                  annotation (extent=[112,-158; 80,-190],
            rotation=180);
      equation 
        connect(steamTurbines.Shaft_b, Shaft_b) annotation (points=[20,0; 200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(steamTurbines.Shaft_a, Shaft_a) annotation (points=[-80,0; -200,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ActuatorsBus, steamTurbines.ActuatorsBus) annotation (points=[200,-140;
              100,-140; 100,-35; 20,-35],         style(color=52, rgbcolor={213,
                255,170}));
        connect(controlledCondeser.SteamIn, steamTurbines.LPT_Out) annotation (
            points=[6,-80; 6,-50; 5,-50],  style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.IPT_In, From_RH_IP) annotation (points=[-40,50;
              -40,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_Out, To_RH_IP) annotation (points=[-55,50;
              -55,140; -100,140; -100,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.LPT_In, From_SH_LP) annotation (points=[-10,50;
              -10,100; 80,100; 80,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_In, From_SH_HP) annotation (points=[-70,50;
              -70,100; -160,100; -160,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(SensorsBus, steamTurbines.SensorsBus) annotation (points=[200,
              -80; 80,-80; 80,-20; 20,-20], style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(SensorsBus, controlledCondeser.SensorsBus) annotation (points=[200,-80;
              80,-80; 80,-122; 35.4,-122],          style(
            color=84,
            rgbcolor={255,170,213},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ActuatorsBus, controlledCondeser.ActuatorsBus) annotation (
            points=[200,-140; 100,-140; 100,-131.6; 35.4,-131.6], style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ActuatorsBus.nPump_feedLP, totalFeedPump.pumpSpeed_rpm) 
          annotation (points=[200,-140; 60,-140; 60,-164.4; 78.72,-164.4],
            style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.inlet, controlledCondeser.WaterOut) annotation (
            points=[80,-174; 6,-174; 6,-140], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.outlet, WaterOut) annotation (points=[112,-174;
              160,-174; 160,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end STG_3LRh_valve_cc;
      
      model STG_3LRh_bypass_cc 
        "Steam turbine group (three pressure levels, reheat) with inlet and bypass valves and controlled condenser" 
        extends Interfaces.STGroup3LRh;
        annotation (Diagram);
        ST3LRh_bypass steamTurbines(
          redeclare package FluidMedium = FluidMedium,
          steamHPNomFlowRate=67.6,
          steamHPNomPressure=1.28e7,
          steamIPNomPressure=2.68e6,
          steamLPNomPressure=6e5,
          pcond=5.3982e3,
          HPT_hstart_in=3.47e6,
          HPT_eta_iso_nom=0.833,
          IPT_eta_iso_nom=0.903,
          LPT_eta_iso_nom=0.876,
          HPT_Kt=0.0032078,
          IPT_Kt=0.018883,
          LPT_Kt=0.078004,
          HPT_hstart_out=3.10745e6,
          IPT_hstart_in=3.54348e6,
          IPT_hstart_out=3.12011e6,
          LPT_hstart_in=3.11911e6,
          LPT_hstart_out=2.39102e6,
          mixLP_hstart=3.11911e6,
          mixLP_V=5,
          steamIPNomFlowRate=81.3524 - 67.6,
          steamLPNomFlowRate=89.438 - 81.3524,
          valveHP_Cv=1165,
          valveHP_dpnom=1.6e5,
          valveIP_Cv=5625,
          valveIP_dpnom=5e4,
          valveLP_Cv=14733,
          valveLP_dpnom=2.964e4,
          bypassHP_Cv=272,
          bypassIP_Cv=1595,
          bypassLP_Cv=7540) 
                       annotation (extent=[-80,-50; 20,50]);
        ThermoPower.PowerPlants.SteamTurbineGroup.Examples.CondPlant_cc 
          controlledCondeser(
          redeclare package FluidMedium = FluidMedium,
          p=5398.2,
          Vtot=10,
          Vlstart=1.5,
          setPoint_ratio=0.85) annotation (extent=[-26,-140; 34,-80],
                                                                    rotation=0);
        HRSG.Components.PrescribedSpeedPump totalFeedPump(
          redeclare package WaterMedium = FluidMedium,
          rho_nom=1000,
          q_nom={0.0898,0,0.1},
          head_nom={72.74,130,0},
          nominalCondensationPressure=5398.2,
          nominalSteamFlow=89.8,
          n0=1500,
          hstart=1.43495e5,
          nominalSteamPressure=6e5) 
                                  annotation (extent=[104,-160; 72,-192],
            rotation=180);
      equation 
        connect(steamTurbines.Shaft_b, Shaft_b) annotation (points=[20,0; 200,0],
            style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(steamTurbines.Shaft_a, Shaft_a) annotation (points=[-80,0; -200,
              0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(ActuatorsBus, controlledCondeser.ActuatorsBus) annotation (
            points=[200,-140; 80,-140; 80,-132; 33.4,-132; 33.4,-131.6],
                                                         style(color=52,
              rgbcolor={213,255,170}));
        connect(ActuatorsBus, steamTurbines.ActuatorsBus) annotation (points=[200,-140;
              80,-140; 80,-35; 20,-35],             style(color=52, rgbcolor={
                213,255,170}));
        connect(SensorsBus, steamTurbines.SensorsBus) annotation (points=[200,-80;
              52,-80; 52,-20; 20,-20],       style(color=84, rgbcolor={255,170,
                213}));
        connect(SensorsBus, controlledCondeser.SensorsBus) annotation (points=[200,-80;
              52,-80; 52,-122; 33.4,-122],     style(color=84, rgbcolor={255,
                170,213}));
        connect(controlledCondeser.SteamIn, steamTurbines.LPT_Out) annotation (
            points=[4,-80; 4,-50; 5,-50],  style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.LPT_In, From_SH_LP) annotation (points=[-10,50;
              -10,100; 80,100; 80,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.IPT_In, From_RH_IP) annotation (points=[-40,50;
              -40,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_Out, To_RH_IP) annotation (points=[-55,50;
              -55,140; -100,140; -100,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(steamTurbines.HPT_In, From_SH_HP) annotation (points=[-70,50;
              -70,100; -160,100; -160,200], style(
            thickness=2,
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(ActuatorsBus.nPump_feedLP, totalFeedPump.pumpSpeed_rpm) 
          annotation (points=[200,-140; 52,-140; 52,-166.4; 70.72,-166.4],
            style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.inlet, controlledCondeser.WaterOut) annotation (
            points=[72,-176; 4,-176; 4,-140], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(totalFeedPump.outlet, WaterOut) annotation (points=[104,-176;
              160,-176; 160,200], style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end STG_3LRh_bypass_cc;
    end Examples;
    
    package Tests "Test cases" 
      model computation_states 
        package FluidMedium = ThermoPower.Water.StandardWater;
        parameter SI.SpecificEnthalpy h=2.39102e6 "value of specific enthalpy";
        parameter SI.Pressure p=2e5 "value of pressure";
        parameter SI.Temperature T=288 "value of temperature";
        
        SI.Temperature Tf "Temperature (p,h)";
        SI.SpecificEnthalpy hf "Specific enthalpy (p,T)";
        SI.Temperature Ts "Saturation temperature (p)";
        SI.SpecificEnthalpy hv "Dew specific enthalpy (p)";
        SI.SpecificEnthalpy hl "Bubble specific enthalpy (p)";
        SI.Density rhov "Dew density (p)";
        SI.Density rhol "Bubble density (p)";
        FluidMedium.SaturationProperties sat "Saturation properties";
        
      equation 
        sat.psat = p;
        sat.Tsat = FluidMedium.saturationTemperature(p);
        
        Ts = sat.Tsat;
        hl=FluidMedium.bubbleEnthalpy(sat);
        hv=FluidMedium.dewEnthalpy(sat);
        Tf=FluidMedium.temperature_ph(p,h);
        hf=FluidMedium.specificEnthalpy_pT(p,T);
        rhov=FluidMedium.density_ph(p,h);
        rhol=FluidMedium.bubbleDensity(sat);
      end computation_states;
      
      model TestSettingTurbineHP 
        Water.SteamTurbineStodola steamTurbineStodola(
          pstart_in=1.28e7,
          pstart_out=2.98e6,
          wstart=67.6,
          hstartin=3.47e6,
          hstartout=3.1076e6,
          wnom=67.6,
          eta_iso_nom=0.833,
          Kt=0.0032078) 
                      annotation (extent=[-40,-40; 20,20]);
        Water.SinkP sinkP(p0=29.8e5, h=3.1076e6) 
          annotation (extent=[30,60; 50,80]);
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
                      annotation (extent=[70,-20; 50,0]);
        Water.SourceP sourceP(h=3.47e6, p0=1.28e7) 
          annotation (extent=[-80,60; -60,80]);
      equation 
        connect(sinkP.flange, steamTurbineStodola.outlet) annotation (points=[30,70;
              14,70; 14,14],                   style(thickness=2));
        annotation (Diagram);
        connect(constantSpeed.flange, steamTurbineStodola.shaft_b) annotation (
            points=[50,-10; 9.2,-10],                             style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(sourceP.flange, steamTurbineStodola.inlet) annotation (points=[-60,70;
              -34,70; -34,14],               style(thickness=2));
      end TestSettingTurbineHP;
      
      model TestSettingTurbineIP 
        Water.SteamTurbineStodola steamTurbineStodola(
          pstart_in=2.68e6,
          pstart_out=6e5,
          wstart=81.10,
          hstartin=3.554e6,
          hstartout=3.1281e6,
          wnom=81.10,
          eta_iso_nom=0.903,
          Kt=0.018883)     annotation (extent=[-40,-40; 20,20]);
        Water.SinkP sinkP(p0=6e5, h=3.1281e6) 
          annotation (extent=[30,60; 50,80]);
        Water.SourceP sourceP(            h=3.554e6, p0=2.68e6) 
          annotation (extent=[-80,60; -60,80]);
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
                      annotation (extent=[70,-20; 50,0]);
      equation 
        connect(sinkP.flange, steamTurbineStodola.outlet) annotation (points=[30,70;
              14,70; 14,14],                   style(thickness=2));
        annotation (Diagram);
        connect(sourceP.flange, steamTurbineStodola.inlet) annotation (points=[-60,70;
              -34,70; -34,14],               style(thickness=2));
        connect(constantSpeed.flange, steamTurbineStodola.shaft_b) annotation (
            points=[50,-10; 9.2,-10],                             style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
      end TestSettingTurbineIP;
      
      model TestSettingTurbineLP 
        Water.SteamTurbineStodola steamTurbineStodola(
          pstart_in=6e5,
          pstart_out=5.3982e3,
          wstart=89.82,
          hstartin=3.109e6,
          hstartout=2.3854e6,
          wnom=89.82,
          eta_iso_nom=0.903,
          Kt=0.078004)     annotation (extent=[-40,-40; 20,20]);
        Water.SinkP sinkP(p0=5.3982e3, h=2.3854e6) 
          annotation (extent=[30,60; 50,80]);
        Water.SourceP sourceP(        h=3.109e6, p0=6e5) 
          annotation (extent=[-80,60; -60,80]);
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
                      annotation (extent=[70,-20; 50,0]);
      equation 
        connect(sinkP.flange, steamTurbineStodola.outlet) annotation (points=[30,70;
              14,70; 14,14],                   style(thickness=2));
        annotation (Diagram);
        connect(sourceP.flange, steamTurbineStodola.inlet) annotation (points=[-60,70;
              -34,70; -34,14],               style(thickness=2));
        connect(constantSpeed.flange, steamTurbineStodola.shaft_b) annotation (
            points=[50,-10; 9.2,-10],                             style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
      end TestSettingTurbineLP;
      
      model TestValveTurbineHP 
        Water.SteamTurbineStodola steamTurbineStodola(
          pstart_in=1.28e7,
          pstart_out=2.98e6,
          wstart=67.6,
          hstartin=3.47e6,
          hstartout=3.1076e6,
          wnom=67.6,
          eta_iso_nom=0.833,
          Kt=0.0032078) 
                      annotation (extent=[-42,-40; 18,20]);
        Water.SinkP sinkP(p0=29.8e5, h=3.1076e6) 
          annotation (extent=[30,60; 50,80]);
        Water.ValveVap valveHP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          CheckValve=true,
          wnom=67.6,
          hstart=3.47e6,
          pnom=1.296e7,
          dpnom=1.6e5,
          Cv=1165)             annotation (extent=[-24,60; -44,40],rotation=90);
        Water.SourceP sourceP(h=3.47e6, p0=1.296e7) 
          annotation (extent=[-80,60; -60,80]);
      public 
        Modelica.Blocks.Sources.Constant com_valveHP(k=0.7) 
          annotation (extent=[4,40; -16,60]);
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
                      annotation (extent=[70,-20; 50,0]);
      equation 
        connect(sinkP.flange, steamTurbineStodola.outlet) annotation (points=[30,70;
              12,70; 12,14],                   style(thickness=2));
        annotation (Diagram);
        connect(valveHP.outlet, steamTurbineStodola.inlet) annotation (points=[-34,40;
              -34,14; -36,14],               style(thickness=2));
        connect(sourceP.flange, valveHP.inlet) 
          annotation (points=[-60,70; -34,70; -34,60], style(thickness=2));
        connect(com_valveHP.y, valveHP.theta) annotation (points=[-17,50; -26,
              50],
            style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(constantSpeed.flange, steamTurbineStodola.shaft_b) annotation (
            points=[50,-10; 7.2,-10],                             style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
      end TestValveTurbineHP;
      
      model TestValveTurbineIP 
        Water.SteamTurbineStodola steamTurbineStodola(
          pstart_in=2.68e6,
          pstart_out=6e5,
          wstart=81.10,
          hstartin=3.554e6,
          hstartout=3.1281e6,
          wnom=81.10,
          eta_iso_nom=0.903,
          Kt=0.018883)     annotation (extent=[-42,-40; 18,20]);
        Water.SinkP sinkP(p0=6e5, h=3.1281e6) 
          annotation (extent=[30,60; 50,80]);
        Water.ValveVap valveIP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          CheckValve=true,
          wnom=81.1,
          hstart=3.554e6,
          dpnom=5e4,
          pnom=2.73e6,
          Cv=5625)             annotation (extent=[-24,60; -44,40],rotation=90);
        Water.SourceP sourceP(            h=3.554e6, p0=2.73e6) 
          annotation (extent=[-80,60; -60,80]);
      public 
        Modelica.Blocks.Sources.Constant com_valveHP 
          annotation (extent=[4,40; -16,60]);
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
                      annotation (extent=[70,-20; 50,0]);
      equation 
        connect(sinkP.flange, steamTurbineStodola.outlet) annotation (points=[30,70;
              12,70; 12,14],                   style(thickness=2));
        annotation (Diagram);
        connect(valveIP.outlet, steamTurbineStodola.inlet) annotation (points=[-34,40;
              -34,14; -36,14],               style(thickness=2));
        connect(sourceP.flange,valveIP. inlet) 
          annotation (points=[-60,70; -34,70; -34,60], style(thickness=2));
        connect(com_valveHP.y, valveIP.theta) 
                                           annotation (points=[-17,50; -26,50],
            style(color=74, rgbcolor={0,0,127}));
        connect(constantSpeed.flange, steamTurbineStodola.shaft_b) annotation (
            points=[50,-10; 7.2,-10],                             style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
      end TestValveTurbineIP;
      
      model TestValveTurbineLP 
        Water.SteamTurbineStodola steamTurbineStodola(
          pstart_in=6e5,
          pstart_out=5.3982e3,
          wstart=89.82,
          hstartin=3.109e6,
          hstartout=2.3854e6,
          wnom=89.82,
          eta_iso_nom=0.903,
          Kt=0.078004)     annotation (extent=[-42,-40; 18,20]);
        Water.SinkP sinkP(p0=5.3982e3, h=2.3854e6) 
          annotation (extent=[30,60; 50,80]);
        Water.ValveVap valveLP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          CheckValve=true,
          wnom=89.82,
          hstart=3.109e6,
          pnom=6.296e5,
          dpnom=2.964e4,
          Cv=14733)            annotation (extent=[-24,60; -44,40],rotation=90);
        Water.SourceP sourceP(        h=3.109e6, p0=6.296e5) 
          annotation (extent=[-80,60; -60,80]);
      public 
        Modelica.Blocks.Sources.Constant com_valveHP 
          annotation (extent=[4,40; -16,60]);
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
                      annotation (extent=[70,-20; 50,0]);
      equation 
        connect(sinkP.flange, steamTurbineStodola.outlet) annotation (points=[30,70;
              12,70; 12,14],                   style(thickness=2));
        annotation (Diagram);
        connect(valveLP.inlet, sourceP.flange) annotation (points=[-34,60; -34,
              70; -60,70], style(
            color=3,
            rgbcolor={0,0,255},
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(valveLP.outlet, steamTurbineStodola.inlet) annotation (points=[-34,40;
              -34,14; -36,14],               style(
            thickness=2,
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(com_valveHP.y, valveLP.theta) annotation (points=[-17,50; -26,
              50], style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
        connect(constantSpeed.flange, steamTurbineStodola.shaft_b) annotation (
            points=[50,-10; 7.2,-10],                             style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
      end TestValveTurbineLP;
      
      model TestTurbineHPefficiency 
        
        Components.SteamTurbineVarEta steamTurbineVarEta(
          pstart_in=1.28e7,
          pstart_out=2.98e6,
          wstart=67.6,
          hstartin=3.47e6,
          hstartout=3.1076e6,
          wnom=67.6,
          eta_iso_nom=0.833,
          Kt=0.0032078,
          n=20,
          Rm=0.936)   annotation (extent=[-58,-88; 2,-28]);
        Water.SinkP sinkP(p0=29.8e5, h=3.1076e6) 
          annotation (extent=[14,12; 34,32]);
        Water.ValveVap valveHP(
          CvData=ThermoPower.Choices.Valve.CvTypes.Cv,
          CheckValve=true,
          wnom=67.6,
          hstart=3.47e6,
          pnom=1.296e7,
          dpnom=1.6e5,
          Cv=1165)             annotation (extent=[-40,12; -60,-8],rotation=90);
        Water.SourceP sourceP(h=3.47e6, p0=1.296e7) 
          annotation (extent=[-96,12; -76,32]);
        annotation (Diagram,
          experiment(StopTime=1000),
          experimentSetupOutput(equdistant=false));
      public 
        Modelica.Blocks.Sources.Ramp com_valveHP(
          offset=1,
          height=-0.9,
          duration=50,
          startTime=1000) 
          annotation (extent=[14,74; -6,94]);
        ThermoPower.PowerPlants.SteamTurbineGroup.Components.PrescribedSpeed 
          constantSpeed 
          annotation (extent=[54,-68; 34,-48]);
      public 
        Modelica.Blocks.Sources.Ramp rif_velocity(
          height=-70,
          offset=314.16/2,
          duration=50,
          startTime=10) 
          annotation (extent=[100,36; 80,56]);
        Modelica.Blocks.Math.Add add 
          annotation (extent=[-36,54; -24,66], rotation=180);
        Modelica.Blocks.Math.Add add1 
          annotation (extent=[60,2; 72,14], rotation=180);
      public 
        Modelica.Blocks.Sources.Ramp com_valveHP1(
          offset=0,
          height=0.2,
          duration=50,
          startTime=80000) 
          annotation (extent=[14,40; -6,60]);
      public 
        Modelica.Blocks.Sources.Ramp rif_velocity1(
          height=30,
          offset=0,
          duration=50,
          startTime=80) 
          annotation (extent=[100,-40; 80,-20]);
      equation 
        connect(sinkP.flange, steamTurbineVarEta.outlet)  annotation (points=[14,22;
              -4,22; -4,-34],                  style(thickness=2));
        connect(valveHP.outlet, steamTurbineVarEta.inlet)  annotation (points=[-50,-8;
              -50,-34; -52,-34],             style(thickness=2));
        connect(sourceP.flange, valveHP.inlet) 
          annotation (points=[-76,22; -50,22; -50,12], style(thickness=2));
        connect(constantSpeed.flange, steamTurbineVarEta.shaft_b)  annotation (
            points=[34,-58; -8.8,-58],style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(com_valveHP1.y, add.u1) annotation (points=[-7,50; -14,50; -14,
              56.4; -22.8,56.4], style(color=74, rgbcolor={0,0,127}));
        connect(com_valveHP.y, add.u2) annotation (points=[-7,84; -14,84; -14,
              64; -22,64; -22.8,63.6], style(color=74, rgbcolor={0,0,127}));
        connect(add.y, valveHP.theta) annotation (points=[-36.6,60; -48,60; -48,
              28; -24,28; -24,2; -42,2], style(color=74, rgbcolor={0,0,127}));
        connect(rif_velocity.y, add1.u2) annotation (points=[79,46; 70,46; 70,
              24; 88,24; 88,11.6; 73.2,11.6], style(color=74, rgbcolor={0,0,127}));
        connect(rif_velocity1.y, add1.u1) annotation (points=[79,-30; 70,-30;
              70,-8; 88,-8; 88,4.4; 73.2,4.4], style(color=74, rgbcolor={0,0,
                127}));
        connect(add1.y, constantSpeed.w_fixed) annotation (points=[59.4,8; 50,8;
              50,-38; 66,-38; 66,-52; 54,-52],   style(color=74, rgbcolor={0,0,
                127}));
      end TestTurbineHPefficiency;
      
      model TestST3LRh_base 
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        Examples.ST3LRh_base steamTurbines(
          redeclare package FluidMedium = FluidMedium,
          steamHPNomFlowRate=67.6,
          steamHPNomPressure=1.28e7,
          steamIPNomFlowRate=81.10 - 67.5,
          steamIPNomPressure=2.68e6,
          steamLPNomPressure=6e5,
          pcond=5.3982e3,
          steamLPNomFlowRate=89.82 - 81.10,
          HPT_hstart_in=3.47e6,
          HPT_hstart_out=3.1076e6,
          IPT_hstart_in=3.554e6,
          IPT_hstart_out=3.12810e6,
          LPT_hstart_in=3.109e6,
          LPT_hstart_out=2.3854e6,
          mixLP_hstart=3.109e6,
          HPT_eta_iso_nom=0.833,
          IPT_eta_iso_nom=0.903,
          LPT_eta_iso_nom=0.876,
          HPT_Kt=0.0032078,
          IPT_Kt=0.018883,
          LPT_Kt=0.078004,
          mixLP_V=10)  annotation (extent=[-80,-40; 0,40]);
        annotation (Diagram,
          experiment(StopTime=1000),
          experimentSetupOutput(equdistant=false));
        Water.SinkP sinkLPT(
          redeclare package Medium = FluidMedium,
          p0=5.3982e3,
          h=2.3854e6)      annotation (extent=[4,-66; 16,-54],  rotation=0);
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
          annotation (extent=[66,-10; 46,10]);
        Water.SourceP sourceHPT(h=3.47e6, p0=1.28e7) 
          annotation (extent=[-78,68; -66,80], rotation=270);
        Water.SourceP sourceLPT(h=3.109e6, p0=6e5) 
          annotation (extent=[-30,54; -18,66], rotation=270);
        Components.EffectHE Rh(dh=3.544e6 - 3.108e6, dp=2.98e6 - 2.68e6) 
          annotation (extent=[-48,54; -60,66], rotation=180);
      equation 
        connect(constantSpeed.flange, steamTurbines.Shaft_b) 
                                                           annotation (points=[46,0; 0,
              0],      style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(sinkLPT.flange, steamTurbines.LPT_Out) 
          annotation (points=[4,-60; -12,-60; -12,-40],
                                                      style(thickness=2));
        connect(steamTurbines.LPT_In, sourceLPT.flange) 
          annotation (points=[-24,40; -24,54], style(thickness=2));
        connect(steamTurbines.HPT_In, sourceHPT.flange) 
          annotation (points=[-72,40; -72,68], style(thickness=2));
        connect(steamTurbines.IPT_In, Rh.out1) 
          annotation (points=[-48,40; -48,60], style(thickness=2));
        connect(steamTurbines.HPT_Out, Rh.in1) annotation (points=[-60,40; -60,
              60; -60.06,60],         style(thickness=2));
      end TestST3LRh_base;
      
      model TestST3LRh_valve 
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        Examples.ST3LRh_valve steamTurbines(
          redeclare package FluidMedium = FluidMedium,
          steamHPNomFlowRate=67.6,
          steamHPNomPressure=1.28e7,
          steamIPNomFlowRate=81.10 - 67.5,
          steamIPNomPressure=2.68e6,
          steamLPNomPressure=6e5,
          pcond=5.3982e3,
          steamLPNomFlowRate=89.82 - 81.10,
          HPT_hstart_in=3.47e6,
          HPT_hstart_out=3.1076e6,
          IPT_hstart_in=3.554e6,
          IPT_hstart_out=3.12810e6,
          LPT_hstart_in=3.109e6,
          LPT_hstart_out=2.3854e6,
          valveHP_dpnom=1.6e5,
          valveIP_dpnom=5e4,
          mixLP_hstart=3.109e6,
          HPT_eta_iso_nom=0.833,
          IPT_eta_iso_nom=0.903,
          LPT_eta_iso_nom=0.876,
          mixLP_V=20,
          valveHP_Cv=1165,
          valveIP_Cv=5625,
          valveLP_Cv=14733,
          valveLP_dpnom=2.964e4,
          HPT_Kt=0.0032078,
          IPT_Kt=0.018883,
          LPT_Kt=0.078004) 
                       annotation (extent=[-80,-40; 0,40]);
        annotation (Diagram,
          experiment(StopTime=1000),
          experimentSetupOutput(equdistant=false));
        Water.SinkP sinkLPT_p(
          redeclare package Medium = FluidMedium,
          p0=5.3982e3,
          h=2.3854e6)      annotation (extent=[-26,-64; -38,-52],
                                                                rotation=0);
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
          annotation (extent=[66,30; 46,50]);
      protected 
        Buses.Actuators actuators annotation (extent=[32,-60; 42,-20], rotation=
             180);
      public 
        Modelica.Blocks.Sources.Constant com_valveHP 
          annotation (extent=[92,-10; 72,10]);
      public 
        Modelica.Blocks.Sources.Constant com_valveLP(k=1) 
          annotation (extent=[92,-90; 72,-70]);
        Water.SourceP sourceHPT_p(h=3.47e6, p0=1.296e7) 
          annotation (extent=[-78,68; -66,80], rotation=270);
        Water.SourceP sourceLPT(h=3.109e6, p0=6.296e5) 
          annotation (extent=[-30,54; -18,66], rotation=270);
        Modelica.Blocks.Sources.Ramp com_valveIP(
          duration=10,
          offset=1,
          startTime=40,
          height=-0.5)  annotation (extent=[92,-50; 72,-30]);
        Components.EffectHE Rh(dh=3.544e6 - 3.108e6, dp=2.98e6 - 2.73e6) 
          annotation (extent=[-48,54; -60,66], rotation=180);
      equation 
        connect(actuators, steamTurbines.ActuatorsBus)   annotation (points=[37,-40;
              10,-40; 10,-28; 0,-28],             style(color=52, rgbcolor={213,
                255,170}));
        connect(constantSpeed.flange, steamTurbines.Shaft_b)   annotation (
            points=[46,40; 24,40; 24,0; 0,0], style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(sinkLPT_p.flange, steamTurbines.LPT_Out) 
          annotation (points=[-26,-58; -12,-58; -12,-40],
                                                        style(thickness=2));
        connect(steamTurbines.LPT_In, sourceLPT.flange) 
          annotation (points=[-24,40; -24,54], style(thickness=2));
        connect(steamTurbines.HPT_In, sourceHPT_p.flange) 
          annotation (points=[-72,40; -72,68], style(thickness=2));
        connect(com_valveHP.y, actuators.Opening_valveHP) annotation (points=[
              71,0; 54,0; 54,-40; 34,-40; 34,-42], style(color=74, rgbcolor={0,
                0,127}));
        connect(com_valveIP.y, actuators.Opening_valveIP) annotation (points=[
              71,-40; 36,-40], style(color=74, rgbcolor={0,0,127}));
        connect(com_valveLP.y, actuators.Opening_valveLP) annotation (points=[
              71,-80; 54,-80; 54,-40; 38,-40], style(color=74, rgbcolor={0,0,
                127}));
        connect(steamTurbines.IPT_In, Rh.out1) 
          annotation (points=[-48,40; -48,60], style(thickness=2));
        connect(steamTurbines.HPT_Out, Rh.in1) 
          annotation (points=[-60,40; -60,60; -60.06,60], style(thickness=2));
      end TestST3LRh_valve;
      
      model TestST3LRh_bypass 
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        Examples.ST3LRh_bypass steamTurbines(
          redeclare package FluidMedium = FluidMedium,
          steamHPNomFlowRate=67.6,
          steamHPNomPressure=1.28e7,
          steamIPNomFlowRate=81.10 - 67.5,
          steamIPNomPressure=2.68e6,
          steamLPNomPressure=6e5,
          pcond=5.3982e3,
          mixLP_V=50,
          steamLPNomFlowRate=89.82 - 81.10,
          HPT_hstart_in=3.47e6,
          HPT_hstart_out=3.1076e6,
          IPT_hstart_in=3.554e6,
          IPT_hstart_out=3.12810e6,
          LPT_hstart_in=3.109e6,
          LPT_hstart_out=2.3854e6,
          valveHP_dpnom=1.6e5,
          valveIP_dpnom=5e4,
          valveLP_dpnom=2.64e4,
          mixLP_hstart=3.109e6,
          HPT_eta_iso_nom=0.833,
          IPT_eta_iso_nom=0.903,
          LPT_eta_iso_nom=0.876,
          valveHP_Cv=1165,
          valveIP_Cv=5625,
          valveLP_Cv=14733,
          bypassHP_Cv=272,
          bypassIP_Cv=1595,
          bypassLP_Cv=7540,
          HPT_Kt=0.0032078,
          IPT_Kt=0.018883,
          LPT_Kt=0.078004,
          valveDrumIP_Cv=810,
          valveDrumLP_Cv=1670) 
                       annotation (extent=[-94,-40; -14,40]);
        annotation (Diagram,
          experiment(StopTime=1000),
          experimentSetupOutput(equdistant=false));
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
          annotation (extent=[20,-10; 0,10]);
      protected 
        Buses.Actuators actuators annotation (extent=[46,-80; 86,-68], rotation=
             180);
      public 
        Modelica.Blocks.Sources.Constant com_valveHP(k=1) 
          annotation (extent=[96,30; 76,50]);
      public 
        Modelica.Blocks.Sources.Constant com_valveIP(k=1) 
          annotation (extent=[96,-10; 76,10]);
      public 
        Modelica.Blocks.Sources.Constant com_valveLP(k=1) 
          annotation (extent=[96,-50; 76,-30]);
      public 
        Modelica.Blocks.Sources.Constant com_bypassHP(k=0) 
          annotation (extent=[36,30; 56,50]);
      public 
        Modelica.Blocks.Sources.Constant com_bypassIP(k=0) 
          annotation (extent=[36,-10; 56,10]);
      public 
        Modelica.Blocks.Sources.Constant com_bypassLP(k=0) 
          annotation (extent=[36,-50; 56,-30]);
        Water.SinkP sinkLPT_p(
          redeclare package Medium = FluidMedium,
          p0=5.3982e3,
          h=2.3854e6)      annotation (extent=[-34,-66; -46,-54],
                                                                rotation=0);
        Water.SourceP sourceHPT_p(h=3.47e6, p0=1.296e7) 
          annotation (extent=[-92,68; -80,80], rotation=270);
        Water.SourceP sourceLPT(h=3.109e6, p0=6.296e5) 
          annotation (extent=[-44,54; -32,66], rotation=270);
        Components.EffectHE Rh(dh=3.544e6 - 3.108e6, dp=2.98e6 - 2.73e6) 
          annotation (extent=[-62,54; -74,66], rotation=180);
      equation 
        connect(com_valveHP.y, actuators.Opening_valveHP) 
                                                      annotation (points=[75,40;
              66,40; 66,-74],        style(color=74, rgbcolor={0,0,127}));
        connect(com_valveIP.y, actuators.Opening_valveIP) 
                                                      annotation (points=[75,0; 66,
              0; 66,-74],   style(color=74, rgbcolor={0,0,127}));
        connect(com_valveLP.y, actuators.Opening_valveLP) 
                                                      annotation (points=[75,-40;
              66,-40; 66,-74],              style(color=74, rgbcolor={0,0,127}));
        connect(actuators,steamTurbines. ActuatorsBus) annotation (points=[66,-74;
              18,-74; 18,-28; -14,-28],   style(color=52, rgbcolor={213,255,170}));
        connect(steamTurbines.Shaft_b, constantSpeed.flange) annotation (points=[-14,0; 0,
              0],          style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(com_bypassHP.y, actuators.Opening_byPassHP) 
                                                        annotation (points=[57,
              40; 66,40; 66,-74], style(color=74, rgbcolor={0,0,127}));
        connect(com_bypassIP.y, actuators.Opening_byPassIP) 
                                                        annotation (points=[57,
              0; 66,0; 66,-74], style(color=74, rgbcolor={0,0,127}));
        connect(com_bypassLP.y, actuators.Opening_byPassLP) 
                                                        annotation (points=[57,
              -40; 66,-40; 66,-74], style(color=74, rgbcolor={0,0,127}));
        connect(sinkLPT_p.flange, steamTurbines.LPT_Out) 
          annotation (points=[-34,-60; -26,-60; -26,-40], style(thickness=2));
        connect(steamTurbines.LPT_In,sourceLPT. flange) 
          annotation (points=[-38,40; -38,54], style(thickness=2));
        connect(steamTurbines.HPT_In,sourceHPT_p. flange) 
          annotation (points=[-86,40; -86,68], style(thickness=2));
        connect(steamTurbines.IPT_In, Rh.out1) 
          annotation (points=[-62,40; -62,60], style(thickness=2));
        connect(steamTurbines.HPT_Out, Rh.in1) annotation (points=[-74,40; -74,
              60; -74.06,60],         style(thickness=2));
      end TestST3LRh_bypass;
      
      model TestCondenserControl 
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        annotation (Diagram,
          experiment,
          experimentSetupOutput(equdistant=false));
        Components.StateReader_water stateTapping(
                       redeclare package Medium = FluidMedium) 
          annotation (extent=[-30,-2; -10,18],  rotation=0);
        Control.PID pID(
          PVmax=1,
          PVmin=0.1,
          CSmin=-0.6,
          CSmax=0.6,
          Ti=2000,
          Kp=-3)      annotation (extent=[-20,-80; 0,-60], rotation=0);
        Water.SourceW sourceSteam(
          redeclare package Medium = FluidMedium,
          w0=89.8,
          p0=5389.2,
          h=2.3854e6) annotation (extent=[-76,50; -56,70]);
        Components.StateReader_water stateCondIn(
                                           redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[-10,50; 10,70],  rotation=0);
        Components.StateReader_water stateCondOut(
                                           redeclare package Medium = 
              FluidMedium) 
          annotation (extent=[34,-20; 54,0],   rotation=0);
        Modelica.Blocks.Sources.Ramp const(
          offset=0.85,
          duration=500,
          startTime=8000,
          height=0.1) 
                    annotation (extent=[-60,-66; -48,-54]);
        ThermoPower.PowerPlants.SteamTurbineGroup.Components.CondenserPreP_tap 
          condenserIdeal_tap(                            redeclare package 
            Medium = FluidMedium, p=5398.2) annotation (extent=[0,0; 40,40]);
        Water.SinkW sinkWater(
          redeclare package Medium = FluidMedium,
          p0=5398.2,
          h=1.43495e5,
          w0=89.8) annotation (extent=[70,-20; 90,0]);
        Water.SourceW sourceTap(
          redeclare package Medium = FluidMedium,
          p0=5389.2,
          h=1.43495e5) 
                      annotation (extent=[-60,-2; -40,18]);
      equation 
        connect(stateCondIn.inlet, sourceSteam.flange) 
          annotation (points=[-6,60; -56,60],  style(thickness=2));
        connect(const.y, pID.SP) annotation (points=[-47.4,-60; -34,-60; -34,
              -66; -20,-66], style(color=74, rgbcolor={0,0,127}));
        connect(stateCondOut.inlet, condenserIdeal_tap.waterOut) 
          annotation (points=[38,-10; 20,-10; 20,0], style(thickness=2));
        connect(stateCondIn.outlet, condenserIdeal_tap.steamIn) 
          annotation (points=[6,60; 20,60; 20,40], style(thickness=2));
        connect(sinkWater.flange, stateCondOut.outlet) 
          annotation (points=[70,-10; 50,-10], style(thickness=2));
        connect(stateTapping.outlet, condenserIdeal_tap.tapWater) 
          annotation (points=[-14,8; -5,8; -5,4; 4,4],
                                           style(thickness=2));
        connect(sourceTap.flange, stateTapping.inlet) 
          annotation (points=[-40,8; -26,8], style(thickness=2));
        connect(pID.CS, sourceTap.in_w0) annotation (points=[0.6,-70; 20,-70;
              20,-20; -70,-20; -70,20; -54,20; -54,14], style(color=74,
              rgbcolor={0,0,127}));
        connect(pID.PV, condenserIdeal_tap.ratio_Vv_Vtot) annotation (points=[-20,-74;
              -80,-74; -80,24; -20,24; -20,16; 0,16],          style(color=74,
              rgbcolor={0,0,127}));
      end TestCondenserControl;
      
      model TestSTG_cc "Test of STG with condenser control" 
        package FluidMedium = ThermoPower.Water.StandardWater;
        
        annotation (Diagram, experiment(StopTime=20000, NumberOfIntervals=10000),
          experimentSetupOutput(equdistant=false));
        
        Modelica.Mechanics.Rotational.ConstantSpeed constantSpeed(w_fixed=
              314.16/2) 
          annotation (extent=[30,-26; 10,-6]);
        Water.SourceP sourceHPT(h=3.47e6, p0=1.28e7,
          redeclare package Medium = FluidMedium) 
          annotation (extent=[-88,72; -76,84], rotation=270);
        Water.SourceP sourceLPT(h=3.109e6, p0=6e5,
          redeclare package Medium = FluidMedium) 
          annotation (extent=[-40,58; -28,70], rotation=270);
        Components.EffectHE Rh(dh=3.544e6 - 3.108e6, dp=2.98e6 - 2.68e6) 
          annotation (extent=[-70,70; -58,58], rotation=0);
        Examples.STG_3LRh_bypass_cc sTG_3LRh(
                                   redeclare package FluidMedium = FluidMedium,
            steamTurbines(SSInit=true)) 
                                   annotation (extent=[-90,-56; -10,24]);
        Water.SinkP sinkWater(
          redeclare package Medium = FluidMedium,
          h=1.43495e5,
          p0=6e5)  annotation (extent=[-12,68; -24,80],
                                                      rotation=90);
      public 
        Modelica.Blocks.Sources.Constant com_valveHP(k=1) 
          annotation (extent=[100,30; 80,50]);
      public 
        Modelica.Blocks.Sources.Constant com_valveIP(k=1) 
          annotation (extent=[100,-10; 80,10]);
      public 
        Modelica.Blocks.Sources.Constant com_valveLP(k=1) 
          annotation (extent=[100,-50; 80,-30]);
      public 
        Modelica.Blocks.Sources.Constant com_bypassHP(k=0) 
          annotation (extent=[40,30; 60,50]);
      public 
        Modelica.Blocks.Sources.Constant com_bypassIP(k=0) 
          annotation (extent=[40,-10; 60,10]);
      public 
        Modelica.Blocks.Sources.Constant com_bypassLP(k=0) 
          annotation (extent=[40,-50; 60,-30]);
      protected 
        Buses.Actuators actuators annotation (extent=[50,-80; 90,-68], rotation=
             180);
      public 
        Modelica.Blocks.Sources.Constant n_pump(k=1425) 
          annotation (extent=[40,66; 60,86]);
      equation 
        connect(constantSpeed.flange, sTG_3LRh.Shaft_b) annotation (points=[10,-16;
              -10,-16],     style(
            color=0,
            rgbcolor={0,0,0},
            thickness=2));
        connect(com_valveHP.y,actuators. Opening_valveHP) 
                                                      annotation (points=[79,40;
              70,40; 70,-74],        style(color=74, rgbcolor={0,0,127}));
        connect(com_valveIP.y,actuators. Opening_valveIP) 
                                                      annotation (points=[79,0; 70,
              0; 70,-74],   style(color=74, rgbcolor={0,0,127}));
        connect(com_valveLP.y,actuators. Opening_valveLP) 
                                                      annotation (points=[79,-40;
              70,-40; 70,-74],              style(color=74, rgbcolor={0,0,127}));
        connect(com_bypassHP.y,actuators. Opening_byPassHP) 
                                                        annotation (points=[61,40;
              70,40; 70,-74],     style(color=74, rgbcolor={0,0,127}));
        connect(com_bypassIP.y,actuators. Opening_byPassIP) 
                                                        annotation (points=[61,0; 70,
              0; 70,-74],       style(color=74, rgbcolor={0,0,127}));
        connect(com_bypassLP.y,actuators. Opening_byPassLP) 
                                                        annotation (points=[61,-40;
              70,-40; 70,-74],      style(color=74, rgbcolor={0,0,127}));
        connect(actuators, sTG_3LRh.ActuatorsBus) annotation (points=[70,-74; 6,
              -74; 6,-44; -10,-44], style(
            color=52,
            rgbcolor={213,255,170},
            fillColor=7,
            rgbfillColor={255,255,255},
            fillPattern=1));
        connect(sTG_3LRh.WaterOut, sinkWater.flange) 
          annotation (points=[-18,24; -18,68], style(thickness=2));
        connect(sTG_3LRh.From_SH_LP, sourceLPT.flange) 
          annotation (points=[-34,24; -34,58], style(thickness=2));
        connect(sTG_3LRh.From_RH_IP, Rh.out1) 
          annotation (points=[-58,24; -58,64],         style(thickness=2));
        connect(sTG_3LRh.To_RH_IP, Rh.in1) annotation (points=[-70,24; -70,64;
              -70.06,64],         style(thickness=2));
        connect(sTG_3LRh.From_SH_HP, sourceHPT.flange) 
          annotation (points=[-82,24; -82,72], style(thickness=2));
        connect(n_pump.y, actuators.nPump_feedLP) annotation (points=[61,76; 70,
              76; 70,-74], style(
            color=74,
            rgbcolor={0,0,127},
            fillColor=30,
            rgbfillColor={230,230,230},
            fillPattern=1));
      end TestSTG_cc;
      
    end Tests;
    
    package Functions "Function definitions" 
      
      function curveEfficiency "curve for efficiency calculation" 
        input Real eta_nom "Nominal efficiency (maximum)";
        input Real k "Ratio uf/wiso";
        input Real x "Degree of reaction";
        output Real eta "Value of efficient correspondent";
      protected 
        Real a;
      algorithm 
        a:=sqrt(1 - x);
        eta:=2*k*((a - k) + sqrt((a - k)^2 + 1 - (a^2)))*eta_nom;
      end curveEfficiency;
      
    end Functions;
    annotation (Documentation(revisions="<html>
<ul>
<li><i>15 Apr 2008</i>
    by <a>Luca Savoldelli</a>:<br>
       First release.</li>
</ul>
</html>"));
  end SteamTurbineGroup;
  
  package ElectricGeneratorGroup 
    "Models and tests of the electrical group (generator and network) and its main components" 
    package Interfaces "Interface definitions" 
      partial model SingleShaft 
        "Base Class for alternator group, configuration single-shaft (one generator)" 
        parameter SI.Frequency fn=50 "Nominal frequency of the grid";
        parameter SI.Power Pn "Nominal power installed on the grid";
        parameter Real eta=1 "Conversion efficiency of the electric generator";
        parameter SI.MomentOfInertia J_shaft=0 "Total inertia of the system";
        parameter Real d_shaft=0 "Damping constant of the shaft";
        parameter SI.AngularVelocity omega_nom=2*Modelica.Constants.pi*fn/2 
          "Nominal angular velocity of the shaft";
        
        annotation (Diagram, Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-202,14; -122,-14],
                                               style(
                color=10,
                rgbcolor={95,95,95},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[140,160; 140,-160], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[120,160; 120,-160], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[100,160; 100,-160], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,20; 100,20], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,0; 120,0], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,-20; 140,-20], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Ellipse(extent=[96,24; 104,16], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[116,4; 124,-4], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[136,-16; 144,-24], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-34,20; 20,20; 44,34], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-30,0; 20,0; 44,14], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-44,-20; 20,-20; 44,-6], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[-140,60; -20,-60], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Text(
              extent=[-120,40; -40,-40],
              style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="G")),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        Modelica.Mechanics.Rotational.Interfaces.Flange_a shaft 
          annotation (extent=[-220,-20; -180,20]);
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end SingleShaft;
      
      partial model DoubleShaft 
        "Base Class for alternator group, configuration double-shaft (two generator)" 
        
        //grid
        parameter SI.Frequency fn=50 "Nominal frequency of the grid";
        parameter SI.Power Pn "Nominal power installed on the grid";
        
        //generators
        parameter Real eta_A=1 
          "Conversion efficiency of the electric generator"                          annotation (Dialog(group = "Generator-Shaft A"));
        parameter Real eta_B=1 
          "Conversion efficiency of the electric generator"                          annotation (Dialog(group = "Generator-Shaft B"));
        
        //other parameter
        parameter SI.MomentOfInertia J_shaft_A=0 
          "Total inertia of the steam turbogenerator"                             annotation (Dialog(group = "Generator-Shaft A"));
        parameter Real d_shaft_A=0 "Damping constant of the shaft" annotation (Dialog(group = "Generator-Shaft A"));
        parameter SI.AngularVelocity omega_nom_A=2*Modelica.Constants.pi*fn/2 
          "Nominal angular velocity of the shaft"                                  annotation (Dialog(group = "Generator-Shaft A"));
        parameter SI.MomentOfInertia J_shaft_B=0 
          "Total inertia of the steam turbogenerator"                             annotation (Dialog(group = "Generator-Shaft B"));
        parameter Real d_shaft_B=0 "Damping constant of the shaft" annotation (Dialog(group = "Generator-Shaft B"));
        parameter SI.AngularVelocity omega_nom_B=2*Modelica.Constants.pi*fn/2 
          "Nominal angular velocity of the shaft"                                  annotation (Dialog(group = "Generator-Shaft B"));
        
        Modelica.Mechanics.Rotational.Interfaces.Flange_a shaft_B 
          annotation (extent=[-220,-120; -180,-80]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a shaft_A 
          annotation (extent=[-222,80; -182,120]);
        annotation (Diagram, Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-204,114; -124,86],
                                               style(
                color=10,
                rgbcolor={95,95,95},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[140,160; 140,-160], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[120,160; 120,-160], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[100,160; 100,-160], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[48,120; 98,120],style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[48,100; 118,100],
                                       style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[48,80; 138,80],   style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Ellipse(extent=[96,124; 104,116],
                                            style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[116,104; 124,96],
                                            style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[136,84; 144,76],   style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-36,120; 18,120; 42,134],
                                                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-32,100; 18,100; 42,114],
                                              style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-46,80; 18,80; 42,94],   style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[-142,160; -22,40], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Text(
              extent=[-122,140; -42,60],
              style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="G"),
            Rectangle(extent=[-202,-86; -122,-114],
                                               style(
                color=10,
                rgbcolor={95,95,95},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[50,-80; 100,-80],
                                         style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,-100; 120,-100],
                                       style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,-120; 140,-120],
                                           style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[-34,-80; 20,-80; 44,-66],
                                                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-30,-100; 20,-100; 44,-86],
                                              style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-44,-120; 20,-120; 44,-106],
                                                  style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[-140,-40; -20,-160],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Text(
              extent=[-120,-60; -40,-140],
              style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="G"),
            Ellipse(extent=[96,-76; 104,-84],
                                            style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[116,-96; 124,-104],
                                            style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[136,-116; 144,-124],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1))),
          Coordsys(extent=[-200,-200; 200,200], scale=0.1));
        Buses.Sensors SensorsBus annotation (extent=[180,-100; 220,-60]);
        Buses.Actuators ActuatorsBus annotation (extent=[220,-160; 180,-120]);
      end DoubleShaft;
      
      partial model TripleShaft 
        "Base Class for alternator group, configuration triple-shaft (three generator)" 
        
        //grid
        parameter SI.Frequency fn=50 "Nominal frequency of the grid";
        parameter SI.Power Pn "Nominal power installed on the grid";
        
        //generators
        parameter Real eta_A=1 
          "Conversion efficiency of the electric generator"                          annotation (Dialog(group = "Generator-Shaft A"));
        parameter Real eta_B=1 
          "Conversion efficiency of the electric generator"                          annotation (Dialog(group = "Generator-Shaft B"));
        parameter Real eta_C=1 
          "Conversion efficiency of the electric generator"                          annotation (Dialog(group = "Generator-Shaft C"));
        
        //other parameter
        parameter SI.MomentOfInertia J_shaft_A=0 
          "Total inertia of the steam turbogenerator"                             annotation (Dialog(group = "Generator-Shaft A"));
        parameter Real d_shaft_A=0 "Damping constant of the shaft" annotation (Dialog(group = "Generator-Shaft A"));
        parameter SI.AngularVelocity omega_nom_A=2*Modelica.Constants.pi*fn/2 
          "Nominal angular velocity of the shaft"                                  annotation (Dialog(group = "Generator-Shaft A"));
        parameter SI.MomentOfInertia J_shaft_B=0 
          "Total inertia of the steam turbogenerator"                             annotation (Dialog(group = "Generator-Shaft B"));
        parameter Real d_shaft_B=0 "Damping constant of the shaft" annotation (Dialog(group = "Generator-Shaft B"));
        parameter SI.AngularVelocity omega_nom_B=2*Modelica.Constants.pi*fn/2 
          "Nominal angular velocity of the shaft"                                  annotation (Dialog(group = "Generator-Shaft B"));
        parameter SI.MomentOfInertia J_shaft_C=0 
          "Total inertia of the steam turbogenerator"                             annotation (Dialog(group = "Generator-Shaft C"));
        parameter Real d_shaft_C=0 "Damping constant of the shaft" annotation (Dialog(group = "Generator-Shaft C"));
        parameter SI.AngularVelocity omega_nom_C=2*Modelica.Constants.pi*fn/2 
          "Nominal angular velocity of the shaft"                                  annotation (Dialog(group = "Generator-Shaft C"));
        
        Modelica.Mechanics.Rotational.Interfaces.Flange_a shaft_B 
          annotation (extent=[-220,-20; -180,20]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a shaft_A 
          annotation (extent=[-220,100; -180,140]);
        Modelica.Mechanics.Rotational.Interfaces.Flange_a shaft_C 
          annotation (extent=[-220,-140; -180,-100]);
        annotation (Diagram, Icon(
            Rectangle(extent=[-200,200; 200,-200], style(
                color=76,
                rgbcolor={170,170,255},
                fillColor=30,
                rgbfillColor={230,230,230})),
            Rectangle(extent=[-200,134; -120,106],
                                               style(
                color=10,
                rgbcolor={95,95,95},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[140,180; 140,-180], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[120,180; 120,-180], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[100,180; 100,-180], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,140; 100,140],
                                         style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,120; 120,120],
                                       style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,100; 140,100], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Ellipse(extent=[96,144; 104,136],
                                            style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[116,124; 124,116],
                                            style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[136,104; 144,96],  style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-34,140; 20,140; 44,154],
                                                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-30,120; 20,120; 44,134],
                                              style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-44,100; 20,100; 44,114],style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[-136,176; -24,64], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Text(
              extent=[-120,160; -40,80],
              style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="G"),
            Rectangle(extent=[-200,-106; -120,-134],
                                               style(
                color=10,
                rgbcolor={95,95,95},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[50,-100; 100,-100],
                                         style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,-120; 120,-120],
                                       style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,-140; 140,-140],
                                           style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[-34,-100; 20,-100; 44,-86],
                                                style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-30,-120; 20,-120; 44,-106],
                                              style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Line(points=[-44,-140; 20,-140; 44,-126],
                                                  style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[-136,-64; -24,-176],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1)),
            Text(
              extent=[-120,-80; -40,-160],
              style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=7,
                rgbfillColor={255,255,255},
                fillPattern=1),
              string="G"),
            Ellipse(extent=[96,-96; 104,-104],
                                            style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[116,-116; 124,-124],
                                            style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Ellipse(extent=[136,-136; 144,-144],
                                               style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillColor=0,
                rgbfillColor={0,0,0},
                fillPattern=1)),
            Rectangle(extent=[-200,14; -120,-14],
                                               style(
                color=10,
                rgbcolor={95,95,95},
                thickness=2,
                gradient=2,
                fillColor=10,
                rgbfillColor={135,135,135})),
            Line(points=[50,20; 100,20], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,0; 120,0], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2,
                fillPattern=1)),
            Line(points=[50,-20; 140,-20], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=2