within TRANSFORM.Fluid.ClosureRelations.PressureLoss.Functions.TubesAndConduits.SinglePhase.LaminarAndQuadraticTurbulent_MSL;
function dp_DP_staticHead
  "Return pressure loss dp as function of mass flow rate m_flow, i.e., dp = f(m_flow), due to wall friction and static head"

  //input records
  input dp_IN_con IN_con "Input record for function dp_overall_DP"
    annotation (Dialog(group="Constant inputs"));
  input dp_IN_var IN_var "Input record for function dp_overall_DP"
    annotation (Dialog(group="Variable inputs"));
  input SI.MassFlowRate m_flow "Mass flow rate"
    annotation (Dialog(group="Input"));
  input SI.MassFlowRate m_flow_small=0.01
    "Regularization of zero flow if |m_flow| < m_flow_small (dummy if use_m_flow_small = false)";
  input Real g_times_height_ab
    "Gravity times (Height(port_b) - Height(port_a))";

  //Outputs
  output SI.Pressure DP "Output for function dp_overall_DP";

protected
  SI.Length diameter = 0.5*(IN_con.diameter_a+IN_con.diameter_b);
  SI.Area crossArea = 0.5*(IN_con.crossArea_a+IN_con.crossArea_b);
  SI.Height roughness = 0.5*(IN_con.roughness_a+IN_con.roughness_b);

  Real Delta = roughness/diameter "Relative roughness";
  SI.ReynoldsNumber Re1 = min(745*exp(if Delta <= 0.0065 then 1 else 0.0065/Delta), IN_con.Re_turbulent)
    "Boundary between laminar regime and transition";
  SI.ReynoldsNumber Re2 = IN_con.Re_turbulent
    "Boundary between transition and turbulent regime";

  SI.MassFlowRate m_flow_a
    "Upper end of regularization domain of the dp(m_flow) relation";
  SI.MassFlowRate m_flow_b
    "Lower end of regularization domain of the dp(m_flow) relation";

  SI.Pressure dp_a "Value at upper end of regularization domain";
  SI.Pressure dp_b "Value at lower end of regularization domain";

  SI.Pressure dp_grav_a = g_times_height_ab*IN_var.rho_a
    "Static head if mass flows in design direction (a to b)";
  SI.Pressure dp_grav_b = g_times_height_ab*IN_var.rho_b
    "Static head if mass flows against design direction (b to a)";

  Real ddp_dm_flow_a
    "Derivative of pressure drop with mass flow rate at m_flow_a";
  Real ddp_dm_flow_b
    "Derivative of pressure drop with mass flow rate at m_flow_b";

  // Properly define zero mass flow conditions
  SI.MassFlowRate m_flow_zero = 0;
  SI.Pressure dp_zero = (dp_grav_a + dp_grav_b)/2;
  Real ddp_dm_flow_zero;

algorithm
  assert(roughness > 1e-10,
    "roughness > 0 required for quadratic turbulent wall friction characteristic");

  m_flow_a := if dp_grav_a<dp_grav_b then
    Internal.m_flow_of_dp_fric(IN_con, IN_var, dp_grav_b - dp_grav_a, Re1, Re2, Delta)+m_flow_small else
    m_flow_small;
  m_flow_b := if dp_grav_a<dp_grav_b then
    Internal.m_flow_of_dp_fric(IN_con, IN_var, dp_grav_a - dp_grav_b, Re1, Re2, Delta)-m_flow_small else
    -m_flow_small;

  if m_flow>=m_flow_a then
    // Positive flow outside regularization
    DP := Internal.dp_fric_of_m_flow(IN_con, IN_var, m_flow, Re1, Re2, Delta) + dp_grav_a;
  elseif m_flow<=m_flow_b then
    // Negative flow outside regularization
    DP := Internal.dp_fric_of_m_flow(IN_con, IN_var, m_flow, Re1, Re2, Delta) + dp_grav_b;
  else
    // Regularization parameters
    (dp_a, ddp_dm_flow_a) := Internal.dp_fric_of_m_flow(IN_con, IN_var, m_flow_a, Re1, Re2, Delta);
    dp_a := dp_a + dp_grav_a "Adding dp_grav to dp_fric to get dp";
    (dp_b, ddp_dm_flow_b) := Internal.dp_fric_of_m_flow(IN_con, IN_var, m_flow_b, Re1, Re2, Delta);
    dp_b := dp_b + dp_grav_b "Adding dp_grav to dp_fric to get dp";
    // Include a properly defined zero mass flow point
    // Obtain a suitable slope from the linear section slope c (value of dp is overwritten later)
    (DP,ddp_dm_flow_zero) := Modelica.Fluid.Utilities.regFun3(
        m_flow_zero,
        m_flow_b,
        m_flow_a,
        dp_b,
        dp_a,
        ddp_dm_flow_b,
        ddp_dm_flow_a);
    // Do regularization
    if m_flow>m_flow_zero then
      DP := Modelica.Fluid.Utilities.regFun3(
          m_flow,
          m_flow_zero,
          m_flow_a,
          dp_zero,
          dp_a,
          ddp_dm_flow_zero,
          ddp_dm_flow_a);
    else
      DP := Modelica.Fluid.Utilities.regFun3(
          m_flow,
          m_flow_b,
          m_flow_zero,
          dp_b,
          dp_zero,
          ddp_dm_flow_b,
          ddp_dm_flow_zero);
    end if;
  end if;

  annotation (smoothOrder=1, Documentation(info="<html>

</html>"));
end dp_DP_staticHead;
